/*
   This file is part of the XXCalc Library - version 3.2
   Copyright (C)  2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010,
   2011, 2012, 2013    Ivano Primi ( ivprimi@libero.it )    

   The XXCalc Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The XXCalc library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include<stdlib.h>
#include<limits.h> /* for ULONG_MAX */
#include<math.h>
#include<errno.h>
#include"ioctype.h"
#include"compl.h"
#include"mathtok.h" /* for MAX_VNSIZE and MAX_NDIM */
#include"utils.h"

extern int errno;		/* Used by xx_str2r() */

#ifdef USE_LONG_DOUBLE
extern long double strtold (const char *nptr, char **endptr);

#if defined (__GLIBC__) && __GLIBC__ <= 2 && __GLIBC_MINOR__ <= 3

#include<string.h> /* for strlen() */
#include<float.h>

/* The following code looks terrible but it is a workaround for */
/* a bug found in the implementation of the function strtold()  */
/* by glibc <= 2.3.x                                            */

static long double strtoldbl (const char *nptr, char **endptr)
{
  char *t, *it1, *it2;
  unsigned long i;
  long double retval;

  t = (char*) malloc ( (strlen(nptr) + 1) * sizeof (char) );
  if (!t)
    return HUGE_VAL;
  else
    {
      /* Skip the leading spaces */
      for (it1 = (char*)nptr, it2 = t; io_isspace (*it1) != 0; *it2 = *it1, it1++, it2++);
      /* Skip the possible sign */
      if (*it1 == '+' || *it1 == '-')
	{
	  *it2 = *it1;
	  it1++, it2++;
	}
    }
  if ( !io_isdigit (*it1) )
    {
      retval = strtold(nptr, endptr);
      free ((void*)t);
    }
  else /* *IT1 is a digit. We'll try to read a decimal number */
    {
      for (; io_isdigit (*it1) != 0; *it2 = *it1, it1++, it2++);
      if ( *it1 == '.' )
	{
	  *it2 = '.';
	  for (++it1, ++it2, i=0; io_isdigit(*it1) != 0 && i <= LDBL_DIG; *it2 = *it1, it1++, it2++, i++);
	  for (i = 0; io_isdigit (*it1) != 0; i++, it1++);
	  for (; *it1 != '\0'; *it2 = *it1, it1++, it2++);
	  *it2 = '\0';
	  retval = strtold(t, endptr);
	  *endptr = (char*)nptr + (*endptr - t) + i;
	  free ((void*)t);
	}
      else
	{
	  /* T should point to a copy of the string pointed to by NPTR. */
	  /* Thus, we may call strtold() directly on NPTR.              */
	  retval = strtold(nptr, endptr);
	  free ((void*)t);
	}
    }
  return retval;
}

#else  /* __GLIBC__ */

#define strtoldbl strtold

#endif /* __GLIBC__ */

#endif /* USE_LONG_DOUBLE */

#ifdef DMALLOC
#include <dmalloc.h>
#endif

/* The following function returns !0 if C is equal to ' ', '\t' or '\n', */
/* otherwise returns 0.                                                  */
int
xx_isspace (char c)
{
  return (c == ' ') || (c == '\t') || (c == '\n');
}

/* The following function returns !0 if C can appear in an operator-token, */
/* otherwise 0 (see mathtok.h).                                            */
int
xx_maybeinop (char c)
{
  int retval;

  retval = (c == '+') || (c == '-') || (c == '*') || (c == '/');
  if ((retval))
    return retval;
  else
    retval = (c == '%') || (c == '$') || (c == '\\') 
      || (c == '^') || (c == '|') || (c == '&');
  if ((retval))
    return retval;
  else
    retval = (c == '<') || (c == '>') || (c == '!') || (c == '=');
  if ((retval))
    return retval;
  else
    return 0;
}

/*
  The following function check if C may appear in
  a valid identifier (variable or function name).
*/
int
xx_maybeinid (char c)
{
  return (io_isalpha (c) || io_isdigit (c) || c == '_');
}

/*
  The following function returns !0 if C can appear in a valid expression,
  otherwise 0.
*/
int
xx_isacceptable (char c)
{
  int retval;

  retval = io_isalnum (c) || xx_maybeinop (c) || xx_isspace (c);
  if ((retval))
    return retval;
  else
    retval = (c == '.') || (c == '{') || (c == '}') || (c == ',');
  if ((retval))
    return retval;
  else
    retval = (c == '_') || (c == '(') || (c == ')') || (c == '[') || (c == ']');
  return retval;
}

/* The function xx_str2id() tries to convert the initial portion of S in a  */
/* valid identifier. If it is able to do it (i.e. S starts with an alphabe- */
/* tic character) returns !0 and *ENDP points to the first  character of S  */
/* that may not appear in a valid identifier.                               */
/* Otherwise, xx_str2id() returns 0 and *ENDP==S.                           */
int
xx_str2id (const char *s, char **endp)
{
  char *t;

  if (!io_isalpha (*s))
    {
      *endp = (char *) s;
      return 0;
    }
  else
    for (t = (char *) s; xx_maybeinid (*t) != 0; t++);
  *endp = t;
  return (t != s);
}

/* The function xx_str2op() tries to convert the initial portion of S in  a   */
/* valid operator.If it is able to do it (i.e. S starts with a xx_maybeinop() */
/* character) returns !0 and *ENDP points to the first  character of S that   */
/* may not appear in a valid operator name.                                   */
/* Otherwise, xx_str2op() returns 0 and *ENDP==S.                             */
int
xx_str2op (const char *s, char **endp)
{
  char *t;

  for (t = (char *) s; xx_maybeinop (*t) != 0; t++)
    ;
  *endp = t;
  return (t != s);
}

/* xx_str2r() is a simple wrapper to strtod().It tries to convert the */
/* initial portion of the string pointed to by 'S' in a valid real    */
/* number and for this purpose uses the strtod() function. xx_str2r() */
/* can return:                                                        */
/* XX_AGNUM - to mean that the conversion has been done successfully, */
/* XX_UFLOW - to mean that the converted value would cause underflow, */
/* XX_OFLOW - to mean that the converted value would cause overflow,  */
/* XX_NAN   - to mean that the conversion was not possible.           */
/* In any case, *ENDP points to the first character of S that stopped */
/* the conversion.                                                    */
/* If the return value is XX_OFLOW, *r is set to  +/-HUGE_VAL, if it  */
/* is XX_UFLOW or XX_NAN, then *r is set to 0.                        */
int
xx_str2r (const char *s, char **endp, r_eal * r)
{
  errno = 0;
#ifdef USE_LONG_DOUBLE
  *r = strtoldbl (s, endp);
#else
  *r = strtod (s, endp);
#endif
  if (errno == ERANGE)
    {
      if (*r == 0)
	return XX_UFLOW;
      else			/* *r== +/-HUGEVAL */
	return XX_OFLOW;
    }
  else
    {
      if (*r == 0 && *endp == s)
	return XX_NAN;
      else
	return XX_AGNUM;
    }
}


/* xx_str2c() tries to convert the initial portion of the string pointed to */
/* by 'S' in a valid complex number and for this purpose uses the strtod()  */
/* function.                                                                */
/* xx_str2c() can return:                                                   */
/* XX_AGNUM - to mean that the conversion has been done successfully,       */
/* XX_UFLOW - to mean that the converted value would cause underflow,       */
/* XX_OFLOW - to mean that the converted value would cause overflow,        */
/* XX_NAN   - to mean that the conversion was not possible.                 */
/* In any case, *ENDP points to the first character of S that stopped       */
/* the conversion.                                                          */
/* When the return value is XX_NAN, *r is set to 0=={0,0}.                  */

int
xx_str2c (const char *s, char **endp, c_omplex * z)
{
  char *t;
  int errno1, errno2;

  for (t = (char *) s; io_isspace (*t) != 0; t++)
    ;				/* Skip initial spaces */
  /* *** */
  if (*t != '{')
    {
      *endp = t;
      *z = c_convert (0);
      return XX_NAN;
    }
  else
    {
      errno = 0;
#ifdef USE_LONG_DOUBLE
      z->re = strtoldbl (t + 1, endp);
#else
      z->re = strtod (t + 1, endp);
#endif
      errno1 = errno;
    }
  if (z->re == 0 && *endp == t + 1)
    {
      *z = c_convert (0);
      return XX_NAN;
    }
  else
    for (t = *endp; io_isspace (*t) != 0; t++)
      ;				/* Skip central spaces */
  /* *** */
  if (*t != ',')
    {
      *endp = t;
      *z = c_convert (0);
      return XX_NAN;
    }
  else
    {
      errno = 0;
#ifdef USE_LONG_DOUBLE
      z->im = strtoldbl (t + 1, endp);
#else
      z->im = strtod (t + 1, endp);
#endif
      errno2 = errno;
    }
  if (z->im == 0 && *endp == t + 1)
    {
      *z = c_convert (0);
      return XX_NAN;
    }
  else
    for (t = *endp; io_isspace (*t) != 0; t++)
      ;				/* Skip final spaces */
  /* *** */
  if (*t != '}')
    {
      *endp = t;
      *z = c_convert (0);
      return XX_NAN;
    }
  else
    {
      *endp = t + 1;
      if ((errno1 == ERANGE) && (z->re != 0))
	return XX_OFLOW;
      else if ((errno2 == ERANGE) && (z->im != 0))
	return XX_OFLOW;
      else if ((errno1 == ERANGE) || (errno2 == ERANGE))
	return XX_UFLOW;
      else
	return XX_AGNUM;
    }
}

/* 
   xx_c2ui() tries to convert Z into an unsigned integer number.
   If it is not possible to do it, then sets *ISANUI to 0 and   
   returns 0, otherwise sets *ISANUI to 1 and returns the number Z
   in the form of an unsigned integer.                           
*/
ui_nteger xx_c2ui (c_omplex z, int* isanui)
{
  r_eal re;

  if (z.re >= 0)
    {
#ifdef USE_LONG_DOUBLE
      re = floorl (z.re);
#else
      re = floor (z.re);
#endif
    }
  else
    {
      *isanui = 0;
      return 0;
    }
  if (z.im != 0 || z.re != re || z.re > ULONG_MAX)
    {
      *isanui = 0;
      return 0;
    }
  else
    {
      *isanui = 1;
      return (ui_nteger)z.re;
    }
}
