/*
   This file is part of the XXCalc Library - version 3.2
   Copyright (C)  2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010,
   2011, 2012, 2013    Ivano Primi ( ivprimi@libero.it )    

   The XXCalc Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The XXCalc library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<float.h> /* for DBL_DIG and LDBL_DIG */
#include"ioctype.h"
#include"evaluate.h"
#include"heapsort.h"
#include"utils.h"
#include"expander.h"
#ifdef DMALLOC
#include <dmalloc.h>
#endif

/* Other useful constants */
#define SUCCESS 1
#define FAILURE 0

extern const c_omplex c_0;
extern int xx_chkind (char* expr);
extern int xx_calc (xx_mathtoken * toklist, ui_nteger tllen, xx_varlist vl,
		    c_omplex * w, int rm);
extern xx_mathtoken *xx_parser (const char *expr, xx_tokstack * ts, 
				ui_nteger * len, int *errcode);

/* errstate is used to store an errorstate, i.e. one of the constants */
/* defined inside parser.h.                                           */
static int errstate;

#define XX_ERRMSG_DIM  34

static const char *errmsg[XX_ERRMSG_DIM] = {
  "No error",
  "Empty expression",
  "Illegal character found",
  "Too long expression",
  "Too long variable name",
  "Too long token",
  "Invalid numerical format",
  "Overflow occurred",
  "Missing closed parenthesis",
  "Too many closed parentheses",
  "Cannot add another level of parenthesis",
  "Reference to unknown operator",
  "Illegal call to function",
  "Reference to unknown function",
  "Division by zero",
  "Out of domain",
  "Invalid exponent",
  "Invalid token at the start of the expression",
  "Invalid token at the end of the expression",
  "Function not followed by open parenthesis",
  "Missing value",
  "Missing operator",
  "Missing open square parenthesis",
  "Missing closed square parenthesis",
  "Square parenthesis in wrong position",
  "Invalid index",
  "Invalid index value",
  "Unrecognized token",
  "Too many operators and functions",
  "Reference to unknown parameter or constant",
  "Reference to non-existent stack",
  "Reference to non-existent list of variables",
  "Cannot add more variables to the selected list",
  "Failed assignment"
};				/* Only Fatal error should not be listed */

/* number of currently allocated stacks */
static unsigned numstacks = 0;
static xx_tokstack *stack = NULL;
/* vnl[] is used only by xx_eval() */
static char* vnl[XX_MAX_NUM_STACKS];


/* number of currently allocated lists of variables */
static unsigned numlists = 0;
static xx_varlist *vlist = NULL;

void
xx_clearerr (void)
{
  errstate = XX_OK;
}

int
xx_errno (void)
{
  return errstate;
}

const char *
xx_errmsg (void)
{
  if (errstate != XX_FATAL_ERROR)
    return errmsg[errstate];
  else
    return "Fatal (memory-)error";
}

void
xx_perror (void)
{
  if (errstate != XX_FATAL_ERROR)
    fprintf (stderr, "\n%s\n\n", errmsg[errstate]);
  else
    fprintf (stderr, "\nFatal error (out of memory?)\n\n");
}

const char *
xx_strerror (int n)
{
  if (n == XX_FATAL_ERROR)
    return "Fatal error (out of memory?)";
  else if (n < 0 || n >= XX_ERRMSG_DIM)
    return "Invalid error code";
  else
    return errmsg[n];
}

int
xx_mkstacks (unsigned how_much)
{
  unsigned i;

  if (how_much > XX_MAX_NUM_STACKS || numstacks != 0 || (stack))
    return FAILURE;
  else if (how_much == 0)
    numstacks = XX_DEF_NUM_STACKS;
  else
    numstacks = how_much;
  stack = (xx_tokstack *) malloc (numstacks * sizeof (xx_tokstack));
  if (!stack)
    return FAILURE;
  else
    {
      for (i = 0; i < numstacks; i++)
	xx_ts_init (&stack[i]);
      return SUCCESS;
    }
}

int
xx_addnewstack (void)
{
  xx_tokstack *tmp;

  if (numstacks == XX_MAX_NUM_STACKS)
    return FAILURE;
  else
    numstacks++;
  /* If you arrive here then numstacks >= 1 */
  tmp = (xx_tokstack *) realloc (stack, numstacks * sizeof (xx_tokstack));
  if (!tmp)
    return FAILURE;
  else
    {
      stack = tmp;
      xx_ts_init (&stack[numstacks - 1]);
      return SUCCESS;
    }
}

unsigned
xx_gettns (void)
{
  return numstacks;
}

void
xx_rmstackno (unsigned which_stack)
{
  if (which_stack < numstacks)
    xx_ts_destroy (&stack[which_stack]);
}

void
xx_rmstacks (void)
{
  unsigned i;

  for (i = 0; i < numstacks; i++)
    xx_ts_destroy (&stack[i]);
  free ((void *) stack);
  stack = NULL;
  numstacks = 0;
  for(i = 0; i < XX_MAX_NUM_STACKS; i++) /* Added on Oct 2, 2002 */
    {
      if( (vnl[i]) )
	{
	  free((void*)vnl[i]);
	  vnl[i] = NULL;
	}
    }
}

/*
  SARRAY should point to an array of dinamycally allocated strings,
  LENGTH should be the number of elements of the array.
  This function filters adjacent matching strings and
  merges them to the first occurrence freeing the memory allocated 
  for the other occurrences (these are practically removed from the array).
  
  The function returns the new length of the array, i.e. the number
  of its non-NULL strings. This can only be zero if either SARRAY == NULL
  or SARRAY != NULL but LENGTH == 0.

  Remark: The array pointed by SARRAY is not resized to the value of
          the new length. 
*/
static
ui_nteger remove_duplicates (char** sarray, ui_nteger length)
{
  if (!sarray)
    return 0;
  else
    {
      ui_nteger i,j,k, new_length = length;

      for (i = 0; i < new_length; i++)
	{
	  for (j = i+1; 
	       j < new_length && 
		 sarray[j] != NULL && 
		 strcmp(sarray[i],sarray[j]) == 0;
	       j++)
	    {
	      free ((void*) sarray[j]);
	      sarray[j] = NULL;
	    }
	  for (k = j; k < new_length; k++)
	    sarray[i+1+(k-j)] = sarray[k];
	  new_length -= (j - i - 1);
	}
      for (i = new_length; i < length; i++)
	{
	  /* These are duplicated pointers, we do not need them */
	  sarray[i] = NULL;
	}
      return new_length;
    }
} 

char**
xx_findvars (unsigned which_stack, ui_nteger* number_of_vars)
{
  ui_nteger i, j, stack_len = 0;
  char** varlist = NULL;
  const xx_mathtoken* ptoken;

  if (which_stack >= numstacks || !number_of_vars)
    {
      if ((number_of_vars))
	*number_of_vars = 0;
      return NULL;
    }
  stack_len = xx_ts_len (stack[which_stack]);
  
  if ( !(varlist = (char **) malloc (stack_len * sizeof (char *))) )
    {
      *number_of_vars = 0;
      return NULL;
    }
  else
    {
      for (i = 0; i < stack_len; varlist[i] = NULL, i++);
      i = 0;
      ptoken = xx_ts_run (&stack[which_stack]);
      while ( (ptoken) )
	{
	  if ( ptoken->name[0] != '\0' )
	    {
	      varlist[i] = (char *) malloc (XX_MAX_NDIM * sizeof (char));
	      if (!varlist[i])
		{
		  for (j = 0; j < i; j++)
		    free ((void *) varlist[j]); /* To avoid memory leaks */
		  free ((void *) varlist);
		  *number_of_vars = 0;
		  return NULL;
		}
	      else
		{
		  strcpy (varlist[i], ptoken->name);
		  i++;
		}
	    }
	  ptoken = xx_ts_run (NULL);
	} /* end while */
      if ( i > 0 && xx_sheapsort ((const char**)varlist, (long)i) != 0 )
	{
	  /* This should only be the case if I > LONG_MAX */
	  for (j = 0; j < i; j++)
	    free ((void *) varlist[j]); /* To avoid memory leaks */
	  free ((void *) varlist);
	  *number_of_vars = 0;
	  return NULL;	  
	}
      else
	{
	  *number_of_vars = remove_duplicates (varlist, i);
	  return varlist;
	}
    }
}

int
xx_validname (const char *name)
{
  const char *t;

  if (!name || !io_isalpha (*name))
    return 0;
  else
    for (t = name; xx_maybeinid (*t) != 0 && t - name < XX_MAX_VNSIZE-1; t++);
  return (*t == '\0' ? 1 : 0);
}

i_nteger
xx_validnames (const char **namelist)
{
  const char *t;
  i_nteger i;

  for (i = 0; i < XX_MAX_VLLENGTH && namelist[i] != NULL; i++)
    {
      if (!io_isalpha (*namelist[i]))
	return -i;
      else
	for (t = namelist[i];
	     xx_maybeinid (*t) != 0 && t - namelist[i] < XX_MAX_VNSIZE-1; t++);
      if (*t != '\0')
	return -i;
    }
  return 1;
}

int
xx_mklists (unsigned how_much)
{
  unsigned i;

  if (how_much > XX_MAX_NUM_LISTS || numlists != 0 || (vlist))
    return FAILURE;
  else if (how_much == 0)
    numlists = XX_DEF_NUM_LISTS;
  else
    numlists = how_much;
  vlist = (xx_varlist *) malloc (numlists * sizeof (xx_varlist));
  if (!vlist)
    return FAILURE;
  else
    {
      for (i = 0; i < numlists; i++)
	xx_vl_init (&vlist[i]);
      return SUCCESS;
    }
}

int
xx_addnewlist (void)
{
  xx_varlist *tmp;

  if (numlists == XX_MAX_NUM_LISTS)
    return FAILURE;
  else
    numlists++;
  /* If you arrive here then numlists >= 1 */
  tmp = (xx_varlist *) realloc (vlist, numlists * sizeof (xx_varlist));
  if (!tmp)
    return FAILURE;
  else
    {
      vlist = tmp;
      xx_vl_init (&vlist[numlists - 1]);
      return SUCCESS;
    }
}

unsigned
xx_gettnl (void)
{
  return numlists;
}

int
xx_fload (FILE * rf, unsigned which_list)
{
  char name[XX_MAX_VNSIZE];
  c_omplex z;
  int n = 0; /* Initialization provided only to avoid bothering warnings! */

  if (which_list >= numlists)
    return FAILURE;
  else
    {
      while (xx_vl_len (vlist[which_list]) < XX_MAX_VLLENGTH
	     && (n = fscanf (rf, XX_FMTSTR_FOR_VAR, name, &z.re, &z.im)) == 3)
	if (!xx_validname (name) || !xx_vl_ins (&vlist[which_list], name, z))
	  return FAILURE;
      if ( (n>=0 && n<3)  ||  ferror (rf) != 0)
	return FAILURE;
      else
	return SUCCESS;
    }
}

int
xx_fbuild (FILE * rf, unsigned which_list)
{
  char name[XX_MAX_VNSIZE];
  c_omplex z;
  int n = 0; /* Initialization provided only to avoid bothering warnings! */

  if (which_list >= numlists)
    return FAILURE;
  else
    {
      xx_vl_destroy (&vlist[which_list]);
      while (xx_vl_len (vlist[which_list]) < XX_MAX_VLLENGTH
	     && (n = fscanf (rf, XX_FMTSTR_FOR_VAR, name, &z.re, &z.im)) == 3)
	if (!xx_validname (name) || !xx_vl_add (&vlist[which_list], name, z))
	  return FAILURE;
      if ( (n>=0 && n<3)  ||  ferror (rf) != 0)
	return FAILURE;
      else
	{
	  xx_vl_sort( vlist[which_list] );
	  return SUCCESS;
	}
    }
}

#define FABS(x) ((x) >= 0 ? (x) : -(x))

int
xx_fwrt (FILE * wf, int prec, unsigned which_list)
{
  const char *name;
  c_omplex z;

  if ( prec < 0 )
    prec = 6;
  else if ( prec > R_MAX_PREC )
    prec = R_MAX_PREC;
  if (which_list >= numlists)
    return FAILURE;
  else
    {
      for (name = xx_vl_run (&vlist[which_list], &z); name != NULL;
	   name = xx_vl_run (NULL, &z))
	{
	  fprintf (wf, "%-*s ", XX_MAX_VNSIZE, name);
#ifdef USE_LONG_DOUBLE

	  if (FABS (z.re) >= 1.0E7)
	    fprintf (wf, "%31.*LE  ", prec, z.re);
	  else
	    fprintf (wf, "%27.*Lf      ", prec, z.re);
	  if (FABS (z.im) >= 1.0E7)
	    fprintf (wf, "%31.*LE\n", prec, z.im);
	  else
	    fprintf (wf, "%27.*Lf\n", prec, z.im);

#else /* USE_LONG_DOUBLE */

	  if (FABS (z.re) >= 1.0E7)
	    fprintf (wf, "%28.*E  ", prec, z.re);
	  else
	    fprintf (wf, "%24.*f      ", prec, z.re);
	  if (FABS (z.im) >= 1.0E7)
	    fprintf (wf, "%28.*E\n", prec, z.im);
	  else
	    fprintf (wf, "%24.*f\n", prec, z.im);

#endif /* USE_LONG_DOUBLE */
	}
      return SUCCESS;
    }
}

int
xx_is_list_empty (unsigned which_list)
{
  if (which_list < numlists)
    return (xx_vl_isempty (vlist[which_list]) != 0 ? 1 : 0);
  else
    return -1;
}

ui_nteger
xx_getll (unsigned which_list)
{
  if (which_list < numlists)
    return xx_vl_len (vlist[which_list]);
  else
    return 0;
}

i_nteger
xx_getpos (unsigned which_list, const char* name, i_nteger from, i_nteger to)
{
  if( which_list >= numlists )
    return -1;
  else
    return xx_vl_getpos_r (vlist[which_list], name, from, to); 
}

i_nteger
xx_getpointer (unsigned which_list, const char* name)
{
  if( which_list >= numlists )
    return -1;
  else
    return xx_vl_getpos (vlist[which_list], name);
}

c_omplex
xx_getvar (unsigned which_list, const char *name, int *isdef)
{
  if (which_list < numlists)
    return xx_vl_get (vlist[which_list], name, isdef);
  else
    {
      *isdef = 0;
      return c_0;
    }
}

c_omplex *
xx_getvars (unsigned from, const char **namelist)
{
  c_omplex *pz;
  ui_nteger i, l;
  int yesno;

  if (from >= numlists)
    return NULL;
  else
    for (l = 0; l < XX_MAX_VLLENGTH && namelist[l] != NULL; l++);
  if (!(pz = (c_omplex *) malloc (l * sizeof (c_omplex))))
    return NULL;
  else
    {
      for (i = 0; i < l; i++)
	pz[i] = xx_vl_get (vlist[from], namelist[i], &yesno);
      return pz;
    }
}

const char*
xx_at (unsigned which_list, ui_nteger pos, c_omplex *value)
{
  if (which_list >= numlists)
    {
      if ((value))
	*value = c_0;
      return NULL;
    }
  else
    return xx_vl_at (vlist[which_list], pos, value);
}

int
xx_readlist (unsigned which_list, char ***varname, c_omplex ** value)
{
  ui_nteger length;
  long i, j;			/* length cannot be > LONG_MAX! */
  const char *name;
  c_omplex z;

  if (which_list >= numlists)
    {
      *varname = NULL;
      *value = NULL;      
      return FAILURE;
    }
  else
    length = xx_vl_len (vlist[which_list]);
  if (length == 0)
    {
      *varname = NULL;
      *value = NULL;
      return SUCCESS;
    }
  else /* length > 0 */
    {
      *value = (c_omplex *) malloc (length * sizeof (c_omplex));
      if (!(*value))
	{
	  *varname = NULL;
	  return FAILURE;
	}
      else
	if (!(*varname = (char **) malloc ((length + 1) * sizeof (char *))))
	{
	  free ((void *) *value);
	  *value = NULL;
	  return FAILURE;
	}
      else
	{
	  for (i = 0; i < length; i++)
	    {
	      (*varname)[i] = (char *) malloc (XX_MAX_VNSIZE * sizeof (char));
	      if (!(*varname)[i])
		{
		  for (j = i - 1; j >= 0; j--)
		    free ((void *) (*varname)[j]); /* To avoid memory leaks */
		  free ((void *) *varname);
		  free ((void *) *value);
		  *varname = NULL;
		  *value = NULL;
		  return FAILURE;
		}
	    }
	  (*varname)[length] = NULL;
	  i = 0;
	  for (name = xx_vl_run (&vlist[which_list], &z); name != NULL;
	       name = xx_vl_run (NULL, &z))
	    {
	      strcpy ((*varname)[i], name);
	      (*value)[i++] = z;
	    }
	  return SUCCESS;
	}
    }	/* end of if(length == 0) ... else */
}

void xx_getrange (unsigned which_list, const char* prefix,
		  i_nteger* from, i_nteger* to)
{
  if ( which_list >= numlists )
    *from = *to = -1;
  else
    xx_vl_getrange (vlist[which_list], prefix, from, to);
}

char **
xx_seekprefix (unsigned where, const char *prefix, c_omplex ** value)
{
  char **namelist;
  const char *name;
  ui_nteger j, n, length;
  i_nteger i, from, to;

  if (where >= numlists || !prefix)
    {
      *value = NULL;
      return NULL;
    }
  else
    length = xx_vl_len (vlist[where]);
  if (length == 0)
    {
      *value = NULL;
      return NULL;
    }
  else if (!(*value = (c_omplex *) malloc (length * sizeof (c_omplex))))
    return NULL;
  else if (!(namelist = (char **) malloc ((length + 1) * sizeof (char *))))
    {
      free ((void *) *value);
      *value = NULL;
      return NULL;
    }
  else
    {
      for (n = 0; n < length; (*value)[n] = c_0, namelist[n++] = NULL);
      namelist[length] = NULL;
      xx_vl_getrange (vlist[where], prefix, &from, &to);
    }
  if( from >= 0 )
    {
      for(n = 0, i = from; i <= to; i++, n++)
	{
	  name = xx_vl_at (vlist[where], i, &(*value)[n]);
	  namelist[n] = (char *) malloc ((strlen (name) + 1) * sizeof (char));
	  if (!namelist[n])
	    {
	      for (j = 0; j < n; j++)
		free ((void *) namelist[j]);
	      free ((void *) namelist);
	      free ((void *) *value);
	      *value = NULL;
	      return NULL;
	    }
	  else
	    strcpy (namelist[n], name);
	}
    }
  return namelist;
}

char **
xx_seeksuffix (unsigned where, const char *suffix, c_omplex ** value)
{
  char **namelist;
  const char *name;
  c_omplex z;
  ui_nteger n, length;
  long i, j;

  if (where >= numlists || !suffix)
    {
      *value = NULL;
      return NULL;
    }
  else
    length = xx_vl_len (vlist[where]);
  if (length == 0)
    {
      *value = NULL;
      return NULL;
    }
  else if (!(*value = (c_omplex *) malloc (length * sizeof (c_omplex))))
    return NULL;
  else if (!(namelist = (char **) malloc ((length + 1) * sizeof (char *))))
    {
      free ((void *) *value);
      *value = NULL;
      return NULL;
    }
  else
    {
      for (n = 0; n < length; (*value)[n] = c_0, namelist[n++] = NULL);
      namelist[length] = NULL;
    }
  for (n = 0, name = xx_vl_run (&vlist[where], &z); name != NULL;
       name = xx_vl_run (NULL, &z))
    {
      i = strlen (suffix);
      j = strlen (name);
      for (; i >= 0 && j >= 0 && suffix[i] == name[j]; i--, j--);
      if (i < 0)
	{
	  namelist[n] = (char *) malloc ((strlen (name) + 1) * sizeof (char));
	  if (!namelist[n])
	    {
	      for (i = 0; i < n; i++)
		free ((void *) namelist[i]);
	      free ((void *) namelist);
	      free ((void *) *value);
	      *value = NULL;
	      return NULL;
	    }
	  else
	    strcpy (namelist[n], name);
	  (*value)[n++] = z;
	}
      /* 
	 If the previous condition is false, then
         either i>=0 && j <0,     or
	 i>=0 && j>=0 &&  suffix[i]!=name[j].
	 In both these cases we may say that name
	 does not end by 'suffix'.
      */
    }
  return namelist;
}

int
xx_mergevl (unsigned to, unsigned from)
{
  if (to >= numlists || from >= numlists || from == to)
    return FAILURE;
  else
    return ( !xx_vl_merge (&vlist[to], vlist[from]) ? FAILURE : SUCCESS );
}

int
xx_dupvl (unsigned dest, unsigned src)
{
  ui_nteger i, length = vlist[src].length;
  const char* name;
  c_omplex value;

  if (dest >= numlists || src >= numlists || dest == src)
    return FAILURE;
  else
    {
      xx_vl_destroy (&vlist[dest]);
      /* We may assume that the list vlist[src] is already sorted. */
      /* Then we can use xx_vl_add() instead of xx_vl_ins().       */
      for (i = 0; i < length; i++)
	{
	  name =  xx_vl_at (vlist[src], i, &value);
	  if (!xx_vl_add (&vlist[dest], name, value))
	    return FAILURE;
	}
      return SUCCESS;
    }
}

int
xx_copyvars (unsigned to, unsigned from, ui_nteger first, ui_nteger last)
{
  xx_varlist tmp;
  ui_nteger i, l;
  const char* name;
  c_omplex z;

  if (to >= numlists || from >= numlists || to == from)
    return FAILURE;
  else if ( last >= vlist[from].length )
    return FAILURE;
  else
    {
      xx_vl_init (&tmp);
      l = xx_vl_len (vlist[to]);
      for( i = first; i <= last &&  l< XX_MAX_VLLENGTH ; i++, l++)
	{
	  name = xx_vl_at (vlist[from], i, &z);
	  if (!xx_vl_add (&tmp, name, z)!=0)
	    {
	      xx_vl_destroy (&tmp);
	      return FAILURE;
	    }
	}
      if( !xx_vl_merge (&vlist[to], tmp) )
	{
	  xx_vl_destroy (&tmp);
	  return FAILURE;
	}
      else
	{
	  xx_vl_destroy (&tmp);
	  return SUCCESS;	  
	}
    }
}

int
xx_assign (unsigned which_list, ui_nteger pos, c_omplex z)
{
  if (which_list >= numlists)
    return FAILURE;
  else
    return ( !xx_vl_assign (vlist[which_list], pos, z) ? FAILURE : SUCCESS );
}

int
xx_setvar (unsigned where, const char *name, c_omplex z)
{
  if (where < numlists)
    return (xx_vl_set (vlist[where], name, z) != 0 ? SUCCESS : FAILURE);
  else
    return FAILURE;
}

int
xx_setvars (unsigned where, const char **namelist, const c_omplex * z)
{
  ui_nteger i;

  if (where >= numlists)
    return FAILURE;
  else
    {
      for (i = 0; i < XX_MAX_VLLENGTH && namelist[i] != NULL; i++)
	xx_vl_set (vlist[where], namelist[i], z[i]);
      return SUCCESS;
    }
}

int
xx_addvar (unsigned where, const char *name, c_omplex z)
{
  if (where >= numlists || name == NULL || *name == '\0')
    return FAILURE;
  else if (xx_vl_len (vlist[where]) >= XX_MAX_VLLENGTH
	   || !xx_vl_ins (&vlist[where], name, z))
    return FAILURE;
  else
    return SUCCESS;
}

int
xx_addvars (unsigned which_list, const char **name, const c_omplex * value)
{
  ui_nteger i;

  if (which_list >= numlists)
    return FAILURE;
  else
    {
      for (i = 0;
	   xx_vl_len (vlist[which_list]) < XX_MAX_VLLENGTH && name[i] != NULL
	   && name[i][0] != '\0'; i++)
	if (!xx_vl_ins (&vlist[which_list], name[i], value[i]))
	  return FAILURE;
      return SUCCESS;
    }
}

int
xx_rebuild (unsigned which_list, const char **name, const c_omplex * value)
{
  ui_nteger i;

  if (which_list >= numlists)
    return FAILURE;
  else
    {
      xx_vl_destroy (&vlist[which_list]);
      for (i = 0;
	   xx_vl_len (vlist[which_list]) < XX_MAX_VLLENGTH && name[i] != NULL
	   && name[i][0] != '\0'; i++)
	if (!xx_vl_add (&vlist[which_list], name[i], value[i]))
	  return FAILURE;
      xx_vl_sort (vlist[which_list]);
      return SUCCESS;
    }
}

int
xx_rmvar (unsigned where, const char *name)
{
  if (where >= numlists)
    return FAILURE;
  else
    {
      xx_vl_del (&vlist[where], name);
      return SUCCESS;
    }
}

int
xx_rmvars (unsigned where, const char **namelist)
{
  ui_nteger i;

  if (where >= numlists)
    return FAILURE;
  else
    {
      for (i = 0; i < XX_MAX_VLLENGTH && namelist[i] != NULL; i++)
	xx_vl_del (&vlist[where], namelist[i]);
      return SUCCESS;
    }
}

void
xx_rmlistno (unsigned which_list)
{
  if (which_list < numlists)
    xx_vl_destroy (&vlist[which_list]);
}

void
xx_rmlists (void)
{
  unsigned i;

  for (i = 0; i < numlists; i++)
    xx_vl_destroy (&vlist[i]);
  free ((void *) vlist);
  vlist = NULL;
  numlists = 0;
}

extern const c_omplex c_0;

c_omplex
xx_comp (const char *expr, unsigned which_stack, unsigned which_list,
	  const char *vname, int rm)
{
  char *varname, *expvname; /* expanded variable name */
  xx_mathtoken *toklist;
  ui_nteger tllen;
  c_omplex w;

  errstate = XX_OK;
  if (which_stack >= numstacks)
    {
      errstate = XX_INVALID_REF_TO_STACK;
      return c_0;
    }
  else if (which_list >= numlists)
    {
      errstate = XX_INVALID_REF_TO_VLIST;
      return c_0;
    }
  else
    if (!(toklist = xx_parser (expr, &stack[which_stack], &tllen, &errstate)))
    return c_0;
  else if ((errstate = xx_calc (toklist, tllen, vlist[which_list], &w, rm)) != XX_OK)
    {
      free ((void *) toklist);
      return c_0;
    }
  else
    free ((void *) toklist);
  if (vname != NULL && vname[0] != '\0')
    {
      varname = (char*)malloc ((strlen(vname)+1) * sizeof(char));
      if( !varname )
	{
	  errstate = XX_FATAL_ERROR;
	  return c_0;
	}
      else
	strcpy (varname, vname);
      if( (errstate = xx_chkind (varname)) != XX_OK  ||  !(expvname = xx_expander (varname, vlist[which_list], &errstate)) )
	{
	  free ((void*)varname);
	  return c_0;
	}
      else if ( strlen(expvname) > XX_MAX_VNSIZE-1 )
	{
	  errstate = XX_VNAME2LONG;
	  free ((void*)varname);
	  return c_0;
	}
      else if (!(xx_vl_ins (&vlist[which_list], expvname, w)))
	errstate = XX_FAILED_ASSIGNMENT;
      else if (xx_vl_len (vlist[which_list]) > XX_MAX_VLLENGTH)
	{
	  xx_vl_del (&vlist[which_list], vname);
	  errstate = XX_SELECTED_VLIST_FULL;
	}
      free ((void*)varname);
      return w;
    }
  else
    return w;
}

const char *
xx_split (const char *expr, const char **rightp)
{
  static char vname[XX_MAX_NDIM];
  char *endp;
  str_s_ize i, j;

  /* Warning: xx_split() is called only for expr != NULL  and */
  /* strstr (expr, ":=") != NULL.                             */
  for (i = 0; xx_isspace (expr[i]) != 0; i++)
    ; /* Skip the leading spaces so that */
  /* i points to the first significant character */
  /* of the string expr.                         */
  if (!io_isalpha (expr[i]))
    {
      errstate = XX_UNRECTOK;
      *rightp = NULL;
      return NULL;
    }
  else
    xx_str2id (expr+i, &endp); 
  /* The return-value of xx_str2id() can be ignored */
  if( endp - expr - i >= XX_MAX_VNSIZE )
    {
      errstate = XX_VNAME2LONG;
      *rightp = NULL;
      return NULL;
    }
  else
    for (j = i; j < endp - expr; vname[j - i] = expr[j], j++);
  /* Added on Oct 14, 2002, modified on Nov 2, 2002 */
  if (*endp == '[') /* Now   expr[j] == *endp */
    {
      do
	{
	  for(++endp; *endp != ']' && *endp != ':'; endp++);
	  if(*endp == ':')/* This means that no ']' has been found before */
	    {             /* the ":=".                                    */
	      errstate = XX_MISSCSQP;
	      *rightp = NULL;
	      return NULL;
	    }
	  else
	    for(++endp; xx_maybeinid(*endp); endp++);
	}while (*endp == '[');
      if (endp - expr - i >= XX_MAX_NDIM)
	{
	  errstate = XX_TOKEN2LONG;
	  *rightp = NULL;
	  return NULL;
	}
      else
	{
	  for(; j < endp - expr; vname[j - i] = expr[j], j++);
	  vname[j - i] = '\0'; /* Now   expr[j] == *endp */
	}
    }
  else /* End of part added on Oct 14, 2002 */
    vname[j - i] = '\0';
  for (i = j; xx_isspace (expr[i]) != 0; i++)
    ;
  if (expr[i] != ':' || expr[i + 1] != '=')
    {
      errstate = XX_UNRECTOK;
      *rightp = NULL;
      return NULL;
    }
  else
    {
      *rightp = expr + i + 2;
      return (const char *) vname;
    }
}

c_omplex
xx_eval (const char *expr, unsigned which_stack, unsigned which_list, int rm)
{
  const char *rightp, *p, *vname;
  static int first_call = 1;
  c_omplex w;

  if( (first_call) ) /* We have to initialize vnl[] */
    {
      unsigned i;

      for(i=0; i< XX_MAX_NUM_STACKS; vnl[i++] = NULL);
      first_call = 0;
    }
  if (which_stack >= numstacks)
    {
      errstate = XX_INVALID_REF_TO_STACK;
      return c_0;
    }
  else if (!expr)
    w = xx_comp (NULL, which_stack, which_list, vnl[which_stack], rm);
  else if (!strstr (expr, ":="))
    {
      if( (vnl[which_stack]) )
	{
	  free((void*)vnl[which_stack]);
	  vnl[which_stack] = NULL;
	}
      w = xx_comp (expr, which_stack, which_list, NULL, rm);
    }
  else if (!(vname = xx_split (expr, &rightp)))
    return c_0;
  else
    {
      if( (vnl[which_stack]) )
	strcpy(vnl[which_stack], vname);
      else
	{
	  vnl[which_stack] = (char*) malloc (XX_MAX_NDIM * sizeof(char));
	  if(!vnl[which_stack])
	    {
	      errstate = XX_FATAL_ERROR;
	      return c_0;
	    }
	  else
	    strcpy(vnl[which_stack], vname);
	}
      for (p = rightp; xx_isspace (*p) != 0; p++);
      if (*p == '\0')
	w = xx_comp (NULL, which_stack, which_list, vnl[which_stack], rm);
      else
	w = xx_comp (rightp, which_stack, which_list, vnl[which_stack], rm);
    }
  return w;
}
