/*
    GTKmathplot - a simple GTK+ based program
    to plot mathematical functions.
    Copyright (C) 2012, 2013  Ivano Primi  <ivprimi@libero.it>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<ctype.h>
#include<errno.h>
#include<float.h> /* for DBL_MAX */
#include<math.h>
#include<xxcalc-3.2.0/evaluate.h>
#include"controls.h"  /* for graphic_mode */
#include"utils.h"

int is_value_finite (double x)
{
  return (x != DBL_MAX ? 1 : 0);
}

grchar* xstralloc (size_t n)
{
  grchar* tmpstr;

  if( n==0 )
    return NULL; /* Nothing must be done ! */
  tmpstr = (grchar*)malloc( n*sizeof(grchar) );
  if(!tmpstr)
    {
      fputs ("WARNING!!! No memory available for new allocations,\n", stderr);
      fputs ("           the program is going to terminate now!!!\n", stderr);
      exit (EXIT_FAILURE);
    }
  else
    return tmpstr;
}

grchar* xstrdup (const grchar* caption)
{
  const grchar* str = (caption != NULL ? caption : "");
  grchar* copy;

  copy = (grchar*) malloc ( strlen(str) + 1 );
  if (copy == NULL)
    {
      fputs ("WARNING!!! No memory available for new allocations,\n", stderr);
      fputs ("           the program is going to terminate now!!!\n", stderr);
      exit (EXIT_FAILURE);
    }
  strcpy (copy, str);
  return copy;
}

size_t xstrlen (const grchar* s)
{
  if (!s)
    return 0;
  else
    return strlen(s);
}

void xfree (grchar** pstr)
{
  if ((pstr) && (*pstr))
    {
      free ((void*)(*pstr));
      *pstr = NULL;
    }
}

char *xstrrealloc(char *ptr, size_t size)
{
  if (size == 0)
    {
      if ((ptr))
	free((void*)ptr);
      return NULL;
    }
  else
    {
      char *mem_ptr;

      if (!ptr)
	mem_ptr = (char*) calloc (size, sizeof(char));
      else
	mem_ptr = (char*) realloc ((void*)ptr, size * sizeof(char));
      if(!mem_ptr)
	{
	  fputs ("WARNING!!! No memory available for new allocations,\n", stderr);
	  fputs ("           the program is going to terminate now!!!\n", stderr);
	  exit (EXIT_FAILURE);
	}
      else
	return mem_ptr;
    }
}

static char* math_error = NULL;

void notify_math_error (const char* prefix, 
                        const char* error_msg, 
                        const char* message)
{
  size_t total_size = 0;

  total_size = xstrlen (prefix) + xstrlen (error_msg) + xstrlen (message) + 1;
  xfree (&math_error);
  math_error = xstralloc (total_size);
  *math_error = '\0';
  if ((prefix))
    strcat (math_error, prefix);
  if ((error_msg))
    strcat (math_error, error_msg);
  if ((message))
    strcat (math_error, message);
}

const char* get_last_math_error (void)
{
  return (const char*) math_error;
}

grint gstrcmp (const grchar* s1, const grchar* s2)
{
  if (s1 == NULL)
    return (s2 == NULL || strlen(s2) == 0 ? 0 : -1);
  if (s2 == NULL)
    return (s1 == NULL || strlen(s1) == 0 ? 0 : +1);
  return strcmp (s1, s2);
} 

int str_2_ulng (const grchar* str, unsigned long* value)
{
  grchar* endp; /* points to the non converted tail of STR */
  grchar* t;
  size_t l;

  errno = 0;
  for(l = 0 ; str[l] != '\0' && isspace(str[l]) != 0 ; l++)
    ;
  if(str[l] == '-')
    {
      *value = 0;
      return STR_2_NAN;
    }
  else
    l = strlen(str);
  t = xstralloc(l+2);
  strncpy(t,str,l+1);
  strcat(t,":");
  *value = strtoul(t,&endp,10);
  if( (strcmp(endp,":")) )
    {
      free((void*)t);
      return STR_2_NAN;
    }
  else if(errno == ERANGE)
    {
      free((void*)t);
      return STR_2_OFLOW;
    }
  else
    {
      free((void*)t);
      return STR_2_OK;
    }
}

const char* str_2_double (const grchar* str, double* value)
{
  c_omplex z;
  const char* illegal_chars = "[]{}";

  if ( (strpbrk(str, illegal_chars)) )
    return xx_strerror (XX_ILLCHAR);
  else
    {
      xx_clearerr();    
      z = xx_comp (str, EVAL_TMP, 0, NULL, 1); 
      *value = c_re(z);
      return (xx_errno() == XX_OK ? NULL : xx_errmsg());
    }
}

static
int is_digit (char c)
{
  return (c >= '0' && c <= '9' ? 1 : 0);
}

/*
  Return the non-negative number contained in the head of the
  string pointed to by STR.
  N is the highest number of blanks that can precede the first digit.
  Return zero if STR does not start with a numerical value (after skipping the
  first N blanks).
 */
grint str_2_number (const char* str, unsigned N)
{
  const char* ptr;
  grint n;

  for (ptr = str; *ptr == ' ' && N > 0; ptr++, N--);
  for (n = 0; is_digit (*ptr); ptr++)
    n = *ptr - '0' + n * 10;
  return n;
}

int read_from_string (const char* str, const char* prefix_to_skip, 
		      char* buffer, unsigned long nbytes)
{
  const char* ptr;

  if (!str)
    return 0;
  if ((prefix_to_skip) && (*prefix_to_skip))
    {
      ptr = strstr (str, prefix_to_skip);
      if (ptr != str)
	return 0; /* The prefix has not been found */
      else
	ptr = str + strlen (prefix_to_skip);
    }
  else
    ptr = str;
  /* 
     Now ptr points either to the beginning of STR
     or, if there is a prefix to skip, to the first
     character after it.
  */

  /*
    If NBYTES == 0, then all characters from STR after
    the prefix to skip are copied into BUFFER.
    If NBYTES > 0, then at most NBYTES characters
    of STR after the prefix to skip are copied into BUFFER,
    and then a terminating null byte is appended.
    
    It is left to the caller to ensure that BUFFER
    has enough space to contain all copied characters
    and the terminating null byte.
   */
  if (nbytes > 0)
    {
      strncpy (buffer, ptr, nbytes);
      buffer[nbytes] = '\0';
    }
  else
    strcpy (buffer, ptr);

  /*
    If there is a newline character before the terminating
    null byte, then it is removed.
   */
  for (; *buffer != '\0' && *buffer != '\n'; buffer++);
  *buffer = '\0';
  return 1;
}

int evaluator_set (unsigned evaluator, const char* eq)
{
  const char* illegal_chars = "[]{}";

  if ( (strpbrk(eq, illegal_chars)) )
    return XX_ILLCHAR;
  else
    {
      xx_clearerr();
      xx_rmstackno (evaluator);
      xx_comp (eq, evaluator, 1, NULL, 1);
      return xx_errno();
    }
}

double evaluator_evaluate (unsigned evaluator, int count,
			   const char** names, const double* values)
{
  c_omplex z;
  int i;

  for (i = 0; i < count; i++)
    xx_setvar(1, names[i], c_reset (values[i], 0.0));
  xx_clearerr();
  z = xx_comp (NULL, evaluator, 1, NULL, 1);
  return (xx_errno() == XX_OK ? c_re(z) : DBL_MAX);
} 

static void free_string_array (char** parray, ui_nteger count)
{
  ui_nteger i;

  for (i = 0; i < count; i++)
    free ((void*) parray[i]);
  free ((void*)parray);
}

gruint which_variables (unsigned func)
{
  char** var_names;
  ui_nteger i, j, n, var_number;
  const char* list_of_constants[] = {
    "e", "pi", NULL
  };

  var_names = xx_findvars (func, &var_number);
  if(!var_names)
    {
      fputs ("WARNING!!! No memory available for new allocations,\n", stderr);
      fputs ("           the program is going to terminate now!!!\n", stderr);
      exit (EXIT_FAILURE);
    }

  /* Remove all constants from the list of variable names */
  for (n = 0; list_of_constants[n] != NULL; n++)
    {
      for (i = 0; i < var_number && gstrcmp (var_names[i], list_of_constants[n]) != 0; i++);
      if (i < var_number)
	{
	  free ((void*)var_names[i]);
	  /* Here we use that the list of variable names is lexicographically ordered */
	  for (j = i+1; j < var_number; j++)
	    var_names[j-1] = var_names[j]; /* J is at least 1, so J-1 cannot overflow */
	  var_names[var_number-1] = NULL;
	  var_number--;
	}
    }

  if (var_number > 2)
    {
      free_string_array (var_names, var_number);
      return 0x100; /* Error: too many variables */
    }
  else if ( var_number < 1)
    {
      free_string_array (var_names, var_number);
      return 0x0;   /* Neither s nor t has been found */
    }
  else if (var_number == 1)
    {
      gruint rv;

      if ( gstrcmp (var_names[0], "t") == 0 )
	rv = 0x01; /* only t has been found */
      else if ( gstrcmp (var_names[0], "s") == 0 )
	rv = 0x10; /* only s has been found */
      else
	rv = 0x100; /* Error: a variable appears which is neither s nor t */
      free_string_array (var_names, var_number);
      return rv;
    }
  else
    {
      gruint rv;

      /* var_number == 2 */
      if ( gstrcmp (var_names[0], "s") == 0 )
	{
	  if ( gstrcmp (var_names[1], "t") == 0 )
	    rv = 0x11; /* Both s and t have been found */
	  else
	    rv = 0x100; /* Error: a variable appears which is neither s nor t */
	}
      else
	{
	  /* Error: a variable appears which is neither s nor t */
	  /* Here we use that the list of variable names is lexicographically ordered */
	  rv = 0x100; 
	}
      free_string_array (var_names, var_number);
      return rv;
    }
}

static
int is_separator (char ch, int consider_digits_as_separators)
{
  if ( (consider_digits_as_separators) && (is_digit(ch) || ch == '.') )
    return 1;
  switch (ch)
    {
    case ' ':
    case '\t':
    case '\v':
    case '\n':
    case '\r':
    case '\f':
    case '+':
    case '-':
    case '*':
    case '/':
    case '^':
    case '%':
    case '\\':
    case '$':
    case '<':
    case '>':
    case '=':
    case '!':
    case '&':
    case '|':
    case '(':
    case ')':
    case '\0':
      return 1;
    default:
      return 0;
    }
}

int is_space (char ch)
{
  switch (ch)
    {
    case ' ':
    case '\t':
    case '\v':
    case '\n':
    case '\r':
    case '\f':
      return 1;
    default:
      return 0;
    }
}

static
int is_admissible (char ch)
{
  return (ch == 'x' || ch == 't' || ch == 'y' || ch == 's');
}

static
int string_contains_var_from_varlist (const char* str, const char* varlist)
{
  const char *ptr, *ptr2;
  int ans = 0;

  for (ptr = str; ans == 0 && *ptr != '\0'; ptr++)
    {
      for (ptr2 = varlist; ans == 0 && *ptr2 != '\0'; ptr2++)
	{
	  if (*ptr == *ptr2)
	    {
	      if ( !(( !is_separator(*(ptr+1), 0) ) ||
		     ( ptr > str && !is_separator(*(ptr-1), 1) )) )
		ans = 1;
	    }
	}
    }  
  return ans;
}

char* convert (const char* str, const char* s1, const char* s2)
{
  char *cstr, *ptr;
  const char *ptr1, *ptr2;

  if (!s1 || !s2 || strlen(s1) != strlen(s2))
    return NULL;
  for (ptr1 = s1; *ptr1 != '\0'; ptr1++)
    {
      if (!is_admissible(*ptr1))
	return NULL;
    }
  for (ptr2 = s2; *ptr2 != '\0'; ptr2++)
    {
      if (!is_admissible(*ptr2))
	return NULL;
    }
  cstr = xstrdup (str);
  for (ptr = cstr; *ptr != '\0'; ptr++)
    {
      for (ptr1 = s1, ptr2 = s2; *ptr1 != '\0'; ptr1++, ptr2++)
	{
	  if (*ptr == *ptr1 && *ptr1 != *ptr2) 
	    {
	      if ( !(( !is_separator(*(ptr+1), 0) ) ||
		     ( ptr > cstr && !is_separator(*(ptr-1), 1) )) )
		{
		  *ptr = *ptr2;
		  break;
		}
	    }
	}
    }
  return cstr;
}

int no_other_char_in_string (const char* s, char ch)
{
  const char *ptr;

  if (!s || ch == '\0')
    return 1;
  /* If we arrive here, then ch != '\0' */
  for (ptr = s; (is_space(*ptr)); ptr++);
  if (*ptr != ch) /* This is also the case if *ptr == '\0' */
    return 0;
  else
    {
      /* *ptr == ch && *ptr != '\0' */
      for (ptr++; (is_space(*ptr)); ptr++);
      return (*ptr == '\0' ? 1 : 0);
    }
}

int is_string_blank (const char* str)
{
  const char *ptr;

  if (!str)
    return -1;
  for (ptr = str; (is_space(*ptr)); ptr++);
  return (*ptr == '\0' ? 1 : 0); 
}

void cartesian_to_parametric (graphic_mode mode,
			      const char* eq, 
			      char** x_eq, char** y_eq, char** z_eq)
{
  if (!x_eq || !y_eq)
    {
      fputs ("WARNING!!! Found pointer which does not refer to a valid memory location,\n", stderr);
      fputs ("           the program is going to terminate now!!!\n", stderr);
      exit (EXIT_FAILURE);
    }
  if (mode == GRAPHIC_2D)
    {
     if ( !(is_string_blank(eq)) )
	{
	  *x_eq = xstrdup("t");
	  if ( (string_contains_var_from_varlist (eq, "ts")) )
	    *y_eq = NULL;
	  else
	    *y_eq = convert (eq, "x", "t");
	}
     else
       *x_eq = *y_eq = NULL;
    }
  else
    {
      if (!z_eq)
	{
	  fputs ("WARNING!!! Found pointer which does not refer to a valid memory location,\n", stderr);
	  fputs ("           the program is going to terminate now!!!\n", stderr);
	  exit (EXIT_FAILURE);
	}
      if ( !(is_string_blank(eq)) )
	{
	  *x_eq = xstrdup("s");
	  *y_eq = xstrdup("t");
	  if ( (string_contains_var_from_varlist (eq, "ts")) )
	    *z_eq = NULL;
	  else
	    *z_eq = convert (eq, "xy", "st");
	}
      else
	*x_eq = *y_eq = *z_eq = NULL;      
    }
}

int parametric_to_cartesian (graphic_mode mode, grmathtype type,
			     const char* x_eq, const char* y_eq, 
			     const char* z_eq,
			     char** eq)
{
  if (!eq || !x_eq || !y_eq)
    return -2;

  if (mode == GRAPHIC_2D)
    {
      if (type != GRMATH_CURVE || !no_other_char_in_string(x_eq, 't'))
	return -1;
      else
	{
	  *eq = convert (y_eq, "t", "x");
	  return 0;
	}
    }
  else
    {
      if (!z_eq)
	return -2;

      if ( !no_other_char_in_string(x_eq, 's') ||
	   !no_other_char_in_string(y_eq, 't') )
	return -1;
      else
	{
	  *eq = convert (z_eq, "st", "xy");
	  return 0;
	}
    }
}

static
int specs_are_valid (print_specs settings)
{
  if (settings.Mw < 1.0 || settings.Mh < 1.0)
    return 0;
  if (settings.tab_width < 1 || settings.line_spacing < 0.0)
    return 0;
  if (settings.right < settings.left + 2 * settings.tab_width * settings.Mw)
    return 0;
  if (settings.bottom < settings.top + settings.Mh + settings.line_spacing)
    return 0;
  return 1;
}

static
int is_position_valid (cursor_position pos, print_specs settings)
{
  if (pos.x >= settings.left && pos.x < settings.right &&
      pos.y >= settings.top && pos.y < settings.bottom)
    return 1;
  else
    return 0;
}

/*
  The following function returns the rounding of X to the nearest 
  integer multiple of G / pow(10, n). Halfway cases
  are always rounded away from zero. 
*/

double smart_round (double x, double g, unsigned n)
{
  double mod;
  
  for (; n > 0; g /= 10, n--);
  if (g <= 0.0)
    return x;

  /* If we arrive here, G has positive sign */
  mod = fmod (x, g);  /* MOD has always the same sign as X */
  x -= mod;
  if (mod >= 0.5 * g)
    x += g;
  if (mod <= -0.5 * g)
    x -= g;    
  return x;
}

int add_line_to_text (char** pfmt_text, size_t nleading_spaces,
		      const char* str, size_t nbytes)
{
  size_t new_size;

  if (!pfmt_text || !str || nbytes > strlen(str))
    return -1;
  new_size = nleading_spaces + nbytes + xstrlen(*pfmt_text) + 2;
  if ((*pfmt_text = (char*) xstrrealloc (*pfmt_text, new_size)) == NULL)
    return -1;
  else
    {
      char* ptr;

      for (ptr = *pfmt_text; *ptr != '\0'; ptr++);
      for (; nleading_spaces > 0; ptr++, nleading_spaces--)
	*ptr = ' ';
      *ptr = '\0';
      strncat (*pfmt_text, str, nbytes);
      for (; nbytes > 0; ptr++, nbytes--);
      *ptr = '\n';
      *(ptr+1) = '\0';
      return 0;
    }
}

static
char* force_text_to_area (const char* text, print_specs settings, 
			  unsigned int tabs, cursor_position start_pos,
			  fmt_error_code* err, double* end_pos_y)
{
  cursor_position curr_pos = start_pos;
  const char *ptr, *prow;
  char *fmt_text = NULL;
  unsigned int N;

  if (text == NULL || !specs_are_valid(settings) || err == NULL || end_pos_y == NULL)
    {
      if ((err))
	*err = Invalid_parameters;
      return NULL;
    }
  if (!is_position_valid (curr_pos, settings))
    {
      *err = Cursor_outside_print_area;
      return NULL;
    }
  for (ptr = prow = text; *ptr != '\0' && *ptr != '\n'; ptr++)
    {
      if ( (curr_pos.y + settings.Mh + settings.line_spacing >= settings.bottom) ||
	   (curr_pos.x >= settings.right) )
	{
	  *err = Missing_space;
	  *end_pos_y = curr_pos.y;
	  return fmt_text;
	}
      if (*ptr == '\t')
	curr_pos.x += (settings.tab_width * settings.Mw);
      else
	curr_pos.x += settings.Mw;
      if (curr_pos.x >= settings.right)
	{
	  N = (prow == text ? 0 : tabs); 
	  while ( (isspace(*prow)) )
	    prow++;
	  add_line_to_text (&fmt_text, N * settings.tab_width, prow, ptr-prow);
	  prow = ptr;
	  curr_pos.y += (settings.Mh + settings.line_spacing);
	  /* curr_pos.x = settings.left + tabs * settings.tab_width * settings.Mw; */
	  curr_pos.x = start_pos.x;
	  if (*ptr == '\t')
	    curr_pos.x += (settings.tab_width * settings.Mw);
	  else
	    curr_pos.x += settings.Mw;	  
	}	
    }
  N = (prow == text ? 0 : tabs); 
  while ( (isspace(*prow)) )
    prow++;
  add_line_to_text (&fmt_text, N * settings.tab_width, prow, ptr-prow);
  *end_pos_y = curr_pos.y + settings.Mh + settings.line_spacing;
  *err = No_error;
  return fmt_text;
}

int is_special_char (char ch)
{
  switch (ch)
    {
    case ' ':
    case '\t':
    case '+':
    case '-':
    case '*':
    case '/':
    case '^':
    case ')':
    case '\\':
    case ',':
      return 1;
    default:
      return 0;
    }
}

static
double length_of_first_token (const char* str, double char_width, 
			      unsigned int tab_width)
{
  double l;

  for (l = 0.0; *str != '\0' && !is_special_char(*str); l += char_width, str++);
  for (; (is_special_char (*str)); str++)
    {
      if (*str == '\t')
	l += (tab_width * char_width);
      else
	l += char_width;
    }
  return l;
}

static
const char* next_word (const char* str)
{
  while (*str != '\0' && !is_special_char(*str))
    str++;
  while ((is_special_char(*str)))
    str++;
  return str;
}

static
char* adapt_text_to_area (const char* text, print_specs settings, 
			  unsigned int tabs, cursor_position start_pos,
			  fmt_error_code* err, double* end_pos_y)
{
  cursor_position curr_pos = start_pos;
  const char *ptr, *prow;
  char *fmt_text = NULL;
  unsigned int N;

  if (text == NULL || !specs_are_valid(settings) || err == NULL || end_pos_y == NULL)
    {
      if ((err))
	*err = Invalid_parameters;
      return NULL;
    }
  if (!is_position_valid (curr_pos, settings))
    {
      *err = Cursor_outside_print_area;
      return NULL;
    }
  /* while ( (is_special_char(*text)) ) */
  /*   text++; */
  ptr = prow = text;
  while (*ptr != '\0' && *ptr != '\n')
    {
      if ( (curr_pos.y + settings.Mh + settings.line_spacing >= settings.bottom) ||
	   (curr_pos.x >= settings.right) )
	{
	  *err = Missing_space;
	  *end_pos_y = curr_pos.y;
	  return fmt_text;
	}
      
      curr_pos.x += length_of_first_token (ptr, settings.Mw, settings.tab_width);

      if (curr_pos.x >= settings.right)
	{
	  N = (prow == text ? 0 : tabs); 
	  add_line_to_text (&fmt_text, N * settings.tab_width, prow, ptr-prow);
	  prow = ptr;
	  curr_pos.y += (settings.Mh + settings.line_spacing);
	  /* curr_pos.x = settings.left + tabs * settings.tab_width * settings.Mw; */
	  curr_pos.x = start_pos.x;
	  curr_pos.x += length_of_first_token (ptr, settings.Mw, settings.tab_width);
	}	
      
      ptr = next_word (ptr);
    }
  N = (prow == text ? 0 : tabs); 
  add_line_to_text (&fmt_text, N * settings.tab_width, prow, ptr-prow);
  *end_pos_y = curr_pos.y + settings.Mh + settings.line_spacing;
  *err = No_error;
  return fmt_text;
}

char* write_text_to_area (const char* text, print_specs settings, 
			  unsigned int tabs, cursor_position start_pos,
			  fmt_error_code* err, double* end_pos_y)
{
  char *fmt_text = NULL;

  fmt_text = adapt_text_to_area (text, settings, tabs, 
				 start_pos, err, end_pos_y);
  if (!fmt_text || *err == Missing_space)
    {
      xfree (&fmt_text);
      fmt_text = force_text_to_area (text, settings, tabs, 
				     start_pos, err, end_pos_y);
    }
  return fmt_text;
}

#ifdef MINOR_TEST

static
const char* error_msg[Nerror_codes] = {
  "No error",
  "Invalid parameters",
  "Cursor outside printable area",
  "Not enough space"
};

int main (void)
{
  char* fmt_text = NULL;
  char line[1024];
  unsigned int nbytes, nleading_spaces;

  do
    {
      printf ("Give me a line of text (Ctrl+C to terminate):\n");
      fgets (line, 1024, stdin);
      printf ("Number of leading spaces> ");
      scanf ("%u", &nleading_spaces);
      printf ("Number of bytes to add> ");
      scanf ("%u", &nbytes);
      getchar();
      if (add_line_to_text (&fmt_text, nleading_spaces, line, nbytes) != 0)
	printf ("Invalid input, retry please!\n");
      else
	printf ("New text:\n%s#END#\n", fmt_text);
    } while ((line));
  return 0;
}

#endif

#ifdef MAJOR_TEST

static
const char* error_msg[Nerror_codes] = {
  "No error",
  "Invalid parameters",
  "Cursor outside printable area",
  "Not enough space"
};

int main (void)
{
  char *ptr, *fmt_text = NULL;
  char text[1024];
  cursor_position start_pos;
  print_specs settings;
  fmt_error_code err;
  double end_pos_y;

  settings.Mw = settings.Mh = 1.0;
  settings.tab_width = 8;
  settings.line_spacing = 0.0;
  settings.left = 0.0;
  settings.right = 80.0;
  settings.top = 0.0;
  settings.bottom = 24.0;
  start_pos.x = settings.left;
  start_pos.y = settings.top;
  do
    {
      printf ("\nGive me a line of text (Ctrl+C to terminate):\n");
      fgets (text, 1024, stdin);
      for (ptr = text; *ptr != '\0' && *ptr != '\n'; ptr++);
      *ptr = '\0';
      fmt_text = adapt_text_to_area (text, settings, 1, start_pos, &err, &end_pos_y);
      if (!fmt_text)
	printf ("fmt_text:\n<NULL>#END#\n");
      else
	printf ("fmt_text:\n%s#END#\n", fmt_text);
      printf ("Error Message: %s\n", error_msg[err]);
      printf ("y-Coordinate of the final position = %f\n", end_pos_y);
      xstrrealloc (fmt_text, 0);
      if ((err))
	{
	  fmt_text = force_text_to_area (text, settings, 1, start_pos, &err, &end_pos_y);
	  printf ("After forcing text to area:\n");
	  if (!fmt_text)
	    printf ("fmt_text:\n<NULL>#END#\n");
	  else
	    printf ("fmt_text:\n%s#END#\n", fmt_text);
	  xstrrealloc (fmt_text, 0);
	}
    } while ((text));
  return 0;
}
#endif

