/*
    GTKmathplot - a simple GTK+ based program
    to plot mathematical functions.
    Copyright (C) 2012, 2013  Ivano Primi  <ivprimi@libero.it>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef __PAR3D__
#define __PAR3D__

#include"gr3d.h"

typedef struct {
  /* Parametric equations of the object, with s and t as parameters */
  grchar *x_eq, *y_eq, *z_eq; 

  /* Evaluators to use with X_EQ, Y_EQ and Z_EQ, respectively */
  unsigned x_func, y_func, z_func; 

  /* Infima and suprema of the intervals of variation of s and t respectively */
  double inf_s, sup_s, inf_t, sup_t;

  /* 
     Number of subintervals which the interval of variation of s / t
     has to be split into                                            
  */
  gruint nsteps_s, nsteps_t; 

  /* 
     The drawing style of the associated geometric object, either
     GRSTYLE_NONE, or GRSTYLE_DOTTED, or GRSTYLE_STROKED or GRSTYLE_FILLED.
     GRSTYLE_NONE can be used only by empty parametric objects, see below
     for the definition of empty object.
  */
  grstyle style;

  /* This is just an identifier, but can be used to codify color and
     line style to be used when drawing the object                    */
  grint color;

  /* A caption for the object, NULL to mean no caption */
  grchar *caption;

  /* 
     This must be considered a private field. It is automatically
     updated by par3d_new(), par3d_define(), par3d_load() and
     par3d_clear().
  */
  grmathtype type;
} par3d_object;


/*
  Create a new empty object, precisely an object OBJ with 
  OBJ.X_EQ = OBJ.Y_EQ = OBJ.Z_EQ = NULL,
  OBJ.X_FUNC = X_EVAL, OBJ.Y_FUNC = Y_EVAL, OBJ.Z_FUNC = Z_EVAL, 
  OBJ.INF_S = OBJ.SUP_S = OBJ.INF_T = OBJ.SUP_T = 0.0,
  OBJ.NSTEPS_S = OBJ.NSTEPS_T = 0, OBJ.STYLE = GRSTYLE_NONE,
  OBJ.COLOR = 0, OBJ.CAPTION = NULL, and OBJ.TYPE = GRMATH_EMPTY.
*/
par3d_object par3d_new (unsigned x_eval, unsigned y_eval, unsigned z_eval);

/*
  Return 1 if OBJ is an empty object, 0 otherwise.
  OBJ is an empty object if and only if
  OBJ.X_EQ = OBJ.Y_EQ = OBJ.Z_EQ = (NULL || ""),
  OBJ.INF_S = OBJ.SUP_S = OBJ.INF_T = OBJ.SUP_T = 0.0,
  OBJ.NSTEPS_S = OBJ.NSTEPS_T = 0, and OBJ.CAPTION = (NULL || ""). 
*/
grint par3d_is_empty (par3d_object obj);

/*
  Define the object pointed to by POBJ according to the values
  of the remaining arguments. While doing this make a deep-copy
  of the strings pointed to by X_EQ, Y_EQ, Z_EQ and CAPTION.
  Do nothing if either POBJ == NULL or if every argument after the
  first one has the same value as the field of the object with the same name.
  Return GR_OK if the object pointed to by POBJ has been
  actually and successfully (re)defined, GR_NO_ACTION if nothing has been done,
  GR_FAILURE if something went wrong while (re)defining the object
  (for example in case of a failed memory allocation).
*/
grexitstatus par3d_define (par3d_object* pobj,
			   grchar* x_eq, grchar* y_eq, grchar* z_eq,
			   double inf_s, double sup_s, gruint nsteps_s,
			   double inf_t, double sup_t, gruint nsteps_t,
			   grstyle style, grint color, grchar* caption);

/*
  Add to the gr3d_object pointed to by PGR3D_OBJ the
  shapes computed from the equations of the parametric object OBJ.
  Do nothing if PGR3D_OBJ == NULL or/and OBJ is an empty 
  or a badly defined object.
  
  Return GR_OK if the object pointed to by PGR3D_OBJ has been
  actually and successfully updated, GR_NO_ACTION if nothing has been done,
  GR_FAILURE if something went wrong (for example in case of a failed memory 
  allocation while adding a new shape).

  Rem.: This function can be called on OBJ only after that
        OBJ.TYPE has been set through a call to par3d_define.
*/
grexitstatus par3d_add_to_gr3d_object (gr3d_object* pgr3d_obj,
				      par3d_object obj);


/*
  Save on the file pointed to by FP the contents of the object OBJ.
  Do nothing if OBJ is the empty object.
  Return GR_OK if OBJ has been actually and successfully saved,
  GR_NO_ACTION if nothing has been done,
  GR_FAILURE if something went wrong while writing on file.
*/
grexitstatus par3d_save (par3d_object obj, FILE* fp);


/*
  Evaluate the object OBJ and save on the file pointed to by FP 
  the results of the computations (use a format compatible with
  GNUplot).
  Do nothing if OBJ is the empty object.
  Return GR_OK if OBJ has been evaluated and 
  the results of the computations have been successfully saved,
  GR_NO_ACTION if nothing has been done,
  GR_FAILURE if something went wrong while writing on file.
*/
grexitstatus par3d_eval_save (par3d_object obj, FILE* fp);


/*
  Load a parametric object from the file pointed to by FP 
  and store it in the structure pointed to by POBJ.
  Return the mathematical type of the retrieved object, which must be
  GRMATH_BAD if POBJ == NULL or in case of failure while reading from file.

  Rem.: 1. Before loading the values from the file, the
           object pointed to by POBJ is cleared through a call
	   to par3d_clear().
	2. The values of POBJ->X_FUNC, POBJ->Y_FUNC, and POBJ->Z_FUNC
	   are left unchanged.
	3. Before returning, POBJ->TYPE is properly set.
*/
grmathtype par3d_load (par3d_object* pobj, FILE* fp);

/*
  Free the memory allocated for POBJ->X_EQ, POBJ->Y_EQ, POBJ->Z_EQ,
  and POBJ->CAPTION, then set all the fields of the structure pointed 
  to by POBJ either to zero or to NULL. Only exception:
  POBJ->X_FUNC, POBJ->Y_FUNC, and POBJ->Z_FUNC must be left unchanged.

  Do nothing if POBJ == NULL or the object pointed to by POBJ is empty.
*/
void par3d_clear (par3d_object* pobj);

#endif /* __PAR3D__ */
