/*
    GTKmathplot - a simple GTK+ based program
    to plot mathematical functions.
    Copyright (C) 2012, 2013  Ivano Primi  <ivprimi@libero.it>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef __PAR2D__
#define __PAR2D__

#include"gr2d.h"

typedef struct {
  /* Parametric equations of the object, with s and t as parameters */
  grchar *x_eq, *y_eq; 

  /* Evaluators to use with X_EQ and Y_EQ, respectively */
  unsigned x_func, y_func; 

  /* Infima and suprema of the intervals of variation of s and t respectively */
  double inf_s, sup_s, inf_t, sup_t;

  /* 
     Number of subintervals which the interval of variation of s / t
     has to be split into                                            
  */
  gruint nsteps_s, nsteps_t; 

  /* 
     The drawing style of the associated geometric object, either
     GRSTYLE_NONE, or GRSTYLE_DOTTED, or GRSTYLE_STROKED or GRSTYLE_FILLED.
     GRSTYLE_NONE can be used only by empty parametric objects, see below
     for the definition of empty object.
  */
  grstyle style;

  /* This is just an identifier, but can be used to codify color and
     line style to be used when drawing the object                    */
  grint color;

  /* A caption for the object, NULL to mean no caption */
  grchar *caption;

  /* 
     This must be considered a private field. It is automatically
     updated by par2d_new(), par2d_define(), par2d_load() and
     par2d_clear().
  */
  grmathtype type;
} par2d_object;


/*
  Create a new empty object, precisely an object OBJ with 
  OBJ.X_EQ = OBJ.Y_EQ = NULL, 
  OBJ.X_FUNC = X_EVAL, OBJ.Y_FUNC = Y_EVAL,
  OBJ.INF_S = OBJ.SUP_S = OBJ.INF_T = OBJ.SUP_T = 0.0,
  OBJ.NSTEPS_S = OBJ.NSTEPS_T = 0, OBJ.STYLE = GRSTYLE_NONE,
  OBJ.COLOR = 0, OBJ.CAPTION = NULL, and OBJ.TYPE = GRMATH_EMPTY.
*/
par2d_object par2d_new (unsigned x_eval, unsigned y_eval);

/*
  Return 1 if OBJ is an empty object, 0 otherwise.
  OBJ is an empty object if and only if
  OBJ.X_EQ = OBJ.Y_EQ = (NULL || ""),
  OBJ.INF_S = OBJ.SUP_S = OBJ.INF_T = OBJ.SUP_T = 0.0,
  OBJ.NSTEPS_S = OBJ.NSTEPS_T = 0, and OBJ.CAPTION = (NULL || "").
*/
grint par2d_is_empty (par2d_object obj);

/*
  Return GRMATH_CURVE if OBJ is a curve, GRMATH_SURFACE if it is a surface,
  GRMATH_EMPTY if it is empty, and GRMATH_BAD if it is badly defined.
  
  Rem.: A parametric object is a curve if and only if:
 
        1. the strings pointed to by X_EQ and Y_EQ are valid 
	   mathematical equations containing at most the parameter t,

	2. SUP_T - INF_T > GR_EPSILON,

	3. NSTEPS_T > 0, and

	4. STYLE != GRSTYLE_NONE.

	A parametric object is a surface if and only if:

	1. the strings pointed to by X_EQ and Y_EQ are valid mathematical 
	   equations and at least one of them contains both the parameters s and t,

	2. SUP_S - INF_S > GR_EPSILON, SUP_T - INF_T > GR_EPSILON,

	3. both NSTEPS_S and NSTEPS_T are greater than zero, and

	4. STYLE != GRSTYLE_NONE.
*/
grmathtype par2d_type (par2d_object obj);

/*
  Define the object pointed to by POBJ according to the values
  of the remaining arguments. While doing this make a deep-copy
  of the strings pointed to by X_EQ, Y_EQ and CAPTION.
  Do nothing if either POBJ == NULL or if every argument after the
  first one has the same value as the field of the object with the same name.
  Return GR_OK if the object pointed to by POBJ has been
  actually and successfully (re)defined, GR_NO_ACTION if nothing has been done,
  GR_FAILURE if something went wrong while (re)defining the object
  (for example in case of a failed memory allocation).
*/
grexitstatus par2d_define (par2d_object* pobj,
			   grchar* x_eq, grchar* y_eq,
			   double inf_s, double sup_s, gruint nsteps_s,
			   double inf_t, double sup_t, gruint nsteps_t,
			   grstyle style, grint color, grchar* caption);

/*
  Add to the gr2d_object pointed to by PGR2D_OBJ the
  shapes computed from the equations of the parametric object OBJ.
  Do nothing if PGR2D_OBJ == NULL or/and OBJ is an empty 
  or a badly defined object.
  
  Return GR_OK if the object pointed to by PGR2D_OBJ has been
  actually and successfully updated, GR_NO_ACTION if nothing has been done,
  GR_FAILURE if something went wrong (for example in case of a failed memory 
  allocation while adding a new shape).
*/
grexitstatus par2d_add_to_gr2d_object (gr2d_object* pgr2d_obj,
				       par2d_object obj);


/*
  Save on the file pointed to by FP the contents of the object OBJ.
  Do nothing if OBJ is the empty object.
  Return GR_OK if OBJ has been actually and successfully saved,
  GR_NO_ACTION if nothing has been done,
  GR_FAILURE if something went wrong while writing on file.
*/
grexitstatus par2d_save (par2d_object obj, FILE* fp);


/*
  Evaluate the object OBJ and save on the file pointed to by FP 
  the results of the computations (use a format compatible with
  GNUplot).
  Do nothing if OBJ is the empty object.
  Return GR_OK if OBJ has been evaluated and 
  the results of the computations have been successfully saved,
  GR_NO_ACTION if nothing has been done,
  GR_FAILURE if something went wrong while writing on file.
*/
grexitstatus par2d_eval_save (par2d_object obj, FILE* fp);


/*
  Load a parametric object from the file pointed to by FP 
  and store it in the structure pointed to by POBJ.
  Return the mathematical type of the retrieved object, which must be
  GRMATH_BAD if POBJ == NULL or in case of failure while reading from file.

  Rem.: 1. Before loading the values from the file, the
           object pointed to by POBJ is cleared through a call
	   to par2d_clear().
	2. The values of POBJ->X_FUNC and POBJ->Y_FUNC
	   are left unchanged.
	3. Before returning, POBJ->TYPE is properly set.
*/
grmathtype par2d_load (par2d_object* pobj, FILE* fp);

/*
  Free the memory allocated for POBJ->X_EQ, POBJ->Y_EQ,
  and POBJ->CAPTION, then set all the fields of the structure pointed 
  to by POBJ either to zero or to NULL. Only exception:
  POBJ->X_FUNC and POBJ->Y_FUNC must be left unchanged.

  Do nothing if POBJ == NULL or the object pointed to by POBJ is empty.
*/
void par2d_clear (par2d_object* pobj);

#endif /* __PAR2D__ */
