/*
    GTKmathplot - a simple GTK+ based program
    to plot mathematical functions.
    Copyright (C) 2012, 2013  Ivano Primi  <ivprimi@libero.it>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include<gtk/gtk.h>
#include<gdk/gdkkeysyms.h>
#include<string.h>
#include"cairo_driver.h"
#include"controls.h"
#include"callbacks.h"

/* 
   _DOCDIR is defined in the command line of the call to the compiler 
   together with _DOC1, _DOC2, _DOC3, _DOC4, _DOC5, _DOC6, _DOC7, and _DOC8.
*/
#define HELP_FILE1   _DOCDIR "/" _DOC1
#define HELP_FILE2   _DOCDIR "/" _DOC2
#define HELP_FILE3   _DOCDIR "/" _DOC3
#define HELP_FILE4   _DOCDIR "/" _DOC4
#define HELP_FILE5   _DOCDIR "/" _DOC5
#define HELP_FILE6   _DOCDIR "/" _DOC6
#define HELP_FILE7   _DOCDIR "/" _DOC7
#define HELP_FILE8   _DOCDIR "/" _DOC8

extern cairo_paper_format_prop supported_paper_formats[];
extern const gchar* supported_orientations[];

const gchar* helpfiles_path[] = {
  HELP_FILE1,
  HELP_FILE2,
  HELP_FILE3,
  HELP_FILE4,
  HELP_FILE5,
  HELP_FILE6,
  HELP_FILE7,
  HELP_FILE8
};

const gchar* help_sections[] = {
  "INTRODUCTION",
  "OVERVIEW",
  "INPUT PAGE",
  "OUTPUT PAGE",
  "CUSTOMIZATION",
  "EXAMPLES",
  "MATH STUFF",
  "FORMAT OF .GDAT FILES",
};

#define helpfiles_no (sizeof(helpfiles_path) / sizeof(helpfiles_path[0]))

const gchar* warning_string = "No error, everything is correct\n";
const gchar* error_string = "Fatal error occurred, program is going to terminate\n";

const gchar* about_string = "GTKmathplot, a simple GTK+ based program\n" \
  "to plot mathematical functions\n" \
  "Copyright (C) 2012, 2013 Ivano Primi <ivprimi@libero.it>\n";

const gchar* license_string = "This program is free software: you can redistribute it and/or modify\n" \
  "it under the terms of the GNU General Public License as published by\n" \
  "the Free Software Foundation, either version 3 of the License, or\n" \
  "(at your option) any later version.\n\n" \
  "This program is distributed in the hope that it will be useful,\n" \
  "but WITHOUT ANY WARRANTY; without even the implied warranty of\n" \
  "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n" \
  "GNU General Public License for more details.\n\n" \
  "You should have received a copy of the GNU General Public License\n" \
  "along with this program.  If not, see <http://www.gnu.org/licenses/>.\n";

static const gchar* label_2d[4] = {
  "Grid",
  "Tics",
  "X-axis",
  "Y-axis"
};

static const gchar* label_3d[7] = {
  "Tics",
  "XY-grid",
  "XZ-grid",
  "YZ-grid",
  "X-axis",
  "Y-axis",
  "Z-axis"
};

static GtkWidget *win, *outer_vbox, *menubar, *notebook;
static GtkWidget *main_menu, *main_menu_item, *quit_menu_item;
static GtkWidget *edit_menu, *edit_menu_item, *pref_menu_item;
static GtkWidget *help_menu, *help_menu_item, *contents_menu_item, *license_menu_item, *about_menu_item;

/* Widgets forming the input page */
static GtkWidget *label1, *vbox1;
static GtkWidget *selection_hbox, *mode_frame, *mode_hbox, *type_frame, *type_hbox;
static GtkWidget *twod_radiob, *threed_radiob, *cart_radiob, *param_radiob, *selection_combo_box;
static GtkWidget *object_def_frame, *object_def_vbox;
static GtkWidget *x_hbox, *y_hbox, *z_hbox; 
static GtkWidget *x_label, *y_label, *z_label;
static GtkWidget *x_entry, *y_entry, *z_entry;

static GtkWidget *s_frame, *s_hbox, *inf_s_label, *sup_s_label, *nsteps_s_label; 
static GtkWidget *inf_s_entry, *sup_s_entry, *nsteps_s_entry;

static GtkWidget *t_frame, *t_hbox, *inf_t_label, *sup_t_label, *nsteps_t_label;
static GtkWidget *inf_t_entry, *sup_t_entry, *nsteps_t_entry;

static GtkWidget *prop_hbox;
static GtkWidget *caption_label, *caption_entry;
static GtkWidget *style_frame, *style_hbox;
static GtkWidget *dotted_radiob, *stroked_radiob, *filled_radiob;
static GtkWidget *loaddef_button, *savedef_button, *eval_save_button, *clear_button;
static GtkWidget *central_hbox, *empty_frame, *display_button;
static GtkWidget *empty_space_frame, *bottom_hbox, *quit_button;

/* Widgets forming the output page */
static GtkWidget *label2, *vbox2, *draw_hbox, *obs_vbox, *draw_frame, *draw_area, *legend_frame;

static GtkWidget *long_label, *lat_label, *rotangle_label;
#ifdef _USE_GTK3_
static GtkAdjustment *long_adj, *lat_adj, *rotangle_adj;
#else
static GtkObject *long_adj, *lat_adj, *rotangle_adj;
#endif
static GtkWidget *long_spinb, *lat_spinb, *rotangle_spinb;

static GtkWidget *zoom_hbox, *smoke_hbox;
static GtkWidget *zoom_label, *smoke_label;
#ifdef _USE_GTK3_
static GtkAdjustment *zoom_adj, *smoke_adj;
#else
static GtkObject *zoom_adj, *smoke_adj;
#endif
static GtkWidget *zoom_hscale, *smoke_hscale;

static GtkWidget *show_frame, *options_vbox, *axes_checkb, *grid_checkb, *tics_checkb, *jumps_checkb;
static GtkWidget *load_button, *save_button, *back_button;

static GtkWidget *legend_evbox, *legend_label;

static GtkWidget *tr_frame, *tr_button_box;
static GtkWidget *ximage_up_button, *ximage_down_button;
static GtkWidget *yimage_up_button, *yimage_down_button;
static GtkWidget *zimage_up_button, *zimage_down_button, *reset_button;

/* Widget forming the dialog windows */
static GtkWidget *help_dialog, *help_area, *help_notebook;
static GtkWidget *help_text[helpfiles_no], *help_labels[helpfiles_no];
static GtkWidget *scrolled_win[helpfiles_no];
static GtkTextBuffer *help_text_buffer[helpfiles_no];

static GtkWidget *about_dialog, *about_area, *about_label;

static GtkWidget *license_dialog, *license_area, *license_label;

static GtkWidget *warning_dialog, *warning_area, *warning_label;

static GtkWidget *error_dialog, *error_area, *error_label;

static GtkWidget *settings_dialog, *settings_area;
static GtkWidget *color_2dsettings_frame, *symbols_2dsettings_frame, *paperfmt_frame, *orientation_frame;
static GtkWidget *color_2dsettings_table, *symbols_2dsettings_table, *paperfmt_combo_box, *orientation_combo_box;
static GtkWidget *symbols_2dsettings_miniframe[NUMBER_OF_2DOBJECTS], *symbols_2dsettings_combo_box[NUMBER_OF_2DOBJECTS];
static GtkWidget *color_2dsettings_miniframe[NUMBER_OF_2DOBJECTS+4], *color_2dsettings_combo_box[NUMBER_OF_2DOBJECTS+4];
static GtkWidget *color_3dsettings_frame, *symbols_3dsettings_frame;
static GtkWidget *color_3dsettings_table, *symbols_3dsettings_table;
static GtkWidget *symbols_3dsettings_miniframe[NUMBER_OF_3DOBJECTS], *symbols_3dsettings_combo_box[NUMBER_OF_3DOBJECTS];
static GtkWidget *color_3dsettings_miniframe[NUMBER_OF_3DOBJECTS+7], *color_3dsettings_combo_box[NUMBER_OF_3DOBJECTS+7];
static GtkWidget *addinfo_checkb, *addC_checkb;

/* Other stuff */
static GtkAccelGroup *accel_group = NULL;

static pointers_to_fields fields = { NULL,
				     NULL, NULL,
				     NULL, NULL, NULL,
				     NULL, NULL, NULL,
				     NULL, NULL,
				     NULL, NULL,
				     NULL, NULL, NULL,
				     NULL, NULL, NULL,
				     NULL, DEFAULT_STYLE,
				     NULL, NULL, NULL,
				     NULL,
                                     NULL, NULL, NULL,
				     NULL, NULL,
                                     NULL, NULL, NULL,
                                     NULL, NULL,
                                     NULL};

static PangoFontDescription* monospaced_font_14 = NULL; 

static
gchar* get_object_label_2d (int object_id)
{
  gchar* label;

  if (object_id >= 0 && object_id < NUMBER_OF_2DOBJECTS)
    {
      label = g_strdup_printf ("Shape %d", object_id + 1);
      return label;
    }
  else if (object_id >= NUMBER_OF_2DOBJECTS && object_id < NUMBER_OF_2DOBJECTS + 4)
    return g_strdup (label_2d[object_id - NUMBER_OF_2DOBJECTS]);
  else
    return (gchar*)NULL;
}

static
gchar* get_object_label_3d (int object_id)
{
  gchar* label;

  if (object_id >= 0 && object_id < NUMBER_OF_3DOBJECTS)
    {
      label = g_strdup_printf ("Shape %d", object_id + 1);
      return label;
    }
  else if (object_id >= NUMBER_OF_3DOBJECTS && object_id < NUMBER_OF_3DOBJECTS + 7)
    return g_strdup (label_3d[object_id - NUMBER_OF_3DOBJECTS]);
  else
    return (gchar*)NULL;
}

void create_gui (void)
{
  const int color_2dsettings_table_ncols = 7;
  const int  symbols_2dsettings_table_ncols = 5;
  const int color_3dsettings_table_ncols = 6;
  const int  symbols_3dsettings_table_ncols = 5;
  gchar* lbl;
  int i;

  win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  accel_group = gtk_accel_group_new();
  gtk_window_add_accel_group (GTK_WINDOW(win), accel_group);
#ifdef _USE_GTK3_
  outer_vbox = gtk_box_new (GTK_ORIENTATION_VERTICAL, 5);
#else
  outer_vbox = gtk_vbox_new (FALSE, 5);
#endif
  menubar = gtk_menu_bar_new ();
  main_menu = gtk_menu_new();
  edit_menu = gtk_menu_new();
  help_menu = gtk_menu_new();

  main_menu_item = gtk_menu_item_new_with_mnemonic ("_Main");
  quit_menu_item = gtk_image_menu_item_new_from_stock (GTK_STOCK_QUIT, accel_group);
#ifdef _USE_GTK3_
  gtk_widget_add_accelerator (quit_menu_item, "activate", accel_group,
			      GDK_KEY_q, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
#else
  gtk_widget_add_accelerator (quit_menu_item, "activate", accel_group,
			      GDK_q, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
#endif

  edit_menu_item = gtk_menu_item_new_with_mnemonic ("_Edit");  
  pref_menu_item = gtk_image_menu_item_new_from_stock (GTK_STOCK_PREFERENCES, accel_group);
#ifdef _USE_GTK3_
  gtk_widget_add_accelerator (pref_menu_item, "activate", accel_group,
			      GDK_KEY_p, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);  
#else
  gtk_widget_add_accelerator (pref_menu_item, "activate", accel_group,
			      GDK_p, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);  
#endif

  help_menu_item = gtk_menu_item_new_with_mnemonic ("_Help");
  contents_menu_item = gtk_image_menu_item_new_from_stock (GTK_STOCK_HELP, accel_group);
#ifdef _USE_GTK3_
  gtk_widget_add_accelerator (contents_menu_item, "activate", accel_group,
			      GDK_KEY_h, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
#else
  gtk_widget_add_accelerator (contents_menu_item, "activate", accel_group,
			      GDK_h, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
#endif
  license_menu_item =  gtk_menu_item_new_with_mnemonic ("_License");
  /* about_menu_item = gtk_menu_item_new_with_mnemonic ("_About"); */
  about_menu_item = gtk_image_menu_item_new_from_stock (GTK_STOCK_ABOUT, accel_group);
#ifdef _USE_GTK3_
  gtk_widget_add_accelerator (about_menu_item, "activate", accel_group,
			      GDK_KEY_a, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
#else
  gtk_widget_add_accelerator (about_menu_item, "activate", accel_group,
			      GDK_a, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
#endif

  notebook = gtk_notebook_new();
  /* Input page */
  label1 = gtk_label_new("Input");
#ifdef _USE_GTK3_
  vbox1 = gtk_box_new (GTK_ORIENTATION_VERTICAL, 20);
  selection_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 50);
  mode_frame = gtk_frame_new("Graphic mode");
  mode_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 10);
#else
  vbox1 = gtk_vbox_new (FALSE, 20);
  selection_hbox = gtk_hbox_new (FALSE, 50);
  mode_frame = gtk_frame_new("Graphic mode");
  mode_hbox = gtk_hbox_new (FALSE, 10);
#endif
  twod_radiob = gtk_radio_button_new_with_label (NULL, "2D");
  threed_radiob = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(twod_radiob), "3D");
#ifdef _USE_GTK3_
  selection_combo_box = gtk_combo_box_text_new();
#else
  selection_combo_box = gtk_combo_box_new_text();
#endif

  object_def_frame = gtk_frame_new ("Object definition");
#ifdef _USE_GTK3_
  object_def_vbox = gtk_box_new (GTK_ORIENTATION_VERTICAL, 20);
  type_frame = gtk_frame_new("Type");
  type_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 10);
#else
  object_def_vbox = gtk_vbox_new (FALSE, 20);
  type_frame = gtk_frame_new("Type");
  type_hbox = gtk_hbox_new (FALSE, 10);
#endif
  cart_radiob = gtk_radio_button_new_with_label (NULL, "Cartesian");
  param_radiob = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(cart_radiob), "Parametric");

#ifdef _USE_GTK3_
  x_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 5);
#else
  x_hbox = gtk_hbox_new (FALSE, 5);
#endif
  x_label = gtk_label_new ("x(s,t) = ");
  x_entry = gtk_entry_new ();

#ifdef _USE_GTK3_
  y_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 5);
#else
  y_hbox = gtk_hbox_new (FALSE, 5);
#endif
  y_label = gtk_label_new ("y(s,t) = ");
  y_entry = gtk_entry_new ();

#ifdef _USE_GTK3_
  z_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 5);
#else
  z_hbox = gtk_hbox_new (FALSE, 5);
#endif
  z_label = gtk_label_new ("z(s,t) = ");
  z_entry = gtk_entry_new ();

  s_frame = gtk_frame_new ("Variation Interval of s");
#ifdef _USE_GTK3_
  s_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 10);
#else
  s_hbox = gtk_hbox_new (FALSE, 10);
#endif
  inf_s_label = gtk_label_new ("Min:");
  sup_s_label = gtk_label_new ("Max:");
  nsteps_s_label = gtk_label_new ("Number of subintervals:");
  inf_s_entry = gtk_entry_new ();
  sup_s_entry = gtk_entry_new ();
  nsteps_s_entry = gtk_entry_new ();

  t_frame = gtk_frame_new ("Variation Interval of t");
#ifdef _USE_GTK3_
  t_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 10);
#else
  t_hbox = gtk_hbox_new (FALSE, 10);
#endif
  inf_t_label = gtk_label_new ("Min:");
  sup_t_label = gtk_label_new ("Max:");
  nsteps_t_label = gtk_label_new ("Number of subintervals:");
  inf_t_entry = gtk_entry_new ();
  sup_t_entry = gtk_entry_new ();
  nsteps_t_entry = gtk_entry_new ();

#ifdef _USE_GTK3_
  prop_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 30);
#else
  prop_hbox = gtk_hbox_new (FALSE, 30);
#endif
  caption_label = gtk_label_new ("caption: ");
  caption_entry =gtk_entry_new();
  style_frame = gtk_frame_new ("Plot style");
#ifdef _USE_GTK3_
  style_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 10);
#else
  style_hbox = gtk_hbox_new (TRUE, 10);
#endif
  dotted_radiob = gtk_radio_button_new_with_label (NULL, "dotted");
  stroked_radiob = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(dotted_radiob), "stroked");
  filled_radiob = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(dotted_radiob), "filled");  
  /*   clear_button = gtk_button_new_with_mnemonic ("  _Clear  "); */
  clear_button = gtk_button_new_from_stock (GTK_STOCK_CLEAR);
  loaddef_button = gtk_button_new_with_mnemonic ("_Load from...");
  savedef_button = gtk_button_new_with_mnemonic ("_Save to...");
  eval_save_button = gtk_button_new_with_mnemonic ("E_valuate & Save");

#ifdef _USE_GTK3_
  central_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 10);
#else
  central_hbox = gtk_hbox_new (FALSE, 10);
#endif
  empty_frame = gtk_frame_new("");
  display_button = gtk_button_new_with_mnemonic ("_Display/Update graphic");

  empty_space_frame = gtk_frame_new("");
#ifdef _USE_GTK3_
  bottom_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 40);
#else
  bottom_hbox = gtk_hbox_new (FALSE, 40);
#endif
  quit_button = gtk_button_new_from_stock (GTK_STOCK_QUIT);

  /* Output page */
  label2 = gtk_label_new("Output");
#ifdef _USE_GTK3_
  vbox2 = gtk_box_new (GTK_ORIENTATION_VERTICAL, 5);
  draw_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 5);
  obs_vbox = gtk_box_new (GTK_ORIENTATION_VERTICAL, 5);
#else
  vbox2 = gtk_vbox_new (FALSE, 5);
  draw_hbox = gtk_hbox_new (FALSE, 5);
  obs_vbox = gtk_vbox_new (FALSE, 5);
#endif
  draw_frame = gtk_frame_new("");
  draw_area = gtk_drawing_area_new();
  legend_frame = gtk_frame_new ("Legend");
  legend_evbox = gtk_event_box_new();

  long_label = gtk_label_new ("Longitude: ");
  lat_label = gtk_label_new ("Latitude: ");
  rotangle_label = gtk_label_new ("Rotation angle: ");
  long_adj = gtk_adjustment_new (DEFAULT_LONGITUDE, 0, 360, 1, 15, 0);
  lat_adj = gtk_adjustment_new (DEFAULT_LATITUDE, -90, 90, 1, 15, 0);
  rotangle_adj = gtk_adjustment_new (DEFAULT_ROTATION_ANGLE, 0, 360, 1, 15, 0);
  long_spinb = gtk_spin_button_new (GTK_ADJUSTMENT(long_adj), 0.5, 0);
  lat_spinb = gtk_spin_button_new (GTK_ADJUSTMENT(lat_adj), 0.5, 0);
  rotangle_spinb = gtk_spin_button_new (GTK_ADJUSTMENT(rotangle_adj), 0.5, 0);

#ifdef _USE_GTK3_
  zoom_hbox = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 5);
  smoke_hbox  = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 5);
#else
  zoom_hbox = gtk_hbox_new (FALSE, 5);
  smoke_hbox  = gtk_hbox_new (FALSE, 5);
#endif
  zoom_label = gtk_label_new ("Zoom (%): ");
  smoke_label = gtk_label_new ("Smoke density: ");
  zoom_adj = gtk_adjustment_new (DEFAULT_ZOOM, 50.0, 1000.0, 10.0, 100.0, 0.0);
  smoke_adj = gtk_adjustment_new (DEFAULT_SMOKE_DENSITY, 0.0, 1.0, 0.01, 0.1, 0.0);
#ifdef _USE_GTK3_
  zoom_hscale = gtk_scale_new (GTK_ORIENTATION_HORIZONTAL, GTK_ADJUSTMENT(zoom_adj));
  smoke_hscale = gtk_scale_new (GTK_ORIENTATION_HORIZONTAL, GTK_ADJUSTMENT(smoke_adj));
#else
  zoom_hscale = gtk_hscale_new (GTK_ADJUSTMENT(zoom_adj));
  smoke_hscale = gtk_hscale_new (GTK_ADJUSTMENT(smoke_adj));
#endif

  show_frame = gtk_frame_new ("Show...");
#ifdef _USE_GTK3_
  options_vbox = gtk_box_new (GTK_ORIENTATION_VERTICAL, 10);
#else
  options_vbox = gtk_vbox_new (FALSE, 10);
#endif
  axes_checkb = gtk_check_button_new_with_label ("x, y and z directions");
  grid_checkb = gtk_check_button_new_with_label ("grid");
  tics_checkb = gtk_check_button_new_with_label ("labelled tics");
  jumps_checkb = gtk_check_button_new_with_label ("discontinuity points");

  load_button = gtk_button_new_with_mnemonic ("_Load object from file");
  save_button = gtk_button_new_with_mnemonic ("_Save as...");
  back_button = gtk_button_new_with_mnemonic ("_Back to input mask");

  legend_label = gtk_label_new ("");

  tr_frame = gtk_frame_new ("Image translation commands along x, y and z");
#ifdef _USE_GTK3_
  tr_button_box = gtk_button_box_new (GTK_ORIENTATION_HORIZONTAL);
#else
  tr_button_box = gtk_hbutton_box_new ();
#endif
  ximage_up_button = gtk_button_new_with_label ("x+");
  ximage_down_button = gtk_button_new_with_label ("x-");
  yimage_up_button = gtk_button_new_with_label ("y+");
  yimage_down_button = gtk_button_new_with_label ("y-");
  zimage_up_button = gtk_button_new_with_label ("z+");
  zimage_down_button = gtk_button_new_with_label ("z-");
  reset_button = gtk_button_new_with_label ("Reset");

  /*
   Window {
     outer_vbox [
     menubar
     +-----------+------------+
     |Input Page | Output Page|
     +           +------------+---------------------+
     |                                              |
     |                                              |
     |             notebook                         |
     |                                              |
     |                                              |
     +----------------------------------------------+
     ]
   }

   menubar has currently the following layout:

     menubar [ main_menu | main_menu_item |   edit_menu | edit_menu_item |      help_menu | help_menu_item     | ]
                         | quit_menu_item |             | pref_menu_item |                | contents_menu_item |
                                                                                          | license_menu_item  |
                                                                                          | about_menu_item    |
  */

  gtk_menu_item_set_submenu (GTK_MENU_ITEM(main_menu_item), main_menu);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM(edit_menu_item), edit_menu);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM(help_menu_item), help_menu);
  gtk_menu_shell_append (GTK_MENU_SHELL(menubar), main_menu_item);
  gtk_menu_shell_append (GTK_MENU_SHELL(menubar), edit_menu_item);
  gtk_menu_shell_append (GTK_MENU_SHELL(menubar), help_menu_item);
  gtk_menu_shell_append (GTK_MENU_SHELL(main_menu), quit_menu_item);
  gtk_menu_shell_append (GTK_MENU_SHELL(edit_menu), pref_menu_item);
  gtk_menu_shell_append (GTK_MENU_SHELL(help_menu), contents_menu_item);
  gtk_menu_shell_append (GTK_MENU_SHELL(help_menu), license_menu_item);
  gtk_menu_shell_append (GTK_MENU_SHELL(help_menu), about_menu_item);
  gtk_container_add (GTK_CONTAINER(win), outer_vbox);
  gtk_box_pack_start (GTK_BOX(outer_vbox), menubar, FALSE, FALSE,  0);
  gtk_box_pack_start (GTK_BOX(outer_vbox), notebook, TRUE, TRUE,  0);


  /* Create layout for input page */
  /*
    Input Page
    
    vbox1 {
    selection_hbox [  mode_frame [ mode_hbox [ twod_radiob  threed_radiob ] ]  selection_combo_box                     ]
    object_def_frame {
    object_def_vbox[
    [   type_frame [ type_hbox [ cart_radiob  param_radiob ] ] ]
    x_hbox         [  x_label x_entry..................................................................................]
    y_hbox         [  y_label y_entry..................................................................................]
    z_hbox         [  z_label z_entry..................................................................................]
    s_frame        [  s_hbox  [  inf_s_label inf_s_entry   sup_s_label sup_s_entry   nsteps_s_label nsteps_s_entry  ]  ]
    t_frame        [  t_hbox  [  inf_t_label inf_t_entry   sup_t_label sup_t_entry   nsteps_t_label nsteps_t_entry  ]  ]
    prop_hbox      [  cpt_label cpt_entry  style_frame [ style_hbox [ dotted_radiob stroked_radiob filled_radiob ] ]  clear_button  loaddef_button  savedef_button  eval_save_button  ]
                   ]
          }
    central_hbox   [  empty_frame                                                                      display_button  ]
    empty_space_frame
    bottom_hbox    [                                          quit_button                                              ]
          }
  */
  gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox1, label1); 
  gtk_box_pack_start (GTK_BOX(vbox1), selection_hbox, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(vbox1), object_def_frame, FALSE, FALSE, 5);
  gtk_container_add (GTK_CONTAINER(object_def_frame), object_def_vbox);
  gtk_box_pack_start (GTK_BOX(object_def_vbox), type_frame, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER(type_frame), type_hbox);
  gtk_box_pack_start (GTK_BOX(type_hbox), cart_radiob, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(type_hbox), param_radiob, FALSE, FALSE, 0);  
  gtk_box_pack_start (GTK_BOX(object_def_vbox), x_hbox, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(object_def_vbox), y_hbox, FALSE, FALSE, 0);  
  gtk_box_pack_start (GTK_BOX(object_def_vbox), z_hbox, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(object_def_vbox), s_frame, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER(s_frame), s_hbox);
  gtk_box_pack_start (GTK_BOX(object_def_vbox), t_frame, FALSE, FALSE, 0);  
  gtk_container_add (GTK_CONTAINER(t_frame), t_hbox);
  gtk_box_pack_start (GTK_BOX(object_def_vbox), prop_hbox, FALSE, FALSE, 0);  
  gtk_box_pack_start (GTK_BOX(vbox1), central_hbox, FALSE, FALSE, 0);  
  gtk_box_pack_start (GTK_BOX(vbox1), empty_space_frame, TRUE, FALSE, 0);  
  gtk_box_pack_start (GTK_BOX(vbox1), bottom_hbox, FALSE, FALSE, 0);  

  gtk_box_pack_start (GTK_BOX(selection_hbox), mode_frame, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER(mode_frame), mode_hbox);
  gtk_box_pack_start (GTK_BOX(mode_hbox), twod_radiob, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(mode_hbox), threed_radiob, FALSE, FALSE, 0);  
  gtk_box_pack_start (GTK_BOX(selection_hbox), selection_combo_box, FALSE, FALSE, 0);

  gtk_box_pack_start (GTK_BOX(x_hbox), x_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(x_hbox), x_entry, TRUE, TRUE, 0);

  gtk_box_pack_start (GTK_BOX(y_hbox), y_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(y_hbox), y_entry, TRUE, TRUE, 0);

  gtk_box_pack_start (GTK_BOX(z_hbox), z_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(z_hbox), z_entry, TRUE, TRUE, 0);

  gtk_box_pack_start (GTK_BOX(s_hbox), inf_s_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(s_hbox), inf_s_entry, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(s_hbox), sup_s_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(s_hbox), sup_s_entry, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(s_hbox), nsteps_s_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(s_hbox), nsteps_s_entry, FALSE, FALSE, 0);

  gtk_box_pack_start (GTK_BOX(t_hbox), inf_t_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(t_hbox), inf_t_entry, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(t_hbox), sup_t_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(t_hbox), sup_t_entry, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(t_hbox), nsteps_t_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(t_hbox), nsteps_t_entry, FALSE, FALSE, 0);

  gtk_box_pack_start (GTK_BOX(prop_hbox), caption_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(prop_hbox), caption_entry, FALSE, FALSE, 0);  
  gtk_box_pack_start (GTK_BOX(prop_hbox), style_frame, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER(style_frame), style_hbox);
  gtk_box_pack_start (GTK_BOX(style_hbox), dotted_radiob, FALSE, FALSE, 0); 
  gtk_box_pack_start (GTK_BOX(style_hbox), stroked_radiob, FALSE, FALSE, 0); 
  gtk_box_pack_start (GTK_BOX(style_hbox), filled_radiob, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX(prop_hbox), eval_save_button, FALSE, FALSE, 0); 
  gtk_box_pack_end (GTK_BOX(prop_hbox), savedef_button, FALSE, FALSE, 0); 
  gtk_box_pack_end (GTK_BOX(prop_hbox), loaddef_button, FALSE, FALSE, 0); 
  gtk_box_pack_end (GTK_BOX(prop_hbox), clear_button, FALSE, FALSE, 0);

  gtk_box_pack_start (GTK_BOX(central_hbox), empty_frame, TRUE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(central_hbox), display_button, FALSE, FALSE, 0);

  gtk_box_pack_start (GTK_BOX(bottom_hbox), quit_button, TRUE, FALSE, 0);

  /* Create layout for output page */
  /*
    Output page

    vbox2 {
           draw_hbox        [  obs_vbox   draw_frame [ draw_area ]   legend_frame [ legend_evbox [legend_label] ]  ]
           tr_frame
          }

    obs_vbox {
     long_label 
     long_spinb  
     lat_label 
     lat_spinb  
     rotangle_label 
     rotangle_spinb
     zoom_hbox [  zoom_label zoom_hscale  ]
     smoke_hbox  [  smoke_label   smoke_hscale  ]

     show_frame [  options_vbox [ 
                                  axes_checkb
                                  grid_checkb
				  tics_checkb
				  jumps_checkb
                                ]  ]
     load_button
     save_button
     back_button
    }

    tr_frame 
    {
     tr_button_box [ ximage_up_button ximage_down_button  yimage_up_button yimage_down_button  zimage_up_button zimage_down_button  reset_button  ]
    }
  */
  gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox2, label2);
  gtk_box_pack_start (GTK_BOX(vbox2), draw_hbox, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(draw_hbox), obs_vbox, FALSE, FALSE, 5);
  gtk_box_pack_end (GTK_BOX(draw_hbox), legend_frame, TRUE, TRUE, 5);
  gtk_box_pack_end (GTK_BOX(draw_hbox), draw_frame, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER(draw_frame), draw_area);
  gtk_box_pack_start (GTK_BOX(vbox2), tr_frame, FALSE, FALSE, 0);

  gtk_box_pack_start (GTK_BOX(obs_vbox), long_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(obs_vbox), long_spinb, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(obs_vbox), lat_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(obs_vbox), lat_spinb, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(obs_vbox), rotangle_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(obs_vbox), rotangle_spinb, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(obs_vbox), zoom_hbox, TRUE, TRUE, 0);  
  gtk_box_pack_start (GTK_BOX(obs_vbox), smoke_hbox, TRUE, TRUE, 0);  

  gtk_box_pack_start (GTK_BOX(zoom_hbox), zoom_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(zoom_hbox), zoom_hscale, TRUE, TRUE, 0);

  gtk_box_pack_start (GTK_BOX(smoke_hbox), smoke_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(smoke_hbox), smoke_hscale, TRUE, TRUE, 0);

  gtk_box_pack_start (GTK_BOX(obs_vbox), show_frame, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER(show_frame), options_vbox);
  gtk_box_pack_start (GTK_BOX(options_vbox), axes_checkb, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(options_vbox), grid_checkb, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(options_vbox), tics_checkb, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(options_vbox), jumps_checkb, FALSE, FALSE, 0);

  gtk_box_pack_start (GTK_BOX(obs_vbox), load_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(obs_vbox), save_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(obs_vbox), back_button, FALSE, FALSE, 0);
  
  gtk_container_add (GTK_CONTAINER(legend_frame), legend_evbox);
  gtk_container_add (GTK_CONTAINER(legend_evbox), legend_label);

  gtk_container_add (GTK_CONTAINER(tr_frame), tr_button_box);
  gtk_box_pack_start (GTK_BOX(tr_button_box), ximage_up_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(tr_button_box), ximage_down_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(tr_button_box), yimage_up_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(tr_button_box), yimage_down_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(tr_button_box), zimage_up_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(tr_button_box), zimage_down_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX(tr_button_box), reset_button, FALSE, FALSE, 0);

  /* 
     Modal dialogs
  */
  help_dialog = gtk_dialog_new_with_buttons ("GTKmathplot help",
					     GTK_WINDOW(win),
					     GTK_DIALOG_DESTROY_WITH_PARENT,
					     GTK_STOCK_CLOSE,
					     GTK_RESPONSE_ACCEPT,
					     NULL);
  help_area = gtk_dialog_get_content_area (GTK_DIALOG(help_dialog));
  help_notebook = gtk_notebook_new();
  gtk_container_add (GTK_CONTAINER(help_area), help_notebook);
  for (i = 0; i < helpfiles_no; i++)
    {
      scrolled_win[i] = gtk_scrolled_window_new (NULL, NULL);
      help_text[i] = gtk_text_view_new();
      help_text_buffer[i] = 
	gtk_text_view_get_buffer (GTK_TEXT_VIEW(help_text[i]));
      help_labels[i] = gtk_label_new(help_sections[i]);
      gtk_container_add (GTK_CONTAINER(scrolled_win[i]), help_text[i]);
      gtk_notebook_append_page (GTK_NOTEBOOK(help_notebook), 
				scrolled_win[i], 
				help_labels[i]); 
    }

  license_dialog = gtk_dialog_new_with_buttons ("GTKmathplot license disclaimer",
						GTK_WINDOW(win),
						GTK_DIALOG_MODAL,
						GTK_STOCK_CLOSE,
						GTK_RESPONSE_ACCEPT,
						NULL);
  license_area = gtk_dialog_get_content_area (GTK_DIALOG(license_dialog));
  license_label = gtk_label_new (license_string);
  gtk_container_add (GTK_CONTAINER(license_area), license_label);

  about_dialog = gtk_dialog_new_with_buttons ("About GTKmathplot",
					      GTK_WINDOW(win),
					      GTK_DIALOG_MODAL,
					      GTK_STOCK_CLOSE,
					      GTK_RESPONSE_ACCEPT,
					      NULL);
  about_area = gtk_dialog_get_content_area (GTK_DIALOG(about_dialog));
  about_label = gtk_label_new (about_string);
  gtk_container_add (GTK_CONTAINER(about_area), about_label);

  warning_dialog = gtk_dialog_new_with_buttons ("GTKmathplot - Warning",
						GTK_WINDOW(win),
						GTK_DIALOG_MODAL,
						GTK_STOCK_OK,
						GTK_RESPONSE_ACCEPT,
						NULL);
  warning_area = gtk_dialog_get_content_area (GTK_DIALOG(warning_dialog));
  warning_label = gtk_label_new (warning_string);
  gtk_container_add (GTK_CONTAINER(warning_area), warning_label);

  error_dialog = gtk_dialog_new_with_buttons ("GTKmathplot - Error",
					      GTK_WINDOW(win),
					      GTK_DIALOG_MODAL,
					      GTK_STOCK_QUIT,
					      GTK_RESPONSE_ACCEPT,
					      NULL);
  error_area = gtk_dialog_get_content_area (GTK_DIALOG(error_dialog));
  error_label = gtk_label_new (error_string);
  gtk_container_add (GTK_CONTAINER(error_area), error_label);
  
  settings_dialog = gtk_dialog_new_with_buttons ("GTKmathplot settings",
					         GTK_WINDOW(win),
					         GTK_DIALOG_MODAL,
					         GTK_STOCK_CLOSE,
					         GTK_RESPONSE_ACCEPT,
					         NULL);
  settings_area = gtk_dialog_get_content_area (GTK_DIALOG(settings_dialog));
  color_2dsettings_frame = gtk_frame_new ("Color settings for 2D shapes");
  symbols_2dsettings_frame = gtk_frame_new ("Symbols for 2D shapes");
  color_3dsettings_frame = gtk_frame_new ("Color settings for 3D shapes");
  symbols_3dsettings_frame = gtk_frame_new ("Symbols for 3D shapes");
  paperfmt_frame = gtk_frame_new ("Paper format");
  orientation_frame = gtk_frame_new ("Orientation");
#ifdef _USE_GTK3_
  color_2dsettings_table = gtk_grid_new ();
  symbols_2dsettings_table  = gtk_grid_new ();
  color_3dsettings_table = gtk_grid_new ();
  symbols_3dsettings_table  = gtk_grid_new ();
  paperfmt_combo_box = gtk_combo_box_text_new();
  orientation_combo_box = gtk_combo_box_text_new();
#else
  color_2dsettings_table = gtk_table_new ((NUMBER_OF_2DOBJECTS+4-1)/color_2dsettings_table_ncols + 1, 
					  color_2dsettings_table_ncols, TRUE);
  symbols_2dsettings_table  = gtk_table_new ((NUMBER_OF_2DOBJECTS-1)/symbols_2dsettings_table_ncols + 1, 
					  symbols_2dsettings_table_ncols, TRUE);
  color_3dsettings_table = gtk_table_new ((NUMBER_OF_3DOBJECTS+7-1)/color_3dsettings_table_ncols + 1, 
					  color_3dsettings_table_ncols, TRUE);
  symbols_3dsettings_table  = gtk_table_new ((NUMBER_OF_3DOBJECTS-1)/symbols_3dsettings_table_ncols + 1, 
					  symbols_3dsettings_table_ncols, TRUE);
  paperfmt_combo_box = gtk_combo_box_new_text();
  orientation_combo_box = gtk_combo_box_new_text();
#endif

  for (i = 0; i < NUMBER_OF_2DOBJECTS; i++)
    {
      lbl = get_object_label_2d(i);
      color_2dsettings_miniframe[i] = gtk_frame_new (lbl);
      symbols_2dsettings_miniframe[i] = gtk_frame_new (lbl);
#ifdef _USE_GTK3_
      color_2dsettings_combo_box[i] = gtk_combo_box_text_new();
      symbols_2dsettings_combo_box[i] = gtk_combo_box_text_new();
#else
      color_2dsettings_combo_box[i] = gtk_combo_box_new_text();
      symbols_2dsettings_combo_box[i] = gtk_combo_box_new_text();
#endif
      g_free (lbl);
    }
  for (i = NUMBER_OF_2DOBJECTS; i < NUMBER_OF_2DOBJECTS + 4; i++)
    {
      lbl = get_object_label_2d(i);
      color_2dsettings_miniframe[i] = gtk_frame_new (lbl);
#ifdef _USE_GTK3_
      color_2dsettings_combo_box[i] = gtk_combo_box_text_new();
#else      
      color_2dsettings_combo_box[i] = gtk_combo_box_new_text();
#endif
      g_free (lbl);
    }

  for (i = 0; i < NUMBER_OF_3DOBJECTS; i++)
    {
      lbl = get_object_label_3d (i);
      color_3dsettings_miniframe[i] = gtk_frame_new (lbl);
      symbols_3dsettings_miniframe[i] = gtk_frame_new (lbl);
#ifdef _USE_GTK3_
      color_3dsettings_combo_box[i] = gtk_combo_box_text_new();
      symbols_3dsettings_combo_box[i] = gtk_combo_box_text_new();
#else
      color_3dsettings_combo_box[i] = gtk_combo_box_new_text();
      symbols_3dsettings_combo_box[i] = gtk_combo_box_new_text();
#endif
      g_free (lbl);
    }
  for (i = NUMBER_OF_3DOBJECTS; i < NUMBER_OF_3DOBJECTS + 7; i++)
    {
      lbl = get_object_label_3d(i);
      color_3dsettings_miniframe[i] = gtk_frame_new (lbl);
#ifdef _USE_GTK3_
      color_3dsettings_combo_box[i] = gtk_combo_box_text_new();
#else
      color_3dsettings_combo_box[i] = gtk_combo_box_new_text();
#endif
      g_free (lbl);
    }

  addinfo_checkb = gtk_check_button_new_with_label ("Add more information to the exported plots");
  addC_checkb = gtk_check_button_new_with_label ("Add copyright statement to the exported plots");
  gtk_box_pack_start (GTK_BOX(settings_area), color_2dsettings_frame, FALSE, FALSE, 1);
  gtk_box_pack_start (GTK_BOX(settings_area), symbols_2dsettings_frame, FALSE, FALSE, 1); 
  gtk_box_pack_start (GTK_BOX(settings_area), color_3dsettings_frame, FALSE, FALSE, 1);
  gtk_box_pack_start (GTK_BOX(settings_area), symbols_3dsettings_frame, FALSE, FALSE, 1); 
  gtk_box_pack_start (GTK_BOX(settings_area), paperfmt_frame, FALSE, FALSE, 1);
  gtk_box_pack_start (GTK_BOX(settings_area), orientation_frame, FALSE, FALSE, 1);
  gtk_container_add (GTK_CONTAINER(paperfmt_frame), paperfmt_combo_box);
  gtk_container_add (GTK_CONTAINER(orientation_frame), orientation_combo_box);
  gtk_container_add (GTK_CONTAINER(color_2dsettings_frame), color_2dsettings_table);
  gtk_container_add (GTK_CONTAINER(symbols_2dsettings_frame), symbols_2dsettings_table);
  for (i = 0; i < NUMBER_OF_2DOBJECTS + 4; i++)
    {
#ifdef _USE_GTK3_
      gtk_grid_attach (GTK_GRID(color_2dsettings_table), 
		       color_2dsettings_miniframe[i],
		       i % color_2dsettings_table_ncols, 
		       i / color_2dsettings_table_ncols, 
		       1, 1);
#else
      gtk_table_attach_defaults (GTK_TABLE(color_2dsettings_table), 
				 color_2dsettings_miniframe[i],
				 i % color_2dsettings_table_ncols, 
				 i % color_2dsettings_table_ncols + 1, 
				 i / color_2dsettings_table_ncols, 
				 i / color_2dsettings_table_ncols + 1);
#endif
      gtk_container_add (GTK_CONTAINER(color_2dsettings_miniframe[i]), 
			 color_2dsettings_combo_box[i]);
    }
  for (i = 0; i < NUMBER_OF_2DOBJECTS; i++)
    {
#ifdef _USE_GTK3_
      gtk_grid_attach (GTK_GRID(symbols_2dsettings_table), 
		       symbols_2dsettings_miniframe[i],
		       i % symbols_2dsettings_table_ncols, 
		       i / symbols_2dsettings_table_ncols, 
		       1, 1);
#else
      gtk_table_attach_defaults (GTK_TABLE(symbols_2dsettings_table), 
				 symbols_2dsettings_miniframe[i],
				 i % symbols_2dsettings_table_ncols, 
				 i % symbols_2dsettings_table_ncols + 1, 
				 i / symbols_2dsettings_table_ncols, 
				 i / symbols_2dsettings_table_ncols + 1);
#endif
      gtk_container_add (GTK_CONTAINER(symbols_2dsettings_miniframe[i]),
			 symbols_2dsettings_combo_box[i]);
    }
  gtk_container_add (GTK_CONTAINER(color_3dsettings_frame), color_3dsettings_table);
  gtk_container_add (GTK_CONTAINER(symbols_3dsettings_frame), symbols_3dsettings_table);
  for (i = 0; i < NUMBER_OF_3DOBJECTS + 7; i++)
    {
#ifdef _USE_GTK3_
      gtk_grid_attach (GTK_GRID(color_3dsettings_table),
		       color_3dsettings_miniframe[i],
		       i % color_3dsettings_table_ncols, 
		       i / color_3dsettings_table_ncols, 
		       1, 1);
#else
      gtk_table_attach_defaults (GTK_TABLE(color_3dsettings_table), 
				 color_3dsettings_miniframe[i],
				 i % color_3dsettings_table_ncols, 
				 i % color_3dsettings_table_ncols + 1, 
				 i / color_3dsettings_table_ncols, 
				 i / color_3dsettings_table_ncols + 1);
#endif
      gtk_container_add (GTK_CONTAINER(color_3dsettings_miniframe[i]), color_3dsettings_combo_box[i]);
    }
  for (i = 0; i < NUMBER_OF_3DOBJECTS; i++)
    {
#ifdef _USE_GTK3_
      gtk_grid_attach (GTK_GRID(symbols_3dsettings_table), 
		       symbols_3dsettings_miniframe[i],
		       i % symbols_3dsettings_table_ncols, 
		       i / symbols_3dsettings_table_ncols, 
		       1, 1);
#else
      gtk_table_attach_defaults (GTK_TABLE(symbols_3dsettings_table), 
				 symbols_3dsettings_miniframe[i],
				 i % symbols_3dsettings_table_ncols, 
				 i % symbols_3dsettings_table_ncols + 1, 
				 i / symbols_3dsettings_table_ncols, 
				 i / symbols_3dsettings_table_ncols + 1);
#endif
      gtk_container_add (GTK_CONTAINER(symbols_3dsettings_miniframe[i]), symbols_3dsettings_combo_box[i]);
    }
  gtk_box_pack_start (GTK_BOX(settings_area), addinfo_checkb, FALSE, FALSE, 1);
  gtk_box_pack_start (GTK_BOX(settings_area), addC_checkb, FALSE, FALSE, 1);  
  
  fields.notebook = notebook;
  fields.cart_radiob = cart_radiob;
  fields.param_radiob = param_radiob;
  fields.x_hbox = x_hbox;
  fields.y_hbox = y_hbox;
  fields.z_hbox = z_hbox;
  fields.x_entry = x_entry;
  fields.y_entry = y_entry;
  fields.z_entry = z_entry;
  fields.y_label = y_label;
  fields.z_label = z_label;
  fields.s_frame = s_frame;
  fields.t_frame = t_frame;
  fields.inf_s_entry = inf_s_entry;
  fields.sup_s_entry = sup_s_entry;
  fields.nsteps_s_entry = nsteps_s_entry;
  fields.inf_t_entry = inf_t_entry;
  fields.sup_t_entry = sup_t_entry;
  fields.nsteps_t_entry = nsteps_t_entry;
  fields.caption_entry = caption_entry;
  fields.dotted_radiob = dotted_radiob;
  fields.stroked_radiob = stroked_radiob;
  fields.filled_radiob = filled_radiob;
  fields.selection_combo_box = selection_combo_box;
  fields.long_spinb = long_spinb;
  fields.lat_spinb = lat_spinb;
  fields.rotangle_spinb = rotangle_spinb;
  fields.zoom_hscale = zoom_hscale;
  fields.smoke_hscale = smoke_hscale;
  fields.axes_checkb = axes_checkb;
  fields.grid_checkb = grid_checkb;
  fields.tics_checkb = tics_checkb;
  fields.jumps_checkb = jumps_checkb;
  fields.zimage_up_button = zimage_up_button;
  fields.zimage_down_button = zimage_down_button;
  fields.draw_area = draw_area;
}

void set_gui_properties (void)
{
  const int nobjects = MAX_NUMBER_OF_OBJECTS;
  mathobj_description desc;
  int i, n;
  gchar* lbl;
#ifdef _USE_GTK3_
  GdkRGBA white;
  GdkRGBA gold;
#else
  GdkColor white;
  GdkColor gold;
#endif

#ifdef _USE_GTK3_
  white.red = white.green = white.blue = white.alpha = 1.0;
  gold.red = 0.74;
  gold.green = 0.62;
  gold.blue = 0.09;
  gold.alpha = 1.0;
#else
  white.red = white.green = white.blue = 0xFFFF;
  gold.red = 0xBC00;
  gold.green = 0x9D00;
  gold.blue = 0x1600;
#endif

  desc = retrieve_math_object ();
  gtk_window_set_title (GTK_WINDOW(win), "GtkMathPlot");
  gtk_container_set_border_width (GTK_CONTAINER(win), 2);
  gtk_container_set_border_width (GTK_CONTAINER(vbox1), 5);
  gtk_container_set_border_width (GTK_CONTAINER(object_def_vbox), 5);
  gtk_container_set_border_width (GTK_CONTAINER(x_hbox), 0);
  gtk_container_set_border_width (GTK_CONTAINER(y_hbox), 0);
  gtk_container_set_border_width (GTK_CONTAINER(z_hbox), 0);
  gtk_container_set_border_width (GTK_CONTAINER(s_frame), 0);
  gtk_container_set_border_width (GTK_CONTAINER(s_hbox), 5);
  gtk_container_set_border_width (GTK_CONTAINER(t_frame), 0);
  gtk_container_set_border_width (GTK_CONTAINER(t_hbox), 5);
  gtk_container_set_border_width (GTK_CONTAINER(prop_hbox), 0);
  /*
    gtk_container_set_border_width (GTK_CONTAINER(central_hbox), 5);
    gtk_container_set_border_width (GTK_CONTAINER(bottom_hbox), 5);
  */
  gtk_frame_set_shadow_type (GTK_FRAME(empty_frame), GTK_SHADOW_NONE);
  gtk_frame_set_shadow_type (GTK_FRAME(empty_space_frame), GTK_SHADOW_NONE);
#ifdef _USE_GTK3_
  gtk_widget_override_background_color (display_button, GTK_STATE_FLAG_NORMAL, &gold);
  gtk_widget_override_background_color (display_button, GTK_STATE_FLAG_PRELIGHT, &gold);
#else
  gtk_widget_modify_bg (display_button, GTK_STATE_NORMAL, &gold);
  gtk_widget_modify_bg (display_button, GTK_STATE_PRELIGHT, &gold);
#endif
  gtk_container_set_border_width (GTK_CONTAINER(vbox2), 5);
  gtk_container_set_border_width (GTK_CONTAINER(obs_vbox), 5);
  gtk_container_set_border_width (GTK_CONTAINER(legend_frame), 0);
  gtk_container_set_border_width (GTK_CONTAINER(tr_button_box), 5);
#ifdef _USE_GTK3_
  gtk_box_set_spacing (GTK_BOX(tr_button_box), 5);
#else
  gtk_button_box_set_spacing (GTK_BUTTON_BOX(tr_button_box), 5);
#endif
  gtk_widget_set_size_request (draw_area, DEFAULT_WIDTH, DEFAULT_HEIGHT);
  gtk_widget_set_size_request (zoom_hscale, 120, 20);
  gtk_widget_set_size_request (smoke_hscale, 120, 20);

  gtk_event_box_set_visible_window (GTK_EVENT_BOX (legend_evbox), TRUE);
#ifdef _USE_GTK3_
  gtk_widget_override_background_color (legend_evbox, GTK_STATE_FLAG_NORMAL, &white);
#else
  gtk_widget_modify_bg (legend_evbox, GTK_STATE_NORMAL, &white);
#endif
  gtk_label_set_line_wrap (GTK_LABEL(legend_label), FALSE);

  gtk_entry_set_text (GTK_ENTRY(x_entry), (const gchar*)desc.x_eq);
  gtk_entry_set_text (GTK_ENTRY(y_entry), (const gchar*)desc.y_eq);
  /* z_entry is set below if it is active */
  gtk_entry_set_text (GTK_ENTRY(inf_s_entry), (const gchar*)desc.inf_s);
  gtk_entry_set_text (GTK_ENTRY(sup_s_entry), (const gchar*)desc.sup_s);
  gtk_entry_set_text (GTK_ENTRY(nsteps_s_entry), (const gchar*)desc.nsteps_s);
  gtk_entry_set_text (GTK_ENTRY(inf_t_entry), (const gchar*)desc.inf_t);
  gtk_entry_set_text (GTK_ENTRY(sup_t_entry), (const gchar*)desc.sup_t);
  gtk_entry_set_text (GTK_ENTRY(nsteps_t_entry), (const gchar*)desc.nsteps_t);
  gtk_entry_set_text (GTK_ENTRY(caption_entry), (const gchar*)desc.caption);

  if (fields.style == GRSTYLE_DOTTED)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dotted_radiob), TRUE);
  else if (fields.style == GRSTYLE_STROKED)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (stroked_radiob), TRUE);
  else
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (filled_radiob), TRUE);

  if (DEFAULT_GRAPHIC_MODE == GRAPHIC_2D)
    {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (twod_radiob), TRUE);
      gtk_widget_set_sensitive (z_entry, FALSE);
      gtk_widget_set_sensitive (long_spinb, FALSE);
      gtk_widget_set_sensitive (lat_spinb, FALSE);
      gtk_widget_set_sensitive (smoke_hscale, FALSE);
      gtk_button_set_label (GTK_BUTTON(axes_checkb), "x and y directions");
      gtk_widget_set_sensitive (zimage_up_button, FALSE);
      gtk_widget_set_sensitive (zimage_down_button, FALSE);
    }
  else
    {
      gtk_entry_set_text (GTK_ENTRY(z_entry), (const gchar*)desc.z_eq);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (threed_radiob), TRUE);
      gtk_widget_set_sensitive (jumps_checkb, FALSE);
    }

  if ((DEFAULT_SHOW_MAIN_DIRS))
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (axes_checkb), TRUE);
  else
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (axes_checkb), FALSE);

  if ((DEFAULT_SHOW_GRID))
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (grid_checkb), TRUE);
  else
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (grid_checkb), FALSE);

  if ((DEFAULT_SHOW_TICS))
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (tics_checkb), TRUE);
  else
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (tics_checkb), FALSE);

  if ((DEFAULT_SHOW_DISC))
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (jumps_checkb), TRUE);
  else
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (jumps_checkb), FALSE);

  for (n = 0; n < nobjects; n++)
    {
      lbl = g_strdup_printf ("Mathematical shape no. %d", n + 1);
#ifdef _USE_GTK3_
      gtk_combo_box_text_insert (GTK_COMBO_BOX_TEXT(selection_combo_box), n, NULL, lbl);
#else
      gtk_combo_box_insert_text (GTK_COMBO_BOX(selection_combo_box), n, lbl);
#endif
      g_free (lbl);
    }
  gtk_combo_box_set_active (GTK_COMBO_BOX(selection_combo_box),
			    DEFAULT_CURRENT_OBJECT);
  gtk_window_set_default_size (GTK_WINDOW(help_dialog), 790, 400);
  monospaced_font_14 = pango_font_description_from_string ("monospace 14");
  for (i =0; i < helpfiles_no; i++)
    {
      gtk_text_view_set_editable (GTK_TEXT_VIEW(help_text[i]), FALSE);
#ifdef _USE_GTK3_
      gtk_widget_override_font (help_text[i], monospaced_font_14);
#else
      gtk_widget_modify_font (help_text[i], monospaced_font_14);
#endif      
      gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(scrolled_win[i]),
				      GTK_POLICY_AUTOMATIC,
				      GTK_POLICY_AUTOMATIC);		
#ifdef _USE_GTK3_
      gtk_scrolled_window_set_min_content_width (GTK_SCROLLED_WINDOW(scrolled_win[i]),
						  750);
      gtk_scrolled_window_set_min_content_height (GTK_SCROLLED_WINDOW(scrolled_win[i]),
						  300);
#endif
    }

#ifdef _USE_GTK3_
  gtk_grid_set_column_homogeneous (GTK_GRID(color_2dsettings_table), TRUE);
  gtk_grid_set_column_homogeneous (GTK_GRID(symbols_2dsettings_table), TRUE);
  gtk_grid_set_column_homogeneous (GTK_GRID(color_3dsettings_table), TRUE);
  gtk_grid_set_column_homogeneous (GTK_GRID(symbols_3dsettings_table), TRUE);
  gtk_grid_set_row_spacing (GTK_GRID(color_2dsettings_table), 0);
  gtk_grid_set_row_spacing (GTK_GRID(symbols_2dsettings_table), 0);
  gtk_grid_set_row_spacing (GTK_GRID(color_3dsettings_table), 0);
  gtk_grid_set_row_spacing (GTK_GRID(symbols_3dsettings_table), 0);
#endif

  gtk_container_set_border_width (GTK_CONTAINER(paperfmt_frame), 5);
  for (n = 0; n < NSUPPORTED_PAPER_FORMATS; n++)
    {
#ifdef _USE_GTK3_
      gtk_combo_box_text_insert (GTK_COMBO_BOX_TEXT(paperfmt_combo_box), n,
				 NULL, supported_paper_formats[n].name);
#else
      gtk_combo_box_insert_text (GTK_COMBO_BOX(paperfmt_combo_box), n, 
				 supported_paper_formats[n].name);
#endif
    }
  gtk_combo_box_set_active (GTK_COMBO_BOX(paperfmt_combo_box),
			    DEFAULT_PAPER_FORMAT);

  gtk_container_set_border_width (GTK_CONTAINER(orientation_frame), 5);
  for (n = 0; n < NSUPPORTED_ORIENTATIONS; n++)
    {
#ifdef _USE_GTK3_
      gtk_combo_box_text_insert (GTK_COMBO_BOX_TEXT(orientation_combo_box), n, 
				 NULL, supported_orientations[n]);
#else
      gtk_combo_box_insert_text (GTK_COMBO_BOX(orientation_combo_box), n, 
				 supported_orientations[n]);
#endif
    }
  gtk_combo_box_set_active (GTK_COMBO_BOX(orientation_combo_box),
			    DEFAULT_ORIENTATION);
  gtk_widget_set_sensitive (orientation_combo_box, FALSE);
  /* This widget is going to be made sensitive again after the
     Landscape-bug will have been removed from PS and PDF drivers
     of the Cairo library */

  gtk_container_set_border_width (GTK_CONTAINER(color_2dsettings_frame), 5);
  gtk_container_set_border_width (GTK_CONTAINER(symbols_2dsettings_frame), 5);
  for (i = 0; i < NUMBER_OF_2DOBJECTS + 4; i++)
    gtk_container_set_border_width (GTK_CONTAINER(color_2dsettings_miniframe[i]), 3);
  for (i = 0; i < NUMBER_OF_2DOBJECTS; i++)
    gtk_container_set_border_width (GTK_CONTAINER(symbols_2dsettings_miniframe[i]), 3);

  gtk_container_set_border_width (GTK_CONTAINER(color_3dsettings_frame), 5);
  gtk_container_set_border_width (GTK_CONTAINER(symbols_3dsettings_frame), 5);
  for (i = 0; i < NUMBER_OF_3DOBJECTS + 7; i++)
    gtk_container_set_border_width (GTK_CONTAINER(color_3dsettings_miniframe[i]), 3);
  for (i = 0; i < NUMBER_OF_3DOBJECTS; i++)
    gtk_container_set_border_width (GTK_CONTAINER(symbols_3dsettings_miniframe[i]), 3);

  for (i = 0; i < NUMBER_OF_2DOBJECTS + 4; i++)
    {
      for (n = Black; n < Nr_Colors_2d; n++)
	{
#ifdef _USE_GTK3_
	  gtk_combo_box_text_insert (GTK_COMBO_BOX_TEXT(color_2dsettings_combo_box[i]), 
				     n, NULL, get_color_2d_name(n));
#else
	  gtk_combo_box_insert_text (GTK_COMBO_BOX(color_2dsettings_combo_box[i]), 
				     n, get_color_2d_name(n));
#endif
	}
      gtk_combo_box_set_active (GTK_COMBO_BOX(color_2dsettings_combo_box[i]),
				get_default_color_2d(i));
    }
  for (i = 0; i < NUMBER_OF_3DOBJECTS + 7; i++)
    {
      for (n = Red; n < Nr_Colors_3d; n++)
	{
#ifdef _USE_GTK3_
	  gtk_combo_box_text_insert (GTK_COMBO_BOX_TEXT(color_3dsettings_combo_box[i]), 
				     n, NULL, get_color_3d_name(n));
#else
	  gtk_combo_box_insert_text (GTK_COMBO_BOX(color_3dsettings_combo_box[i]), 
				     n, get_color_3d_name(n));
#endif
	}
      gtk_combo_box_set_active (GTK_COMBO_BOX(color_3dsettings_combo_box[i]),
				get_default_color_3d(i));
    }

  for (i = 0; i < NUMBER_OF_2DOBJECTS; i++)
    {
      for (n = 0; n < Nr_Symbols; n++)
	{
#ifdef _USE_GTK3_
	  gtk_combo_box_text_insert (GTK_COMBO_BOX_TEXT(symbols_2dsettings_combo_box[i]),
				     n, NULL, get_symbol_name(n));
#else
	  gtk_combo_box_insert_text (GTK_COMBO_BOX(symbols_2dsettings_combo_box[i]),
				     n, get_symbol_name(n));
#endif
	}
      gtk_combo_box_set_active (GTK_COMBO_BOX(symbols_2dsettings_combo_box[i]),
  				get_default_symbol(i));
    }
  for (i = 0; i < NUMBER_OF_3DOBJECTS; i++)
    {
      for (n = 0; n < Nr_Symbols; n++)
	{
#ifdef _USE_GTK3_
	  gtk_combo_box_text_insert (GTK_COMBO_BOX_TEXT(symbols_3dsettings_combo_box[i]),
				     n, NULL, get_symbol_name(n));
#else
	  gtk_combo_box_insert_text (GTK_COMBO_BOX(symbols_3dsettings_combo_box[i]),
				     n, get_symbol_name(n));
#endif
	}
      gtk_combo_box_set_active (GTK_COMBO_BOX(symbols_3dsettings_combo_box[i]),
  				get_default_symbol(i));
    }

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(addinfo_checkb), TRUE);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(addC_checkb), TRUE);
}

void load_help_file (void)
{
  int i;
  gchar* contents;
  gsize len;
  GError* error;

  for (i = 0; i < helpfiles_no; i++)
    {
      error = NULL;
      if ( g_file_get_contents (helpfiles_path[i], 
				&contents, &len, &error) == FALSE )
	{
	  contents = g_strdup_printf ("Could not find file\n%s\n",
				      helpfiles_path[i]);
	  gtk_text_buffer_insert_at_cursor (help_text_buffer[i], contents, 
					    (gint)strlen(contents));
	}
      else
	gtk_text_buffer_insert_at_cursor (help_text_buffer[i], 
					  contents, (gint)len);
      g_free (contents);
    }
}

void show_warning_message (const gchar* warning_message)
{
  gtk_label_set_text (GTK_LABEL(warning_label), warning_message);
  gtk_widget_show_all (warning_dialog);
}

void handle_fatal_error (const gchar* error_message)
{
  gtk_label_set_text (GTK_LABEL(error_label), error_message);
  gtk_widget_show_all (error_dialog);
}

void issue_redraw_command (void)
{
  gtk_widget_queue_draw (draw_area);
}

#ifdef _USE_GTK3_
#define gtk_widget_hide_all gtk_widget_hide
#define GTK_SIGNAL_FUNC(f) G_CALLBACK(f)
#endif

static gboolean delete_event_causes_hiding (GtkWidget* widget, GdkEvent* event, gpointer data)
{
  gtk_widget_hide_all (widget);
  return TRUE;
}

void bind_callbacks (void)
{
  int i;

  g_signal_connect (G_OBJECT(win), "delete-event", GTK_SIGNAL_FUNC(on_close), NULL);
  g_signal_connect (G_OBJECT(win), "destroy", G_CALLBACK(gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT(quit_button), "clicked", G_CALLBACK(gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT(quit_menu_item), "activate", G_CALLBACK(gtk_main_quit), NULL);
  g_signal_connect_swapped (G_OBJECT(pref_menu_item), "activate", G_CALLBACK(gtk_widget_show_all), settings_dialog);
  g_signal_connect_swapped (G_OBJECT(contents_menu_item), "activate", G_CALLBACK(gtk_widget_show_all), help_dialog);
  g_signal_connect_swapped (G_OBJECT(license_menu_item), "activate", G_CALLBACK(gtk_widget_show_all), license_dialog);
  g_signal_connect_swapped (G_OBJECT(about_menu_item), "activate", G_CALLBACK(gtk_widget_show_all), about_dialog);
  g_signal_connect (G_OBJECT(help_dialog), "response", G_CALLBACK(gtk_widget_hide_all), 
		    NULL);
  g_signal_connect (G_OBJECT(license_dialog), "response", G_CALLBACK(gtk_widget_hide_all), 
		    NULL);
  g_signal_connect (G_OBJECT(about_dialog), "response", G_CALLBACK(gtk_widget_hide_all), 
		    NULL);
  g_signal_connect (G_OBJECT(warning_dialog), "response", G_CALLBACK(gtk_widget_hide_all), 
		    NULL);
  g_signal_connect (G_OBJECT(error_dialog), "response", G_CALLBACK(gtk_main_quit), 
		    NULL);
  g_signal_connect (G_OBJECT(settings_dialog), "response", G_CALLBACK(gtk_widget_hide_all), 
		    NULL);  

  g_signal_connect (G_OBJECT(help_dialog), "delete-event", G_CALLBACK(delete_event_causes_hiding), 
		    NULL);
  g_signal_connect (G_OBJECT(license_dialog), "delete-event", G_CALLBACK(delete_event_causes_hiding),
		    NULL);
  g_signal_connect (G_OBJECT(about_dialog), "delete-event", G_CALLBACK(delete_event_causes_hiding), 
		    NULL);
  g_signal_connect (G_OBJECT(warning_dialog), "delete-event", G_CALLBACK(delete_event_causes_hiding),
		    NULL);
  g_signal_connect (G_OBJECT(error_dialog), "delete-event", G_CALLBACK(gtk_main_quit),
		    NULL);
  g_signal_connect (G_OBJECT(settings_dialog), "delete-event", G_CALLBACK(delete_event_causes_hiding),
		    NULL);  

  g_signal_connect (G_OBJECT(help_dialog), "close", G_CALLBACK(delete_event_causes_hiding), 
		    NULL);
  g_signal_connect (G_OBJECT(license_dialog), "close", G_CALLBACK(delete_event_causes_hiding),
		    NULL);
  g_signal_connect (G_OBJECT(about_dialog), "close", G_CALLBACK(delete_event_causes_hiding), 
		    NULL);
  g_signal_connect (G_OBJECT(warning_dialog), "close", G_CALLBACK(delete_event_causes_hiding),
		    NULL);
  g_signal_connect (G_OBJECT(error_dialog), "close", G_CALLBACK(gtk_main_quit),
		    NULL);
  g_signal_connect (G_OBJECT(settings_dialog), "close", G_CALLBACK(delete_event_causes_hiding),
		    NULL);  

  /*
    Now we bind the callbacks which are relevant for the graphic stuff 
  */
  g_signal_connect (G_OBJECT(twod_radiob), "toggled", G_CALLBACK(on_set_2dmode), &fields);
  g_signal_connect (G_OBJECT(threed_radiob), "toggled", G_CALLBACK(on_set_3dmode), &fields);
  g_signal_connect (G_OBJECT(selection_combo_box), "changed", G_CALLBACK(on_changed_selection), &fields);
  g_signal_connect (G_OBJECT(cart_radiob), "toggled", G_CALLBACK(on_set_cartesian), &fields);
  g_signal_connect (G_OBJECT(param_radiob), "toggled", G_CALLBACK(on_set_parametric), &fields);
  g_signal_connect (G_OBJECT(dotted_radiob), "clicked", G_CALLBACK(on_set_dotted_style), &fields.style);
  g_signal_connect (G_OBJECT(stroked_radiob), "clicked", G_CALLBACK(on_set_stroked_style), &fields.style);
  g_signal_connect (G_OBJECT(filled_radiob), "clicked", G_CALLBACK(on_set_filled_style), &fields.style);
  g_signal_connect (G_OBJECT(loaddef_button), "clicked", G_CALLBACK(on_load_definition), &fields);
  g_signal_connect (G_OBJECT(savedef_button), "clicked", G_CALLBACK(on_save_definition), &fields);
  g_signal_connect (G_OBJECT(eval_save_button), "clicked", G_CALLBACK(on_save_evaluation), &fields);
  g_signal_connect (G_OBJECT(clear_button), "clicked", G_CALLBACK(on_clear_definition), &fields);
  g_signal_connect (G_OBJECT(display_button), "clicked", G_CALLBACK(on_display), &fields);

#ifdef _USE_GTK3_
  g_signal_connect (G_OBJECT(draw_area), "draw", GTK_SIGNAL_FUNC(on_draw), NULL);  
#else
  g_signal_connect (G_OBJECT(draw_area), "expose_event", GTK_SIGNAL_FUNC(on_draw), NULL);
#endif

  g_signal_connect (G_OBJECT(long_spinb), "value-changed", G_CALLBACK (on_set_longitude), draw_area);
  g_signal_connect (G_OBJECT(lat_spinb), "value-changed", G_CALLBACK (on_set_latitude), draw_area);
  g_signal_connect (G_OBJECT(rotangle_spinb), "value-changed", G_CALLBACK (on_set_rotation_angle), draw_area);
  g_signal_connect (G_OBJECT(axes_checkb), "toggled", G_CALLBACK(on_toggle_show_main_dirs), draw_area);
  g_signal_connect (G_OBJECT(grid_checkb), "toggled", G_CALLBACK(on_toggle_show_grid), draw_area);
  g_signal_connect (G_OBJECT(tics_checkb), "toggled", G_CALLBACK(on_toggle_show_tics), draw_area);
  g_signal_connect (G_OBJECT(jumps_checkb), "toggled", G_CALLBACK(on_toggle_show_discontinuities), draw_area);
  g_signal_connect (G_OBJECT(zoom_hscale), "value-changed", G_CALLBACK(on_set_zoom_factor), &fields);
  g_signal_connect (G_OBJECT(smoke_hscale), "value-changed", G_CALLBACK(on_set_smoke_density), draw_area);
    
  g_signal_connect (G_OBJECT(load_button), "clicked", G_CALLBACK(on_load_image), draw_area);
  g_signal_connect (G_OBJECT(save_button), "clicked", G_CALLBACK(on_save_image), win);
  g_signal_connect (G_OBJECT(back_button), "clicked", G_CALLBACK(on_back_input_mask), notebook);
  
  g_signal_connect_swapped (G_OBJECT(ximage_up_button), "button_press_event", GTK_SIGNAL_FUNC (on_ximage_up), draw_area);
  g_signal_connect_swapped (G_OBJECT(ximage_down_button), "button_press_event", GTK_SIGNAL_FUNC (on_ximage_down), draw_area);
  g_signal_connect_swapped (G_OBJECT(yimage_up_button), "button_press_event", GTK_SIGNAL_FUNC (on_yimage_up), draw_area);
  g_signal_connect_swapped (G_OBJECT(yimage_down_button), "button_press_event", GTK_SIGNAL_FUNC (on_yimage_down), draw_area);
  g_signal_connect_swapped (G_OBJECT(zimage_up_button), "button_press_event", GTK_SIGNAL_FUNC (on_zimage_up), draw_area);
  g_signal_connect_swapped (G_OBJECT(zimage_down_button), "button_press_event", GTK_SIGNAL_FUNC (on_zimage_down), draw_area);
  g_signal_connect (G_OBJECT(reset_button), "clicked", G_CALLBACK (on_reset_imagepos), draw_area);

  g_signal_connect (G_OBJECT(addinfo_checkb), "toggled", G_CALLBACK(on_toggle_print_addinfo), NULL);
  g_signal_connect (G_OBJECT(addC_checkb), "toggled", G_CALLBACK(on_toggle_print_disclaimer), NULL);  
  g_signal_connect (G_OBJECT(paperfmt_combo_box), "changed", G_CALLBACK(on_set_paper_format), NULL);
  g_signal_connect (G_OBJECT(orientation_combo_box), "changed", G_CALLBACK(on_set_orientation), NULL);
  for (i = 0; i < NUMBER_OF_2DOBJECTS+4; i++)
    g_signal_connect (G_OBJECT(color_2dsettings_combo_box[i]), "changed",
		      G_CALLBACK(on_set_color_2d), GINT_TO_POINTER(i));
  for (i = 0; i < NUMBER_OF_3DOBJECTS+7; i++)
    g_signal_connect (G_OBJECT(color_3dsettings_combo_box[i]), "changed",
		      G_CALLBACK(on_set_color_3d), GINT_TO_POINTER(i));

  for (i = 0; i < NUMBER_OF_2DOBJECTS; i++)
    g_signal_connect (G_OBJECT(symbols_2dsettings_combo_box[i]), "changed",
		      G_CALLBACK(on_set_symbol_2d), GINT_TO_POINTER(i));
  for (i = 0; i < NUMBER_OF_3DOBJECTS; i++)
    g_signal_connect (G_OBJECT(symbols_3dsettings_combo_box[i]), "changed",
		      G_CALLBACK(on_set_symbol_3d), GINT_TO_POINTER(i));
}

void set_legend (void)
{
  gchar* legend = get_formatted_captions ();

  if (!legend)
    {
      handle_fatal_error ("The program has not enough memory to keep on working\n" \
			  "please quit now.\n");      
    }
  else
    {
      gtk_label_set_markup (GTK_LABEL(legend_label), legend);
      g_free (legend);
    }
}

void switch_2dgmode_button_on(void)
{
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(twod_radiob), TRUE);
}

void switch_3dgmode_button_on(void)
{
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(threed_radiob), TRUE);
}

void show_gui (void)
{
  gtk_widget_show_all (win);
  if (DEFAULT_REPR_TYPE == CARTESIAN)
    {
      if (DEFAULT_GRAPHIC_MODE == GRAPHIC_2D)
	{
	  gtk_widget_hide_all (x_hbox);
	  gtk_widget_hide_all (z_hbox);	  
	  gtk_label_set_text (GTK_LABEL(y_label), "y(x) =   ");
	  gtk_widget_hide_all (s_frame);
	  gtk_frame_set_label (GTK_FRAME(t_frame), "Variation Interval of x");	 
	}
      else
	{
	  gtk_widget_hide_all (x_hbox);
	  gtk_widget_hide_all (y_hbox);	  
	  gtk_label_set_text (GTK_LABEL(z_label), "z(x,y) = ");
	  gtk_frame_set_label (GTK_FRAME(s_frame), "Variation Interval of x");
	  gtk_frame_set_label (GTK_FRAME(t_frame), "Variation Interval of y");
	}
    }
}
