/*
    GTKmathplot - a simple GTK+ based program
    to plot mathematical functions.
    Copyright (C) 2012, 2013  Ivano Primi  <ivprimi@libero.it>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef __GR3D__
#define __GR3D__

#include<stdio.h> /* for FILE */
#include"gr.h"

typedef struct {
  double x, y, z; /* Coordinates of a point in a 3D space */
} gr3d_point;

typedef struct {
  grtype type;
  grint  color;       /* This can be used to mark the single basic shapes */
                      /* as parts of a bigger shape                       */ 
  grchar* caption;    /* A caption to be used as identifier               */
  gr3d_point A, B, C; /* vertices. If type is POINT, then only A is used. */
                      /* If type is SEGMENT, then only A and B are used.  */
                      /* Not used vertices are set to {0.0, 0.0, 0.0}.    */

  gr3d_point M;  /* middle point, always computed from the vertices. */
                 /* M must be handled as a private field.            */

  double length; /* The length or perimeter of the shape             */

  double depth;  /* depth of the shape in the z buffer.              */
                 /* This must be handled as a private field.         */

  double XA, YA; /* Coordinates of A on the plane orthogonal to the z-buffer */
  double XB, YB; /* Coordinates of B on the plane orthogonal to the z-buffer */
  double XC, YC; /* Coordinates of C on the plane orthogonal to the z-buffer */
                 /* These must be handled as private fields.                 */
} gr3d_basic_shape;

typedef struct {
  gr3d_basic_shape** list;     /* This is an array of pointers to shapes */
                     /* If LIST == NULL, we say that the object is empty */
  grint len, size; /* LEN is the number of shapes stored in the array,
                        SIZE is the number of available slots in the array */ 
  double R;          /* The radius of the smallest sphere centered at
                        list[0]->M which contains the whole object, or
			alternatively all the shapes of the object whose length
			is below a given threshold, see gr3d_compute_bounding_sphere().      
                        After the creation of an object this is set to 0.0 
                        and can only be changed by calling 
                        gr3d_compute_bounding_sphere() on the object.      */
  /*
	 After adding one or more shapes to an object by using gr3d_add_*(),
	 LIST[0]->M.x, LIST[0]->M.y and LIST[0]->M.z  are the coordinates
	 of the middle point of the object, LIST[0]->length is the mean
	 length of the shapes forming the object. They are both computed
	 by considering only the shapes whose COLOR lies in the range
	 FIRST_COLOR..LAST_COLOR.
  */
  grint first_color, last_color;

  /*
    Number of the shapes of the object whose COLOR lies in the range
    FIRST_COLOR..LAST_COLOR.
  */
  grint rlen;
} gr3d_object; 



/*
  Return 1 if P.x, P.y and P.z are all finite values, otherwise 0. 
*/
grint gr3d_point_is_finite (gr3d_point P);

/* 
   Create a new gr3d object with FIRST_COLOR = COLOR1 and
   LAST_COLOR = COLOR2. Return an empty object if no memory is available 
   to create the object.

   Rem.: The new object has R = 0.0, RLEN = LEN = 0, SIZE = 1 and LIST 
         is formed by just one slot, LIST[0]. 
	 This is a pointer to a shape with TYPE = GR_NONE,
	 COLOR = 0, CAPTION = NULL,
	 A.x = A.y = A.z = B.x = B.y = B.z = C.x = C.y = C.z = 0.0,
	 M.x = M.y = M.z = 0.0, depth = length = 0.0, and
	 XA = YA = XB = YB = XC = YC = 0.0.  
*/
gr3d_object gr3d_new (grint color1, grint color2);

/*
  Return 1 if OBJ is an empty object, otherwise 0.

  Remark: empty means that OBJ.LIST == NULL.
*/
int gr3d_is_empty (gr3d_object obj);

/* 
   Add a triangle to an existing gr3d object. Return the new number
   of shapes forming the object, or -1 if it was not possible
   to add the new shape due to memory issues. Return -1 also
   if POBJ == NULL or *POBJ is an empty object.

   Rem.: POBJ is a pointer to an existing object, the other parameters
         are the coordinates of the three vertices A, B and C of the
	 triangle to be added.
*/
grint gr3d_add_triangle (gr3d_object* pobj, 
			 grint color,
			 const grchar* caption,
			 double xA, double yA, double zA,
			 double xB, double yB, double zB,
			 double xC, double yC, double zC);

/* 
   Add a segment to an existing gr3d object. Return the new number
   of shapes forming the object, or -1 if it was not possible
   to add the new shape due to memory issues. Return -1 also
   if POBJ == NULL or *POBJ is an empty object.

   Rem.: POBJ is a pointer to an existing object, the other parameters
         are the coordinates of the vertices A, B of the
	 segment to be added.
*/
grint gr3d_add_segment (gr3d_object* pobj, 
			grint color,
			const grchar* caption,
			double xA, double yA, double zA,
			double xB, double yB, double zB);

/* 
   Add a point to an existing gr3d object. Return the new number
   of shapes forming the object, or -1 if it was not possible
   to add the new shape due to memory issues. Return -1 also
   if POBJ == NULL or *POBJ is an empty object.

   Rem.: POBJ is a pointer to an existing object, the other parameters
         are the coordinates of the point to be added.
*/
grint gr3d_add_point (gr3d_object* pobj, 
		      grint color,			
		      const grchar* caption,
		      double x, double y, double z);

/* 
   Reorder the shapes of a given object according to the order
   they are seen from an observer placed very far away in the direction
   
   d := 
   (cos(longitude)*cos(latitude), sin(longitude)*cos(latitude), sin(latitude))

   In addition, compute the coordinates of the projections
   of the points forming the object on the plane orthogonal to d 
   and passing through the point O.
   These coordinates are computed with respect to the ortogonal
   system defined by the rotation angle "rot_angle".

   Rem.: POBJ should be a pointer to an existing object. 
         Do nothing if POBJ == NULL or *POBJ is an empty object.
*/
void gr3d_view_from_direction (gr3d_object* pobj, gr3d_point O,
			       double longitude, double latitude, 
			       double rot_angle); 

/*
  Sort (POBJ->LIST[1],...,POBJ->LIST[POBJ->LEN])
  in ascending order according to the value of the depth field.
  Do nothing if POBJ == NULL or *POBJ is an empty object.

   Rem.: POBJ should be a pointer to an existing object. 
*/
void gr3d_sort_shapes (gr3d_object* pobj);

/*
  Compute depth and coordinates of the points of the basic shapes marked by COLOR
  for the object pointed to by POBJ with respect to the orthogonal reference system 
  having the point O as origin and defined by the angles
  LONGITUDE, LATITUDE and ROT_ANGLE.

  Rem.: POBJ should be a pointer to an existing object. 
        Nothing is done if POBJ == NULL or *POBJ is an empty object.

        This function and gr3d_sort_shapes() are useful whenever 
	the basic shapes marked by COLOR have been removed using gr3d_remove_items_with_color()
	and then new shapes with the same color have been added
	by means of gr3d_add*(). If this is the case, the couple
	gr3d_partial_upgrade() + gr3d_sort_shapes()
	allows to save time compared to gr3d_view_from_direction().
*/
void gr3d_partial_upgrade (gr3d_object* pobj, grint color,
			   gr3d_point O, double longitude, double latitude, 
			   double rot_angle);

/*
  If THRESHOLD >= 0, then set POBJ->R to the radius of the smallest 
  sphere centered at (POBJ->LIST[0])->M which contains all the shapes 
  of the object pointed to by POBJ whose length does not exceed the 
  given THRESHOLD and whose color lies in the range
  POBJ->FIRST_COLOR..POBJ->LAST_COLOR.
  If THRESHOLD < 0, simply set POBJ->R to the radius of the smallest 
  sphere centered at (POBJ->LIST[0])->M which contains all the shapes 
  of the object pointed to by POBJ whose color lies in the range
  POBJ->FIRST_COLOR..POBJ->LAST_COLOR.

  Rem.: POBJ should be a pointer to an existing object. 
        Do nothing if POBJ == NULL or *POBJ is an empty object.
*/
void gr3d_compute_bounding_sphere (gr3d_object* pobj, double threshold);

/*
  Return the middle point of the object OBJ, i.e. OBJ.LIST[0]->M.

  Remark: if OBJ is an empty object, then {0.0, 0.0, 0.0} is returned.
*/
gr3d_point gr3d_get_middle_point (gr3d_object obj);

/*
  If FACTOR z <= 0.0, then return the mean length of the object
  OBJ, i.e. OBJ.LIST[0]->LENGTH.
  Otherwise, return the mean length of all the shapes from OBJ
  whose length does not exceed FACTOR * OBJ.LIST[0]->LENGTH and
  whose color lies in the range OBJ.FIRST_COLOR..OBJ.LAST_COLOR.

  Remark: if OBJ is an empty object, then 0.0 is returned.
*/
double gr3d_get_mean_length (gr3d_object obj, double factor);

/*
  Return the mean length of all basic shapes from OBJ marked
  by COLOR. If OBJ is an empty object, return zero.
*/
double gr3d_get_mean_length_by_color (gr3d_object obj, grint color);

/*
  Print to the file pointed to by FP the shapes forming the
  object OBJ. Return the number of printed shapes in case of
  success (0 if OBJ is an empty object), -1 in case of failure.
*/
grint gr3d_print (gr3d_object obj, FILE* fp);

/*
  Save in the file pointed to by FP the shapes forming the
  object OBJ. Return the number of saved shapes in case of
  success (0 if OBJ is an empty object), -1 in case of failure.
*/
grint gr3d_save (gr3d_object obj, FILE* fp);

/*
  Load an object from the file pointed to by FP and
  stores it in the structure pointed to by POBJ.
  Return the number of loaded shapes in case of
  success, -1 in case of failure (for example,
  if POBJ == NULL or the contents of the file
  could not be completely read).

  Remark: POBJ should be a pointer to an existing object. 
          The shapes read from file are appended to this object.
*/
grint gr3d_load (gr3d_object* pobj, FILE* fp);

/*
  Remove from the object pointed to by POBJ all basic shapes
  marked by COLOR whose length/perimeter is greater than
  THRESHOLD. If TRASHCAN != NULL and *TRASHCAN is not an empty object,
  put the removed shapes in the object pointed to by TRASHCAN.
  Do nothing but return 0 if POBJ == NULL or *POBJ is an empty object.

  Return the number of shapes forming the object pointed to by TRASHCAN
  if TRASHCAN != NULL , *TRASHCAN is not an empty object, and the shapes
  removed from *POBJ have been successfully added to *TRASHCAN, otherwise
  return either 0, to mean that no shape has been actually added to *TRASHCAN,
  or -1, to indicate a failure while adding shapes to *TRASHCAN.

  Remarks: POBJ should be a pointer to an existing object.
           This function recomputes (POBJ->LIST[0]).M,
           (POBJ->LIST[0]).LENGTH and POBJ->RLEN if necessary.
	   If a memory failure occurs while trying to add new shapes to
	   the object pointed to by TRASHCAN, then this function
           immediately stops and return -1.
*/
grint gr3d_remove_items_with_color (gr3d_object* pobj, 
				    grint color, double threshold,
				    gr3d_object* trashcan);

/*
  Append to the object pointed to by POBJ the basic shapes from OBJ.
  Return the new number of shapes forming the object pointed to by POBJ.
  Do nothing if POBJ == NULL, *POBJ is an empty object, or OBJ is empty.
  In this case returns -1.

  Remark: POBJ should be a pointer to an existing object.
          If the number returned is less than the sum of OBJ.LEN
	  with the initial value of POBJ->LEN, then a memory
	  failure occurred while trying to add a new shape.
*/
grint gr3d_add_object (gr3d_object* pobj, gr3d_object obj);

/*
  Delete the object pointed to by POBJ. Do nothing if POBJ == NULL
  or *POBJ is an empty object.

  Rem.: POBJ->FIRST_COLOR and POBJ->LAST_COLOR are left unchanged,
        all other fields are zeroed.
*/
void gr3d_delete (gr3d_object* pobj);

#endif /* __GR3D__ */
