/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 


#ifndef _G_PERMS_H_
#define _G_PERMS_H_

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>

#include <glib.h>
#include <glib-object.h>

#ifdef HAVE_LIBACL
#	include <acl/libacl.h>
#endif

typedef struct _GPermsEntry GPermsEntry;
typedef struct _GPermsSpecialEntry GPermsSpecialEntry;
typedef enum _GPermsType GPermsType;
typedef enum _GPermsAttribut GPermsAttribut;
typedef enum _GPermsMode GPermsMode;



//Containe one entrie wich define the permissions
struct _GPermsEntry
{
	gint      type;
	guint     id;
	gchar    *name;
	gboolean  read;
	gboolean  write;
	gboolean  exec;
};

//
struct _GPermsSpecialEntry
{
	gboolean  suid;
	gboolean  sgid;
	gboolean  sticky;
};


//
enum _GPermsAttribut
{
	P_SUID    = 32,
	P_SGID    = 16,
	P_STICKY  = 8,
	P_READ    = 4,
	P_WRITE   = 2,
	P_EXECUTE = 1
};



//Type of an entry
enum _GPermsType 
{
	P_USER_OWNER  = 1,
	P_USER = 2,
	P_GROUP_OWNER = 4,
	P_GROUP = 8,
	P_OTHER = 16,
	P_MASK  = 32
};

//
typedef enum _GPermsWalk 
{
	#ifdef HAVE_LIBACL
	P_FIRST_ENTRY = ACL_FIRST_ENTRY,
	P_NEXT_ENTRY  = ACL_NEXT_ENTRY
	#else
	P_FIRST_ENTRY = 1,
	P_NEXT_ENTRY  = 2
	#endif
} GPermsWalk;

//
enum _GPermsMode
{
	#ifdef HAVE_LIBACL
	P_ACCESS  = ACL_TYPE_ACCESS,	
	P_DEFAULT = ACL_TYPE_DEFAULT,
	#else
	P_ACCESS,	
	P_DEFAULT,
	#endif
	
	P_UNIX,
	P_ACL
};


typedef struct _GPerms GPerms;
typedef struct _GPermsClass GPermsClass;

//
struct _GPerms 
{
	GObject parent_instance;

	#ifdef HAVE_LIBACL
	acl_t acl;
	acl_entry_t acl_entry;
	#endif
	
	
	//
	guint nb_users, nb_groups, mask;
	
	
	gint unix_entry_ptr;
	GPermsMode mode;	//P_ACL or P_UNIX
	
	GPermsMode  acl_type;	//P_ACCESS, P_DEFAULT
	
	gboolean is_directory;

	
	GPermsEntry *(*func_get_entry) (GPerms *, const gint);
	gint (*func_set_perm) (GPerms *, const gint, const guint, gint, const gboolean);

	
	/***** PUBLIC ****/
	
	gchar *filename;
	GPermsEntry *entry;	//Current Entry
};


struct _GPermsClass
{
	GObjectClass parent_class;
	
	//void (* toggle_read) (GPerms *g_perms);
	void (* add_user)  (GPerms *g_perms);
	void (* del_user)  (GPerms *g_perms);
	void (* add_group) (GPerms *g_perms);
	void (* del_group) (GPerms *g_perms);
	void (* init_default)    (GPerms *g_perms);
	void (* clear_default)  (GPerms *g_perms);
	void (* set_user)  (GPerms *g_perms);
	void (* set_group)  (GPerms *g_perms);
};


#define TYPE_G_PERMS (g_perms_get_type())
#define G_PERMS(object) (G_TYPE_CHECK_INSTANCE_CAST ((object), TYPE_G_PERSM, GPerms))
#define G_PERMS_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), TYPE_G_PERMS, GPermsClass))
#define IS_G_PERMS(object) (G_TYPE_CHECK_INSTANCE_TYPE ((object), TYPE_G_PERMS))
#define IS_TEST_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), TYPE_TEST))
#define G_PERMS_GET_CLASS(object) (G_TYPE_INSTANCE_GET_CLASS ((object), TYPE_G_PERMS, GPermsClass))

GType g_perms_get_type();


GPerms *g_perms_new(void);


void g_perms_delete(GPerms *self);
gboolean g_perms_init(GPerms *self);
void g_perms_end(GPerms *self);

//
gint g_perms_load_file(GPerms *g_perms, const gchar *filename, const GPermsMode mode);

//
GPermsEntry *g_perms_get_entry(GPerms *g_perms, const GPermsType type);

#ifdef HAVE_LIBACL
acl_entry_t g_perms_found_entry(GPerms *g_perms, const gint type , const guint id);
#endif

gint g_perms_add_user (GPerms *g_perms, const uid_t uid);
gint g_perms_add_group(GPerms *g_perms, const gid_t gid);
gint g_perms_del_user (GPerms *g_perms, const gid_t gid);
gint g_perms_del_group(GPerms *g_perms, const gid_t gid);

gboolean g_perms_set_perm(GPerms *g_perms, const GPermsType type, const guint id, 
	const GPermsAttribut attr, const gboolean state);

gint g_perms_set_user_owner (GPerms *g_perms, uid_t uid);
gint g_perms_set_group_owner(GPerms *g_perms, gid_t gid);

GPermsSpecialEntry *g_perms_get_special_perms(GPerms *g_perms);
gint g_perms_set_special_perms(GPerms *g_perms, GPermsAttribut attr, const gboolean state);

/* Clear all ACL
 * If the file is open in DEFAULT mode : delete all default ACL
 * If the file is open il ACCESS mode : delete all ACL  */
gboolean g_perms_clear(GPerms *g_perms);


/* Create default entry for defaults ACL by copying currents
 */
gboolean g_perms_init_default(GPerms *g_perms);

#endif	/*_GPERMS_H_*/
