/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "gperms.h"
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>
#include <gtk/gtkmarshal.h>


enum {
	ADD_USER,
	DEL_USER,
	ADD_GROUP,
	DEL_GROUP,
	INIT_DEFAULT,
	CLEAR_DEFAULT,
	SET_USER,
	SET_GROUP,
	//TOGGLE_READ,
	LAST_SIGNAL
};
static gint g_perms_signals[LAST_SIGNAL] = { 0 };



static void g_perms_class_init(GObjectClass *klass, gpointer data);


GPermsEntry *g_perms_get_entry__unix(GPerms *g_perms, const gint type);
GPermsEntry *g_perms_get_entry__acl(GPerms *g_perms, const GPermsType type);

//MANIPULATION FUNCTIONS
gint g_perms_set_perm__acl(GPerms *g_perms, const GPermsType type, const guint id, 
	const GPermsAttribut attr, const gboolean state);
gint g_perms_set_perm__unix(GPerms *g_perms, const GPermsType type, const guint id, 
	const GPermsAttribut attr, const gboolean state);


/********** OBJECT  CREATION **********/

/* this function returns the type identifier for the class */
GType g_perms_get_type()
{
 	/* we use static so we don't have to use a global variable for the type */
 	static GType g_perms_type = 0 ;

 	if (!g_perms_type)
	{
 		static const GTypeInfo g_perms_info =
 		{
 			sizeof (GPermsClass),
 			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) g_perms_class_init,
			(GClassFinalizeFunc) NULL,
			NULL,
			sizeof(GPerms),
			0,
			NULL //(GInstanceInitFunc) g_perms_instance_init
		};
	
		/* we register our new type with the (g)object system */
		g_perms_type = g_type_register_static ( G_TYPE_OBJECT, "GPerms", &g_perms_info, 0);
	}
 	
	return g_perms_type;
}


static void g_perms_class_init(GObjectClass *klass, gpointer data)
{
	GObjectClass *object_class;

	/* register a destructor */
	object_class = G_OBJECT_CLASS(klass);
	
	g_perms_signals[ADD_USER] = 
		g_signal_new ("add-user",
				G_OBJECT_CLASS_TYPE(object_class),
				G_SIGNAL_RUN_FIRST,
				G_STRUCT_OFFSET (GPermsClass, add_user),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

	g_perms_signals[DEL_USER] = 
		g_signal_new ("del-user",
				G_OBJECT_CLASS_TYPE(object_class),
				G_SIGNAL_RUN_FIRST,
				G_STRUCT_OFFSET (GPermsClass, del_group),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
				
	g_perms_signals[ADD_GROUP] = 
		g_signal_new ("add-group",
				G_OBJECT_CLASS_TYPE(object_class),
				G_SIGNAL_RUN_FIRST,
				G_STRUCT_OFFSET (GPermsClass, add_group),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
			
	g_perms_signals[DEL_GROUP] = 
		g_signal_new ("del-group",
				G_OBJECT_CLASS_TYPE(object_class),
				G_SIGNAL_RUN_FIRST,
				G_STRUCT_OFFSET (GPermsClass, del_group),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
	

	g_perms_signals[INIT_DEFAULT] = 
		g_signal_new ("init-default",
				G_OBJECT_CLASS_TYPE(object_class),
				G_SIGNAL_RUN_FIRST,
				G_STRUCT_OFFSET (GPermsClass, init_default),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
				
	g_perms_signals[CLEAR_DEFAULT] = 
		g_signal_new ("clear-default",
				G_OBJECT_CLASS_TYPE(object_class),
				G_SIGNAL_RUN_FIRST,
				G_STRUCT_OFFSET (GPermsClass, clear_default),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
	
	g_perms_signals[SET_USER] = 
		g_signal_new ("set-user",
				G_OBJECT_CLASS_TYPE(object_class),
				G_SIGNAL_RUN_FIRST,
				G_STRUCT_OFFSET (GPermsClass, set_user),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
			
	g_perms_signals[SET_GROUP] = 
		g_signal_new ("set-group",
				G_OBJECT_CLASS_TYPE(object_class),
				G_SIGNAL_RUN_FIRST,
				G_STRUCT_OFFSET (GPermsClass, set_group),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
			
				
	/*object_class->toogle_read = NULL;
	object_class->add_user = NULL;*/
}


//
GPerms* g_perms_new(void)
{	
	return g_object_new(g_perms_get_type(), NULL);
}


//
gboolean g_perms_init(GPerms *g_perms)
{
	#ifdef HAVE_LIBACL
	g_perms->acl = NULL;
	#endif
	g_perms->filename = NULL;

	g_perms->entry = NULL;
	
	return TRUE;
}


//
void g_perms_end(GPerms *self)
{
	/* TODO: put deinit code here */
}

//
void g_perms_delete(GPerms *self)
{
	g_return_if_fail(NULL != self);
	g_perms_end(self);
	g_free(self);
}


/********** WRAP **********/

//
GPermsEntry *g_perms_get_entry(GPerms *g_perms, const GPermsType type)
{
	g_perms->func_get_entry(g_perms, type);
}

/*Define a permission of an attribut (read, write, exec) to a type (P_USER, P_GROUP,  
  P_OTHER, P_MASK) to an id 
 */
gboolean g_perms_set_perm(GPerms *g_perms, const GPermsType type, const guint id, 
	const GPermsAttribut attr, const gboolean state)
{
	return g_perms->func_set_perm(g_perms, type, id, attr, state);
}


/********** MANIPULATION FUNCTIONS **********/

/*Load permissions from file
PARAMETERS : 
	- filename : name of file
	- mode : Mode of opening : P_ACCESS or P_DEFAULT
RETURN : 0 ok, -1 fail
 */
gint g_perms_load_file(GPerms *g_perms, const gchar *filename, const GPermsMode mode)
{
	#ifdef HAVE_LIBACL
	acl_t acl = NULL;
	#endif
	
	g_log(0, G_LOG_LEVEL_DEBUG, "GPerms - Load File : %s", filename);
	
	if (g_perms->filename)	
		g_free(g_perms->filename);
	
	g_perms->filename = g_strdup(filename);
	g_perms->nb_users  = 0;
	g_perms->nb_groups = 0;
	g_perms->mask = 0;
	
	//Check if file or directory exist
	struct stat s_stat;
	
	if (stat(filename, &s_stat) == -1)
	{
		g_log(0, G_LOG_LEVEL_DEBUG, "GPerms - Error stat(%s)", filename);
		return FALSE;
	}
	
	g_perms->is_directory = S_ISDIR(s_stat.st_mode);
		
	
	#ifdef HAVE_LIBACL
	g_perms->acl_type = mode;
	acl = acl_get_file(filename, g_perms->acl_type);
	g_perms->acl = acl;
	
	if (acl != NULL) //An ACL File or dir
	{
		g_log(0, G_LOG_LEVEL_DEBUG, "GPerms - Load file have ACL");
		g_perms->mode = P_ACL;
		g_perms->func_get_entry = g_perms_get_entry__acl;
		g_perms->func_set_perm  = g_perms_set_perm__acl;
		
		//count
		GPermsEntry *entry_perms;
		
		entry_perms = g_perms_get_entry(g_perms, P_FIRST_ENTRY);	
		while (entry_perms) 
		{			
			if (entry_perms->type == P_USER)
				g_perms->nb_users++;
			
			if (entry_perms->type == P_GROUP)
				g_perms->nb_groups++;
			
			if (entry_perms->type == P_MASK)
				g_perms->mask = 1;
			
			entry_perms = g_perms_get_entry(g_perms, P_NEXT_ENTRY);
		}

	} 
	else
	{
		g_log(0, G_LOG_LEVEL_DEBUG, "GPerms - Load file haven't ACL");
		g_perms->mode = P_UNIX;
		g_perms->func_get_entry = g_perms_get_entry__unix;
		g_perms->func_set_perm  = g_perms_set_perm__unix;			
	}
	
	#else
	g_perms->func_get_entry = g_perms_get_entry__unix;
	g_perms->func_set_perm  = g_perms_set_perm__unix;
	
	#endif
	
	
	return 0;	
}


/* Add an new user to the file or directory
 *PARAMETERS :
 *	- uid_t uid : 
 *RETURN : 0 if all is ok, and set g_perms->entry with the add entry
 *	Emit the add-user signal
 */
gint g_perms_add_user(GPerms *g_perms, uid_t uid)
{
	g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Add user %d", uid);
#ifdef HAVE_LIBACL
	acl_permset_t permset;
	acl_entry_t acl_entry;
	uid_t *uid_acl;
	
	if (g_perms->entry)
		g_free(g_perms->entry);
	
	if (acl_create_entry(&g_perms->acl, &acl_entry) == -1 )
	{
		g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Cannot create ACL entry", 0);
		return -1;
	}
	
	acl_set_tag_type(acl_entry, ACL_USER);
	
	uid_acl = (uid_t *) g_malloc(sizeof(uid_t));
	*uid_acl = (uid_t) uid;
	
	acl_set_qualifier(acl_entry, uid_acl);
	
	g_perms->nb_users++;
	if (g_perms_valid__acl(g_perms) == -1)
	{
		g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : ACL is not valid", 0);
		return -1;
	}
		
	if (acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl) == -1) 
	{
		acl_delete_entry(g_perms->acl, acl_entry);
		g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Cannot set ACL to file %s", g_perms->filename);
		return -1;
	}

	
	uid_t *id;
	struct passwd *passwd;
	g_perms->entry = g_new(GPermsEntry, 1);
	//id = (uid_t *) acl_get_qualifier(&acl_entry);
	passwd = getpwuid(*uid_acl);
	g_perms->entry->type = P_USER;		
	g_perms->entry->id   = *uid_acl;
	if (passwd)		
		g_perms->entry->name = g_strdup(passwd->pw_name);
	else
		g_perms->entry->name = g_strdup("--Unknow--");
	//g_perms->entry = g_perms_found_entry(g_perms, P_USER, uid);

	g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Emit Add User Signal", 0);
	g_signal_emit(g_perms, g_perms_signals[ADD_USER], 0);
	//g_signal_emit_by_name(g_perms, "add-user");
	
	
	return 0;
#else	
	return -1;
#endif
}

/*Remove User from ACL
 RETURN : 0 ok, -1 fail
 */
gint g_perms_del_user(GPerms *g_perms, uid_t uid)
{
#ifdef HAVE_LIBACL
	g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Delete user %d", uid);
	
	acl_entry_t acl_entry;
	acl_tag_t e_type;
	gint ret;
	//uid_t *uid_acl;
	
	acl_entry = g_perms_found_entry(g_perms, P_USER, (guint) uid);
	if (acl_entry == NULL)
		return -1;
	
	acl_delete_entry(g_perms->acl, acl_entry);
	acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl);

	//uid_t *id;
	struct passwd *passwd;
	g_perms->entry = g_new(GPermsEntry, 1);
	//id = (uid_t *) acl_get_qualifier(&acl_entry);
	passwd = getpwuid(uid);
	g_perms->entry->type = P_USER;		
	g_perms->entry->id   = uid;
	if (passwd)		
		g_perms->entry->name = g_strdup(passwd->pw_name);
	else
		g_perms->entry->name = g_strdup("--Unknow--");
	
	
	//g_perms_del_user(g_perms, uid);
	g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Emit Del User Signal", 0);
	g_signal_emit_by_name(g_perms, "del-user");
	
	return 0;
#else
	return -1;
#endif	
}

//
gint g_perms_add_group(GPerms *g_perms, const gid_t gid)
{
	g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Add Group %d", gid);
#ifdef HAVE_LIBACL
	acl_permset_t permset;
	acl_entry_t acl_entry;
	gid_t *gid_acl;
	
	if (g_perms->entry)
		g_free(g_perms->entry);
	
	if (acl_create_entry(&g_perms->acl, &acl_entry) == -1 )
	{
		g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Cannot create ACL entry", 0);
		return -1;
	}
	
	acl_set_tag_type(acl_entry, ACL_GROUP);
	
	gid_acl = (gid_t *) g_malloc(sizeof(uid_t));
	*gid_acl = (gid_t) gid;
	
	acl_set_qualifier(acl_entry, gid_acl);
	
	g_perms->nb_groups++;
	if (g_perms_valid__acl(g_perms) == -1)
	{
		g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : ACL is not valid", 0);
		return -1;
	}
		
	if (acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl) == -1) 
	{
		acl_delete_entry(g_perms->acl, acl_entry);
		g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Cannot set ACL to file %s", g_perms->filename);
		return -1;
	}

	
	uid_t *id;
	struct group *group;
	g_perms->entry = g_new(GPermsEntry, 1);
	group = getgrgid(*gid_acl);
	g_perms->entry->type = P_GROUP;
	g_perms->entry->id    = *gid_acl;
	if (group)		
		g_perms->entry->name = g_strdup(group->gr_name);
	else
		g_perms->entry->name = g_strdup("--Unknow--");

	g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Emit Add Group Signal", 0);
	g_signal_emit_by_name(g_perms, "add-group");
	
	return 0;
#else	
	return -1;
#endif

}


//
gint g_perms_del_group(GPerms *g_perms, gid_t gid)
{
#ifdef HAVE_LIBACL
	g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Delete group %d", gid);
	
	acl_entry_t acl_entry;
	acl_tag_t e_type;
	gint ret;
	//uid_t *uid_acl;
	
	acl_entry = g_perms_found_entry(g_perms, P_GROUP, (guint) gid);
	if (acl_entry == NULL)
		return -1;
	
	acl_delete_entry(g_perms->acl, acl_entry);
	acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl);

	//uid_t *id;
	struct group *group;
	g_perms->entry = g_new(GPermsEntry, 1);
	//id = (uid_t *) acl_get_qualifier(&acl_entry);
	group = getgrgid(gid);
	g_perms->entry->type = P_USER;		
	g_perms->entry->id   = gid;
	if (group)		
		g_perms->entry->name = g_strdup(group->gr_name);
	else
		g_perms->entry->name = g_strdup("--Unknow--");
	
	
	//g_perms_del_user(g_perms, uid);
	g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Emit Del Group Signal with gid=%d", g_perms->entry->id);
	g_signal_emit_by_name(g_perms, "del-group");
	
	return 0;
#else
	return -1;
#endif	
	/*g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Delete group %d", gid);
#ifdef HAVE_LIBACL
	acl_entry_t acl_entry;
	acl_tag_t e_type;
	gint ret;
	
	acl_entry = g_perms_found_entry(g_perms, P_GROUP, (guint) gid);
	if (acl_entry == NULL)
		return -1;
	
	acl_delete_entry(g_perms->acl, acl_entry);
	acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl);

	
	//uid_t *id;
	struct group *group;
	g_perms->entry = g_new(GPermsEntry, 1);
	group = getgrgid(gid);
	g_perms->entry->type = P_GROUP;
	g_perms->entry->id    = gid;
	if (group)		
		g_perms->entry->name = g_strdup(group->gr_name);
	else
		g_perms->entry->name = g_strdup("--Unknow--");

	/*g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Emit Add Group Signal", 0);
	g_signal_emit_by_name(g_perms, "add-group");*/
	
	/*
	g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Emit Del Group Signal", 0);
	g_signal_emit_by_name(g_perms, "del-group");
	
	return 0;
#else
	return -1;
#endif*/
}



//
GPermsSpecialEntry *g_perms_get_special_perms(GPerms *g_perms)
{
	GPermsSpecialEntry *special_perms;
	struct stat state;
	
	special_perms = g_new0(GPermsSpecialEntry, 1);
	
	g_assert(g_perms->filename);
	stat(g_perms->filename, &state);
	
	special_perms->suid   = state.st_mode & S_ISUID;
	special_perms->sgid   = state.st_mode & S_ISGID;
	special_perms->sticky = state.st_mode & S_ISVTX;
		
	return special_perms;
}

//
gint g_perms_set_special_perms(GPerms *g_perms, GPermsAttribut attr, const gboolean state)
{
	struct stat file_stat;
	mode_t mode;
	
	
	g_assert(g_perms->filename);
	stat(g_perms->filename, &file_stat);
	

	if (attr == P_SUID)
		attr = S_ISUID;
	if (attr == P_SGID)
		attr= S_ISGID;
	if (attr == P_STICKY)
		attr = S_ISVTX;


	
	mode = file_stat.st_mode;
	
	if (state)
		mode = mode | attr;
	else
		mode = mode & ~attr;
		
	return chmod(g_perms->filename, mode);
}


#ifdef HAVE_LIBACL
//
acl_entry_t g_perms_found_entry(GPerms *g_perms, const gint type , const guint id)
{
	GPermsEntry *entry_perms;
		
	entry_perms = g_perms_get_entry(g_perms, P_FIRST_ENTRY);
 	if (entry_perms == NULL)
		return NULL;

	while (entry_perms) 
	{
		gint cur_id;
		struct stat file_stat;
		acl_tag_t e_type;
		
		if ((entry_perms->type & type) == 0) {
			entry_perms = g_perms_get_entry(g_perms, P_NEXT_ENTRY);
			continue;
		}
		
		
		if (entry_perms->type  == P_USER_OWNER)
			if (entry_perms->id == id)
				return g_perms->acl_entry;
		
		if (entry_perms->type == P_USER)
			if (entry_perms->id == id)
				return g_perms->acl_entry;
		
		if (entry_perms->type == P_GROUP_OWNER)
			if (entry_perms->id == id)
				return g_perms->acl_entry;
		
		if (entry_perms->type == P_GROUP)
			if (entry_perms->id == id)
				return g_perms->acl_entry;
		
		if (entry_perms->type == P_OTHER)
			return g_perms->acl_entry;
		
		if (entry_perms->type == P_MASK)
			return g_perms->acl_entry;
		
		
		entry_perms = g_perms_get_entry(g_perms, P_NEXT_ENTRY);
	}

	return NULL;
}
#endif




//
gint g_perms_set_user_owner(GPerms *g_perms, uid_t uid)
{
	//GPermsEntry *entry;
	
	if (chown(g_perms->filename, uid, -1) == -1)
		return -1;
	
	/*entry = g_new0(GPermsEntry, 1);
	entry->id = uid;
	entry->name = */
	
	uid_t *id;
	struct passwd *passwd;
	g_perms->entry = g_new(GPermsEntry, 1);
	passwd = getpwuid(uid);
	g_perms->entry->type = P_USER;		
	g_perms->entry->id   = uid;
	if (passwd)		
		g_perms->entry->name = g_strdup(passwd->pw_name);
	else
		g_perms->entry->name = g_strdup("--Unknow--");
	
	
	g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Emit set-user Signal with", 0);
	g_signal_emit_by_name(g_perms, "set-user");
	
	return 0;
}


/*
 Set The owner of file
 Emit set-user signal
 */
gint g_perms_set_group_owner(GPerms *g_perms, gid_t gid)
{
	if (chown(g_perms->filename, -1, gid) == -1)
		return -1;
	
	gid_t *gd;
	struct group *group;
	g_perms->entry = g_new(GPermsEntry, 1);
	group = getgrgid(gid);
	g_perms->entry->type = P_USER;		
	g_perms->entry->id   = gid;
	if (group)		
		g_perms->entry->name = g_strdup(group->gr_name);
	else
		g_perms->entry->name = g_strdup("--Unknow--");
	
	
	g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Emit set-group Signal with", 0);
	g_signal_emit_by_name(g_perms, "set-group");
	
	return 0;
}


/*
 Remove default ACLs and emit clear-default signal
 */
gboolean g_perms_clear(GPerms *g_perms)
{
#ifdef HAVE_LIBACL
	GPermsEntry *entry_perms;
	
	entry_perms = g_perms_get_entry(g_perms, P_FIRST_ENTRY);
	
	while (entry_perms)
	{
		acl_delete_entry(g_perms->acl, g_perms->acl_entry);
		acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl);
		
		entry_perms = g_perms_get_entry(g_perms, P_NEXT_ENTRY);
	}

	g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Emit clear-default Signal", 0);
	g_signal_emit(g_perms, g_perms_signals[CLEAR_DEFAULT], 0);
	
	return TRUE;
#endif
	return FALSE;
}

/*
 Initialize the defaul AC to current acces ACLs
 */
gboolean g_perms_init_default(GPerms *g_perms)
{
#ifdef HAVE_LIBACL
	acl_entry_t acl_entry;
	
	acl_create_entry(&g_perms->acl, &acl_entry);
	acl_set_tag_type(acl_entry, ACL_USER_OBJ);
	
	acl_create_entry(&g_perms->acl, &acl_entry);
	acl_set_tag_type(acl_entry, ACL_GROUP_OBJ);
	
	acl_create_entry(&g_perms->acl, &acl_entry);
	acl_set_tag_type(acl_entry, ACL_OTHER);
		
	if (acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl) == -1)
		return FALSE;
	
	g_log(0, G_LOG_LEVEL_DEBUG, "GPerms : Emit init-default Signal", 0);
	g_signal_emit(g_perms, g_perms_signals[INIT_DEFAULT], 0);
	
	return TRUE;
#endif
	return FALSE;
}

/********** ACL **********/

#ifdef HAVE_LIBACL

//
gint g_perms_valid__acl(GPerms *g_perms)
{
	acl_entry_t acl_entry;
	acl_permset_t permset;
	
	if (g_perms->mask == 0) 
	{	
		acl_create_entry(&g_perms->acl, &acl_entry);
		acl_set_tag_type(acl_entry, ACL_MASK);

		acl_get_permset(acl_entry, &permset);
		acl_add_perm(permset, ACL_READ);
		acl_add_perm(permset, ACL_WRITE);
		acl_add_perm(permset, ACL_EXECUTE);
		
		acl_set_permset(acl_entry, permset);
		
		acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl);

		
		g_perms->mask = 1;
	}
	
	
}


//
GPermsEntry *g_perms_get_entry__acl(GPerms *g_perms, const GPermsType type)
{
	GPermsEntry *entry_perms;
	acl_permset_t permset;
	acl_tag_t acl_tag;
	struct stat    file_stat;
	struct passwd *passwd;
	struct group  *group;
	gint *id;
	gint  ret;

	
	ret = acl_get_entry(g_perms->acl, type, &g_perms->acl_entry);
	if (ret == -1)
		return NULL;

		
	if (acl_get_tag_type(g_perms->acl_entry, &acl_tag) == -1)
		return NULL;
		
	entry_perms = g_new(GPermsEntry, 1);

	//		
	switch (acl_tag)
	{
		case ACL_USER_OBJ :
			if (stat(g_perms->filename, &file_stat) == -1)
			{
				g_free(entry_perms);
				return NULL;
			}
			
			passwd = getpwuid(file_stat.st_uid);
			
			entry_perms->type = P_USER_OWNER;
			entry_perms->id   = passwd->pw_uid;
			entry_perms->name = g_strdup(passwd->pw_name);
			g_log(0, G_LOG_LEVEL_DEBUG, "GPerms - Get Entry ACL_USER_OBJ %d - %s", entry_perms->id, entry_perms->name);
			break;
	
		case ACL_USER :
			if ( (id = acl_get_qualifier(g_perms->acl_entry)) == NULL) 
			{
				g_free(entry_perms);
				return NULL;
			}
		
			passwd = getpwuid(*id);
			
			entry_perms->type = P_USER;		
			entry_perms->id    = *id;
			if (passwd)		
				entry_perms->name  = g_strdup(passwd->pw_name);
			else
				entry_perms->name  = g_strdup("--Unknow--");
			g_log(0, G_LOG_LEVEL_DEBUG, "GPerms - Get Entry ACL_USER %d - %s", entry_perms->id, entry_perms->name);
			break;
		

		case ACL_GROUP_OBJ :				
			if (stat(g_perms->filename, &file_stat) == -1)
			{
				g_free(entry_perms);
				return NULL;
			}
			
			group = getgrgid(file_stat.st_gid);
		
			entry_perms->type = P_GROUP_OWNER;
			entry_perms->id   = file_stat.st_gid;
			entry_perms->name = g_strdup(group->gr_name);
			g_log(0, G_LOG_LEVEL_DEBUG, "GPerms - Get Entry ACL_GROUP_OBJ %d - %s", entry_perms->id, entry_perms->name);
			break;
	
			
		case ACL_GROUP :
			id = acl_get_qualifier(g_perms->acl_entry);
			if (id == NULL)
			{
				g_free(entry_perms);
				return NULL;
			}
			
			
			group = getgrgid(*id);

			entry_perms->type = P_GROUP;
			entry_perms->id   = *id;
			if (group)
				entry_perms->name = g_strdup(group->gr_name);	
			else
				entry_perms->name = g_strdup("--Unknow--");
			g_log(0, G_LOG_LEVEL_DEBUG, "GPerms - Get Entry ACL_GROUP %d - %s", entry_perms->id, entry_perms->name);
			break;

		
		case ACL_OTHER :
			entry_perms->type = P_OTHER;
			g_log(0, G_LOG_LEVEL_DEBUG, "GPerms - Get Entry ACL_OTHER", 0);
			break;
		
			
		case ACL_MASK :
			entry_perms->type = P_MASK;
			g_log(0, G_LOG_LEVEL_DEBUG, "GPerms - Get Entry ACL_MASK", 0);
			break;
		
	
		default :
			return NULL;
	}
	
	
	acl_get_permset(g_perms->acl_entry, &permset);
	entry_perms->read  = acl_get_perm(permset, ACL_READ);
	entry_perms->write = acl_get_perm(permset, ACL_WRITE);
	entry_perms->exec  = acl_get_perm(permset, ACL_EXECUTE);
	

	return entry_perms;	
}


/* type : P_USER, P_GROUP, ...
 * perm : P_READ, P_WRITE, P_EXECUTE
 * 
 */
gboolean g_perms_set_perm__acl(GPerms *g_perms, const GPermsType type, const guint id, 
	const GPermsAttribut attr, const gboolean state)
{
	acl_entry_t acl_entry;
	acl_permset_t permset;
	gint attr_acl;
	
	
	if (attr == P_READ)
		attr_acl = ACL_READ;
	if (attr == P_WRITE)
		attr_acl = ACL_WRITE;
	if (attr == P_EXECUTE)
		attr_acl = ACL_EXECUTE;
	
	//For User
	if (type == P_USER) 
	{
		acl_entry = g_perms_found_entry(g_perms, P_USER | P_USER_OWNER, id);
		if (acl_entry == NULL)
			return -1;


		acl_get_permset(acl_entry, &permset);
	
		if (state)
			acl_add_perm(permset, attr_acl);			
		else
			acl_delete_perm(permset, attr_acl);
	}
		
	//For Groups
	if (type == P_GROUP) 
	{
		acl_entry = g_perms_found_entry(g_perms, P_GROUP | P_GROUP_OWNER, id);
		if (acl_entry == NULL)
			return -1;
				
		acl_get_permset(acl_entry, &permset);
	
		if (state)
			acl_add_perm(permset, attr_acl);
		else
			acl_delete_perm(permset, attr_acl);
	}
	
	//For Others
	if (type == P_OTHER) 
	{
		acl_entry = g_perms_found_entry(g_perms, P_OTHER, 0);
		if (acl_entry == NULL)
			return -1;
		
		acl_get_permset(acl_entry, &permset);
	
		if (state)
			acl_add_perm(permset, attr_acl);
		else
			acl_delete_perm(permset, attr_acl);
	}
	
	//For Mask
	if (type == P_MASK) {
		acl_entry = g_perms_found_entry(g_perms, P_MASK, 0);
		if (acl_entry == NULL)
			return -1;
		
		acl_get_permset(acl_entry, &permset);
	
		if (state)
			acl_add_perm(permset, attr_acl);
		else
			acl_delete_perm(permset, attr_acl);
	}
	
	acl_set_permset(acl_entry, permset);
	if (acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl) == -1)
	{
		g_perms->acl = acl_get_file(g_perms->filename, g_perms->acl_type);
		return FALSE;
	}
	
	return TRUE;
}

#endif

/********** UNIX **********/


//
GPermsEntry *g_perms_get_entry__unix(GPerms *g_perms, const gint type)
{
	GPermsEntry *entry_perms;
	struct stat state;
	struct passwd *passwd;
	struct group  *group;
	
	if (type ==  P_FIRST_ENTRY) 	//Return user
	{
		stat(g_perms->filename, &state);
		passwd = getpwuid(state.st_uid);
		
		entry_perms = g_new0(GPermsEntry, 1);
		entry_perms->type = P_USER_OWNER;
		entry_perms->id = state.st_uid;
		entry_perms->name  = g_strdup(passwd->pw_name);
		entry_perms->read  = state.st_mode & S_IRUSR;
		entry_perms->write = state.st_mode & S_IWUSR;
		entry_perms->exec  = state.st_mode & S_IXUSR;
		
		g_perms->unix_entry_ptr = P_USER_OWNER;
	}
	else if (g_perms->unix_entry_ptr == P_USER_OWNER)
	{
		stat(g_perms->filename, &state);
		group = getgrgid(state.st_gid);
		
		entry_perms = g_new0(GPermsEntry, 1);
		entry_perms->type = P_GROUP_OWNER;
		entry_perms->id = state.st_gid;
		entry_perms->name  = g_strdup(group->gr_name);
		entry_perms->read  = state.st_mode & S_IRGRP;
		entry_perms->write = state.st_mode & S_IWGRP;
		entry_perms->exec  = state.st_mode & S_IXGRP;
		
		g_perms->unix_entry_ptr = P_GROUP_OWNER;
	} 
	else if (g_perms->unix_entry_ptr == P_GROUP_OWNER) 
	{
		stat(g_perms->filename, &state);
		group = getgrgid(state.st_gid);
		
		entry_perms = g_new0(GPermsEntry, 1);
		entry_perms->type = P_OTHER;
		entry_perms->read  = state.st_mode & S_IROTH;
		entry_perms->write = state.st_mode & S_IWOTH;
		entry_perms->exec  = state.st_mode & S_IXOTH;
		
		g_perms->unix_entry_ptr = P_OTHER;

	} 
	else
	{
		return NULL;
	}
	
	return entry_perms;
}


//
gboolean g_perms_set_perm__unix(GPerms *g_perms, const GPermsType type, const guint id, 
	const GPermsAttribut attr, const gboolean state)
{
	struct stat file_stat;
	mode_t mode;
	gint   attr_unix;   
	
	
	stat(g_perms->filename, &file_stat);
	
	if (type == P_MASK) 
		return FALSE;

		
	if (type == P_USER_OWNER || type == P_USER) 
	{
		if (getpwuid((uid_t) id) != NULL)
		{
			if (attr == P_READ)
				attr_unix = S_IRUSR;
			if (attr == P_WRITE)
				attr_unix = S_IWUSR;
			if (attr == P_EXECUTE)
				attr_unix = S_IXUSR;
		}
		else
		{
			return FALSE;
		}
	}	

	if (type == P_GROUP_OWNER || type == P_GROUP) 
	{
		if (getgrgid((gid_t) id) != NULL)
		{
			if (attr == P_READ)
				attr_unix = S_IRGRP;
			if (attr == P_WRITE)
				attr_unix = S_IWGRP;
			if (attr == P_EXECUTE)
				attr_unix = S_IXGRP;
		}
		else
		{
			return FALSE;
		}
	}	
	
	
	if (type == P_OTHER) 
	{
		if (attr == P_READ)
			attr_unix = S_IROTH;
		if (attr == P_WRITE)
			attr_unix = S_IWOTH;
		if (attr == P_EXECUTE)
			attr_unix = S_IXOTH;
	}	

	
	mode = file_stat.st_mode;
	
	if (state)
		mode = mode | attr_unix;
	else
		mode = mode & ~attr_unix;
		
	if (chmod(g_perms->filename, mode) == -1)
		return FALSE;
	
	return TRUE;
}



/********** CORE FUNCTIONS **********/

/*GPermsEntry *_get_entry()
{
	
	
}*/
