/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 

#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <libintl.h>
#include <gtk/gtk.h>

#include "gtkusergroup.h"



enum {
	VALIDATE,
	CANCEL,
	SELECT_USER,
	SELECT_GROUP,
	LAST_SIGNAL
};

static gint gtk_usergroup_signals[LAST_SIGNAL] = { 0 };

static void gtk_usergroup_class_init (GtkUserGroupClass *class);
static void gtk_usergroup_init (GtkUserGroup *gtk_usergroup);

static void gtk_usergroup_on_user_clicked(GtkCellRenderer *cell, gchar *path_str, 
	GtkListStore *list_users);
static void gtk_usergroup_on_validate(GtkWidget *button, gpointer *data);
static void gtk_usergroup_on_cancel(GtkWidget *button, gpointer *data);
static void gtk_usergroup_on_select_user(GtkTreeView *tree_view, gpointer user_data);
static void gtk_usergroup_on_select_group(GtkTreeView *tree_view, gpointer user_data);



static GtkWidget *_create_list_view(GtkListStore *list, gchar *text_header);
static void _fill_users_list(GtkListStore *list_users);
static void _fill_groups_list(GtkListStore *list_groups);




/***** WIDGET CREATION *****/

//
GtkType gtk_usergroup_get_type(void)
{
	static GType gtk_usergroup_type = 0;

	if (!gtk_usergroup_type) {
		static const GTypeInfo gtk_usergroup_info =
		{
			sizeof (GtkUserGroupClass),
			NULL,
			NULL,
			(GClassInitFunc)  gtk_usergroup_class_init,
			NULL,
			NULL,
			sizeof (GtkUserGroup),
			0,
			(GInstanceInitFunc) gtk_usergroup_init
		};

		gtk_usergroup_type = g_type_register_static (GTK_TYPE_DIALOG, 
				"GtkUserGroup", &gtk_usergroup_info, 0);
	}

	return gtk_usergroup_type;
}


//
static void gtk_usergroup_class_init (GtkUserGroupClass *class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;
  
	/*gtk_usergroup_signals[VALIDATE] = 
		g_signal_new ("validate",
				G_OBJECT_CLASS_TYPE(object_class),
				GTK_RUN_FIRST,
				G_STRUCT_OFFSET (GtkUserGroupClass, validate),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID, GTK_TYPE_NONE, 0);*/
  
	class->validate = NULL;
}


//
static void gtk_usergroup_init (GtkUserGroup *gtk_usergroup)
{
	GtkDialog *dialog;
	GtkWidget *hbox_top, *hbox_bottom, *vhbox;
	GtkWidget *label;
	GtkWidget *button;
	GtkWidget *widget;
	GtkWidget *scrolled;
	
	GtkWidget    *view_users, *view_groups;
	GtkListStore *list_users, *list_groups;

	
	//DIALOG
	dialog = GTK_DIALOG(gtk_usergroup);
	
	hbox_top = gtk_hbox_new(TRUE, 2);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), GTK_WIDGET(hbox_top), TRUE, TRUE, 0);
	gtk_widget_show(hbox_top);
	
	hbox_bottom = gtk_hbox_new(TRUE, 2);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), GTK_WIDGET(hbox_bottom), TRUE, TRUE, 0);
	gtk_widget_show(hbox_bottom);
	
	
	//User
	list_users = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
	_fill_users_list(list_users);
	view_users = _create_list_view(list_users, _("Users"));
	g_signal_connect(G_OBJECT(view_users), "cursor-changed", 
		G_CALLBACK(gtk_usergroup_on_select_user), gtk_usergroup);
	gtk_usergroup->list_users = list_users;
	gtk_usergroup->view_users = view_users;
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
                                  	GTK_POLICY_AUTOMATIC,
                                  	GTK_POLICY_AUTOMATIC);
	gtk_container_add(GTK_CONTAINER(scrolled), view_users);
	gtk_box_pack_start(GTK_BOX(hbox_top), GTK_WIDGET(scrolled), TRUE, TRUE, 0);
	gtk_widget_show_all(scrolled);
	
	
	//Group
	list_groups = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
	_fill_groups_list(list_groups);
	view_groups = _create_list_view(list_groups, _("Groups"));
	g_signal_connect(G_OBJECT(view_groups), "cursor-changed", 
		G_CALLBACK(gtk_usergroup_on_select_group), gtk_usergroup);
	gtk_usergroup->list_groups = list_groups;
	gtk_usergroup->view_groups = view_groups;
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
                                  	GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add(GTK_CONTAINER(scrolled), view_groups);
	gtk_box_pack_start(GTK_BOX(hbox_top), GTK_WIDGET(scrolled), TRUE, TRUE, 0);
	gtk_widget_show_all(scrolled);


	//Buttons
	button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
	gtk_box_pack_start(GTK_BOX(hbox_bottom), GTK_WIDGET(button), TRUE, TRUE, 0);
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(gtk_usergroup_on_cancel), gtk_usergroup);
	gtk_widget_show(button);
	
	button = gtk_button_new_from_stock(GTK_STOCK_OK);
	gtk_box_pack_start(GTK_BOX(hbox_bottom), GTK_WIDGET(button), TRUE, TRUE, 0);
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(gtk_usergroup_on_validate), gtk_usergroup);
	gtk_widget_show(button);
}



/***** MANUPULATION FUNCTIONS *****/


//
GtkWidget *gtk_usergroup_new()
{
	GtkUserGroup *usergroup;

	usergroup= g_object_new (GTK_TYPE_USERGROUP, NULL);
	gtk_window_set_title (GTK_WINDOW (usergroup), "Users & Groups Selection");
	gtk_dialog_set_has_separator (GTK_DIALOG(usergroup), FALSE);

	return GTK_WIDGET(usergroup);
}


//
void gtk_usergroup_set_validate_user(GtkUserGroup *gtk_usergroup, GtkWidget *widget,
	void *function)
{
	gtk_usergroup->validate_user_widget   = widget;
	gtk_usergroup->validate_user_function = function;
}


//
void gtk_usergroup_set_validate_group(GtkUserGroup *gtk_usergroup, GtkWidget *widget,
	void *function)
{
	gtk_usergroup->validate_group_widget   = widget;
	gtk_usergroup->validate_group_function = function;
}


//
void gtk_usergroup_hide_users(GtkUserGroup *gtk_usergroup)
{
	gtk_widget_hide(gtk_usergroup->view_users);
}


/***** CALLBACKS *****/

//
static void gtk_usergroup_on_select_user(GtkTreeView *tree_view, gpointer user_data)
{
	GtkUserGroup *gtk_usergroup;
	GtkTreeSelection *selection;
	
	gtk_usergroup = GTK_USERGROUP(user_data);
	
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(gtk_usergroup->view_groups));
	gtk_tree_selection_unselect_all(selection);
}						
	

//
static void gtk_usergroup_on_select_group(GtkTreeView *tree_view, gpointer user_data)
{
	GtkUserGroup *gtk_usergroup;
	GtkTreeSelection *selection;
	
	gtk_usergroup = GTK_USERGROUP(user_data);
	
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(gtk_usergroup->view_users));
	gtk_tree_selection_unselect_all(selection);
}


//
static void gtk_usergroup_on_validate(GtkWidget *button, gpointer *data)
{
	GtkUserGroup *gtk_usergroup;
	void (*function) (GtkWidget *, gpointer *);
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	
	
	gtk_usergroup = GTK_USERGROUP(data);
	gtk_widget_hide(GTK_WIDGET(gtk_usergroup));
	
	
	//User selected
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(gtk_usergroup->view_users));
	if (gtk_tree_selection_get_selected(GTK_TREE_SELECTION(selection), 
					&gtk_usergroup->list_users, &iter)) {
	
		gchar *uid, *user;
	
		gtk_tree_model_get(GTK_TREE_MODEL(gtk_usergroup->list_users), &iter, 0, &uid, 1, &user, -1);
		
		function = gtk_usergroup->validate_user_function;
		if (function == NULL)
			return;
		
		function(gtk_usergroup->validate_user_widget, atoi(uid));		
		return;
	}
	
	//Group selected
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(gtk_usergroup->view_groups));
	if (gtk_tree_selection_get_selected(GTK_TREE_SELECTION(selection), 
					&gtk_usergroup->list_users, &iter)) {
		
		gchar *gid, *group;
		
		gtk_tree_model_get(GTK_TREE_MODEL(gtk_usergroup->list_groups), 
					&iter, 0, &gid, 1, &group, -1);
		//g_print("GtkUserGroup : Selected group : %u - %s\n", atoi(gid), group);
		
		function = gtk_usergroup->validate_group_function;
		if (function == NULL)
			return;
		
		function(gtk_usergroup->validate_group_widget, atoi(gid));		
		return;
	}
	
}


//
static void gtk_usergroup_on_cancel(GtkWidget *button, gpointer *data)
{
	GtkUserGroup *gtk_usergroup;
	
	gtk_usergroup = GTK_USERGROUP(data);
	gtk_widget_hide(GTK_WIDGET(gtk_usergroup));
}


//
static void gtk_usergroup_on_user_clicked(GtkCellRenderer *cell, gchar *path_str, 
	GtkListStore *list_users)
{
	GtkTreeIter  iter;
	GtkTreePath *path = gtk_tree_path_new_from_string(path_str);
	gchar *value;
	
	gtk_tree_model_get_iter(GTK_TREE_MODEL(list_users), &iter, path);
	gtk_tree_model_get(GTK_TREE_MODEL(list_users), &iter, 0, &value, -1);
	
	gtk_tree_path_free(path);
	
	g_print("GtkUserGroup : user selected %s\n", value);
}



/***** CORE DATA MANUPULATION FUNCTIONS *****/

//
static GtkWidget *_create_list_view(GtkListStore *list, gchar *text_header)
{
	GtkWidget *view;
	GtkTreeViewColumn *column;
	GtkCellRenderer   *cell_renderer;
	

	view = gtk_tree_view_new();
	gtk_tree_view_set_model(GTK_TREE_VIEW(view), GTK_TREE_MODEL(list));
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), TRUE);
	gtk_widget_set_size_request(view, 200, 300);
	
	column = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(column, "id");
	gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);
	cell_renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(column, cell_renderer, TRUE);
	gtk_tree_view_column_add_attribute(column, cell_renderer, 
		"text", 0);
	
	column = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(column, text_header);
	gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);
	cell_renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(column, cell_renderer, TRUE);
	gtk_tree_view_column_add_attribute(column, cell_renderer, 
		"text", 1);
		
	//gtk_container_add(GTK_CONTAINER(scrolled), view);
	
	return view;
}
	

//
static void _fill_users_list(GtkListStore *list_users)
{
	struct passwd *passwd;
	GtkTreeIter  iter;
	
	//printf("fill_user : \n");
	
	setpwent();
	
	while(1) {
		passwd = getpwent();
		if (passwd == NULL)
			break;
		
		gtk_list_store_append(list_users, &iter);	
		gtk_list_store_set(list_users, &iter, 
			0, g_strdup_printf("%u", passwd->pw_uid),
			1, passwd->pw_name, 
			-1);
		
		//printf("\t - %s\n", passwd->pw_name);
	}
	
	endpwent();
}


//
static void _fill_groups_list(GtkListStore *list_groups)
{
	struct group *group;
	GtkTreeIter   iter;
	
	while(1) {
		group = getgrent();
		if (group == NULL)
			break;
		
		gtk_list_store_append(list_groups, &iter);	
		gtk_list_store_set(list_groups, &iter, 
			0, g_strdup_printf("%u", group->gr_gid),
			1, g_strdup(group->gr_name), 
			-1);
	}
	
	endgrent();
}
