/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 

#include <libintl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "gtkacl.h"
#include "gtkusergroup.h"
#include "gtkacl-callbacks.h"


#define MESSAGE_BOX(format, error)	\
	{GtkWidget *message_box;\
	message_box = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, \
			GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, format, error); \
		gtk_dialog_run (GTK_DIALOG (message_box)); \
 		gtk_widget_destroy (message_box);}
		
#define MESSAGE_BOX_ERROR() MESSAGE_BOX("%s", g_strerror(errno))
		

enum {
	ADD_ACL,
	DEL_ACL,	
	LAST_SIGNAL
};


static gint gtk_acl_signals[LAST_SIGNAL] = { 0 };



static void gtk_acl_class_init (GtkAclClass *class);
static void gtk_acl_init (GtkAcl *gtk_acl);

static GtkWidget *create_menu_type(GtkAcl *gtk_acl);
static GtkWidget *create_perms_view(GtkAcl *gtk_acl, GtkTreeStore *acl_tree);
static GtkWidget *create_frame_specials(GtkAcl *gtk_acl);
static GtkWidget *create_frame_default(GtkAcl *gtk_acl);


/***** WIDGET CREATION *****/

//
GtkType gtk_acl_get_type(void)
{
	static GType gtk_acl_type = 0;

	if (!gtk_acl_type) {
		static const GTypeInfo gtk_acl_info =
		{
			sizeof (GtkAclClass),
			NULL,
			NULL,
			(GClassInitFunc)  gtk_acl_class_init,
			NULL,
			NULL,
			sizeof (GtkAcl),
			0,
			(GInstanceInitFunc) gtk_acl_init
		};

		gtk_acl_type = g_type_register_static (GTK_TYPE_HBOX, "GtkAcl",
				&gtk_acl_info, 0);	
	}
	
	return gtk_acl_type;
}


//
static void gtk_acl_class_init (GtkAclClass *class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;
  
	/*gtk_acl_signals[ADD_ACL] = gtk_signal_new ("add_acl",
					GTK_RUN_FIRST,
					G_OBJECT_CLASS_TYPE(object_class),
					G_STRUCT_OFFSET (GtkAclClass, add_acl),
					gtk_signal_default_marshaller, GTK_TYPE_NONE, 
					0);

	gtk_acl_signals[ADD_ACL] = gtk_signal_new ("del_acl",
					GTK_RUN_FIRST,
					G_OBJECT_CLASS_TYPE(object_class),
					G_STRUCT_OFFSET (GtkAclClass, del_acl),
					gtk_signal_default_marshaller, GTK_TYPE_NONE, 
					0);*/

	class->add_acl = NULL;
	class->del_acl = NULL;
}


//
static void gtk_acl_init (GtkAcl *gtk_acl)
{
	GtkWidget *box_main, *vbox_list, *box_special, *box_buttons,
		*box_advanced;
	
	GtkWidget *button;
	GtkWidget *button_advanced;
	
	
	box_main     = gtk_vbox_new(FALSE, 3);
	gtk_container_set_border_width(GTK_CONTAINER(box_main), 5);
	
	box_special  = gtk_hbox_new(TRUE,  3);	

	
	//HEADER - with check acl and acl type menu
	GtkWidget *box_header;
	GtkWidget *check;
	box_header   = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(box_header),  GTK_BUTTONBOX_EDGE);
	
	check = gtk_check_button_new_with_label(_("Acl support"));
	gtk_widget_set_sensitive(check, FALSE);
	gtk_acl->check_acl = check;
	gtk_container_add(GTK_CONTAINER(box_header), GTK_WIDGET(check));
	
	//Menu Choice  access or default
	GtkWidget *menu_type;
	menu_type = create_menu_type(gtk_acl);
	gtk_container_add(GTK_CONTAINER(box_header), GTK_WIDGET(menu_type ));
	
	gtk_box_pack_start(GTK_BOX(box_main), GTK_WIDGET(box_header), FALSE, FALSE, 0);
	
	
	//PERMS VIEW
	GtkWidget    *perms_view;
	GtkTreeStore *acl_tree;

	acl_tree = gtk_tree_store_new(6, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, 
		G_TYPE_BOOLEAN,	G_TYPE_BOOLEAN, G_TYPE_BOOLEAN);
	gtk_acl->acl_data = acl_tree;
	
	perms_view = create_perms_view(gtk_acl, acl_tree);
	gtk_box_pack_start(GTK_BOX(box_main), GTK_WIDGET(perms_view), TRUE, TRUE, 0);
	
	
	//ADVANCED MENU
	box_advanced  = gtk_hbox_new(FALSE,  2);	
	gtk_container_set_border_width(GTK_CONTAINER(box_advanced), 5);
	
		
	
	//Special perms
	GtkWidget *frame;	
	frame = create_frame_specials(gtk_acl);
	gtk_box_pack_start(GTK_BOX(box_advanced), GTK_WIDGET(frame), FALSE, FALSE, 0);

	frame = create_frame_default(gtk_acl);
	gtk_box_pack_end(GTK_BOX(box_advanced), GTK_WIDGET(frame), FALSE, FALSE, 0);
	
	gtk_box_pack_start(GTK_BOX(box_main), GTK_WIDGET(box_advanced), FALSE, FALSE, 0);
	gtk_widget_show(box_advanced);
	
	
	// ACTIONS BUTTONS
	
	//Action box
	GtkWidget *box_action;
	box_action = gtk_hbox_new(FALSE, 0);
	
	//Button advanced
	button_advanced = gtk_toggle_button_new_with_label(_("Advanced"));

	g_signal_connect(G_OBJECT(button_advanced), "toggled", G_CALLBACK(gtk_acl_on_avanced_toggled), gtk_acl);
	gtk_box_pack_start(GTK_BOX(box_action),  GTK_WIDGET(button_advanced), FALSE, FALSE, 20);
	gtk_acl->advanced = button_advanced ;
	gtk_widget_show(button_advanced);
	
	//Box buttons
	box_buttons = gtk_hbutton_box_new();
	gtk_box_set_spacing(GTK_BOX(box_buttons), 5);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(box_buttons), GTK_BUTTONBOX_END);
	
	//Button owner
	button = gtk_button_new_with_label(_("Owner"));
	gtk_widget_show(button);
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(gtk_acl_on_set_owner), 
		gtk_acl);
	gtk_container_add(GTK_CONTAINER(box_buttons), GTK_WIDGET(button));
	gtk_acl->button_owner = button;

	//Buton "Del ACL"
	button = gtk_button_new_from_stock(GTK_STOCK_DELETE);
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(gtk_acl_on_del_acl), gtk_acl);
	gtk_widget_show(button);
	gtk_container_add(GTK_CONTAINER(box_buttons), GTK_WIDGET(button));
	gtk_acl->button_del = button;
	
	//Buton "Add ACL"
	button = gtk_button_new_from_stock(GTK_STOCK_ADD);
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(gtk_acl_on_add_acl), gtk_acl);
	gtk_widget_show(button);
	gtk_container_add(GTK_CONTAINER(box_buttons), GTK_WIDGET(button));
	gtk_acl->button_add = button;

	//
	gtk_box_pack_end(GTK_BOX(box_action), GTK_WIDGET(box_buttons), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(box_main), GTK_WIDGET(box_action), FALSE, FALSE, 0);

	//
	gtk_container_add (GTK_CONTAINER(gtk_acl), box_main);
	
	
	
	//Create UserGroup Window
	gtk_acl->win_usergroup = gtk_usergroup_new();
	gtk_usergroup_set_validate_user(GTK_USERGROUP(gtk_acl->win_usergroup), 
		GTK_WIDGET(gtk_acl), G_CALLBACK(gtk_acl_add_user));
	gtk_usergroup_set_validate_group(GTK_USERGROUP(gtk_acl->win_usergroup), 
		GTK_WIDGET(gtk_acl), G_CALLBACK(gtk_acl_add_group));
		
	//Create UserGroup for owner
	gtk_acl->win_usergroup_owner = gtk_usergroup_new();
	
	gtk_usergroup_set_validate_user(GTK_USERGROUP(gtk_acl->win_usergroup_owner), 
		GTK_WIDGET(gtk_acl), G_CALLBACK(gtk_acl_set_user_owner));
	gtk_usergroup_set_validate_group(GTK_USERGROUP(gtk_acl->win_usergroup_owner), 
		GTK_WIDGET(gtk_acl), G_CALLBACK(gtk_acl_set_group_owner));
		

	gtk_widget_show_all(box_main);
	//gtk_widget_hide(gtk_acl->special_frame);
	
	//
	gtk_acl->perms = g_perms_new();
}


/***** MANUPULATION FUNCTIONS *****/

//
GtkWidget *gtk_acl_new()
{
	return g_object_new (GTK_TYPE_ACL, NULL);
}


//
gboolean gtk_acl_load_file(GtkAcl *gtk_acl, const gchar *filename, const gint type)
{
	//GtkWidget *tooltips;
	
	if (g_perms_load_file(gtk_acl->perms, filename, type) == FALSE)
	{
		MESSAGE_BOX("Wrong path!", 0);		
		return FALSE;
	}
	
	
	gtk_widget_set_sensitive(GTK_WIDGET(gtk_acl->menu_mode), FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(gtk_acl->frame_default), FALSE);
	
	
	if (gtk_acl->perms->mode == P_MODE_ACL)
	{
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gtk_acl->check_acl), TRUE);
		if (gtk_acl->perms->is_directory == TRUE)
		{
			
			
			gtk_widget_set_sensitive(GTK_WIDGET(gtk_acl->menu_mode), TRUE);
			
			/*gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), GTK_WIDGET(gtk_acl->check_special[0]),
				"The program is executed with the owner PPprivileges", 
				"");*/
		}
	}
	else 
	{
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gtk_acl->check_acl), FALSE);
		
	}
	
	gtk_acl_load(gtk_acl);
	
	/*gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), GTK_WIDGET(gtk_acl->check_special[0]),
				"The program is executed with the owner PPprivileges", 
				"");*/

	return TRUE;
}


//
void gtk_acl_load(GtkAcl *gtk_acl)
{
	gint type;
	GtkTreeIter iter_user_root, iter_group_root, iter_others_root, iter_mask_root;
	GtkTreeIter iter_user, iter_group;
	struct _entry_perms *entry_perms;

	
	gtk_tree_store_clear(gtk_acl->acl_data);

	entry_perms = g_perms_get_entry(gtk_acl->perms, FIRST_ENTRY);
	if (entry_perms == NULL) {
		//MESSAGE_BOX_ERROR();
		return;
	}

	

	gtk_tree_store_append(GTK_TREE_STORE(gtk_acl->acl_data), 
		&iter_user_root, NULL);
	gtk_tree_store_append(GTK_TREE_STORE(gtk_acl->acl_data), 
		&iter_group_root, NULL);
	gtk_tree_store_append(GTK_TREE_STORE(gtk_acl->acl_data), 
		&iter_others_root, NULL);

	
	//
	while (entry_perms) 
	{
		uid_t *uid_user;
		struct stat file_stat;

		GtkTreeIter iter_user;
		

		if (entry_perms->type == P_USER_OWNER)
		{
			gtk_tree_store_set(
				GTK_TREE_STORE(gtk_acl->acl_data), &iter_user_root, 
				COL_TYPE   , _("Users"),
				COL_ID     , g_strdup_printf("%u", entry_perms->id), 
				COL_NAME   , entry_perms->name, 
				//COL_SUID   , entry_perms->suid,
				COL_READ   , entry_perms->read,
				COL_WRITE  , entry_perms->write,
				COL_EXECUTE, entry_perms->exec,
				-1);			
		}

		
		if (entry_perms->type == P_USER)
		{			
			gtk_tree_store_append(
				GTK_TREE_STORE(gtk_acl->acl_data), 
				&iter_user, &iter_user_root);			
			gtk_tree_store_set(
				GTK_TREE_STORE(gtk_acl->acl_data), 
				&iter_user, 
				COL_ID     , g_strdup_printf("%u", entry_perms->id), 
				COL_NAME   , entry_perms->name, 
				COL_READ   , entry_perms->read,
				COL_WRITE  , entry_perms->write,
				COL_EXECUTE, entry_perms->exec,
				-1);
		}

		
		if (entry_perms->type == P_GROUP_OWNER) 
		{
			gtk_tree_store_set(
				GTK_TREE_STORE(gtk_acl->acl_data), 
				&iter_group_root,
				COL_TYPE   , _("Groups"),
				COL_ID     , g_strdup_printf("%u", entry_perms->id), 
				COL_NAME   , entry_perms->name, 
				COL_READ   , entry_perms->read,
				COL_WRITE  , entry_perms->write,
				COL_EXECUTE, entry_perms->exec,
				-1);
		}

		
		if (entry_perms->type == P_GROUP)
		{
			gtk_tree_store_append(
				GTK_TREE_STORE(gtk_acl->acl_data), 
				&iter_user, &iter_group_root);
			gtk_tree_store_set(
				GTK_TREE_STORE(gtk_acl->acl_data), 
				&iter_user, 
				COL_ID     , g_strdup_printf("%u", entry_perms->id), 
				COL_NAME   , entry_perms->name, 
				COL_READ   , entry_perms->read,
				COL_WRITE  , entry_perms->write,
				COL_EXECUTE, entry_perms->exec,
				-1);

		}
			
		
		if (entry_perms->type == P_OTHER)
		{
			gtk_tree_store_set(
				GTK_TREE_STORE(gtk_acl->acl_data), 
				&iter_others_root, 
				COL_TYPE   , _("Others"),
				COL_READ   , entry_perms->read,
				COL_WRITE  , entry_perms->write,
				COL_EXECUTE, entry_perms->exec,
				-1);
		}

			
		if (entry_perms->type == P_MASK)
		{
			gtk_tree_store_append(GTK_TREE_STORE(gtk_acl->acl_data), 
				&iter_mask_root, NULL);
		
			gtk_tree_store_set(
				GTK_TREE_STORE(gtk_acl->acl_data), 
				&iter_mask_root, 
				COL_TYPE   , _("Mask"),
				COL_READ   , entry_perms->read,
				COL_WRITE  , entry_perms->write,
				COL_EXECUTE, entry_perms->exec,
				-1);
		}
									
		
		if (entry_perms)
			g_free(entry_perms);
			
		entry_perms = g_perms_get_entry(gtk_acl->perms, NEXT_ENTRY);
	}
	
		
	gtk_tree_view_expand_all(GTK_TREE_VIEW(gtk_acl->acl_view));	
	
	//Special perms
	struct _special_entry_perms *special_perms;
	
	special_perms = g_perms_get_special_perms(gtk_acl->perms);
	
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gtk_acl->check_special[0]), 
		special_perms->suid);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gtk_acl->check_special[1]), 
		special_perms->sgid);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gtk_acl->check_special[2]), 
		special_perms->sticky);
	
	gtk_acl_gray(gtk_acl);
}


//
void gtk_acl_add_user(GtkAcl *gtk_acl, gpointer uid)
{
	if (g_perms_add_user(gtk_acl->perms, (uid_t) uid) == -1)
		MESSAGE_BOX(_("Cannot add user!"), 0);
	
	gtk_acl_load(gtk_acl);
}


//
void gtk_acl_del_user(GtkAcl *gtk_acl, uid_t uid)
{
	if (g_perms_del_user(gtk_acl->perms, uid) == -1)
		MESSAGE_BOX_ERROR();
	
	gtk_acl_load(gtk_acl);
}


//
void gtk_acl_del_group(GtkAcl *gtk_acl, gid_t gid)
{
	if (g_perms_del_group(gtk_acl->perms, gid) == -1)
		MESSAGE_BOX_ERROR();
		
	gtk_acl_load(gtk_acl);
}


//
void gtk_acl_add_group(GtkAcl *gtk_acl, gpointer gid)
{
	if (g_perms_add_group(gtk_acl->perms, (gid_t) gid) == -1)
		MESSAGE_BOX_ERROR();
	
	gtk_acl_load(gtk_acl);
}



//
void gtk_acl_set_user_owner(GtkAcl *gtk_acl, gpointer uid)
{
	if (g_perms_set_user_owner(gtk_acl->perms, (uid_t) uid) == -1)
		MESSAGE_BOX_ERROR();

	gtk_acl_load(gtk_acl);
}


//
void gtk_acl_set_group_owner(GtkAcl *gtk_acl, gpointer gid)
{
	if (g_perms_set_group_owner(gtk_acl->perms, (gid_t) gid) == -1)
		MESSAGE_BOX_ERROR();
		

	gtk_acl_load(gtk_acl);	
}

//
void gtk_acl_set_advanced_mode(GtkAcl *gtk_acl, const gboolean state)
{ 
	GtkTreeViewColumn *column;
	
	column = gtk_tree_view_get_column(GTK_TREE_VIEW(gtk_acl->acl_view), COL_ID);
	if (column == NULL)
		return;
	
	gtk_tree_view_column_set_visible(GTK_TREE_VIEW_COLUMN(column), state);
	
	
	if (state == FALSE) 
	{
		/*gtk_widget_hide(gtk_acl->special[0]);
		gtk_widget_hide(gtk_acl->special[1]);
		gtk_widget_hide(gtk_acl->special[2]);*/
		
		gtk_widget_hide(gtk_acl->frame_specials);
		gtk_widget_hide(gtk_acl->frame_default);
		
		//gtk_widget_hide(gtk_acl->menu_mode);
	}
	
	
	if (state == TRUE) 
	{
		/*gtk_widget_show(gtk_acl->special[0]);
		gtk_widget_show(gtk_acl->special[1]);
		gtk_widget_show(gtk_acl->special[2]);
		*/
		gtk_widget_show_all(gtk_acl->frame_specials);
		gtk_widget_show_all(gtk_acl->frame_default);
		//gtk_widget_show(gtk_acl->menu_mode);
	}
	
	
}


//
void gtk_acl_gray(GtkAcl *gtk_acl)
{
	uid_t uid;
	GtkTreeIter iter;
	gchar *char_uid;

	//IF USER IS THE OWNER ALLOW ADD AND DEL
	if (gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(gtk_acl->acl_data), &iter, "0"))
	{
		gtk_tree_model_get(GTK_TREE_MODEL(gtk_acl->acl_data), &iter, COL_ID, &char_uid, -1);
		uid = atoi(char_uid);
		if (getuid() == uid)
		{
			gtk_widget_set_sensitive(gtk_acl->button_add, TRUE);
			gtk_widget_set_sensitive(gtk_acl->button_del, TRUE);
		}
		else
		{
			gtk_widget_set_sensitive(gtk_acl->button_add, FALSE);
			gtk_widget_set_sensitive(gtk_acl->button_del, FALSE);
		}
	}
	
	
	//IF USER IS THE OWNER ALLOW ADD, DEL AND CHANGE OWNER
	if (getuid() == 0)	/* Maybe Use geteuid ? */
	{
		gtk_widget_set_sensitive(gtk_acl->button_owner, TRUE);
		gtk_widget_set_sensitive(gtk_acl->button_add, TRUE);
		gtk_widget_set_sensitive(gtk_acl->button_del, TRUE);
	}
	else
	{
		gtk_widget_set_sensitive(gtk_acl->button_owner, FALSE);
	}
	

	
}


/***** CONTROL CREATION *****/

//
static GtkWidget *create_menu_type(GtkAcl *gtk_acl)
{
	GtkWidget *options;
	GtkWidget *menu, *menu_item;
	menu = gtk_menu_new();
	menu_item = gtk_menu_item_new_with_label(_("Access"));
	g_signal_connect(G_OBJECT(menu_item), "activate", gtk_acl_on_show_access, gtk_acl);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);
	menu_item = gtk_menu_item_new_with_label(_("Default"));
	g_signal_connect(G_OBJECT(menu_item), "activate", gtk_acl_on_show_default, gtk_acl);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);
	
	options = gtk_option_menu_new();
	gtk_option_menu_set_menu(GTK_OPTION_MENU(options), menu);
	gtk_acl->menu_mode = options;
	gtk_widget_show(GTK_WIDGET(options));
	
	return options;
}


//
static GtkWidget *create_perms_view(GtkAcl *gtk_acl, GtkTreeStore *acl_tree)
{
	GtkTreeViewColumn *col;
	GtkCellRenderer   *cell_renderer;
	GtkWidget         *scrolled;
	GtkWidget         *acl_view; 
	
	//SCrolled view
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
                                  	GTK_POLICY_AUTOMATIC,
                                  	GTK_POLICY_AUTOMATIC);
					
	
	
	//Acl Tree View
	acl_view = gtk_tree_view_new();
	gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(acl_view), TRUE);
	gtk_tree_view_set_model(GTK_TREE_VIEW(acl_view), GTK_TREE_MODEL(acl_tree));
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(acl_view), TRUE);
	gtk_widget_set_size_request(acl_view, 400, 250);
	gtk_acl->acl_view = acl_view;
	

	//Column type
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(col, (gchar *) _("Type"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(acl_view), col);
	cell_renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(col, cell_renderer, TRUE);
	gtk_tree_view_column_add_attribute(col, cell_renderer, "text", COL_TYPE);

	//Column Name
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(col, (gchar *) _("Name"));
	
	gtk_tree_view_append_column(GTK_TREE_VIEW(acl_view), col);
	cell_renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(col, cell_renderer, TRUE);
	gtk_tree_view_column_add_attribute(col, cell_renderer, "text", COL_NAME);
	
	//Column Id
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(col, (gchar *) _("Id"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(acl_view), col);
	cell_renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(col, cell_renderer, TRUE);
	gtk_tree_view_column_add_attribute(col, cell_renderer, "text", COL_ID);
	
	//Column Read
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(col, _("Read"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(acl_view), col);
	cell_renderer = gtk_cell_renderer_toggle_new();
	g_signal_connect(G_OBJECT(cell_renderer), "toggled", 
		G_CALLBACK(gtk_acl_on_toggle_read), gtk_acl);
	gtk_tree_view_column_pack_start(col, cell_renderer, TRUE);
	gtk_tree_view_column_add_attribute(col, cell_renderer, "active", COL_READ);
	
	//Column Write
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(col, _("Write"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(acl_view), col);
	cell_renderer = gtk_cell_renderer_toggle_new();
	g_signal_connect(G_OBJECT(cell_renderer), "toggled", 
		G_CALLBACK(gtk_acl_on_toggle_write), gtk_acl);	
	gtk_tree_view_column_pack_start(col, cell_renderer, TRUE);
	gtk_tree_view_column_add_attribute(col, cell_renderer, "active", COL_WRITE);
	
	//Column Exec
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(col, _("Exec"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(acl_view), col);
	cell_renderer = gtk_cell_renderer_toggle_new();
	g_signal_connect(G_OBJECT(cell_renderer), "toggled", 
		G_CALLBACK(gtk_acl_on_toggle_execute), gtk_acl);
	gtk_tree_view_column_pack_start(col, cell_renderer, TRUE);
	gtk_tree_view_column_add_attribute(col, cell_renderer, "active", COL_EXECUTE);
	
	
	gtk_widget_show(acl_view);
	gtk_container_add (GTK_CONTAINER(scrolled), acl_view);
	//gtk_box_pack_start(GTK_BOX(vbox_list), GTK_WIDGET(scrolled), TRUE, TRUE, 0);
	
	return scrolled;
	
}

//
static GtkWidget *create_frame_specials(GtkAcl *gtk_acl)
{
	GtkWidget *frame;
	GtkWidget *box_specials;
	GtkWidget *check;
	GtkWidget *tooltips;
	
	
	frame = gtk_frame_new(_("Specials permissions"));
	gtk_acl->frame_specials = frame;
	
	box_specials = gtk_hbox_new(TRUE, 5);
	
	tooltips = gtk_tooltips_new();
	
	check = gtk_check_button_new_with_label("suid");
	g_signal_connect(G_OBJECT(check), "toggled", G_CALLBACK(gtk_acl_on_toggle_suid), 
		gtk_acl);
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), GTK_WIDGET(check),
		_("Executable : the program is executed\nwith the owner privileges"), 
		"");
	gtk_acl->check_special[0] = check;
	gtk_box_pack_start(GTK_BOX(box_specials), GTK_WIDGET(check), FALSE, FALSE, 2);

	check = gtk_check_button_new_with_label("sgid");
	g_signal_connect(G_OBJECT(check), "toggled", G_CALLBACK(gtk_acl_on_toggle_sgid), 
		gtk_acl);
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), GTK_WIDGET(check),
		_("Executable : the program is executed\nwith the group privileges\n"
		"Directory  : all new file in directory\nwill be writable for group of directory"), 
		"");
	gtk_acl->check_special[1] = check;
	gtk_box_pack_start(GTK_BOX(box_specials), GTK_WIDGET(check), FALSE, FALSE, 2);

	check = gtk_check_button_new_with_label("sticky");
	g_signal_connect(G_OBJECT(check), "toggled", G_CALLBACK(gtk_acl_on_toggle_sticky), 
		gtk_acl);
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), GTK_WIDGET(check),
		_("Executable : the program stay in memory after started.\nIt will start also faster\n"
		"Directory  : only edit your files"), 
		"");
	gtk_acl->check_special[2] = check;
	gtk_box_pack_start(GTK_BOX(box_specials), GTK_WIDGET(check), FALSE, FALSE, 2);
	
	gtk_container_add(GTK_CONTAINER(frame), GTK_WIDGET(box_specials));
	
	return frame;
}

//
static GtkWidget *create_frame_default(GtkAcl *gtk_acl)
{
	GtkWidget *frame;
	GtkWidget *box;
	GtkWidget *button;
	
	frame = gtk_frame_new(_("Default"));
	gtk_acl->frame_default = frame;
	
	box = gtk_hbox_new(FALSE, 2);
	
	//Add Default
	button = gtk_button_new_with_label(_("init"));
	//gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
	g_signal_connect(G_OBJECT(button), "clicked", gtk_acl_on_init_default, gtk_acl);
	gtk_box_pack_start(GTK_BOX(box), GTK_WIDGET(button), FALSE, FALSE, 1);

	//Del default
	button = gtk_button_new_with_label(_("del"));
	//gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
	g_signal_connect(G_OBJECT(button), "clicked", gtk_acl_on_clear, gtk_acl);
	gtk_box_pack_start(GTK_BOX(box), GTK_WIDGET(button), FALSE, FALSE, 1);
	
	gtk_container_add(GTK_CONTAINER(frame), GTK_WIDGET(box));
	
	return frame;
}
