/*
 * This file is part of Gterm.
 *
 * Gterm is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <locale.h>

#include <glib-object.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <pango/pango.h>

#include <glib/gi18n.h>

#include <vte/vte.h>

#include "settings.h"
#include "callback.h"

#ifdef USE_MENU
# include "menu.h"
#endif

#include "dnd.h"

GtkWidget *term;

gchar *geometry = NULL;
gchar *win_name = NULL;
gboolean show_ver = FALSE;

static GOptionEntry ents[] = {
  {"audio", 'a', 0, G_OPTION_ARG_NONE, &cfg.audible,
   N_("turn on audible bell"), NULL},
  {"blink", 'b', 0, G_OPTION_ARG_NONE, &cfg.blink, 
   N_("cursor blink on"), NULL},
  {"command", 'e', 0, G_OPTION_ARG_CALLBACK, set_cmd,
   N_("run CMD instead of $SHELL"), "CMD"},
  {"font", 'f', 0, G_OPTION_ARG_CALLBACK, set_font, 
   N_("set font to FONT"), "FONT"},
  {"login", 'l', 0, G_OPTION_ARG_NONE, &cfg.login, 
   N_("use login shell"), NULL},
  {"locale", 'L', 0, G_OPTION_ARG_STRING, &cfg.locale,
   N_("use specific locale"), NULL},
  {"config", 'c', 0, G_OPTION_ARG_CALLBACK, set_cfg,
   N_("use this config file"), "FILE"},
  {"fore", 'F', 0, G_OPTION_ARG_CALLBACK, set_color,
   N_("set foreground color"), "COLOR"},
  {"back", 'B', 0, G_OPTION_ARG_CALLBACK, set_color,
   N_("set background color"), "COLOR"},
  {"cursor", 'C', 0, G_OPTION_ARG_CALLBACK, set_color, 
   N_("set cursor color"), "COLOR"},
  {"geometry", 'g', 0, G_OPTION_ARG_STRING, &geometry, 
   N_("window geometry"), "WxH+X+Y"},
#ifdef USE_MENU
  {"menu", 'm', 0, G_OPTION_ARG_STRING, &cfg.menu,
   N_("use menu description from file"), "FILE"},
#endif
  {"history", 'H', 0, G_OPTION_ARG_INT, &cfg.lines, 
   N_("set history size"), "N"},
  {"scrollbar", 's', 0, G_OPTION_ARG_CALLBACK, set_scroll,
   N_("use scrollbar"), "on/off"},
  {"type", 't', 0, G_OPTION_ARG_STRING, &cfg.term,
   N_("set terminal emulation type"), NULL},
  {"title", 'T', 0, G_OPTION_ARG_STRING, &cfg.title, 
   N_("window title"), NULL},
  {"name", 'n', 0, G_OPTION_ARG_STRING, &win_name,
   N_("window name"), NULL},
  {"dir", 'W', 0, G_OPTION_ARG_STRING, &cfg.wd,
   N_("set initial working directory"), NULL},
  {"version", 'v', 0, G_OPTION_ARG_NONE, &show_ver, 
   N_("show version"), NULL},
  {NULL}
};

static GtkTargetEntry tgt[] = {
  {"text/x-moz-url", 0, TARGET_MOZ_URL},
  {"UTF8_STRING", 0, TARGET_UTF8_STRING},
  {"COMPOUND_TEXT", 0, TARGET_COMPOUND_TEXT},
  {"TEXT", 0, TARGET_TEXT},
  {"STRING", 0, TARGET_STRING},
  {"text/plain", 0, TARGET_TEXT_PLAIN}
};

int
main (int argc, char *argv[])
{
  GOptionContext *context;
  
  GtkWidget *win, *hbox, *sbar;
#ifdef USE_MENU
  GtkMenu *menu;
#endif

  GdkGeometry geom;

  gchar *cmd;
  gchar **sh_argv = NULL;
  gchar **env_add;

  int i, envs_cnt;

#ifdef ENABLE_NLS
  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif

  gtk_init (&argc, &argv);
  
  /* Read default configuration file */
  read_config (NULL);

  setlocale (LC_ALL, cfg.locale);
  
  context = g_option_context_new (_("- terminal emulator based on vte widget"));
  g_option_context_add_main_entries (context, ents, GETTEXT_PACKAGE);
  g_option_context_add_group (context, gtk_get_option_group (TRUE));
  g_option_context_parse (context, &argc, &argv, NULL);
  g_option_context_free (context);
  
  if (show_ver)
    {
      g_printf (_("gterm, version %s\n"), VERSION);
      exit (0);
    }
  
  /* in case of change locale from user defined config file */
  setlocale (LC_ALL, cfg.locale);

  /* Create a window to hold the scrolling shell */
  win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  if (win_name)
    gtk_window_set_wmclass (GTK_WINDOW (win), win_name, win_name);
  gtk_window_set_default_icon_name ("gnome-terminal");
  gtk_container_set_resize_mode (GTK_CONTAINER (win), GTK_RESIZE_IMMEDIATE);

  /* Create a box to hold everything */
  hbox = gtk_hbox_new (0, FALSE);
  gtk_container_add (GTK_CONTAINER (win), hbox);

  /* Create the terminal widget and add it to the possible scrolling shell */
  term = vte_terminal_new ();
  gtk_box_pack_start (GTK_BOX (hbox), term, TRUE, TRUE, 0);

  /* Create the scrollbar for the widget */
  if (cfg.sb)
    {
      sbar = gtk_vscrollbar_new ((VTE_TERMINAL (term))->adjustment);
      gtk_box_pack_start (GTK_BOX (hbox), sbar, FALSE, FALSE, 0);
    }

  /* Set window title */
  g_signal_connect (G_OBJECT (term), "window-title-changed",
                    G_CALLBACK (window_set_title), win);

  /* Set destroy and exit signals */
  g_signal_connect (G_OBJECT (win), "destroy",
                    G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (term), "child-exited",
                    G_CALLBACK (gtk_main_quit), NULL);

  /* Set drag and drop */
  gtk_drag_dest_set (term, GTK_DEST_DEFAULT_ALL,
                     tgt, G_N_ELEMENTS (tgt), GDK_ACTION_COPY);
  g_signal_connect (G_OBJECT (term), "drag_data_received",
                    G_CALLBACK (drop_data), NULL);

  /* Set initial window title */
  gtk_window_set_title (GTK_WINDOW (win), cfg.title);

  /* Set terminal geometry */
  vte_terminal_set_size (VTE_TERMINAL (term), cfg.width, cfg.height);
  vte_terminal_get_padding (VTE_TERMINAL (term),
                            &geom.base_width, &geom.base_height);
  geom.width_inc = (VTE_TERMINAL (term))->char_width;
  geom.height_inc = (VTE_TERMINAL (term))->char_height;
  geom.min_width = geom.base_width + geom.width_inc * 2;
  geom.min_height = geom.base_height + geom.height_inc * 2;
  gtk_window_set_geometry_hints (GTK_WINDOW (win), term, &geom,
                                 GDK_HINT_RESIZE_INC |
                                 GDK_HINT_BASE_SIZE | GDK_HINT_MIN_SIZE);

  if (geometry)
    gtk_window_parse_geometry (GTK_WINDOW (win), geometry);

  /* Set encoding if specified */
  if (cfg.enc) 
    vte_terminal_set_encoding (VTE_TERMINAL (term), cfg.enc);

  /* Set some defaults */
  vte_terminal_set_audible_bell (VTE_TERMINAL (term), cfg.audible);
  vte_terminal_set_visible_bell (VTE_TERMINAL (term), !cfg.audible);
  vte_terminal_set_cursor_blinks (VTE_TERMINAL (term), cfg.blink);
  vte_terminal_set_scroll_on_output (VTE_TERMINAL (term), FALSE);
  vte_terminal_set_scroll_on_keystroke (VTE_TERMINAL (term), TRUE);
  vte_terminal_set_scrollback_lines (VTE_TERMINAL (term), cfg.lines);
  vte_terminal_set_mouse_autohide (VTE_TERMINAL (term), TRUE);
  
  /* Set colors */
  vte_terminal_set_colors (VTE_TERMINAL (term), &cfg.fore, &cfg.back,
                           cfg.palette, cfg.palsize);
  vte_terminal_set_color_cursor (VTE_TERMINAL (term), &cfg.cursor);
  vte_terminal_set_color_highlight (VTE_TERMINAL (term), &cfg.hlite);
  if (cfg.have_bold)
    vte_terminal_set_color_bold (VTE_TERMINAL (term), &cfg.bold);
  if (cfg.have_dim)
    vte_terminal_set_color_dim (VTE_TERMINAL (term), &cfg.dim);

  /* Set environment variables */
  envs_cnt = g_hash_table_size (cfg.envs);
  env_add = (gchar **) g_malloc (sizeof (gchar *) * (envs_cnt + 1));
  g_hash_table_foreach (cfg.envs, add_env, (gpointer) env_add);
  env_add[envs_cnt] = NULL;

  /* Add templates */
  for (i = 1; i < g_list_length (cfg.tmpls); i++)
    {
      vte_terminal_match_add (VTE_TERMINAL (term),
                              g_list_nth_data(cfg.tmpls, i));
    }

  /* Set terminal emulation mode */
  vte_terminal_set_emulation (VTE_TERMINAL (term), cfg.term);

  /* Set the default font */
  vte_terminal_set_font (VTE_TERMINAL (term), cfg.font);

#ifdef USE_MENU
  /* Add terminal context menu */
  menu = create_menu (cfg.menu);
  if (menu)
    g_signal_connect (G_OBJECT (term), "button-press-event",
                      G_CALLBACK (popup_menu), menu);
#endif

  /* Launch a shell */
  if (cfg.is_shell)
    {
      cmd = cfg.cmd;
      if (cfg.login) 
	{
	  sh_argv = g_malloc (sizeof (gchar *) * 3);
	  sh_argv[0] = cfg.cmd;
	  sh_argv[1] = "--login";
	  sh_argv[2] = NULL;
	}
    }
  else
    {
      gchar *ptr = cfg.cmd;

      cmd = ptr;
      while ((*ptr != ' ') && (*ptr != '\0')) ptr++;
      if (*ptr != '\0') 
        {
          sh_argv = g_strsplit (cfg.cmd, " ", 0);
          *ptr = '\0';
        }
    }
  vte_terminal_fork_command (VTE_TERMINAL (term),
                             cmd, sh_argv, env_add,
                             cfg.wd, TRUE, TRUE, TRUE);

  gtk_widget_show_all (win);

  gtk_main ();

  return 0;
}
