/* GSequencer - Advanced GTK Sequencer
 * Copyright (C) 2005-2021 Joël Krähemann
 *
 * This file is part of GSequencer.
 *
 * GSequencer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GSequencer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GSequencer.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ags/plugin/ags_vst3_plugin.h>

#include <ags/plugin/ags_plugin_port.h>

#include <math.h>

#if defined(AGS_W32API)
#include <windows.h>
#else
#include <dlfcn.h>
#endif

void ags_vst3_plugin_class_init(AgsVst3PluginClass *vst3_plugin);
void ags_vst3_plugin_init (AgsVst3Plugin *vst3_plugin);
void ags_vst3_plugin_set_property(GObject *gobject,
				  guint prop_id,
				  const GValue *value,
				  GParamSpec *param_spec);
void ags_vst3_plugin_get_property(GObject *gobject,
				  guint prop_id,
				  GValue *value,
				  GParamSpec *param_spec);
void ags_vst3_plugin_finalize(GObject *gobject);

gpointer ags_vst3_plugin_instantiate(AgsBasePlugin *base_plugin,
				     guint samplerate, guint buffer_size);
void ags_vst3_plugin_connect_port(AgsBasePlugin *base_plugin,
				  gpointer plugin_handle,
				  guint port_index,
				  gpointer data_location);
void ags_vst3_plugin_activate(AgsBasePlugin *base_plugin,
			      gpointer plugin_handle);
void ags_vst3_plugin_deactivate(AgsBasePlugin *base_plugin,
				gpointer plugin_handle);
void ags_vst3_plugin_run(AgsBasePlugin *base_plugin,
			 gpointer plugin_handle,
			 snd_seq_event_t *seq_event,
			 guint frame_count);
void ags_vst3_plugin_load_plugin(AgsBasePlugin *base_plugin);

/**
 * SECTION:ags_vst3_plugin
 * @short_description: The vst3 plugin class
 * @title: AgsVst3Plugin
 * @section_id:
 * @include: ags/plugin/ags_vst3_plugin.h
 *
 * The #AgsVst3Plugin loads/unloads a Vst3 plugin.
 */

enum{
  PROP_0,
};

enum{
  LAST_SIGNAL,
};

static gpointer ags_vst3_plugin_parent_class = NULL;
static guint vst3_plugin_signals[LAST_SIGNAL];

GType
ags_vst3_plugin_get_type (void)
{
  static volatile gsize g_define_type_id__volatile = 0;

  if(g_once_init_enter (&g_define_type_id__volatile)){
    GType ags_type_vst3_plugin = 0;

    static const GTypeInfo ags_vst3_plugin_info = {
      sizeof (AgsVst3PluginClass),
      NULL, /* vst3_init */
      NULL, /* vst3_finalize */
      (GClassInitFunc) ags_vst3_plugin_class_init,
      NULL, /* class_finalize */
      NULL, /* class_data */
      sizeof (AgsVst3Plugin),
      0,    /* n_preallocs */
      (GInstanceInitFunc) ags_vst3_plugin_init,
    };

    ags_type_vst3_plugin = g_type_register_static(AGS_TYPE_BASE_PLUGIN,
						  "AgsVst3Plugin",
						  &ags_vst3_plugin_info,
						  0);

    g_once_init_leave(&g_define_type_id__volatile, ags_type_vst3_plugin);
  }

  return g_define_type_id__volatile;
}

void
ags_vst3_plugin_class_init(AgsVst3PluginClass *vst3_plugin)
{
  AgsBasePluginClass *base_plugin;

  GObjectClass *gobject;
  GParamSpec *param_spec;
  
  ags_vst3_plugin_parent_class = g_type_class_peek_parent(vst3_plugin);

  /* GObjectClass */
  gobject = (GObjectClass *) vst3_plugin;

  gobject->set_property = ags_vst3_plugin_set_property;
  gobject->get_property = ags_vst3_plugin_get_property;

  gobject->finalize = ags_vst3_plugin_finalize;

  /* properties */

  /* AgsBasePluginClass */
  base_plugin = (AgsBasePluginClass *) vst3_plugin;

  base_plugin->instantiate = ags_vst3_plugin_instantiate;

  base_plugin->connect_port = ags_vst3_plugin_connect_port;

  base_plugin->activate = ags_vst3_plugin_activate;
  base_plugin->deactivate = ags_vst3_plugin_deactivate;

  base_plugin->run = ags_vst3_plugin_run;

  base_plugin->load_plugin = ags_vst3_plugin_load_plugin;

  /* AgsVst3PluginClass */
}

void
ags_vst3_plugin_init(AgsVst3Plugin *vst3_plugin)
{
  vst3_plugin->get_plugin_factory = NULL;

  vst3_plugin->host_context = NULL;

  vst3_plugin->icomponent = NULL;
  vst3_plugin->iedit_controller = NULL;
}

void
ags_vst3_plugin_set_property(GObject *gobject,
			     guint prop_id,
			     const GValue *value,
			     GParamSpec *param_spec)
{
  AgsVst3Plugin *vst3_plugin;

  GRecMutex *base_plugin_mutex;

  vst3_plugin = AGS_VST3_PLUGIN(gobject);

  /* get base plugin mutex */
  base_plugin_mutex = AGS_BASE_PLUGIN_GET_OBJ_MUTEX(vst3_plugin);

  switch(prop_id){
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(gobject, prop_id, param_spec);
    break;
  }
}

void
ags_vst3_plugin_get_property(GObject *gobject,
			     guint prop_id,
			     GValue *value,
			     GParamSpec *param_spec)
{
  AgsVst3Plugin *vst3_plugin;

  GRecMutex *base_plugin_mutex;

  vst3_plugin = AGS_VST3_PLUGIN(gobject);

  /* get base plugin mutex */
  base_plugin_mutex = AGS_BASE_PLUGIN_GET_OBJ_MUTEX(vst3_plugin);

  switch(prop_id){
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(gobject, prop_id, param_spec);
    break;
  }
}

void
ags_vst3_plugin_finalize(GObject *gobject)
{
  AgsVst3Plugin *vst3_plugin;

  vst3_plugin = AGS_VST3_PLUGIN(gobject);

  /* call parent */
  G_OBJECT_CLASS(ags_vst3_plugin_parent_class)->finalize(gobject);
}

gpointer
ags_vst3_plugin_instantiate(AgsBasePlugin *base_plugin,
			    guint samplerate, guint buffer_size)
{
  AgsVstIPluginFactory *iplugin_factory;

  gpointer retval;

  guint i, i_stop;
  
  AgsVstIPluginFactory* (*GetPluginFactory)();
  
  GRecMutex *base_plugin_mutex;

  /* get base plugin mutex */
  base_plugin_mutex = AGS_BASE_PLUGIN_GET_OBJ_MUTEX(base_plugin);
  
  /* get instantiate */
  g_rec_mutex_lock(base_plugin_mutex);

  GetPluginFactory = AGS_VST3_PLUGIN(base_plugin)->get_plugin_factory;
    
  g_rec_mutex_unlock(base_plugin_mutex);

  retval = NULL;

  if(GetPluginFactory != NULL){    
    AgsVstPClassInfo *info;

    AgsVstTResult val;
    
    iplugin_factory = GetPluginFactory();

    info = ags_vst_pclass_info_alloc();
    
    i_stop = ags_vst_iplugin_factory_count_classes(iplugin_factory);
    
    for(i = 0; i < i_stop; i++){
      ags_vst_iplugin_factory_get_class_info(iplugin_factory,
					     i, info);

      if(!g_strcmp0(ags_vst_pclass_info_get_category(&info), AGS_VST_KAUDIO_EFFECT_CLASS) == FALSE){
	continue;
      }

      AGS_VST3_PLUGIN(base_plugin)->icomponent = NULL;

      val = ags_vst_iplugin_factory_create_instance(iplugin_factory,
						    ags_vst_pclass_info_get_cid(&info),
						    ags_vst_icomponent_get_iid(),
						    (void **) &(AGS_VST3_PLUGIN(base_plugin)->icomponent));

      if(val != AGS_VST_KRESULT_TRUE){
	g_warning("failed to create VST3 instance with plugin factory");
	
	break;
      }

      ags_vst_icomponent_set_io_mode(AGS_VST3_PLUGIN(base_plugin)->icomponent,
				     AGS_VST_KADVANCED);

      AGS_VST3_PLUGIN(base_plugin)->host_context = ags_vst_host_context_get_instance();
      
      ags_vst_iplugin_base_initialize((AgsVstIPluginBase *) AGS_VST3_PLUGIN(base_plugin)->icomponent,
				      AGS_VST3_PLUGIN(base_plugin)->host_context);


      AGS_VST3_PLUGIN(base_plugin)->iedit_controller = NULL;
      
      val = ags_vst_iplugin_factory_create_instance(iplugin_factory,
						    ags_vst_pclass_info_get_cid(&info),
						    ags_vst_iedit_controller_get_iid(),
						    (void **) &(AGS_VST3_PLUGIN(base_plugin)->iedit_controller));

      if(val != AGS_VST_KRESULT_TRUE){
	g_warning("failed to create VST3 instance with plugin factory");
	
	break;
      }

      ags_vst_iplugin_base_initialize((AgsVstIPluginBase *) AGS_VST3_PLUGIN(base_plugin)->iedit_controller,
				      AGS_VST3_PLUGIN(base_plugin)->host_context);

      break;
    }    
  }
  
  
  return(retval);
}

void
ags_vst3_plugin_connect_port(AgsBasePlugin *base_plugin,
			     gpointer plugin_handle,
			     guint port_index,
			     gpointer data_location)
{
  //TODO:JK: implement me
}

void
ags_vst3_plugin_activate(AgsBasePlugin *base_plugin,
			 gpointer plugin_handle)
{
  //TODO:JK: implement me
}

void
ags_vst3_plugin_deactivate(AgsBasePlugin *base_plugin,
			   gpointer plugin_handle)
{
  //TODO:JK: implement me
}

void
ags_vst3_plugin_run(AgsBasePlugin *base_plugin,
		    gpointer plugin_handle,
		    snd_seq_event_t *seq_event,
		    guint frame_count)
{
  //TODO:JK: implement me
}

void
ags_vst3_plugin_load_plugin(AgsBasePlugin *base_plugin)
{
  AgsVstIPluginFactory *iplugin_factory;

  GList *plugin_port;

  gpointer retval;

  guint i, i_stop;
  gboolean success;  

  GError *error;

  AgsVstIPluginFactory* (*GetPluginFactory)();
  
  GRecMutex *base_plugin_mutex;

  /* get base plugin mutex */
  base_plugin_mutex = AGS_BASE_PLUGIN_GET_OBJ_MUTEX(base_plugin);

  /* dlopen */
  g_rec_mutex_lock(base_plugin_mutex);

#ifdef AGS_W32API
  base_plugin->plugin_so = LoadLibrary(base_plugin->filename);
#else
  base_plugin->plugin_so = dlopen(base_plugin->filename,
				  RTLD_NOW);
#endif
  
  if(base_plugin->plugin_so == NULL){
    g_warning("ags_vst3_plugin.c - failed to load static object file");
    
#ifndef AGS_W32API    
    dlerror();
#endif
    
    g_rec_mutex_unlock(base_plugin_mutex);

    return;
  }

  plugin_port = NULL;

  success = FALSE;

#ifdef AGS_W32API
  GetPluginFactory =
    AGS_VST3_PLUGIN(base_plugin)->get_plugin_factory = GetProcAddress(base_plugin->plugin_so,
								      "GetPluginFactory");
    
  success = (AGS_VST3_PLUGIN(base_plugin)->get_plugin_factory != NULL) ? TRUE: FALSE;
#else
  GetPluginFactory =
    AGS_VST3_PLUGIN(base_plugin)->get_plugin_factory = dlsym(base_plugin->plugin_so,
							     "GetPluginFactory");
  
  success = (dlerror() == NULL) ? TRUE: FALSE;
#endif

  g_rec_mutex_unlock(base_plugin_mutex);

  if(success){
    AgsVstPClassInfo *info;

    AgsVstTResult val;
    
    iplugin_factory = GetPluginFactory();

    info = ags_vst_pclass_info_alloc();
    
    i_stop = ags_vst_iplugin_factory_count_classes(iplugin_factory);
    
    for(i = 0; i < i_stop; i++){
      ags_vst_iplugin_factory_get_class_info(iplugin_factory,
					     i, info);

      if(!g_strcmp0(ags_vst_pclass_info_get_category(&info), AGS_VST_KAUDIO_EFFECT_CLASS) == FALSE){
	continue;
      }

      AGS_VST3_PLUGIN(base_plugin)->icomponent = NULL;

      val = ags_vst_iplugin_factory_create_instance(iplugin_factory,
						    ags_vst_pclass_info_get_cid(&info),
						    ags_vst_icomponent_get_iid(),
						    (void **) &(AGS_VST3_PLUGIN(base_plugin)->icomponent));

      if(val != AGS_VST_KRESULT_TRUE){
	g_warning("failed to create VST3 instance with plugin factory");
	
	break;
      }

      ags_vst_icomponent_set_io_mode(AGS_VST3_PLUGIN(base_plugin)->icomponent,
				     AGS_VST_KADVANCED);

      AGS_VST3_PLUGIN(base_plugin)->host_context = ags_vst_host_context_get_instance();
      
      ags_vst_iplugin_base_initialize((AgsVstIPluginBase *) AGS_VST3_PLUGIN(base_plugin)->icomponent,
				      AGS_VST3_PLUGIN(base_plugin)->host_context);


      AGS_VST3_PLUGIN(base_plugin)->iedit_controller = NULL;
      
      val = ags_vst_iplugin_factory_create_instance(iplugin_factory,
						    ags_vst_pclass_info_get_cid(&info),
						    ags_vst_iedit_controller_get_iid(),
						    (void **) &(AGS_VST3_PLUGIN(base_plugin)->iedit_controller));

      if(val != AGS_VST_KRESULT_TRUE){
	g_warning("failed to create VST3 instance with plugin factory");
	
	break;
      }

      ags_vst_iplugin_base_initialize((AgsVstIPluginBase *) AGS_VST3_PLUGIN(base_plugin)->iedit_controller,
				      AGS_VST3_PLUGIN(base_plugin)->host_context);

      break;
    }

    i_stop = ags_vst_iedit_controller_get_parameter_count(AGS_VST3_PLUGIN(base_plugin)->iedit_controller);

    for(i = 0; i < i_stop; i++){
      AgsPluginPort *current_plugin_port;
      
      AgsVstParameterInfo *info;
      AgsVstParamID *id;
      
      guint flags;
      gint32 step_count;
      AgsVstParamValue default_normalized_value;
      
      current_plugin_port = ags_plugin_port_new();
      g_object_ref(current_plugin_port);

      plugin_port = g_list_prepend(plugin_port,
				   current_plugin_port);

      g_value_init(current_plugin_port->default_value,
		   G_TYPE_FLOAT);
      g_value_init(current_plugin_port->lower_value,
		   G_TYPE_FLOAT);
      g_value_init(current_plugin_port->upper_value,
		   G_TYPE_FLOAT);
      
      info = ags_vst_parameter_info_alloc();
      
      ags_vst_iedit_controller_get_parameter_info(AGS_VST3_PLUGIN(base_plugin)->iedit_controller,
						  i, info);

      flags = ags_vst_parameter_info_get_flags(info);

      step_count = ags_vst_parameter_info_get_step_count(info);

      id = ags_vst_parameter_info_get_param_id(info);
      
      current_plugin_port->port_index = i;

      error = NULL;
      current_plugin_port->port_name = g_utf16_to_utf8(ags_vst_parameter_info_get_title(info),
						       128,
						       NULL,
						       NULL,
						       &error);

      if(error != NULL){
	g_warning("%s", error->message);
      }
      
      default_normalized_value = ags_vst_parameter_info_get_default_normalized_value(info);

      if(step_count == 0){
	/* set lower */
	g_value_set_float(current_plugin_port->lower_value,
			  0.0);
	    
	/* set upper */
	g_value_set_float(current_plugin_port->upper_value,
			  1.0);

	/* set default */
	g_value_set_float(current_plugin_port->default_value,
			  ags_vst_iedit_controller_normalized_param_to_plain(AGS_VST3_PLUGIN(base_plugin)->iedit_controller,
									     id,
									     default_normalized_value));
	
	current_plugin_port->scale_steps = -1;
      }else if(step_count == 1){
	/* set lower */
	g_value_set_float(current_plugin_port->lower_value,
			  0.0);
	    
	/* set upper */
	g_value_set_float(current_plugin_port->upper_value,
			  1.0);

	/* set default */
	current_plugin_port->flags |= AGS_PLUGIN_PORT_TOGGLED;

	g_value_set_float(current_plugin_port->default_value,
			  default_normalized_value);
	
	current_plugin_port->scale_steps = step_count;
      }else{
	/* set lower */
	g_value_set_float(current_plugin_port->lower_value,
			  0.0);
	    
	/* set upper */
	g_value_set_float(current_plugin_port->upper_value,
			  (gfloat) step_count);

	/* set default */
	g_value_set_float(current_plugin_port->default_value,
			  fmin(step_count, default_normalized_value * (step_count + 1)));
	
	current_plugin_port->scale_steps = step_count;
      }

      
      if((AGS_VST_KCAN_AUTOMATE & (flags)) != 0){
	//TODO:JK: implement me
      }
      
      if((AGS_VST_KIS_READ_ONLY & (flags)) != 0){
	current_plugin_port->flags |= AGS_PLUGIN_PORT_OUTPUT;
      }else{
	current_plugin_port->flags |= AGS_PLUGIN_PORT_INPUT;
      }
      
      if((AGS_VST_KIS_WRAP_AROUND & (flags)) != 0){	
	//TODO:JK: implement me
      }
      
      if((AGS_VST_KIS_LIST & (flags)) != 0){
	//TODO:JK: implement me
      }
      
      if((AGS_VST_KIS_HIDDEN & (flags)) != 0){
	//TODO:JK: implement me
      }
      
      if((AGS_VST_KIS_PROGRAM_CHANGE & (flags)) != 0){
	//TODO:JK: implement me
      }
      
      if((AGS_VST_KIS_BYPASS & (flags)) != 0){
	current_plugin_port->flags |= AGS_PLUGIN_PORT_TOGGLED;
      }      
      
      ags_vst_parameter_info_free(info);
    }

    ags_vst_pclass_info_free(info);

    base_plugin->plugin_port = g_list_reverse(plugin_port);
  }
}

/**
 * ags_vst3_plugin_new:
 * @filename: the plugin .so
 * @effect: the effect's string representation
 * @effect_index: the effect's index
 *
 * Create a new instance of #AgsVst3Plugin
 *
 * Returns: the new #AgsVst3Plugin
 *
 * Since: 3.10.2
 */
AgsVst3Plugin*
ags_vst3_plugin_new(gchar *filename, gchar *effect, guint effect_index)
{
  AgsVst3Plugin *vst3_plugin;

  vst3_plugin = (AgsVst3Plugin *) g_object_new(AGS_TYPE_VST3_PLUGIN,
					       "filename", filename,
					       "effect", effect,
					       "effect-index", effect_index,
					       NULL);

  return(vst3_plugin);
}
