/* GSequencer - Advanced GTK Sequencer
 * Copyright (C) 2005-2017 Joël Krähemann
 *
 * This file is part of GSequencer.
 *
 * GSequencer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GSequencer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GSequencer.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ags/X/ags_ui_provider.h>

void ags_ui_provider_class_init(AgsUiProviderInterface *ginterface);

/**
 * SECTION:ags_ui_provider
 * @short_description: provide accoustics
 * @title: AgsUiProvider
 * @section_id: AgsUiProvider
 * @include: ags/X/ags_ui_provider.h
 *
 * The #AgsUiProvider provides you the window capabilities.
 */

GType
ags_ui_provider_get_type()
{
  static volatile gsize g_define_type_id__volatile = 0;

  if(g_once_init_enter (&g_define_type_id__volatile)){
    GType ags_type_ui_provider = 0;
    
    ags_type_ui_provider = g_type_register_static_simple(G_TYPE_INTERFACE,
							 "AgsUiProvider",
							 sizeof(AgsUiProviderInterface),
							 (GClassInitFunc) ags_ui_provider_class_init,
							 0, NULL, 0);

    g_once_init_leave(&g_define_type_id__volatile, ags_type_ui_provider);
  }

  return g_define_type_id__volatile;
}

void
ags_ui_provider_class_init(AgsUiProviderInterface *ginterface)
{
  /* empty */
}

/**
 * ags_ui_provider_get_window:
 * @ui_provider: the #AgsUiProvider
 * 
 * Get window.
 * 
 * Returns: the #AgsWindow
 * 
 * Since: 2.0.0
 */
GtkWidget*
ags_ui_provider_get_window(AgsUiProvider *ui_provider)
{
  AgsUiProviderInterface *ui_provider_interface;

  g_return_val_if_fail(AGS_IS_UI_PROVIDER(ui_provider), NULL);
  ui_provider_interface = AGS_UI_PROVIDER_GET_INTERFACE(ui_provider);
  g_return_val_if_fail(ui_provider_interface->get_window, NULL);

  return(ui_provider_interface->get_window(ui_provider));
}

/**
 * ags_ui_provider_get_window:
 * @ui_provider: the #AgsUiProvider
 * @window: the #AgsWindow
 * 
 * Set window.
 * 
 * Since: 2.0.0
 */
void
ags_ui_provider_set_window(AgsUiProvider *ui_provider,
			   GtkWidget *window)
{
  AgsUiProviderInterface *ui_provider_interface;

  g_return_if_fail(AGS_IS_UI_PROVIDER(ui_provider));
  ui_provider_interface = AGS_UI_PROVIDER_GET_INTERFACE(ui_provider);
  g_return_if_fail(ui_provider_interface->set_window);

  ui_provider_interface->set_window(ui_provider,
				    window);
}

/**
 * ags_ui_provider_get_gui_thread:
 * @ui_provider: the #AgsUiProvider
 * 
 * Get gui thread.
 * 
 * Returns: the #AgsGuiThread
 * 
 * Since: 2.0.0
 */
AgsThread*
ags_ui_provider_get_gui_thread(AgsUiProvider *ui_provider)
{
  AgsUiProviderInterface *ui_provider_interface;

  g_return_val_if_fail(AGS_IS_UI_PROVIDER(ui_provider), NULL);
  ui_provider_interface = AGS_UI_PROVIDER_GET_INTERFACE(ui_provider);
  g_return_val_if_fail(ui_provider_interface->get_gui_thread, NULL);

  return(ui_provider_interface->get_gui_thread(ui_provider));
}

/**
 * ags_ui_provider_get_gui_thread:
 * @ui_provider: the #AgsUiProvider
 * @gui_thread: the #AgsGuiThread
 * 
 * Set gui thread.
 * 
 * Since: 2.0.0
 */
void
ags_ui_provider_set_gui_thread(AgsUiProvider *ui_provider,
			       AgsThread *gui_thread)
{
  AgsUiProviderInterface *ui_provider_interface;

  g_return_if_fail(AGS_IS_UI_PROVIDER(ui_provider));
  ui_provider_interface = AGS_UI_PROVIDER_GET_INTERFACE(ui_provider);
  g_return_if_fail(ui_provider_interface->set_gui_thread);

  ui_provider_interface->set_gui_thread(ui_provider,
					gui_thread);
}

/**
 * ags_ui_provider_get_show_animation:
 * @ui_provider: the #AgsUiProvider
 * 
 * Get show animation.
 * 
 * Returns: %TRUE if animation does show, else %FALSE
 * 
 * Since: 2.0.0
 */
gboolean
ags_ui_provider_get_show_animation(AgsUiProvider *ui_provider)
{
  AgsUiProviderInterface *ui_provider_interface;

  g_return_val_if_fail(AGS_IS_UI_PROVIDER(ui_provider), FALSE);
  ui_provider_interface = AGS_UI_PROVIDER_GET_INTERFACE(ui_provider);
  g_return_val_if_fail(ui_provider_interface->get_show_animation, FALSE);

  return(ui_provider_interface->get_show_animation(ui_provider));
}

/**
 * ags_ui_provider_set_show_animation:
 * @ui_provider: the #AgsUiProvider
 * @do_show_animation: do show animation
 * 
 * Set show animation.
 * 
 * Since: 2.0.0
 */
void
ags_ui_provider_set_show_animation(AgsUiProvider *ui_provider,
				   gboolean do_show_animation)
{
  AgsUiProviderInterface *ui_provider_interface;

  g_return_if_fail(AGS_IS_UI_PROVIDER(ui_provider));
  ui_provider_interface = AGS_UI_PROVIDER_GET_INTERFACE(ui_provider);
  g_return_if_fail(ui_provider_interface->set_show_animation);

  ui_provider_interface->set_show_animation(ui_provider,
					    do_show_animation);
}

/**
 * ags_ui_provider_get_gui_ready:
 * @ui_provider: the #AgsUiProvider
 * 
 * Get GUI ready.
 * 
 * Returns: %TRUE if gui is ready, else %FALSE
 * 
 * Since: 2.0.0
 */
gboolean
ags_ui_provider_get_gui_ready(AgsUiProvider *ui_provider)
{
  AgsUiProviderInterface *ui_provider_interface;

  g_return_val_if_fail(AGS_IS_UI_PROVIDER(ui_provider), FALSE);
  ui_provider_interface = AGS_UI_PROVIDER_GET_INTERFACE(ui_provider);
  g_return_val_if_fail(ui_provider_interface->get_gui_ready, FALSE);

  return(ui_provider_interface->get_gui_ready(ui_provider));
}

/**
 * ags_ui_provider_set_gui_ready:
 * @ui_provider: the #AgsUiProvider
 * @is_gui_ready: is GUI ready
 * 
 * Set GUI ready.
 * 
 * Since: 2.0.0
 */
void
ags_ui_provider_set_gui_ready(AgsUiProvider *ui_provider,
			      gboolean is_gui_ready)
{
  AgsUiProviderInterface *ui_provider_interface;

  g_return_if_fail(AGS_IS_UI_PROVIDER(ui_provider));
  ui_provider_interface = AGS_UI_PROVIDER_GET_INTERFACE(ui_provider);
  g_return_if_fail(ui_provider_interface->set_gui_ready);

  ui_provider_interface->set_gui_ready(ui_provider,
				       is_gui_ready);
}

/**
 * ags_ui_provider_get_animation_window:
 * @ui_provider: the #AgsUiProvider
 * 
 * Get animation window.
 * 
 * Returns: the #GtkWidget
 * 
 * Since: 2.2.33
 */
GtkWidget*
ags_ui_provider_get_animation_window(AgsUiProvider *ui_provider)
{
  AgsUiProviderInterface *ui_provider_interface;

  g_return_val_if_fail(AGS_IS_UI_PROVIDER(ui_provider), NULL);
  ui_provider_interface = AGS_UI_PROVIDER_GET_INTERFACE(ui_provider);
  g_return_val_if_fail(ui_provider_interface->get_animation_window, NULL);

  return(ui_provider_interface->get_animation_window(ui_provider));
}

/**
 * ags_ui_provider_get_animation_window:
 * @ui_provider: the #AgsUiProvider
 * @animation_window: the #GtkWidget
 * 
 * Set animation window.
 * 
 * Since: 2.2.33
 */
void
ags_ui_provider_set_animation_window(AgsUiProvider *ui_provider,
				     GtkWidget *animation_window)
{
  AgsUiProviderInterface *ui_provider_interface;

  g_return_if_fail(AGS_IS_UI_PROVIDER(ui_provider));
  ui_provider_interface = AGS_UI_PROVIDER_GET_INTERFACE(ui_provider);
  g_return_if_fail(ui_provider_interface->set_animation_window);

  ui_provider_interface->set_animation_window(ui_provider,
					      animation_window);
}

/**
 * ags_ui_provider_get_machine:
 * @ui_provider: the #AgsUiProvider
 * 
 * Get machine.
 * 
 * Returns: the #GList-struct containing #AgsMachine
 * 
 * Since: 2.4.0
 */
GList*
ags_ui_provider_get_machine(AgsUiProvider *ui_provider)
{
  AgsUiProviderInterface *ui_provider_interface;

  g_return_val_if_fail(AGS_IS_UI_PROVIDER(ui_provider), NULL);
  ui_provider_interface = AGS_UI_PROVIDER_GET_INTERFACE(ui_provider);
  g_return_val_if_fail(ui_provider_interface->get_machine, NULL);

  return(ui_provider_interface->get_machine(ui_provider));
}

/**
 * ags_ui_provider_get_machine:
 * @ui_provider: the #AgsUiProvider
 * @machine: the #GList-struct containing #AgsMachine
 * 
 * Set machine.
 * 
 * Since: 2.4.0
 */
void
ags_ui_provider_set_machine(AgsUiProvider *ui_provider,
			    GList *machine)
{
  AgsUiProviderInterface *ui_provider_interface;

  g_return_if_fail(AGS_IS_UI_PROVIDER(ui_provider));
  ui_provider_interface = AGS_UI_PROVIDER_GET_INTERFACE(ui_provider);
  g_return_if_fail(ui_provider_interface->set_machine);

  ui_provider_interface->set_machine(ui_provider,
				     machine);
}
