/* GSequencer - Advanced GTK Sequencer
 * Copyright (C) 2005-2018 Joël Krähemann
 *
 * This file is part of GSequencer.
 *
 * GSequencer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GSequencer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GSequencer.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ags/audio/pulse/ags_pulse_server.h>
#include <ags/audio/pulse/ags_pulse_client.h>
#include <ags/audio/pulse/ags_pulse_port.h>

#include <ags/libags.h>

#include <ags/audio/pulse/ags_pulse_devout.h>
#include <ags/audio/pulse/ags_pulse_devin.h>

#include <string.h>

#include <errno.h>

#include <ags/i18n.h>

void ags_pulse_server_class_init(AgsPulseServerClass *pulse_server);
void ags_pulse_server_connectable_interface_init(AgsConnectableInterface *connectable);
void ags_pulse_server_sound_server_interface_init(AgsSoundServerInterface *sound_server);
void ags_pulse_server_init(AgsPulseServer *pulse_server);
void ags_pulse_server_set_property(GObject *gobject,
				   guint prop_id,
				   const GValue *value,
				   GParamSpec *param_spec);
void ags_pulse_server_get_property(GObject *gobject,
				   guint prop_id,
				   GValue *value,
				   GParamSpec *param_spec);
void ags_pulse_server_dispose(GObject *gobject);
void ags_pulse_server_finalize(GObject *gobject);

AgsUUID* ags_pulse_server_get_uuid(AgsConnectable *connectable);
gboolean ags_pulse_server_has_resource(AgsConnectable *connectable);
gboolean ags_pulse_server_is_ready(AgsConnectable *connectable);
void ags_pulse_server_add_to_registry(AgsConnectable *connectable);
void ags_pulse_server_remove_from_registry(AgsConnectable *connectable);
xmlNode* ags_pulse_server_list_resource(AgsConnectable *connectable);
xmlNode* ags_pulse_server_xml_compose(AgsConnectable *connectable);
void ags_pulse_server_xml_parse(AgsConnectable *connectable,
				xmlNode *node);
gboolean ags_pulse_server_is_connected(AgsConnectable *connectable);
void ags_pulse_server_connect(AgsConnectable *connectable);
void ags_pulse_server_disconnect(AgsConnectable *connectable);

void ags_pulse_server_set_url(AgsSoundServer *sound_server,
			      gchar *url);
gchar* ags_pulse_server_get_url(AgsSoundServer *sound_server);
void ags_pulse_server_set_ports(AgsSoundServer *sound_server,
				guint *ports, guint port_count);
guint* ags_pulse_server_get_ports(AgsSoundServer *sound_server,
				  guint *port_count);
void ags_pulse_server_set_soundcard(AgsSoundServer *sound_server,
				    gchar *client_uuid,
				    GList *soundcard);
GList* ags_pulse_server_get_soundcard(AgsSoundServer *sound_server,
				      gchar *client_uuid);
void ags_pulse_server_set_sequencer(AgsSoundServer *sound_server,
				    gchar *client_uuid,
				    GList *sequencer);
GList* ags_pulse_server_get_sequencer(AgsSoundServer *sound_server,
				      gchar *client_uuid);
GObject* ags_pulse_server_register_soundcard(AgsSoundServer *sound_server,
					     gboolean is_output);
void ags_pulse_server_unregister_soundcard(AgsSoundServer *sound_server,
					   GObject *soundcard);
GObject* ags_pulse_server_register_sequencer(AgsSoundServer *sound_server,
					     gboolean is_output);
void ags_pulse_server_unregister_sequencer(AgsSoundServer *sound_server,
					   GObject *sequencer);

void* ags_pulse_server_do_poll_loop(void *ptr);

/**
 * SECTION:ags_pulse_server
 * @short_description: pulseaudio instance
 * @title: AgsPulseServer
 * @section_id:
 * @include: ags/audio/pulse/ags_pulse_server.h
 *
 * The #AgsPulseServer is an object to represent a running pulseaudio instance.
 */

enum{
  PROP_0,
  PROP_APPLICATION_CONTEXT,
  PROP_URL,
  PROP_DEFAULT_SOUNDCARD,
  PROP_DEFAULT_PULSE_CLIENT,
  PROP_PULSE_CLIENT,
};

static gpointer ags_pulse_server_parent_class = NULL;

static pthread_mutex_t ags_pulse_server_class_mutex = PTHREAD_MUTEX_INITIALIZER;

GType
ags_pulse_server_get_type()
{
  static volatile gsize g_define_type_id__volatile = 0;

  if(g_once_init_enter (&g_define_type_id__volatile)){
    GType ags_type_pulse_server = 0;

    static const GTypeInfo ags_pulse_server_info = {
      sizeof(AgsPulseServerClass),
      NULL, /* base_init */
      NULL, /* base_finalize */
      (GClassInitFunc) ags_pulse_server_class_init,
      NULL, /* class_finalize */
      NULL, /* class_data */
      sizeof(AgsPulseServer),
      0,    /* n_preallocs */
      (GInstanceInitFunc) ags_pulse_server_init,
    };

    static const GInterfaceInfo ags_connectable_interface_info = {
      (GInterfaceInitFunc) ags_pulse_server_connectable_interface_init,
      NULL, /* interface_finalize */
      NULL, /* interface_data */
    };
    
    static const GInterfaceInfo ags_sound_server_interface_info = {
      (GInterfaceInitFunc) ags_pulse_server_sound_server_interface_init,
      NULL, /* interface_finalize */
      NULL, /* interface_data */
    };

    ags_type_pulse_server = g_type_register_static(G_TYPE_OBJECT,
						   "AgsPulseServer",
						   &ags_pulse_server_info,
						   0);

    g_type_add_interface_static(ags_type_pulse_server,
				AGS_TYPE_CONNECTABLE,
				&ags_connectable_interface_info);

    g_type_add_interface_static(ags_type_pulse_server,
				AGS_TYPE_SOUND_SERVER,
				&ags_sound_server_interface_info);

    g_once_init_leave(&g_define_type_id__volatile, ags_type_pulse_server);
  }

  return g_define_type_id__volatile;
}

void
ags_pulse_server_class_init(AgsPulseServerClass *pulse_server)
{
  GObjectClass *gobject;

  GParamSpec *param_spec;
  
  ags_pulse_server_parent_class = g_type_class_peek_parent(pulse_server);

  /* GObjectClass */
  gobject = (GObjectClass *) pulse_server;

  gobject->set_property = ags_pulse_server_set_property;
  gobject->get_property = ags_pulse_server_get_property;

  gobject->dispose = ags_pulse_server_dispose;
  gobject->finalize = ags_pulse_server_finalize;

  /* properties */
  /**
   * AgsPulseServer:application-context:
   *
   * The assigned #AgsApplicationContext
   * 
   * Since: 2.0.0
   */
  param_spec = g_param_spec_object("application-context",
				   i18n_pspec("the application context object"),
				   i18n_pspec("The application context object"),
				   AGS_TYPE_APPLICATION_CONTEXT,
				   G_PARAM_READABLE | G_PARAM_WRITABLE);
  g_object_class_install_property(gobject,
				  PROP_APPLICATION_CONTEXT,
				  param_spec);

  /**
   * AgsPulseServer:url:
   *
   * The assigned URL.
   * 
   * Since: 2.0.0
   */
  param_spec = g_param_spec_string("url",
				   i18n_pspec("the URL"),
				   i18n_pspec("The URL"),
				   NULL,
				   G_PARAM_READABLE | G_PARAM_WRITABLE);
  g_object_class_install_property(gobject,
				  PROP_URL,
				  param_spec);

  /**
   * AgsPulseServer:default-soundcard:
   *
   * The default soundcard.
   * 
   * Since: 2.0.0
   */
  param_spec = g_param_spec_object("default-soundcard",
				   i18n_pspec("default soundcard"),
				   i18n_pspec("The default soundcard"),
				   G_TYPE_OBJECT,
				   G_PARAM_READABLE | G_PARAM_WRITABLE);
  g_object_class_install_property(gobject,
				  PROP_DEFAULT_SOUNDCARD,
				  param_spec);

  /**
   * AgsPulseServer:default-pulse-client:
   *
   * The default pulse client.
   * 
   * Since: 2.0.0
   */
  param_spec = g_param_spec_object("default-pulse-client",
				   i18n_pspec("default pulse client"),
				   i18n_pspec("The default pulse client"),
				   AGS_TYPE_PULSE_CLIENT,
				   G_PARAM_READABLE | G_PARAM_WRITABLE);
  g_object_class_install_property(gobject,
				  PROP_DEFAULT_PULSE_CLIENT,
				  param_spec);

  /**
   * AgsPulseServer:pulse-client:
   *
   * The pulse client list.
   * 
   * Since: 2.0.0
   */
  param_spec = g_param_spec_pointer("pulse-client",
				    i18n_pspec("pulse client list"),
				    i18n_pspec("The pulse client list"),
				    G_PARAM_READABLE | G_PARAM_WRITABLE);
  g_object_class_install_property(gobject,
				  PROP_PULSE_CLIENT,
				  param_spec);
}

void
ags_pulse_server_connectable_interface_init(AgsConnectableInterface *connectable)
{
  connectable->get_uuid = ags_pulse_server_get_uuid;
  connectable->has_resource = ags_pulse_server_has_resource;

  connectable->is_ready = ags_pulse_server_is_ready;
  connectable->add_to_registry = ags_pulse_server_add_to_registry;
  connectable->remove_from_registry = ags_pulse_server_remove_from_registry;

  connectable->list_resource = ags_pulse_server_list_resource;
  connectable->xml_compose = ags_pulse_server_xml_compose;
  connectable->xml_parse = ags_pulse_server_xml_parse;

  connectable->is_connected = ags_pulse_server_is_connected;  
  connectable->connect = ags_pulse_server_connect;
  connectable->disconnect = ags_pulse_server_disconnect;

  connectable->connect_connection = NULL;
  connectable->disconnect_connection = NULL;
}

void
ags_pulse_server_sound_server_interface_init(AgsSoundServerInterface *sound_server)
{
  sound_server->set_url = ags_pulse_server_set_url;
  sound_server->get_url = ags_pulse_server_get_url;
  sound_server->set_ports = ags_pulse_server_set_ports;
  sound_server->get_ports = ags_pulse_server_get_ports;
  sound_server->set_soundcard = ags_pulse_server_set_soundcard;
  sound_server->get_soundcard = ags_pulse_server_get_soundcard;
  sound_server->set_sequencer = ags_pulse_server_set_sequencer;
  sound_server->get_sequencer = ags_pulse_server_get_sequencer;
  sound_server->register_soundcard = ags_pulse_server_register_soundcard;
  sound_server->unregister_soundcard = ags_pulse_server_unregister_soundcard;
  sound_server->register_sequencer = ags_pulse_server_register_sequencer;
  sound_server->unregister_sequencer = ags_pulse_server_unregister_sequencer;
}

void
ags_pulse_server_init(AgsPulseServer *pulse_server)
{
  pthread_mutex_t *mutex;
  pthread_mutexattr_t *attr;

  /* flags */
  pulse_server->flags = 0;

  /* server mutex */
  pulse_server->obj_mutexattr = 
    attr = (pthread_mutexattr_t *) malloc(sizeof(pthread_mutexattr_t));
  pthread_mutexattr_init(attr);
  pthread_mutexattr_settype(attr,
			    PTHREAD_MUTEX_RECURSIVE);

  pulse_server->obj_mutex = 
    mutex = (pthread_mutex_t *) malloc(sizeof(pthread_mutex_t));
  pthread_mutex_init(mutex,
		     attr);

  g_atomic_int_set(&(pulse_server->running),
		   TRUE);
  pulse_server->thread = (pthread_t *) malloc(sizeof(pthread_t));

  /* parent */
  pulse_server->application_context = NULL;

  /* uuid */
  pulse_server->uuid = ags_uuid_alloc();
  ags_uuid_generate(pulse_server->uuid);
  
#ifdef AGS_WITH_PULSE
  pulse_server->main_loop = pa_mainloop_new();
  pulse_server->main_loop_api = pa_mainloop_get_api(pulse_server->main_loop);
#else
  pulse_server->main_loop = NULL;
  pulse_server->main_loop_api = NULL;
#endif

  pulse_server->url = NULL;
  
  pulse_server->port = NULL;
  pulse_server->port_count = 0;
  
  pulse_server->n_soundcards = 0;
  pulse_server->n_sequencers = 0;

  pulse_server->default_soundcard = NULL;

  pulse_server->default_client = NULL;
  pulse_server->client = NULL;
}

void
ags_pulse_server_set_property(GObject *gobject,
			      guint prop_id,
			      const GValue *value,
			      GParamSpec *param_spec)
{
  AgsPulseServer *pulse_server;

  pthread_mutex_t *pulse_server_mutex;

  pulse_server = AGS_PULSE_SERVER(gobject);

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  switch(prop_id){
  case PROP_APPLICATION_CONTEXT:
    {
      AgsApplicationContext *application_context;

      application_context = (AgsApplicationContext *) g_value_get_object(value);

      pthread_mutex_lock(pulse_server_mutex);

      if(pulse_server->application_context == (GObject *) application_context){
	pthread_mutex_unlock(pulse_server_mutex);

	return;
      }

      if(pulse_server->application_context != NULL){
	g_object_unref(G_OBJECT(pulse_server->application_context));
      }

      if(application_context != NULL){
	g_object_ref(G_OBJECT(application_context));
      }

      pulse_server->application_context = (GObject *) application_context;

      pthread_mutex_unlock(pulse_server_mutex);
    }
    break;
  case PROP_URL:
    {
      gchar *url;

      url = g_value_get_string(value);

      pthread_mutex_lock(pulse_server_mutex);

      if(pulse_server->url == url){
	pthread_mutex_unlock(pulse_server_mutex);

	return;
      }

      if(pulse_server->url != NULL){
	g_free(pulse_server->url);
      }

      pulse_server->url = g_strdup(url);

      pthread_mutex_unlock(pulse_server_mutex);
    }
    break;
  case PROP_DEFAULT_SOUNDCARD:
    {
      GObject *default_soundcard;

      default_soundcard = (GObject *) g_value_get_object(value);

      pthread_mutex_lock(pulse_server_mutex);

      if(pulse_server->default_soundcard == (GObject *) default_soundcard){
	pthread_mutex_unlock(pulse_server_mutex);

	return;
      }

      if(pulse_server->default_soundcard != NULL){
	g_object_unref(G_OBJECT(pulse_server->default_soundcard));
      }

      if(default_soundcard != NULL){
	g_object_ref(G_OBJECT(default_soundcard));
      }

      pulse_server->default_soundcard = (GObject *) default_soundcard;

      pthread_mutex_unlock(pulse_server_mutex);
    }
    break;
  case PROP_DEFAULT_PULSE_CLIENT:
    {
      AgsPulseClient *default_client;

      default_client = (AgsPulseClient *) g_value_get_object(value);

      pthread_mutex_lock(pulse_server_mutex);

      if(pulse_server->default_client == (GObject *) default_client){
	pthread_mutex_unlock(pulse_server_mutex);

	return;
      }

      if(pulse_server->default_client != NULL){
	g_object_unref(G_OBJECT(pulse_server->default_client));
      }

      if(default_client != NULL){
	g_object_ref(G_OBJECT(default_client));
      }

      pulse_server->default_client = (GObject *) default_client;

      pthread_mutex_unlock(pulse_server_mutex);
    }
    break;
  case PROP_PULSE_CLIENT:
    {
      GObject *client;

      client = (GObject *) g_value_get_pointer(value);

      pthread_mutex_lock(pulse_server_mutex);

      if(!AGS_IS_PULSE_CLIENT(client) ||
	 g_list_find(pulse_server->client, client) != NULL){
	pthread_mutex_unlock(pulse_server_mutex);

	return;
      }

      g_object_ref(G_OBJECT(client));
      pulse_server->client = g_list_prepend(pulse_server->client,
					    client);

      pthread_mutex_unlock(pulse_server_mutex);
    }
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(gobject, prop_id, param_spec);
    break;
  }
}

void
ags_pulse_server_get_property(GObject *gobject,
			      guint prop_id,
			      GValue *value,
			      GParamSpec *param_spec)
{
  AgsPulseServer *pulse_server;

  pthread_mutex_t *pulse_server_mutex;

  pulse_server = AGS_PULSE_SERVER(gobject);

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());
  
  switch(prop_id){
  case PROP_APPLICATION_CONTEXT:
    {
      pthread_mutex_lock(pulse_server_mutex);

      g_value_set_object(value, pulse_server->application_context);

      pthread_mutex_unlock(pulse_server_mutex);
    }
    break;
  case PROP_URL:
    {
      pthread_mutex_lock(pulse_server_mutex);

      g_value_set_string(value, pulse_server->url);

      pthread_mutex_unlock(pulse_server_mutex);
    }
    break;
  case PROP_DEFAULT_SOUNDCARD:
    {
      pthread_mutex_lock(pulse_server_mutex);

      g_value_set_object(value, pulse_server->default_soundcard);

      pthread_mutex_unlock(pulse_server_mutex);
    }
    break;
  case PROP_DEFAULT_PULSE_CLIENT:
    {
      pthread_mutex_lock(pulse_server_mutex);

      g_value_set_object(value, pulse_server->default_soundcard);

      pthread_mutex_unlock(pulse_server_mutex);
    }
    break;
  case PROP_PULSE_CLIENT:
    {
      pthread_mutex_lock(pulse_server_mutex);

      g_value_set_pointer(value,
			  g_list_copy(pulse_server->client));

      pthread_mutex_unlock(pulse_server_mutex);
    }
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(gobject, prop_id, param_spec);
    break;
  }
}

void
ags_pulse_server_dispose(GObject *gobject)
{
  AgsPulseServer *pulse_server;

  GList *list;
  
  pulse_server = AGS_PULSE_SERVER(gobject);

  /* application context */
  if(pulse_server->application_context != NULL){
    g_object_unref(G_OBJECT(pulse_server->application_context));
    
    pulse_server->application_context = NULL;
  }

  /* default soundcard */
  if(pulse_server->default_soundcard != NULL){
    g_object_unref(G_OBJECT(pulse_server->default_soundcard));

    pulse_server->default_soundcard = NULL;
  }
  
  /* default client */
  if(pulse_server->default_client != NULL){
    g_object_unref(G_OBJECT(pulse_server->default_client));

    pulse_server->default_client = NULL;
  }
  
  /* client */
  if(pulse_server->client != NULL){
    list = pulse_server->client;

    while(list != NULL){
      g_object_run_dispose(G_OBJECT(list->data));

      list = list->next;
    }
    
    g_list_free_full(pulse_server->client,
		     g_object_unref);

    pulse_server->client = NULL;
  }

  /* call parent */
  G_OBJECT_CLASS(ags_pulse_server_parent_class)->dispose(gobject);
}

void
ags_pulse_server_finalize(GObject *gobject)
{
  AgsPulseServer *pulse_server;

  pulse_server = AGS_PULSE_SERVER(gobject);

  pthread_mutex_destroy(pulse_server->obj_mutex);
  free(pulse_server->obj_mutex);

  pthread_mutexattr_destroy(pulse_server->obj_mutexattr);
  free(pulse_server->obj_mutexattr);
  
  /* application context */
  if(pulse_server->application_context != NULL){
    g_object_unref(G_OBJECT(pulse_server->application_context));
  }

  /* url */
  g_free(pulse_server->url);

  /* default soundcard */
  if(pulse_server->default_soundcard != NULL){
    g_object_unref(G_OBJECT(pulse_server->default_soundcard));
  }
  
  /* default client */
  if(pulse_server->default_client != NULL){
    g_object_unref(G_OBJECT(pulse_server->default_client));
  }
  
  /* client */
  if(pulse_server->client != NULL){
    g_list_free_full(pulse_server->client,
		     g_object_unref);
  }
  
  /* call parent */
  G_OBJECT_CLASS(ags_pulse_server_parent_class)->finalize(gobject);
}

AgsUUID*
ags_pulse_server_get_uuid(AgsConnectable *connectable)
{
  AgsPulseServer *pulse_server;
  
  AgsUUID *ptr;

  pthread_mutex_t *pulse_server_mutex;

  pulse_server = AGS_PULSE_SERVER(connectable);

  /* get pulse server signal mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  /* get UUID */
  pthread_mutex_lock(pulse_server_mutex);

  ptr = pulse_server->uuid;

  pthread_mutex_unlock(pulse_server_mutex);
  
  return(ptr);
}

gboolean
ags_pulse_server_has_resource(AgsConnectable *connectable)
{
  return(FALSE);
}

gboolean
ags_pulse_server_is_ready(AgsConnectable *connectable)
{
  AgsPulseServer *pulse_server;
  
  gboolean is_ready;

  pthread_mutex_t *pulse_server_mutex;

  pulse_server = AGS_PULSE_SERVER(connectable);

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  /* check is added */
  pthread_mutex_lock(pulse_server_mutex);

  is_ready = (((AGS_PULSE_SERVER_ADDED_TO_REGISTRY & (pulse_server->flags)) != 0) ? TRUE: FALSE);

  pthread_mutex_unlock(pulse_server_mutex);
  
  return(is_ready);
}

void
ags_pulse_server_add_to_registry(AgsConnectable *connectable)
{
  AgsPulseServer *pulse_server;

  if(ags_connectable_is_ready(connectable)){
    return;
  }
  
  pulse_server = AGS_PULSE_SERVER(connectable);

  ags_pulse_server_set_flags(pulse_server, AGS_PULSE_SERVER_ADDED_TO_REGISTRY);
}

void
ags_pulse_server_remove_from_registry(AgsConnectable *connectable)
{
  AgsPulseServer *pulse_server;

  if(!ags_connectable_is_ready(connectable)){
    return;
  }

  pulse_server = AGS_PULSE_SERVER(connectable);

  ags_pulse_server_unset_flags(pulse_server, AGS_PULSE_SERVER_ADDED_TO_REGISTRY);
}

xmlNode*
ags_pulse_server_list_resource(AgsConnectable *connectable)
{
  xmlNode *node;
  
  node = NULL;

  //TODO:JK: implement me
  
  return(node);
}

xmlNode*
ags_pulse_server_xml_compose(AgsConnectable *connectable)
{
  xmlNode *node;
  
  node = NULL;

  //TODO:JK: implement me
  
  return(node);
}

void
ags_pulse_server_xml_parse(AgsConnectable *connectable,
			   xmlNode *node)
{
  //TODO:JK: implement me
}

gboolean
ags_pulse_server_is_connected(AgsConnectable *connectable)
{
  AgsPulseServer *pulse_server;
  
  gboolean is_connected;

  pthread_mutex_t *pulse_server_mutex;

  pulse_server = AGS_PULSE_SERVER(connectable);

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  /* check is connected */
  pthread_mutex_lock(pulse_server_mutex);

  is_connected = (((AGS_PULSE_SERVER_CONNECTED & (pulse_server->flags)) != 0) ? TRUE: FALSE);
  
  pthread_mutex_unlock(pulse_server_mutex);
  
  return(is_connected);
}

void
ags_pulse_server_connect(AgsConnectable *connectable)
{
  AgsPulseServer *pulse_server;

  GList *list_start, *list;  

  pthread_mutex_t *pulse_server_mutex;
  
  if(ags_connectable_is_connected(connectable)){
    return;
  }

  pulse_server = AGS_PULSE_SERVER(connectable);

  ags_pulse_server_set_flags(pulse_server, AGS_PULSE_SERVER_CONNECTED);

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  list =
    list_start = g_list_copy(pulse_server->client);

  while(list != NULL){
    ags_connectable_connect(AGS_CONNECTABLE(list->data));

    list = list->next;
  }

  g_list_free(list_start);
}

void
ags_pulse_server_disconnect(AgsConnectable *connectable)
{
  AgsPulseServer *pulse_server;

  GList *list_start, *list;

  pthread_mutex_t *pulse_server_mutex;

  if(!ags_connectable_is_connected(connectable)){
    return;
  }

  pulse_server = AGS_PULSE_SERVER(connectable);
  
  ags_pulse_server_unset_flags(pulse_server, AGS_PULSE_SERVER_CONNECTED);

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  /* client */
  list =
    list_start = g_list_copy(pulse_server->client);

  while(list != NULL){
    ags_connectable_disconnect(AGS_CONNECTABLE(list->data));

    list = list->next;
  }

  g_list_free(list_start);
}

/**
 * ags_pulse_server_get_class_mutex:
 * 
 * Use this function's returned mutex to access mutex fields.
 *
 * Returns: the class mutex
 * 
 * Since: 2.0.0
 */
pthread_mutex_t*
ags_pulse_server_get_class_mutex()
{
  return(&ags_pulse_server_class_mutex);
}

/**
 * ags_pulse_server_test_flags:
 * @pulse_server: the #AgsPulseServer
 * @flags: the flags
 *
 * Test @flags to be set on @pulse_server.
 * 
 * Returns: %TRUE if flags are set, else %FALSE
 *
 * Since: 2.0.0
 */
gboolean
ags_pulse_server_test_flags(AgsPulseServer *pulse_server, guint flags)
{
  gboolean retval;  
  
  pthread_mutex_t *pulse_server_mutex;

  if(!AGS_IS_PULSE_SERVER(pulse_server)){
    return(FALSE);
  }

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  /* test */
  pthread_mutex_lock(pulse_server_mutex);

  retval = (flags & (pulse_server->flags)) ? TRUE: FALSE;
  
  pthread_mutex_unlock(pulse_server_mutex);

  return(retval);
}

/**
 * ags_pulse_server_set_flags:
 * @pulse_server: the #AgsPulseServer
 * @flags: see #AgsPulseServerFlags-enum
 *
 * Enable a feature of @pulse_server.
 *
 * Since: 2.0.0
 */
void
ags_pulse_server_set_flags(AgsPulseServer *pulse_server, guint flags)
{
  pthread_mutex_t *pulse_server_mutex;

  if(!AGS_IS_PULSE_SERVER(pulse_server)){
    return;
  }

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  //TODO:JK: add more?

  /* set flags */
  pthread_mutex_lock(pulse_server_mutex);

  pulse_server->flags |= flags;
  
  pthread_mutex_unlock(pulse_server_mutex);
}
    
/**
 * ags_pulse_server_unset_flags:
 * @pulse_server: the #AgsPulseServer
 * @flags: see #AgsPulseServerFlags-enum
 *
 * Disable a feature of @pulse_server.
 *
 * Since: 2.0.0
 */
void
ags_pulse_server_unset_flags(AgsPulseServer *pulse_server, guint flags)
{  
  pthread_mutex_t *pulse_server_mutex;

  if(!AGS_IS_PULSE_SERVER(pulse_server)){
    return;
  }

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  //TODO:JK: add more?

  /* unset flags */
  pthread_mutex_lock(pulse_server_mutex);

  pulse_server->flags &= (~flags);
  
  pthread_mutex_unlock(pulse_server_mutex);
}

void
ags_pulse_server_set_url(AgsSoundServer *sound_server,
			 gchar *url)
{
  AgsPulseServer *pulse_server;

  pthread_mutex_t *pulse_server_mutex;

  pulse_server = AGS_PULSE_SERVER(sound_server);

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  /* set URL */
  pthread_mutex_lock(pulse_server_mutex);

  pulse_server->url = g_strdup(url);

  pthread_mutex_unlock(pulse_server_mutex);
}

gchar*
ags_pulse_server_get_url(AgsSoundServer *sound_server)
{
  AgsPulseServer *pulse_server;

  gchar *url;

  pthread_mutex_t *pulse_server_mutex;

  pulse_server = AGS_PULSE_SERVER(sound_server);

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  /* set URL */
  pthread_mutex_lock(pulse_server_mutex);

  url = pulse_server->url;

  pthread_mutex_unlock(pulse_server_mutex);
  
  return(url);
}


void
ags_pulse_server_set_ports(AgsSoundServer *sound_server,
			   guint *port, guint port_count)
{
  AgsPulseServer *pulse_server;

  pthread_mutex_t *pulse_server_mutex;

  pulse_server = AGS_PULSE_SERVER(sound_server);

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  /* set ports */
  pthread_mutex_lock(pulse_server_mutex);

  pulse_server->port = port;
  pulse_server->port_count = port_count;

  pthread_mutex_unlock(pulse_server_mutex);
}

guint*
ags_pulse_server_get_ports(AgsSoundServer *sound_server,
			   guint *port_count)
{
  AgsPulseServer *pulse_server;

  guint *port;
  
  pthread_mutex_t *pulse_server_mutex;

  pulse_server = AGS_PULSE_SERVER(sound_server);

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  /* get ports */
  pthread_mutex_lock(pulse_server_mutex);

  if(port_count != NULL){
    *port_count = AGS_PULSE_SERVER(sound_server)->port_count;
  }

  port = pulse_server->port;

  pthread_mutex_unlock(pulse_server_mutex);
  
  return(port);
}

void
ags_pulse_server_set_soundcard(AgsSoundServer *sound_server,
			       gchar *client_uuid,
			       GList *soundcard)
{
  AgsPulseServer *pulse_server;
  AgsPulseClient *pulse_client;

  GList *list;

  pulse_server = AGS_PULSE_SERVER(sound_server);
  
  pulse_client = (AgsPulseClient *) ags_pulse_server_find_client(pulse_server,
								 client_uuid);

  if(!AGS_IS_PULSE_CLIENT(pulse_client)){
    return;
  }
  
  //NOTE:JK: soundcard won't removed
  list = soundcard;

  while(list != NULL){
    ags_pulse_client_add_device(pulse_client,
				(GObject *) list->data);
    
    list = list->next;
  }
}

GList*
ags_pulse_server_get_soundcard(AgsSoundServer *sound_server,
			       gchar *client_uuid)
{
  AgsPulseServer *pulse_server;
  AgsPulseClient *pulse_client;

  GList *device_start, *device;
  GList *list;
  
  pulse_server = AGS_PULSE_SERVER(sound_server);

  pulse_client = (AgsPulseClient *) ags_pulse_server_find_client(pulse_server,
								 client_uuid);

  if(!AGS_IS_PULSE_CLIENT(pulse_client)){
    return(NULL);
  }

  g_object_get(pulse_client,
	       "device", &device_start,
	       NULL);
  
  device = device_start;
  list = NULL;

  while(device != NULL){
    if(AGS_IS_PULSE_DEVOUT(device->data) ||
       AGS_IS_PULSE_DEVIN(device->data)){
      list = g_list_prepend(list,
			    device->data);
    }

    device = device->next;
  }

  return(g_list_reverse(list));
}


void
ags_pulse_server_set_sequencer(AgsSoundServer *sound_server,
			       gchar *client_uuid,
			       GList *sequencer)
{
  AgsPulseServer *pulse_server;
  AgsPulseClient *pulse_client;

  GList *list;

  pulse_server = AGS_PULSE_SERVER(sound_server);

  pulse_client = (AgsPulseClient *) ags_pulse_server_find_client(pulse_server,
								 client_uuid);

  if(!AGS_IS_PULSE_CLIENT(pulse_client)){
    return;
  }

  //NOTE:JK: sequencer won't removed
  list = sequencer;

  while(list != NULL){
    ags_pulse_client_add_device(pulse_client,
				(GObject *) list->data);
    
    list = list->next;
  }
}

GList*
ags_pulse_server_get_sequencer(AgsSoundServer *sound_server,
			       gchar *client_uuid)
{
  AgsPulseServer *pulse_server;
  AgsPulseClient *pulse_client;

  GList *device_start, *device;
  GList *list;
  
  pulse_server = AGS_PULSE_SERVER(sound_server);

  pulse_client = (AgsPulseClient *) ags_pulse_server_find_client(pulse_server,
								 client_uuid);

  if(!AGS_IS_PULSE_CLIENT(pulse_client)){
    return(NULL);
  }

  g_object_get(pulse_client,
	       "device", &device_start,
	       NULL);
  
  device = device_start;
  list = NULL;

#if 0  
  while(device != NULL){
    if(AGS_IS_PULSE_MIDIIN(device->data)){
      list = g_list_prepend(list,
			    device->data);
    }

    device = device->next;
  }
#endif
  
  return(g_list_reverse(list));
}

GObject*
ags_pulse_server_register_soundcard(AgsSoundServer *sound_server,
				    gboolean is_output)
{
  AgsPulseServer *pulse_server;
  AgsPulseClient *default_client;
  AgsPulsePort *pulse_port;
  AgsPulseDevout *pulse_devout;
  AgsPulseDevin *pulse_devin;

  AgsApplicationContext *application_context;

  GObject *soundcard;
  
#ifdef AGS_WITH_PULSE
  pa_context *context;
#else
  gpointer context;
#endif

  gchar *str;  

  guint n_soundcards;
  gboolean initial_set;
  guint i;  

  pthread_mutex_t *pulse_server_mutex;
  pthread_mutex_t *pulse_client_mutex;

  pulse_server = AGS_PULSE_SERVER(sound_server);

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  /* the default client */
  initial_set = FALSE;
  
  /* get some fields */
  pthread_mutex_lock(pulse_server_mutex);

  application_context= pulse_server->application_context;

  default_client = pulse_server->default_client;

  n_soundcards = pulse_server->n_soundcards;
  
  pthread_mutex_unlock(pulse_server_mutex);

  /* the default client */
  if(default_client == NULL){
    default_client = ags_pulse_client_new((GObject *) pulse_server);
    
    g_object_set(pulse_server,
		 "default-pulse-client", default_client,
		 NULL);
    ags_pulse_server_add_client(pulse_server,
				default_client);
    
    ags_pulse_client_open((AgsPulseClient *) default_client,
			  "ags-default-client");
    initial_set = TRUE;    
  }

  /* get pulse client mutex */
  pthread_mutex_lock(ags_pulse_client_get_class_mutex());
  
  pulse_client_mutex = default_client->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_client_get_class_mutex());

  /* get context */
  pthread_mutex_lock(pulse_client_mutex);

  context = default_client->context;

  pthread_mutex_unlock(pulse_client_mutex);
  
  if(context == NULL){
    g_warning("ags_pulse_server.c - can't open pulseaudio client");
  }

  /* the soundcard */
  soundcard = NULL;

  /* the soundcard */
  if(is_output){
    soundcard = 
      pulse_devout = ags_pulse_devout_new(application_context);
    
    str = g_strdup_printf("ags-pulse-devout-%d",
			  n_soundcards);
    
    g_object_set(AGS_PULSE_DEVOUT(pulse_devout),
		 "pulse-client", default_client,
		 "device", str,
		 NULL);
    g_free(str);
            
    /* register ports */      
    pulse_port = ags_pulse_port_new((GObject *) default_client);

    str = g_strdup_printf("ags-soundcard%d",
			  n_soundcards);

    g_object_set(pulse_port,
		 "pulse-devout", pulse_devout,
		 NULL);
    ags_pulse_client_add_port(default_client,
			      (GObject *) pulse_port);

    g_object_set(pulse_devout,
		 "pulse-port", pulse_port,
		 NULL);

    pulse_devout->port_name = (gchar **) malloc(2 * sizeof(gchar *));
    pulse_devout->port_name[0] = g_strdup(str);
    pulse_devout->port_name[1] = NULL;
    
    ags_pulse_port_register(pulse_port,
			    str,
			    TRUE, FALSE,
			    TRUE);

    ags_pulse_devout_realloc_buffer(pulse_devout);

    g_object_set(default_client,
		 "device", pulse_devout,
		 NULL);

    /* increment n-soundcards */
    pthread_mutex_lock(pulse_server_mutex);

    pulse_server->n_soundcards += 1;

    pthread_mutex_unlock(pulse_server_mutex);
  }else{
    soundcard = 
      pulse_devin = ags_pulse_devin_new(application_context);
    
    str = g_strdup_printf("ags-pulse-devin-%d",
			  pulse_server->n_soundcards);
    
    g_object_set(AGS_PULSE_DEVIN(pulse_devin),
		 "pulse-client", default_client,
		 "device", str,
		 NULL);
    g_free(str);
        
    /* register ports */
    str = g_strdup_printf("ags-soundcard%d",
			  n_soundcards);
    
#ifdef AGS_DEBUG
    g_message("%s", str);
#endif
      
    pulse_port = ags_pulse_port_new((GObject *) default_client);
    g_object_set(pulse_port,
		 "pulse-devin", pulse_devin,
		 NULL);
    ags_pulse_client_add_port(default_client,
			      (GObject *) pulse_port);

    g_object_set(pulse_devin,
		 "pulse-port", pulse_port,
		 NULL);
      
    pulse_devin->port_name = (gchar **) malloc(2 * sizeof(gchar *));
    pulse_devin->port_name[0] = g_strdup(str);
    pulse_devin->port_name[1] = NULL;
    
    ags_pulse_port_register(pulse_port,
			    str,
			    TRUE, FALSE,
			    TRUE);

    ags_pulse_devin_realloc_buffer(pulse_devin);

    g_object_set(default_client,
		 "device", pulse_devin,
		 NULL);
    
    /* increment n-soundcards */
    pthread_mutex_lock(pulse_server_mutex);

    pulse_server->n_soundcards += 1;

    pthread_mutex_unlock(pulse_server_mutex);
  }
  
  return((GObject *) soundcard);
}

void
ags_pulse_server_unregister_soundcard(AgsSoundServer *sound_server,
				      GObject *soundcard)
{
  AgsPulseServer *pulse_server;
  AgsPulseClient *default_client;

  GList *list_start, *list;
  GList *port;
  
  pthread_mutex_t *pulse_server_mutex;

  pulse_server = AGS_PULSE_SERVER(sound_server);

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());
  
  /* the default client */
  g_object_get(pulse_server,
	       "default-client", &default_client,
	       NULL);

  if(default_client == NULL){
    g_warning("GSequencer - no pulse client");
    
    return;
  }
  
  if(AGS_IS_PULSE_DEVOUT(soundcard)){
    g_object_get(soundcard,
		 "pulse-port", &list_start,
		 NULL);

    list = list_start;

    while(list != NULL){
      ags_pulse_port_unregister(list->data);
      ags_pulse_client_remove_port(default_client,
				   list->data);
    
      list = list->next;
    }

    g_list_free(list_start);
  }else if(AGS_IS_PULSE_DEVIN(soundcard)){
    g_object_get(soundcard,
		 "pulse-port", &list_start,
		 NULL);

    list = list_start;

    while(list != NULL){
      ags_pulse_port_unregister(list->data);
      ags_pulse_client_remove_port(default_client,
				   list->data);
    
      list = list->next;
    }

    g_list_free(list_start);
  }
  
  ags_pulse_client_remove_device(default_client,
				 soundcard);
  
  g_object_get(default_client,
	       "port", &port,
	       NULL);
  
  if(port == NULL){
    /* reset n-soundcards */
    pthread_mutex_lock(pulse_server_mutex);

    pulse_server->n_soundcards = 0;

    pthread_mutex_unlock(pulse_server_mutex);
  }

  g_list_free(port);
}

GObject*
ags_pulse_server_register_sequencer(AgsSoundServer *sound_server,
				    gboolean is_output)
{
  g_message("GSequencer - can't register pulseaudio sequencer");
  
  return(NULL);
}

void
ags_pulse_server_unregister_sequencer(AgsSoundServer *sound_server,
				      GObject *sequencer)
{
  g_message("GSequencer - can't unregister pulseaudio sequencer");
}

/**
 * ags_pulse_server_register_default_soundcard:
 * @pulse_server: the #AgsPulseServer
 * 
 * Register default soundcard.
 * 
 * Returns: the instantiated #AgsPulseDevout
 * 
 * Since: 2.0.0
 */
GObject*
ags_pulse_server_register_default_soundcard(AgsPulseServer *pulse_server)
{
  AgsPulseClient *default_client;
  AgsPulseDevout *pulse_devout;
  AgsPulsePort *pulse_port;

  AgsApplicationContext *application_context;
  
#ifdef AGS_WITH_PULSE
  pa_context *context;
#else
  gpointer context;
#endif

  gchar *str;
  
  guint i;

  pthread_mutex_t *pulse_server_mutex;
  pthread_mutex_t *pulse_client_mutex;

  if(!AGS_IS_PULSE_SERVER(pulse_server)){
    return(NULL);
  }

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  /* get some fields */
  pthread_mutex_lock(pulse_server_mutex);

  application_context = pulse_server->application_context;

  default_client = pulse_server->default_client;

  pthread_mutex_unlock(pulse_server_mutex);
  
  /* the default client */
  g_object_get(pulse_server,
	       "default-client", &default_client,
	       NULL);
  
  /* the default client */
  if(default_client == NULL){
    default_client = ags_pulse_client_new((GObject *) pulse_server);
    
    g_object_set(pulse_server,
		 "default-pulse-client", default_client,
		 NULL);
    ags_pulse_server_add_client(pulse_server,
				default_client);
    
    ags_pulse_client_open((AgsPulseClient *) default_client,
			  "ags-default-client");
  }

  /* get pulse client mutex */
  pthread_mutex_lock(ags_pulse_client_get_class_mutex());
  
  pulse_client_mutex = default_client->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_client_get_class_mutex());

  /* get context */
  pthread_mutex_lock(pulse_client_mutex);

  context = default_client->context;

  pthread_mutex_unlock(pulse_client_mutex);
  
  if(context == NULL){
    g_warning("ags_pulse_server.c - can't open pulseaudio client");
  }

  /* the soundcard */
  pulse_devout = ags_pulse_devout_new(application_context);
  
  g_object_set(AGS_PULSE_DEVOUT(pulse_devout),
	       "pulse-client", default_client,
	       "device", "ags-default-devout",
	       NULL);
    
  /* register ports */
  str = g_strdup_printf("ags-default-soundcard");

#ifdef AGS_DEBUG
  g_message("%s", str);
#endif
    
  pulse_port = ags_pulse_port_new((GObject *) default_client);
  g_object_set(pulse_port,
	       "pulse-devout", pulse_devout,
	       NULL);
  ags_pulse_client_add_port(default_client,
			    (GObject *) pulse_port);

  g_object_set(pulse_devout,
	       "pulse-port", pulse_port,
	       NULL);
  
  pulse_devout->port_name = (gchar **) malloc(2 * sizeof(gchar *));
  pulse_devout->port_name[0] = g_strdup(str);
  pulse_devout->port_name[1] = NULL;
  
  ags_pulse_port_register(pulse_port,
			  str,
			  TRUE, FALSE,
			  TRUE);

  g_free(str);

  g_object_set(default_client,
	       "device", pulse_devout,
	       NULL);
  
  return((GObject *) pulse_devout);
}

/**
 * ags_pulse_server_find_url:
 * @pulse_server: the #GList-struct containing #AgsPulseServer
 * @url: the url to find
 *
 * Find #AgsPulseServer by url.
 *
 * Returns: the #GList-struct containing a #AgsPulseServer matching @url or %NULL
 *
 * Since: 2.0.0
 */
GList*
ags_pulse_server_find_url(GList *pulse_server,
			  gchar *url)
{
  GList *retval;
  
  pthread_mutex_t *pulse_server_mutex;

  retval = NULL;
  
  while(pulse_server != NULL){
    /* get pulse server mutex */
    pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
    pulse_server_mutex = AGS_PULSE_SERVER(pulse_server->data)->obj_mutex;
  
    pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

    /* check URL */
    pthread_mutex_lock(pulse_server_mutex);
    
    if(!g_ascii_strcasecmp(AGS_PULSE_SERVER(pulse_server->data)->url,
			   url)){
      retval = pulse_server;

      pthread_mutex_unlock(pulse_server_mutex);
    
      break;
    }

    pthread_mutex_unlock(pulse_server_mutex);
    
    pulse_server = pulse_server->next;
  }

  return(retval);
}

/**
 * ags_pulse_server_find_client:
 * @pulse_server: the #AgsPulseServer
 * @client_uuid: the uuid to find
 *
 * Find #AgsPulseClient by uuid.
 *
 * Returns: the #AgsPulseClient found or %NULL
 *
 * Since: 2.0.0
 */
GObject*
ags_pulse_server_find_client(AgsPulseServer *pulse_server,
			     gchar *client_uuid)
{
  AgsPulseClient *retval;
  
  GList *list_start, *list;

  pthread_mutex_t *pulse_server_mutex;
  pthread_mutex_t *pulse_client_mutex;

  if(!AGS_IS_PULSE_SERVER(pulse_server)){
    return(NULL);
  }

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  /* get some fields */
  pthread_mutex_lock(pulse_server_mutex);

  list =
    list_start = g_list_copy(pulse_server->client);

  pthread_mutex_unlock(pulse_server_mutex);

  retval = NULL;
  
  while(list != NULL){
    /* get pulse client mutex */
    pthread_mutex_lock(ags_pulse_client_get_class_mutex());
  
    pulse_client_mutex = AGS_PULSE_CLIENT(list->data)->obj_mutex;
  
    pthread_mutex_unlock(ags_pulse_client_get_class_mutex());

    /* check client UUID */
    pthread_mutex_lock(pulse_client_mutex);
    
    if(!g_ascii_strcasecmp(AGS_PULSE_CLIENT(list->data)->client_uuid,
			   client_uuid)){
      retval = list->data;

      pthread_mutex_unlock(pulse_client_mutex);

      break;
    }

    pthread_mutex_unlock(pulse_client_mutex);
    
    list = list->next;
  }

  g_list_free(list_start);
  
  return(retval);
}

/**
 * ags_pulse_server_find_port:
 * @pulse_server: the #AgsPulseServer
 * @port_uuid: the uuid to find
 *
 * Find #AgsPulsePort by uuid.
 *
 * Returns: the #AgsPulsePort found or %NULL
 *
 * Since: 2.0.0
 */
GObject*
ags_pulse_server_find_port(AgsPulseServer *pulse_server,
			   gchar *port_uuid)
{
  GList *client_start, *client;
  GList *port_start, *port;

  gboolean success;
  
  pthread_mutex_t *pulse_port_mutex;

  g_object_get(pulse_server,
	       "pulse-client", &client_start,
	       NULL);

  client = client_start;
  
  while(client != NULL){
    g_object_get(pulse_server,
		 "pulse-port", &port_start,
		 NULL);

    port_start = port;
    
    while(port != NULL){
      /* get pulse port mutex */
      pthread_mutex_lock(ags_pulse_port_get_class_mutex());
  
      pulse_port_mutex = AGS_PULSE_PORT(port->data)->obj_mutex;
  
      pthread_mutex_unlock(ags_pulse_port_get_class_mutex());
      
      /* check port UUID */
      pthread_mutex_lock(pulse_port_mutex);
      
      success = (!g_ascii_strcasecmp(AGS_PULSE_PORT(port->data)->port_uuid,
				     port_uuid)) ? TRUE: FALSE;

      pthread_mutex_unlock(pulse_port_mutex);
      
      if(success){
	g_list_free(client_start);
	g_list_free(port_start);
	
	return(port->data);
      }

      /* iterate */
      port = port->next;
    }

    g_list_free(port_start);

    /* iterate */
    client = client->next;
  }

  g_list_free(client_start);
  
  return(NULL);
}

/**
 * ags_pulse_server_add_client:
 * @pulse_server: the #AgsPulseServer
 * @pulse_client: the #AgsPulseClient to add
 *
 * Add @pulse_client to @pulse_server
 *
 * Since: 2.0.0
 */
void
ags_pulse_server_add_client(AgsPulseServer *pulse_server,
			    GObject *pulse_client)
{
  pthread_mutex_t *pulse_server_mutex;

  if(!AGS_IS_PULSE_SERVER(pulse_server) ||
     !AGS_IS_PULSE_CLIENT(pulse_client)){
    return;
  }

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  /* get some fields */
  pthread_mutex_lock(pulse_server_mutex);

  if(g_list_find(pulse_server->client, pulse_client) == NULL){
    g_object_ref(pulse_client);
    pulse_server->client = g_list_prepend(pulse_server->client,
					  pulse_client);
  }

  pthread_mutex_unlock(pulse_server_mutex);
}

/**
 * ags_pulse_server_remove_client:
 * @pulse_server: the #AgsPulseServer
 * @pulse_client: the #AgsPulseClient to remove
 *
 * Remove @pulse_client to @pulse_server
 *
 * Since: 2.0.0
 */
void
ags_pulse_server_remove_client(AgsPulseServer *pulse_server,
			       GObject *pulse_client)
{
  pthread_mutex_t *pulse_server_mutex;

  if(!AGS_IS_PULSE_SERVER(pulse_server) ||
     !AGS_IS_PULSE_CLIENT(pulse_client)){
    return;
  }

  /* get pulse server mutex */
  pthread_mutex_lock(ags_pulse_server_get_class_mutex());
  
  pulse_server_mutex = pulse_server->obj_mutex;
  
  pthread_mutex_unlock(ags_pulse_server_get_class_mutex());

  /* get some fields */
  pthread_mutex_lock(pulse_server_mutex);

  if(g_list_find(pulse_server->client, pulse_client) != NULL){
    pulse_server->client = g_list_remove(pulse_server->client,
					 pulse_client);
    g_object_unref(pulse_client);
  }

  pthread_mutex_unlock(pulse_server_mutex);
}

/**
 * ags_pulse_server_connect_client:
 * @pulse_server: the #AgsPulseServer
 *
 * Connect all clients.
 *
 * Since: 2.0.0
 */
void
ags_pulse_server_connect_client(AgsPulseServer *pulse_server)
{
  GList *client_start, *client;

  gchar *client_name;
  
  if(!AGS_IS_PULSE_SERVER(pulse_server)){
    return;
  }

  g_object_get(pulse_server,
	       "pulse-client", &client_start,
	       NULL);
  
  client = client_start;

  while(client != NULL){
    /* client name */
    g_object_get(client->data,
		 "client-name", &client_name,
		 NULL);
    
    /* open */
    ags_pulse_client_open((AgsPulseClient *) client->data,
			  client_name);
    ags_pulse_client_activate(client->data);
    
    /* iterate */
    client = client->next;
  }

  g_list_free(client_start);
}

void*
ags_pulse_server_do_poll_loop(void *ptr)
{
  AgsPulseServer *pulse_server;

#ifndef __APPLE__
  struct sched_param param;
#endif

  pulse_server = (AgsPulseServer *) ptr;
    
  /* Declare ourself as a real time task */
#ifndef __APPLE__
  param.sched_priority = AGS_RT_PRIORITY;
  
  if(sched_setscheduler(0, SCHED_FIFO, &param) == -1) {
    perror("sched_setscheduler failed");
  }
#endif

#ifdef AGS_WITH_PULSE
  pa_mainloop_run(pulse_server->main_loop,
		  NULL);
#endif

  pthread_exit(NULL);
}

void
ags_pulse_server_start_poll(AgsPulseServer *pulse_server)
{
  pthread_create(pulse_server->thread, NULL,
		 ags_pulse_server_do_poll_loop, pulse_server);
}

/**
 * ags_pulse_server_new:
 * @application_context: the #AgsApplicationContext
 * @url: the URL as string
 *
 * Create a new instance of #AgsPulseServer.
 *
 * Returns: the new #AgsPulseServer
 *
 * Since: 2.0.0
 */
AgsPulseServer*
ags_pulse_server_new(AgsApplicationContext *application_context,
		     gchar *url)
{
  AgsPulseServer *pulse_server;

  pulse_server = (AgsPulseServer *) g_object_new(AGS_TYPE_PULSE_SERVER,
						 "application-context", application_context,
						 "url", url,
						 NULL);

  return(pulse_server);
}
