/********************************************************************************
*
* DrawTickMarks3d.c
*
* Draws tick marks and tick-mark labels for 3-dimensional plots.
*
* Copyright  2008 Spencer A. Buckner
* http://savannah.gnu.org/projects/gsegrafix
*
* This file is part of GNUGraphics.
*
* This file is part of GSEGrafix, a scientific and engineering plotting program.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
********************************************************************************/


#include <math.h>
#include "gsegraf.h"


void DrawTickMarks3d ( void )
   {
   /* Declare variables */
   int quadrant, i, nxticks, nyticks, nzticks, nticks;
   double origin[3], axis1[3], axis2[3], axis3[3],
          xmin, xmax, ymin, ymax, zmin, zmax, delta,
          *tickvalues, xscale, yscale, axismin,
          x1, x2, y1, y2, xtick, ytick, angle,
          xlabel1, xlabel2, ylabel1, ylabel2,
          x11, y11, x22, y22;
   char string[21];
   GnomeCanvasPoints *points;
   GnomeCanvasItem *line, *text;


   /* Check plot box on */
   if ( strcmp(p_plot_param->plot_box, "off") == 0 )
      return;


   /* Get quadrant */
   quadrant = p_plot_param_3d->quadrant;


   /* Get axes */
   for ( i=1; i<=3; i++ )
      {
      axis1[i-1]  = p_plot_param_3d->axis1[i-1];
      axis2[i-1]  = p_plot_param_3d->axis2[i-1];
      axis3[i-1]  = p_plot_param_3d->axis3[i-1];
      }


   /* Get origin */
   for ( i=1; i<=3; i++ )
      origin[i-1] = p_plot_param_3d->origin[i-1];


   /* Get minimum and maximum axis values */
   nxticks = p_ticklabels->nxvalues;
   nyticks = p_ticklabels->nyvalues;
   nzticks = p_ticklabels->nzvalues;
   xmin = p_ticklabels->xvalues[0];
   xmax = p_ticklabels->xvalues[nxticks-1];
   ymin = p_ticklabels->yvalues[0];
   ymax = p_ticklabels->yvalues[nyticks-1];
   zmin = p_ticklabels->zvalues[0];
   zmax = p_ticklabels->zvalues[nzticks-1];
   xmin = xmin - p_ticklabels->xoffset1;
   xmax = xmax + p_ticklabels->xoffset2;
   ymin = ymin - p_ticklabels->yoffset1;
   ymax = ymax + p_ticklabels->yoffset2;
   zmin = zmin - p_ticklabels->zoffset1;
   zmax = zmax + p_ticklabels->zoffset2;


   /* Axis 1 tick-mark parameters */
   if ( quadrant == 1 )
      {
      x1 = origin[1] + axis2[1];
      y1 = origin[2] - axis2[2];
      x2 = origin[1] + axis1[1] + axis2[1];
      y2 = origin[2] - axis1[2] - axis2[2];
      xscale =  axis1[1]/(xmax - xmin);
      yscale = -axis1[2]/(xmax - xmin);
      nticks = nxticks;
      tickvalues = &p_ticklabels->xvalues[0];
      axismin = xmin;
      delta = (xmax - xmin)/(nxticks - 1.0);
      }

   else if ( quadrant == 2 )
      {
      x1 = origin[1] + axis2[1];
      y1 = origin[2] - axis2[2];
      x2 = origin[1] + axis1[1] + axis2[1];
      y2 = origin[2] - axis1[2] - axis2[2];
      xscale =  axis1[1]/(ymax - ymin);
      yscale = -axis1[2]/(ymax - ymin);
      nticks = nyticks;
      tickvalues = &p_ticklabels->yvalues[0];
      axismin = ymin;
      delta = (ymax - ymin)/(nyticks - 1.0);
      }

   else if ( quadrant == 3 )
      {
      x1 = origin[1] + axis1[1] + axis2[1];
      y1 = origin[2] - axis1[2] - axis2[2];
      x2 = origin[1] + axis2[1];
      y2 = origin[2] - axis2[2];
      xscale = -axis1[1]/(xmax - xmin);
      yscale =  axis1[2]/(xmax - xmin);
      nticks = nxticks;
      tickvalues = &p_ticklabels->xvalues[0];
      axismin = xmin;
      delta = (xmax - xmin)/(nxticks - 1.0);
      }

   else
      {
      x1 = origin[1] + axis1[1] + axis2[1];
      y1 = origin[2] - axis1[2] - axis2[2];
      x2 = origin[1] + axis2[1];
      y2 = origin[2] - axis2[2];
      xscale = -axis1[1]/(ymax - ymin);
      yscale =  axis1[2]/(ymax - ymin);
      nticks = nyticks;
      tickvalues = &p_ticklabels->yvalues[0];
      axismin = ymin;
      delta = (ymax - ymin)/(nyticks - 1.0);
      }


   /* Initialize points structure */
   points = gnome_canvas_points_new(2);


   /* Draw axis1 tick marks and labels */
   if ( ((quadrant == 1 || quadrant == 3) && strcmp(p_plot_param->x_tick_marks, "on") == 0) ||
        ((quadrant == 2 || quadrant == 4) && strcmp(p_plot_param->y_tick_marks, "on") == 0) )
      {
      angle = atan2(axis2[2], axis2[1]) - pi;
      width_axis1_tick_labels = 0.0; 
      for ( i=1; i<=nticks; i++ )
         {
         xtick = x1 + (tickvalues[i-1] - axismin)*xscale;
         ytick = y1 + (tickvalues[i-1] - axismin)*yscale;

         memset(string, 0, strlen(string));
         if ( fabs(tickvalues[i-1]) < 0.01*delta )
            sprintf(string, "%1.0f", 0.0);
         else
            sprintf(string, "%g", tickvalues[i-1]);

         if ( ((quadrant == 1 || quadrant == 3) && strcmp(p_plot_param->x_tick_labels, "on") == 0) ||
              ((quadrant == 2 || quadrant == 4) && strcmp(p_plot_param->y_tick_labels, "on") == 0) )
            {
            text = gnome_canvas_item_new(group,
                                         GNOME_TYPE_CANVAS_TEXT,
                                         "text", string,
                                         "x", xtick - tick_major*cos(angle),
                                         "y", ytick + tick_major*sin(angle),
                                         "anchor", GTK_ANCHOR_NORTH_WEST,
                                         "font-desc", font_tick_labels,
                                         "fill_color_rgba", 0x000000FF,
                                         NULL);

            gnome_canvas_item_get_bounds(text, &xlabel1, &ylabel1, &xlabel2, &ylabel2);
            if ( xlabel2 - xlabel1 > width_axis1_tick_labels )
               width_axis1_tick_labels = xlabel2 - xlabel1;
            }

         points->coords[0] = xtick;
         points->coords[1] = ytick;
         points->coords[2] = xtick + tick_major*cos(angle);
         points->coords[3] = ytick - tick_major*sin(angle);

         line = gnome_canvas_item_new(group,
                                      GNOME_TYPE_CANVAS_LINE,
                                      "points", points,
                                      "fill_color_rgba", 0x000000FF,
                                      "width_pixels", 1,
                                      NULL);
         }

      x11 = x1 - axis2[1];
      y11 = y1 + axis2[2];
      x22 = x2 - axis2[1];
      y22 = y2 + axis2[2];
      angle = atan2(axis2[2], axis2[1]);
      for ( i=1; i<=nticks; i++ )
         {
         xtick = x11 + (tickvalues[i-1] - axismin)*xscale;
         ytick = y11 + (tickvalues[i-1] - axismin)*yscale;
         points->coords[0] = xtick;
         points->coords[1] = ytick;
         points->coords[2] = xtick + tick_major*cos(angle);
         points->coords[3] = ytick - tick_major*sin(angle);

         line = gnome_canvas_item_new(group,
                                      GNOME_TYPE_CANVAS_LINE,
                                      "points", points,
                                      "fill_color_rgba", 0x000000FF,
                                      "width_pixels", 1,
                                      NULL);
         }

      x11 = x1 - axis2[1];
      y11 = y1 + axis2[2];
      x22 = x2 - axis2[1];
      y22 = y2 + axis2[2];
      angle = atan2(axis3[2], axis3[1]);
      for ( i=1; i<=nticks; i++ )
         {
         xtick = x11 + (tickvalues[i-1] - axismin)*xscale;
         ytick = y11 + (tickvalues[i-1] - axismin)*yscale;
         points->coords[0] = xtick;
         points->coords[1] = ytick;
         points->coords[2] = xtick + tick_major*cos(angle);
         points->coords[3] = ytick - tick_major*sin(angle);

         line = gnome_canvas_item_new(group,
                                      GNOME_TYPE_CANVAS_LINE,
                                      "points", points,
                                      "fill_color_rgba", 0x000000FF,
                                      "width_pixels", 1,
                                      NULL);
         }

      x11 = x1 - axis2[1] + axis3[1];
      y11 = y1 + axis2[2] - axis3[2];
      x22 = x2 - axis2[1] + axis3[1];
      y22 = y2 + axis2[2] - axis3[2];
      angle = atan2(axis3[2], axis3[1]) - pi;
      for ( i=1; i<=nticks; i++ )
         {
         xtick = x11 + (tickvalues[i-1] - axismin)*xscale;
         ytick = y11 + (tickvalues[i-1] - axismin)*yscale;
         points->coords[0] = xtick;
         points->coords[1] = ytick;
         points->coords[2] = xtick + tick_major*cos(angle);
         points->coords[3] = ytick - tick_major*sin(angle);

         line = gnome_canvas_item_new(group,
                                      GNOME_TYPE_CANVAS_LINE,
                                      "points", points,
                                      "fill_color_rgba", 0x000000FF,
                                      "width_pixels", 1,
                                      NULL);
         }
      }


   /* Axis 2 tick-mark parameters */
   if ( quadrant == 1 )
      {
      x1 = origin[1] + axis1[1];
      y1 = origin[2] - axis1[2];
      x2 = origin[1] + axis1[1] + axis2[1];
      y2 = origin[2] - axis1[2] - axis2[2];
      xscale =  axis2[1]/(ymax - ymin);
      yscale = -axis2[2]/(ymax - ymin);
      nticks = nyticks;
      tickvalues = &p_ticklabels->yvalues[0];
      axismin = ymin;
      delta = (ymax - ymin)/(nyticks - 1.0);
      }

   else if ( quadrant == 2 )
      {
      x1 = origin[1] + axis1[1] + axis2[1];
      y1 = origin[2] - axis1[2] - axis2[2];
      x2 = origin[1] + axis1[1];
      y2 = origin[2] - axis1[2];
      xscale = -axis2[1]/(xmax - xmin);
      yscale =  axis2[2]/(xmax - xmin);
      nticks = nxticks;
      tickvalues = &p_ticklabels->xvalues[0];
      axismin = xmin;
      delta = (xmax - xmin)/(nxticks - 1.0);
      }

   else if ( quadrant == 3 )
      {
      x1 = origin[1] + axis1[1] + axis2[1];
      y1 = origin[2] - axis1[2] - axis2[2];
      x2 = origin[1] + axis1[1];
      y2 = origin[2] - axis1[2];
      xscale = -axis2[1]/(ymax - ymin);
      yscale =  axis2[2]/(ymax - ymin);
      nticks = nyticks;
      tickvalues = &p_ticklabels->yvalues[0];
      axismin = ymin;
      delta = (ymax - ymin)/(nyticks - 1.0);
      }

   else if ( quadrant == 4 )
      {
      x1 = origin[1] + axis1[1];
      y1 = origin[2] - axis1[2];
      x2 = origin[1] + axis1[1] + axis2[1];
      y2 = origin[2] - axis1[2] - axis2[2];
      xscale =  axis2[1]/(xmax - xmin);
      yscale = -axis2[2]/(xmax - xmin);
      nticks = nxticks;
      tickvalues = &p_ticklabels->xvalues[0];
      axismin = xmin;
      delta = (xmax - xmin)/(nxticks - 1.0);
      }


   /* Draw axis2 tick marks and labels */
   if ( ((quadrant == 2 || quadrant == 4) && strcmp(p_plot_param->x_tick_marks, "on") == 0) ||
        ((quadrant == 1 || quadrant == 3) && strcmp(p_plot_param->y_tick_marks, "on") == 0) )
      {
      angle = atan2(axis1[2], axis1[1]) - pi;
      width_axis2_tick_labels = 0.0;
      for ( i=1; i<=nticks; i++ )
         {
         xtick = x1 + (tickvalues[i-1] - axismin)*xscale;
         ytick = y1 + (tickvalues[i-1] - axismin)*yscale;

         memset(string, 0, strlen(string));
         if ( fabs(tickvalues[i-1]) < 0.01*delta )
            sprintf(string, "%1.0f", 0.0);
         else
            sprintf(string, "%g", tickvalues[i-1]);

         if ( ((quadrant == 2 || quadrant == 4) && strcmp(p_plot_param->x_tick_labels, "on") == 0) ||
              ((quadrant == 1 || quadrant == 3) && strcmp(p_plot_param->y_tick_labels, "on") == 0) )
            {
            text = gnome_canvas_item_new(group,
                                         GNOME_TYPE_CANVAS_TEXT,
                                         "text", string,
                                         "x", xtick - tick_major*cos(angle),
                                         "y", ytick + tick_major*sin(angle),
                                         "anchor", GTK_ANCHOR_NORTH_EAST,
                                         "font-desc", font_tick_labels,
                                         "fill_color_rgba", 0x000000FF,
                                         NULL);

            gnome_canvas_item_get_bounds(text, &xlabel1, &ylabel1, &xlabel2, &ylabel2);
            if ( xlabel2 - xlabel1 > width_axis2_tick_labels )
               width_axis2_tick_labels = xlabel2 - xlabel1;
            }

         points->coords[0] = xtick;
         points->coords[1] = ytick;
         points->coords[2] = xtick + tick_major*cos(angle);
         points->coords[3] = ytick - tick_major*sin(angle);

         line = gnome_canvas_item_new(group,
                                      GNOME_TYPE_CANVAS_LINE,
                                      "points", points,
                                      "fill_color_rgba", 0x000000FF,
                                      "width_pixels", 1,
                                      NULL);
         }

      x11 = x1 - axis1[1];
      y11 = y1 + axis1[2];
      x22 = x2 - axis1[1];
      y22 = y2 + axis1[2];
      angle = atan2(axis1[2], axis1[1]);
      for ( i=1; i<=nticks; i++ )
         {
         xtick = x11 + (tickvalues[i-1] - axismin)*xscale;
         ytick = y11 + (tickvalues[i-1] - axismin)*yscale;
         points->coords[0] = xtick;
         points->coords[1] = ytick;
         points->coords[2] = xtick + tick_major*cos(angle);
         points->coords[3] = ytick - tick_major*sin(angle);

         line = gnome_canvas_item_new(group,
                                      GNOME_TYPE_CANVAS_LINE,
                                      "points", points,
                                      "fill_color_rgba", 0x000000FF,
                                      "width_pixels", 1,
                                      NULL);
         }

      x11 = x1 - axis1[1];
      y11 = y1 + axis1[2];
      x22 = x2 - axis1[1];
      y22 = y2 + axis1[2];
      angle = atan2(axis3[2], axis3[1]);
      for ( i=1; i<=nticks; i++ )
         {
         xtick = x11 + (tickvalues[i-1] - axismin)*xscale;
         ytick = y11 + (tickvalues[i-1] - axismin)*yscale;
         points->coords[0] = xtick;
         points->coords[1] = ytick;
         points->coords[2] = xtick + tick_major*cos(angle);
         points->coords[3] = ytick - tick_major*sin(angle);

         line = gnome_canvas_item_new(group,
                                      GNOME_TYPE_CANVAS_LINE,
                                      "points", points,
                                      "fill_color_rgba", 0x000000FF,
                                      "width_pixels", 1,
                                      NULL);
         }

      x11 = x1 - axis1[1] + axis3[1];
      y11 = y1 + axis1[2] - axis3[2];
      x22 = x2 - axis1[1] + axis3[1];
      y22 = y2 + axis1[2] - axis3[2];
      angle = atan2(axis3[2], axis3[1]) - pi;
      for ( i=1; i<=nticks; i++ )
         {
         xtick = x11 + (tickvalues[i-1] - axismin)*xscale;
         ytick = y11 + (tickvalues[i-1] - axismin)*yscale;
         points->coords[0] = xtick;
         points->coords[1] = ytick;
         points->coords[2] = xtick + tick_major*cos(angle);
         points->coords[3] = ytick - tick_major*sin(angle);

         line = gnome_canvas_item_new(group,
                                      GNOME_TYPE_CANVAS_LINE,
                                      "points", points,
                                      "fill_color_rgba", 0x000000FF,
                                      "width_pixels", 1,
                                      NULL);
         }
      }


   /* Draw axis3 tick marks and labels */
   if ( strcmp(p_plot_param->z_tick_marks, "on") == 0 )
      {
      x1 = origin[1] + axis1[1];
      y1 = origin[2] - axis1[2];
      x2 = origin[1] + axis1[1] + axis3[1];
      y2 = origin[2] - axis1[2] - axis3[2];
      xscale =  axis3[1]/(zmax - zmin);
      yscale = -axis3[2]/(zmax - zmin);
      nticks = nzticks;
      delta = (zmax - zmin)/(nxticks - 1.0);
      angle = atan2(axis1[2], axis1[1]) - pi;
      width_axis3_tick_labels = 0.0;
      for ( i=1; i<=nticks; i++ )
         {
         xtick = x1 + (p_ticklabels->zvalues[i-1] - zmin)*xscale;
         ytick = y1 + (p_ticklabels->zvalues[i-1] - zmin)*yscale;

         memset(string, 0, strlen(string));
         if ( fabs(p_ticklabels->zvalues[i-1]) < 0.01*delta )
            sprintf(string, "%1.0f", 0.0);
         else
            sprintf(string, "%g", p_ticklabels->zvalues[i-1]);

         if ( strcmp(p_plot_param->z_tick_labels, "on") == 0 )
            {
            text = gnome_canvas_item_new(group,
                                         GNOME_TYPE_CANVAS_TEXT,
                                         "text", string,
                                         "x", xtick - tick_major,
                                         "y", ytick,
                                         "anchor", GTK_ANCHOR_EAST,
                                         "font-desc", font_tick_labels,
                                         "fill_color_rgba", 0x000000FF,
                                         NULL);

            gnome_canvas_item_get_bounds(text, &xlabel1, &ylabel1, &xlabel2, &ylabel2);
            if ( xlabel2 - xlabel1 > width_axis3_tick_labels )
               width_axis3_tick_labels = xlabel2 - xlabel1;
            }

         points->coords[0] = xtick;
         points->coords[1] = ytick;
         points->coords[2] = xtick + tick_major*cos(angle);
         points->coords[3] = ytick - tick_major*sin(angle);

         line = gnome_canvas_item_new(group,
                                      GNOME_TYPE_CANVAS_LINE,
                                      "points", points,
                                      "fill_color_rgba", 0x000000FF,
                                      "width_pixels", 1,
                                      NULL);
         }

      x1 = origin[1];
      y1 = origin[2];
      x2 = origin[1] + axis3[1];
      y2 = origin[2] - axis3[2];
      angle = atan2(axis1[2], axis1[1]);
      for ( i=1; i<=nticks; i++ )
         {
         xtick = x1 + (p_ticklabels->zvalues[i-1] - zmin)*xscale;
         ytick = y1 + (p_ticklabels->zvalues[i-1] - zmin)*yscale;
         points->coords[0] = xtick;
         points->coords[1] = ytick;
         points->coords[2] = xtick + tick_major*cos(angle);
         points->coords[3] = ytick - tick_major*sin(angle);

         line = gnome_canvas_item_new(group,
                                      GNOME_TYPE_CANVAS_LINE,
                                      "points", points,
                                      "fill_color_rgba", 0x000000FF,
                                      "width_pixels", 1,
                                      NULL);
         }

      x1 = origin[1];
      y1 = origin[2];
      x2 = origin[1] + axis3[1];
      y2 = origin[2] - axis3[2];
      angle = atan2(axis2[2], axis2[1]);
      for ( i=1; i<=nticks; i++ )
         {
         xtick = x1 + (p_ticklabels->zvalues[i-1] - zmin)*xscale;
         ytick = y1 + (p_ticklabels->zvalues[i-1] - zmin)*yscale;
         points->coords[0] = xtick;
         points->coords[1] = ytick;
         points->coords[2] = xtick + tick_major*cos(angle);
         points->coords[3] = ytick - tick_major*sin(angle);

         line = gnome_canvas_item_new(group,
                                      GNOME_TYPE_CANVAS_LINE,
                                      "points", points,
                                      "fill_color_rgba", 0x000000FF,
                                      "width_pixels", 1,
                                      NULL);
         }

      x1 = origin[1] + axis2[1];
      y1 = origin[2] - axis2[2];
      x2 = origin[1] + axis2[1] + axis3[1];
      y2 = origin[2] - axis2[2] - axis3[2];
      angle = atan2(axis2[2], axis2[1]) - pi;
      for ( i=1; i<=nticks; i++ )
         {
         xtick = x1 + (p_ticklabels->zvalues[i-1] - zmin)*xscale;
         ytick = y1 + (p_ticklabels->zvalues[i-1] - zmin)*yscale;
         points->coords[0] = xtick;
         points->coords[1] = ytick;
         points->coords[2] = xtick + tick_major*cos(angle);
         points->coords[3] = ytick - tick_major*sin(angle);

         line = gnome_canvas_item_new(group,
                                      GNOME_TYPE_CANVAS_LINE,
                                      "points", points,
                                      "fill_color_rgba", 0x000000FF,
                                      "width_pixels", 1,
                                      NULL);
         }
      }


   /* Free points structure */
   gnome_canvas_points_unref(points);

   return;
   }
