/*
  This file is part of GSEGrafix, a scientific and engineering plotting 
  program.

  Copyright (C) 2017 John Darrington

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <config.h>
#include <gtk/gtk.h>
#include <cairo.h>
#include "gse-context.h"

#include "gsegraf.h"

#include "InitializePlot.h"
#include "DrawGraph.h"
#include "ZoomIn.h"
#include "ZoomOut.h"
#include "gse-cairo.h"

#include <string.h>

#define _(X) X

struct _GseApp
{
  GtkApplication parent_instance;
  GtkWidget *window;
  GtkWidget *drawing_area;
  char *filename;
  GSimpleAction *about_action;
  GSimpleAction *quit_action;
  GSimpleAction *open_action;
  GSimpleAction *coords_action;
  struct gse_ctx *context;
  char coords_string[256];
};

typedef struct _GseApp GseApp;
  
typedef GtkApplicationClass GseAppClass;


G_DEFINE_TYPE (GseApp, gse_app, GTK_TYPE_APPLICATION)

#define GSE_TYPE_APP            (gse_app_get_type ())

#define GSE_APP(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), GSE_TYPE_APP, GseApp))


static void
gse_app_init (GseApp *app)
{
  app->context = NULL;
  app->about_action = g_simple_action_new ("about", NULL);
  app->quit_action = g_simple_action_new ("quit", NULL);
  app->open_action = g_simple_action_new ("open", NULL);
  app->coords_action =
    g_simple_action_new_stateful ("coords",
				  NULL,
				  g_variant_new_boolean (FALSE));
}

static void gse_app_activate (GApplication *app);

static void
gse_app_open (GApplication *app,
	  GFile **      files,
	  gint          n_files,
	      const gchar        *hint)
{
  GseApp *gse = GSE_APP (app);
  gse->filename = g_file_get_path (files[0]);
  g_application_activate (G_APPLICATION (app));
}


static void
gse_app_class_init (GseAppClass *class)
{
  GApplicationClass *application_class = G_APPLICATION_CLASS (class);
  GObjectClass *object_class = G_OBJECT_CLASS (class);
  
  application_class->open = gse_app_open;
  application_class->activate = gse_app_activate;
}

GseApp *
gse_new (void)
{
  GseApp *gse;

  g_set_application_name ("Gse");

  gse = g_object_new (gse_app_get_type (),
		      "application-id", "org.gnu.gse",
		      "flags", G_APPLICATION_HANDLES_OPEN,
		      NULL);

  return gse;
}





GMenu *
create_menu ()
{
  GMenu *menu = g_menu_new ();
  GMenu *view = g_menu_new ();
  g_menu_append (view, _("Display _Coordinates"), "win.coords");

  GMenu *help = g_menu_new ();
  g_menu_append (help, _("_About"), "win.about");

  GMenu *file = g_menu_new ();
  g_menu_append (file, _("_Open"), "win.open");
  g_menu_append (file, _("_Quit"), "win.quit");

  g_menu_append_submenu (menu, _("_File"), G_MENU_MODEL (file));
  g_menu_append_submenu (menu, _("_View"), G_MENU_MODEL (view));
  g_menu_append_submenu (menu, _("_Help"), G_MENU_MODEL (help));
  
  return menu;
}

gboolean
on_configure (GtkWidget *w, GdkEventConfigure *e, gpointer ud)
{
  GseApp *gse = GSE_APP (ud);
  struct gse_ctx *ctx = gse->context;
  if (ctx == NULL)
    return TRUE;

  ctx->window_width = e->width;
  ctx->window_height = e->height;
  
  return TRUE;
}

static gboolean version = FALSE;


static gboolean
motion_notify (GtkWidget *widget,
	       GdkEventMotion  *event,
	       gpointer   user_data)
{
  GseApp *gse = GSE_APP (user_data);
  struct gse_ctx *ctx = gse->context;
  GVariant *coords = g_action_get_state (G_ACTION (gse->coords_action));
  if (g_variant_get_boolean (coords))
    {
      double x = event->x - ctx->plot_box_data.xmin;
      double y = ctx->plot_box_data.ymax - event->y;

      /* Get plot box minimum and maximum values */
      double x1_box = ctx->plot_box_data.xmin;
      double x2_box = ctx->plot_box_data.xmax;
      double y1_box = ctx->plot_box_data.ymin;
      double y2_box = ctx->plot_box_data.ymax;

      if ((x < 0)
	  ||
	  (y < 0)
	  ||
	  (x > x2_box - x1_box)
	  ||
	  (y > y2_box - y1_box))
	{
	  gse->coords_string[0] = '\0';
	  goto done;
	}

      /* Get minimum and maximum axis values */
      int nxvalues = ctx->xtick_labels.nvalues;
      int nyvalues = ctx->ytick_labels.nvalues;
      int nzvalues = ctx->ztick_labels.nvalues;
      double xmin = ctx->xtick_labels.values[0];
      double xmax = ctx->xtick_labels.values[nxvalues-1];
      double ymin = ctx->ytick_labels.values[0];
      double ymax = ctx->ytick_labels.values[nyvalues-1];
      double zmin = ctx->ztick_labels.values[0];
      double zmax = ctx->ztick_labels.values[nzvalues-1];
      xmin -=  ctx->xtick_labels.offset1;
      xmax +=  ctx->xtick_labels.offset2;
      ymin -=  ctx->ytick_labels.offset1;
      ymax +=  ctx->ytick_labels.offset2;
      zmin -=  ctx->ztick_labels.offset1;
      zmax +=  ctx->ztick_labels.offset2;

      /* Calculate axis scale factors */
      double xscale = (x2_box - x1_box) / (xmax - xmin);
      double yscale = (y2_box - y1_box) / (ymax - ymin);

      snprintf (gse->coords_string, 256, _("%g, %g"), x/xscale + xmin,
		y/yscale + ymin);
    }

 done:
  gtk_widget_queue_draw (gse->drawing_area);
  
  return FALSE;
}

gboolean
draw (GtkWidget *w, cairo_t *cr, gpointer ud)
{
  GseApp *gse = GSE_APP (ud);

  if (gse->context == NULL)
    return FALSE;

  GdkRGBA *color;
  GtkStyleContext *sc = gtk_widget_get_style_context (w);
  gtk_style_context_get (sc,
			 GTK_STATE_FLAG_SELECTED,
			 "background-color",
			 &color, NULL);

  GtkCssProvider *cp = gtk_css_provider_new ();
  gchar *css = g_strdup_printf ("* {background-color: rgba(%d, %d, %d, 0.25);}",
				(int) (100.0 * color->red),
				(int) (100.0 * color->green),
				(int) (100.0 * color->blue));

  gtk_css_provider_load_from_data (cp, css, strlen (css), 0);

  gtk_style_context_add_provider (sc, GTK_STYLE_PROVIDER (cp),
				  GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);

  gse->context->cr = cr;

  DrawGraph (gse->context);

  
  cairo_rectangle_t *zoom = g_object_get_data (G_OBJECT (w), "zoom-area");

  if (zoom)
  {
    gtk_render_background (sc, cr,
		     zoom->x, zoom->y, zoom->width, zoom->height);
  }


  if (gse->coords_string[0])
    {
      gse_cairo_render_text (cr,
			     gse->coords_string,
			     10.0,
			     gse->context->window_height  - 10.0,
			     GSE_ANCHOR_SOUTH_WEST,
			     gse->context->canvas_fg_color,
			     gse->context->font_text
			     );
    }
      

  
  gtk_style_context_remove_provider (sc, GTK_STYLE_PROVIDER (cp));
  g_free (css);
  g_object_unref (cp);
  gdk_rgba_free (color);
  
  return FALSE;
}

static GOptionEntry entries[] =
{
  { "version", 'v', 0, G_OPTION_ARG_NONE, &version, "Show version and exit", NULL },
  { NULL }
};

void
du (GtkGestureDrag *gest, gdouble offsetx, gdouble offsety, gpointer ud)
{
  GseApp *gse = GSE_APP (ud);
  GtkWidget *drawing_area = gse->drawing_area;
  struct gse_ctx *context = gse->context;

  gdouble startx, starty;
  gtk_gesture_drag_get_start_point (gest, &startx, &starty);

  cairo_rectangle_t *zoom = g_malloc (sizeof (*zoom));
  zoom->x = startx;
  zoom->y = starty;
  zoom->width = offsetx;
  zoom->height = offsety;

  if (offsetx < 0)
    {
      zoom->width = -offsetx;
      zoom->x += offsetx;
    }

  if (offsety < 0)
    {
      zoom->height = -offsety;
      zoom->y += offsety;
    }

  g_object_set_data (G_OBJECT (drawing_area), "zoom-area", zoom);
  gtk_widget_queue_draw (drawing_area);
}

void
de (GtkGestureDrag *gest, gdouble offsetx, gdouble offsety, gpointer ud)
{
  GseApp *gse = GSE_APP (ud);
  GtkWidget *drawing_area = gse->drawing_area;
  struct gse_ctx *context = gse->context;
  gdouble startx, starty;
  gtk_gesture_drag_get_start_point (gest, &startx, &starty);

  g_free (g_object_get_data (G_OBJECT (drawing_area), "zoom-area"));
  g_object_set_data (G_OBJECT (drawing_area), "zoom-area", NULL);

  ZoomIn (context, startx, starty, startx + offsetx, starty + offsety);
  gtk_widget_queue_draw (drawing_area);
}

gboolean
released (GtkWidget *drawing_area, GdkEventButton *event, gpointer ud)
{
  GseApp *gse = GSE_APP (ud);
  struct gse_ctx *context = gse->context;

  if (event->button != 3)
    return FALSE;
  ZoomOut (context);
  gtk_widget_queue_draw (drawing_area);

  return FALSE;
}

static void
on_about (GSimpleAction *sa, GVariant *parameter, gpointer user_data)
{
  char *authors[] = { "Spencer A. Buckner", "John Darrington"};
  GtkWidget *dialog = gtk_about_dialog_new ();
  gtk_show_about_dialog (GTK_WINDOW (dialog),
			 "comments", _("This is an experimental fork of GNU GseGrafix."),
			 "website", "http://sv.gnu.org/p/gsegrafix-experimental/",
			 "version", PACKAGE_VERSION,
			 "authors", authors,
			 "license-type", GTK_LICENSE_GPL_3_0,
			 NULL);
}


static void
on_open (GSimpleAction *sa, GVariant *parameter, gpointer user_data)
{
  GseApp *gse = GSE_APP (user_data);
  GtkWidget *dialog = gtk_file_chooser_dialog_new (_("Open Parameter File"),
						   GTK_WINDOW (gse->window),
						   GTK_FILE_CHOOSER_ACTION_OPEN,
						   _("Cancel"), GTK_RESPONSE_CANCEL,
						   _("Open"), GTK_RESPONSE_ACCEPT,
						   NULL);
  if (GTK_RESPONSE_ACCEPT == gtk_dialog_run (GTK_DIALOG (dialog)))
    {
      gchar *filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));

      if (filename)
	{
	  if (gse->context)
	    gse_context_destroy (gse->context);
	  gse->context = gse_context_create (filename);
	  gse->context->window_height = gtk_widget_get_allocated_height (gse->drawing_area);
	  gse->context->window_width = gtk_widget_get_allocated_width (gse->drawing_area);

	  GError **err = &gse->context->err;
	  if (setjmp (gse->context->finish))
	    {
	      GtkWidget *dialog = gtk_message_dialog_new (GTK_WINDOW (gse->window),
							  GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
							  (*err)->message);
	      gtk_dialog_run (GTK_DIALOG (dialog));
	      g_clear_error (err);
	      gtk_widget_destroy (dialog);
	    }
	  else
	    InitializePlot(gse->context);
	  gtk_widget_queue_draw (gse->drawing_area);
	}
    }

  gtk_widget_destroy (dialog);
}


static void
gse_app_activate (GApplication *app)
{
  GseApp *gse = GSE_APP (app);

  g_signal_connect (gse->about_action, "activate", G_CALLBACK (on_about), NULL);
  g_signal_connect (gse->open_action, "activate", G_CALLBACK (on_open), gse);

  g_signal_connect_swapped (gse->quit_action, "activate", G_CALLBACK (g_application_quit), app);

  gse->window = gtk_application_window_new (GTK_APPLICATION (app));

  g_action_map_add_action (G_ACTION_MAP (gse->window), G_ACTION (gse->about_action));
  g_action_map_add_action (G_ACTION_MAP (gse->window), G_ACTION (gse->quit_action));
  g_action_map_add_action (G_ACTION_MAP (gse->window), G_ACTION (gse->coords_action));
  g_action_map_add_action (G_ACTION_MAP (gse->window), G_ACTION (gse->open_action));
  
  g_signal_connect_swapped (gse->window, "destroy", G_CALLBACK (g_application_quit), app);

  GtkWidget *vbox = gtk_box_new (GTK_ORIENTATION_VERTICAL, 5);
  GtkWidget *menu_bar = gtk_menu_bar_new_from_model (G_MENU_MODEL (create_menu ()));
  gse->drawing_area = gtk_drawing_area_new ();

  gtk_box_pack_start (GTK_BOX (vbox), menu_bar, FALSE, FALSE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), gse->drawing_area, TRUE, TRUE, 5);

  gtk_container_add (GTK_CONTAINER (gse->window), vbox);

  if (gse->filename)
    {
      gse->context = gse_context_create (gse->filename);
      GError **err = &gse->context->err;
      if (setjmp (gse->context->finish))
	{
	  GtkWidget *dialog = gtk_message_dialog_new (GTK_WINDOW (gse->window),
						      GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
						      (*err)->message);
	  gtk_dialog_run (GTK_DIALOG (dialog));
	  g_clear_error (err);
	  gtk_widget_destroy (dialog);
	}
      else
	InitializePlot(gse->context);
    }

  GtkGesture *drag = gtk_gesture_drag_new (gse->drawing_area);
  g_signal_connect (drag, "drag-update", G_CALLBACK (du), gse);
  g_signal_connect (drag, "drag-end", G_CALLBACK (de), gse);

  g_signal_connect (gse->drawing_area, "button-release-event",
		    G_CALLBACK (released), gse);
  
  g_signal_connect (gse->drawing_area, "draw", G_CALLBACK (draw), gse);

  g_signal_connect (gse->drawing_area, "motion-notify-event",
		    G_CALLBACK (motion_notify), gse);
  gtk_widget_add_events (gse->drawing_area, GDK_POINTER_MOTION_MASK);

  g_signal_connect (gse->drawing_area, "configure-event",
		    G_CALLBACK (on_configure), gse);

  gtk_widget_show_all (gse->window);
}


int
main (int argc, char **argv)
{
  GError *error = NULL;
  GOptionContext *optctx = g_option_context_new ("- test gsegrafix library");
  g_option_context_add_main_entries (optctx, entries, NULL);
  if (!g_option_context_parse (optctx, &argc, &argv, &error))
    {
      g_print ("option parsing failed: %s\n", error->message);
      return 1;
    }
  g_option_context_free (optctx);

  if (version)
    {
      g_print ("Version: %s\n", PACKAGE_VERSION);
      return 0;
    }

  GseApp *app = gse_new ();
  int status = g_application_run (G_APPLICATION (app), argc, argv);
  g_object_unref (app);
  return status;
}
