/*******************************************************************************
*
* ReadParamFile.c
*
* Reads plot-parameter file.
*
* Copyright  2008, 2009, 2010, 2011, 2012 Spencer A. Buckner
* http://savannah.gnu.org/projects/gsegrafix
*
* This file is part of GSEGrafix, a scientific and engineering plotting program.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*******************************************************************************/

#include "ReadParamFile.h"
#include "InitializePlot.h"
#include "Misc.h"

#include <ctype.h>
#include <locale.h>
#include <math.h>
#include "gsegraf.h"

#include <string.h>
#include <stdlib.h>
#include <math.h> // for abs
#include <errno.h>

#define _(X) X

const char symbol_string[]  = "ld.cCtTsSiIpPhH+xra";
const char symbol_string1[] = "cCtTsSiIpPhH";       
const char symbol_string2[] = "+xra";               
const char color_string[]   = "kaswrylqbfmogtnpx";  


void ReadParamFile (struct gse_ctx *context, struct data_parser *parser)
   {
     int locale = 0;
   /* Declare variables */
   int i, j, ch, iplot, nplots, iformat, nformat, nfilenames_total, nformats_total,
       index, index_filenames, index_formats, index_bin_values,
       index_bin_refs, ibinwidth, index_stemflags, ibinvalue, ibinref, ininterp,
       istemflag, imeshcolor, icontourcolor, stylesize;
   unsigned int i1_str, i2_str, size;
   const char *string;
   char *pchar, character;

   const char *error_str[] =
      { "Missing file_name data.",
        "Missing file_format data.",
        "Invalid or missing axis_type parameter.",
        "Invalid or missing plot_type parameter.",
        "Invalid axis_limits parameter.",
        "Invalid or incomplete view direction angles.",
        "Invalid grid parameter.",
        "Invalid histogram bin width.",
        "Invalid ninterp value.",
        "Invalid contours value.",
        "Invalid or missing contour color.",
        "Invalid or missing mesh color.",
        "Invalid window-size data;\ntwo integers required.",
        "Invalid or missing date-time font size.",
        "Invalid or missing legend font size.",
        "Invalid or missing text font size.",
        "Invalid or missing tick-label font size.",
        "Invalid or missing axis-label font size.",
        "Invalid or missing title font size.",
        "Invalid x-axis limit.",
        "Invalid y-axis limit.",
        "Invalid axis limit." };
   FILE *fptr;


   /* Open plot-parameter file */
   if ( (fptr = fopen(context->p_param_file, "r")) == NULL )
     {
       const char *diag = strerror (errno);
       g_set_error (&context->err, context->domain, 0,
		    _("Cannot open plot-parameter file \"%s\": %s"),
		    context->p_param_file, diag);
       longjmp (context->finish, 1);
     }


   /* Get maximum line length */
   context->maxline = 0;
   i = 0;
   while ( (ch = fgetc(fptr)) != EOF )
      {
      if ( ch != '\n' )
         i++;
      else
         {
         if ( context->maxline < i )
            context->maxline = i;
         i = 0;
         }
      }
   context->maxline++;


   /****************************************************************************
   * Increase context->maxline for use with char *fgets(char *s, int n, FILE *stream).
   * fgets reads at most the next n-1 characters into the array s. It is desired
   * that fgets read the terminating character so that following uses of fgets
   * start with the next line in the file.
   ****************************************************************************/
   context->maxline++;
   context->line = xmalloc(context->maxline*sizeof(char));


   /* Get locale information */
   rewind(fptr);
   while ( fgets(context->line, context->maxline, fptr) != NULL )
      {
      if ( strncmp(context->line, "locale", 6) == 0 )
         {
	   if ( sscanf(&context->line[6], "%d", &locale) != 1 )
            {
	      g_set_error (&context->err, context->domain, 0,
			   "Unable to read locale integer at position %d.",
			   ftell (fptr));
	      longjmp (context->finish, 1);
            }
         }
      }

   /* decimal point is a period */
   if ( locale == 1 )
      {
      if ( setlocale(LC_NUMERIC, "C" ) == NULL )
         g_set_error_literal (&context->err, context->domain, 0, "Error setting \"C\" locale.");
      }

   /* native locale */
   else if ( locale == 2 )
      {
      if ( setlocale(LC_NUMERIC, "" ) == NULL )
         g_set_error_literal (&context->err, context->domain, 0, "Error setting \"\" locale.");
      }

   /* french locale */
   else if ( locale == 3 )
      {
      if ( setlocale(LC_NUMERIC, "french" ) == NULL )
         g_set_error_literal (&context->err, context->domain, 0, "Error setting \"french\" locale.");
      }

   /* german locale */
   else if ( locale == 4 )
      {
      if ( setlocale(LC_NUMERIC, "german" ) == NULL )
         g_set_error_literal (&context->err, context->domain, 0, "Error setting \"german\" locale.");
      }


   /* Get number of file_name entries */
   iplot = 0;
   rewind(fptr);
   while ( fgets(context->line, context->maxline, fptr) != NULL )
      {
      if ( strncmp(context->line, "file_name", 9) == 0 )
         iplot++;
      else if ( strncmp(context->line, "#####", 5) == 0 )
         break;
      }
   nplots = iplot;
   context->plot_param.nplots = nplots;
   if ( nplots > 0 )
      parser->nfilenames = xmalloc(nplots*sizeof(int));


   /* Get number of file_format entries */
   nformat = 0;
   if ( nplots > 0 )
      {
      iformat = 0;
      rewind(fptr);
      while ( fgets(context->line, context->maxline, fptr) != NULL )
         {
         if ( strncmp(context->line, "file_format", 11) == 0 )
            iformat++;
         else if ( strncmp(context->line, "#####", 5) == 0 )
            break;
         }
      nformat = iformat;
      if ( nformat > 0 )
         parser->nformats = xmalloc(nformat*sizeof(int));
      }


   /* Get filename sizes */
   nfilenames_total = 0;
   iplot = 0;
   rewind(fptr);
   if ( nplots > 0 )
      {
      while ( fgets(context->line, context->maxline, fptr) != NULL )
         {
         if ( strncmp(context->line, "file_name", 9) == 0 )
            {
            iplot++;
            size = strlen(&context->line[9]);
            if ( (string = get_string (context, &context->line[9], &i1_str, &i2_str, &size, 0)) != NULL )
               parser->nfilenames[iplot-1] = size + 1;
            else
	    {
	      g_set_error (&context->err, context->domain, 0,
			   _("Expecting a filename at position %d"),
			   ftell (fptr));
	      longjmp (context->finish, 1);
	    }

            nfilenames_total = nfilenames_total + parser->nfilenames[iplot-1];
            }

         else if ( strncmp(context->line, "#####", 5) == 0 )
            break;
         }
      }


   /* Get format sizes */
   nformats_total = 0;
   iformat = 0;
   rewind(fptr);
   if ( nformat > 0 )
      {
      while ( fgets(context->line, context->maxline, fptr) != NULL )
         {
         if ( strncmp(context->line, "file_format", 11) == 0 )
            {
            iformat++;
            size = strlen(&context->line[11]);
            if ( (string = get_string (context, &context->line[11], &i1_str, &i2_str, &size, 0)) != NULL )
               parser->nformats[iformat-1] = size + 1;
            else
               {
               g_set_error (&context->err, context->domain, 0,
			    _("Expecting a scanf format string at position %d"),
			    ftell (fptr));
               longjmp (context->finish, 1);
               }

            nformats_total = nformats_total + parser->nformats[iformat-1];
            }

         else if ( strncmp(context->line, "#####", 5) == 0 )
            break;
         }
      }


   /* Get background color */
   rewind(fptr);
   while ( fgets(context->line, context->maxline, fptr) != NULL )
      {
      if ( strncmp(context->line, "background_color", 16) == 0 )
         {
         size = strlen(&context->line[16]);
         if ( (string = get_string (context, &context->line[16], &i1_str, &i2_str, &size, 0)) != NULL )
            if ( strcmp(string, "black") == 0 )
               {
               context->canvas_bg_color = 0x000000FF;   /* black */
               context->canvas_fg_color = 0xFFFFFFFF;   /* white */
               }
         }

      else if ( strncmp(context->line, "background_image", 16) == 0 )
         {
         size = strlen(&context->line[16]);
         if ( (string = get_string (context, &context->line[16], &i1_str, &i2_str, &size, 0)) != NULL )
            {
            context->background_image_file = xmalloc(size + 1);
            strcpy(context->background_image_file, string);
            }
         index = 16 + i2_str + 2;
         size = strlen(&context->line[index]);
         if ( (string = get_string (context, &context->line[index], &i1_str, &i2_str, &size, 0)) != NULL )
            {
            if ( strcmp(string, "center") == 0 )
               context->background_image_style = 1;
            else if ( strcmp(string, "fill") == 0 )
               context->background_image_style = 2;
            else if ( strcmp(string, "scale") == 0 )
               context->background_image_style = 3;
            else if ( strcmp(string, "zoom") == 0 )
               context->background_image_style = 4;
            }
         }

      else if ( strncmp(context->line, "#####", 5) == 0 )
         break;
      }


   /* Allocate memory for plot parameters */
   parser->filenames           = xmalloc(nfilenames_total*sizeof(char));
   parser->formats             = xmalloc(nformats_total*sizeof(char));
   parser->formats_mod         = xmalloc(nformats_total*sizeof(char));

   context->plot_parameters  = xzalloc (nplots * sizeof (*context->plot_parameters));

   memset(parser->filenames,   0, nfilenames_total*sizeof(char));
   memset(parser->formats,     0, nformats_total*sizeof(char));
   memset(parser->formats_mod, 0, nformats_total*sizeof(char));

   /* Set default number of contours */
   /* (2d and 3d contour plots) */
   context->ncontours = 0;

   /* Set defaults */
   for ( i=1; i<=nplots; i++ )
     {
       context->plot_parameters[i-1].bin_widths = -1.0;
       strncpy(context->plot_parameters[i-1].bin_values, "percent", 9);
       strncpy(context->plot_parameters[i-1].bin_refs, "mean", 9);
       strncpy(context->plot_parameters[i-1].stemflags, "off", 4);
       context->plot_parameters[i-1].ninterp = 20;
       context->plot_parameters[i-1].meshcolors = context->canvas_fg_color;
       context->plot_parameters[i-1].contourcolors = context->canvas_fg_color;
       context->plot_parameters[i-1].styleflags = -1;
       context->plot_parameters[i-1].stylechar1 = '*';
       context->plot_parameters[i-1].stylechar2 = '*';
       context->plot_parameters[i-1].stylesizes = 1;
       context->plot_parameters[i-1].alphacolor = 0xFF;
       context->plot_parameters[i-1].zblack = -DBL_MAX;
       context->plot_parameters[i-1].zwhite =  DBL_MAX;
     }

   /* Read filenames */
   iplot = 0;
   index_filenames = 0;
   rewind(fptr);
   if ( nplots > 0 )
      {
      while ( fgets(context->line, context->maxline, fptr) != NULL )
         {
         if ( strncmp(context->line, "file_name", 9) == 0 )
            {
            iplot++;
            size = strlen(&context->line[9]);
            if ( (string = get_string (context, &context->line[9], &i1_str, &i2_str, &size, 0)) != NULL )
               {
               strcpy(&parser->filenames[index_filenames], string);
               index_filenames = index_filenames + parser->nfilenames[iplot-1];
               }
            }

         else if ( strncmp(context->line, "#####", 5) == 0 )
            break;
         }
      }


   /* Read file formats */
   iformat = 0;
   index_formats = 0;
   rewind(fptr);
   if ( nformat > 0 )
      {
      while ( fgets(context->line, context->maxline, fptr) != NULL )
         {
         if ( strncmp(context->line, "file_format", 11) == 0 )
            {
            iformat++;
            size = strlen(&context->line[11]);
            if ( (string = get_string (context, &context->line[11], &i1_str, &i2_str, &size, 0)) != NULL )
               {
               strcpy(&parser->formats[index_formats], string);
               index_formats = index_formats + parser->nformats[iformat-1];
               }
            }

         else if ( strncmp(context->line, "#####", 5) == 0 )
            break;
         }
      }


   /* Read axis type */
   rewind(fptr);
   while ( fgets(context->line, context->maxline, fptr) != NULL )
      {
      if ( strncmp(context->line, "axis_type", 9) == 0 )
         {
         size = strlen(&context->line[9]);
         if ( (string = get_string (context, &context->line[9], &i1_str, &i2_str, &size, 0)) != NULL )
            strncpy(&context->plot_param.axis_type[0], string, 8);
         }

      else if ( strncmp(context->line, "#####", 5) == 0 )
         break;
      }

   if ( strcmp(context->plot_param.axis_type, "linear")   != 0 &&
        strcmp(context->plot_param.axis_type, "semilogx") != 0 &&
        strcmp(context->plot_param.axis_type, "semilogy") != 0 &&
        strcmp(context->plot_param.axis_type, "loglog")   != 0 &&
        strcmp(context->plot_param.axis_type, "polar")    != 0 &&
        strcmp(context->plot_param.axis_type, "3d")       != 0 )
      {
      g_set_error (&context->err, context->domain, 0,
		   _("Invalid or missing axis_type parameter at position %d"),
		   ftell (fptr));
      longjmp (context->finish, 1);
      }


   /* Read plot types */
   iplot = 0;
   int idx = 0;
   rewind(fptr);
   if ( nplots > 0 )
      {
      while ( fgets(context->line, context->maxline, fptr) != NULL )
         {
         if ( strncmp(context->line, "plot_type", 9) == 0 )
            {
            iplot++;
            size = strlen(&context->line[9]);
            if ( (string = get_string (context, &context->line[9], &i1_str, &i2_str, &size, 0)) != NULL )
               {
	       strncpy (context->plot_parameters[idx].plot_types, string, 10);
	       idx++;
               }
            }

         else if ( strncmp(context->line, "#####", 5) == 0 )
            break;
         }
      }

   idx = 0;
   for ( iplot=1; iplot<=nplots; iplot++ )
      {
	const struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];
	
      if ( strncmp (context->plot_parameters[idx].plot_types, "points", 10)    != 0 &&
           strncmp (context->plot_parameters[idx].plot_types, "histogram", 10) != 0 &&
           strncmp (context->plot_parameters[idx].plot_types, "color", 10)     != 0 &&
           strncmp (context->plot_parameters[idx].plot_types, "contour", 10)   != 0 &&
           strncmp (context->plot_parameters[idx].plot_types, "mesh", 10)      != 0 )
         {
         g_set_error_literal (&context->err, context->domain, 0, error_str[3]);
         longjmp (context->finish, 1);
         }

      /* Increment index */
      idx = 0;
      }


   /* Read plot styles */
   iplot = 0;
   rewind(fptr);
   if ( nplots > 0 )
      {
      while ( fgets(context->line, context->maxline, fptr) != NULL )
         {
         if ( strncmp(context->line, "plot_style", 10) == 0 )
            {
            iplot++;

	    struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];

	    
            if ( strncmp (context->plot_parameters[idx].plot_types, "points", 10)    == 0 ||
                 strncmp (context->plot_parameters[idx].plot_types, "histogram", 10) == 0 )
               {
               if ( (sscanf(&context->line[10], " %c 0x%x", &the_plot->stylechar1, (unsigned int *) &the_plot->stylecolor2) == 2 ||
                     sscanf(&context->line[10], " %c 0X%x", &the_plot->stylechar1, (unsigned int *) &the_plot->stylecolor2) == 2) &&
                    isdigit(the_plot->stylechar1) == 0 )
                  {
                  /************************************************************
                  *
                  * 2d points and 3d points plot types
                  * Read symbol character, hexadecimal color, and symbol size
                  *
                  * 2d histogram
                  * Read symbol character and hexadecimal color
                  *
                  ************************************************************/
                  the_plot->styleflags = 4;
                  the_plot->fill_colors_rgba    = the_plot->stylecolor2;         /* set specified fill color */
                  the_plot->outline_colors_rgba = the_plot->stylecolor2;         /* set specified outline color */
                  if ( (pchar = strchr("ctsiphb", the_plot->stylechar1)) != NULL )
                     the_plot->fill_colors_rgba = context->canvas_bg_color;             /* set fill color to background color */
                  if ( (pchar = strchr("cCtTsSiIpPhH+xra", the_plot->stylechar1)) != NULL )
                     the_plot->stylesizes = 6;                                  /* set default symbol size */
                  if ( sscanf(&context->line[10], " %*c %*s %d", &stylesize) == 1 )
                     the_plot->stylesizes = abs(stylesize);                     /* get symbol size */
                  }

               else if ( sscanf(&context->line[10], " %c %c", &the_plot->stylechar1, &the_plot->stylechar2) == 2 &&
                         (pchar = strchr(color_string, the_plot->stylechar2)) != NULL )
                  {
                  /************************************************************
                  *
                  * 2d points and 3d points plot types
                  * Read symbol character, color character, and symbol size
                  *
                  * 2d histogram
                  * Read symbol character and color character
                  *
                  ************************************************************/
                  the_plot->styleflags = 2;
                  index = pchar - &color_string[0];                            /* get index to color character */
                  the_plot->stylecolor2 = context->color_rgba[index];                    /* set specified color */
                  the_plot->fill_colors_rgba    = the_plot->stylecolor2;         /* set specified fill color */
                  the_plot->outline_colors_rgba = the_plot->stylecolor2;         /* set specified outline color */
                  if ( (pchar = strchr("ctsiphb", the_plot->stylechar1)) != NULL )
                     the_plot->fill_colors_rgba = context->canvas_bg_color;              /* set fill color to background color */
                  if ( (pchar = strchr("cCtTsSiIpPhH+xra", the_plot->stylechar1)) != NULL )
                     the_plot->stylesizes = 6;                                  /* set default symbol size */
                  if ( sscanf(&context->line[10], " %*c %*c %d", &stylesize) == 1 )
                     the_plot->stylesizes = abs(stylesize);                     /* get symbol size */
                  }
               }

            else if ( strncmp (context->plot_parameters[idx].plot_types, "color", 10) == 0 &&
                      strcmp(context->plot_param.axis_type, "linear") == 0 )
               {
               size = strlen(&context->line[10]);
               if ( (string = get_string (context, &context->line[10], &i1_str, &i2_str, &size, 0)) != NULL )
                  {
                  /************************************************************
                  *
                  * 2d color plot type
                  * Read "nearest" or "bilinear" string and two decimal numbers
                  *
                  ************************************************************/
                  if ( strcmp(string, "nearest") == 0 )
                     {
                     the_plot->styleflags = 9;
                     sscanf(&string[size+1], " %lf %lf", &the_plot->zblack, &the_plot->zwhite);
                     }
                  else if ( strcmp(string, "bilinear") == 0 )
                     {
                     the_plot->styleflags = 8;
                     sscanf(&string[size+1], " %lf %lf", &the_plot->zblack, &the_plot->zwhite);
                     }
                  }
               }

            else if ( strncmp (context->plot_parameters[idx].plot_types, "contour", 10) == 0 &&
                      strcmp(context->plot_param.axis_type, "linear") == 0 )
               {
               size = strlen(&context->line[10]);
               if ( (string = get_string (context, &context->line[10], &i1_str, &i2_str, &size, 0)) != NULL )
                  {
                  /************************************************************
                  *
                  * 2d contour plot type
                  * Read "auto" string and line width
                  *
                  ************************************************************/
                  if ( strcmp(string, "auto") == 0 )
                     {
                     the_plot->styleflags = 7;
                     if ( sscanf(&string[size+1], " %d", &stylesize) == 1 )
                        the_plot->stylesizes = abs(stylesize);                  /* get line width */
                     }
                  }

               else if ( sscanf(&context->line[10], " 0x%x", (unsigned int *) &the_plot->stylecolor1) == 1 ||
                         sscanf(&context->line[10], " 0X%x", (unsigned int *) &the_plot->stylecolor1) == 1 )
                  {
                  /************************************************************
                  *
                  * 2d contour plot type
                  * Read hexadecimal color and line width
                  *
                  ************************************************************/
                  the_plot->styleflags = 3;
                  if ( sscanf(&context->line[10], " %*s %d", &stylesize) == 1 )
                     the_plot->stylesizes = abs(stylesize);                     /* get line width */
                  }

               else if ( sscanf(&context->line[10], " %c", &the_plot->stylechar1) == 1 &&
                         (pchar = strchr(color_string, the_plot->stylechar1)) != NULL )
                  {
                  /************************************************************
                  *
                  * 2d contour plot type
                  * Read color character and line width
                  *
                  ************************************************************/
                  the_plot->styleflags = 1;
                  index = pchar - &color_string[0];                            /* get index to color character */
                  the_plot->stylecolor1 = context->color_rgba[index];                    /* set specified color */
                  if ( sscanf(&context->line[10], " %*c %d", &stylesize) == 1 )
                     the_plot->stylesizes = abs(stylesize);                     /* get line width */
                  }
               }

            else if ( (strncmp (context->plot_parameters[idx].plot_types, "contour", 10) == 0 ||
                       strncmp (context->plot_parameters[idx].plot_types, "color", 10)   == 0 ||
                       strncmp (context->plot_parameters[idx].plot_types, "mesh", 10)    == 0) &&
                      strcmp(context->plot_param.axis_type, "3d") == 0 )
               {
               size = strlen(&context->line[10]);
               if ( (strncmp (context->plot_parameters[idx].plot_types, "color", 10) == 0 ||
                     strncmp (context->plot_parameters[idx].plot_types, "mesh", 10)  == 0) &&
                    (string = get_string (context, &context->line[10], &i1_str, &i2_str, &size, 0)) != NULL )
                  {
                  /************************************************************
                  *
                  * 3d color and 3d mesh plot types
                  * Read "auto" string and color alpha value
                  *
                  ************************************************************/
                  if ( strcmp(string, "auto") == 0 )
                     {
                     the_plot->styleflags = 7;
                     if ( sscanf(&string[size+1], " 0x%x", (unsigned int *) &the_plot->alphacolor) == 1 ||
                          sscanf(&string[size+1], " 0X%x", (unsigned int *) &the_plot->alphacolor) == 1 )
                        {
                        if ( the_plot->alphacolor > 0xFF )                      /* check alpha value */
                           the_plot->alphacolor = 0xFF;
                        }
                     }
                  }

               else if ( sscanf(&context->line[10], " 0x%x 0x%x", (unsigned int *) &the_plot->stylecolor1, (unsigned int *) &the_plot->stylecolor2) == 2 ||
                         sscanf(&context->line[10], " 0X%x 0X%x", (unsigned int *) &the_plot->stylecolor1, (unsigned int *) &the_plot->stylecolor2) == 2 ||
                         sscanf(&context->line[10], " 0x%x 0X%x", (unsigned int *) &the_plot->stylecolor1, (unsigned int *) &the_plot->stylecolor2) == 2 ||
                         sscanf(&context->line[10], " 0X%x 0x%x", (unsigned int *) &the_plot->stylecolor1, (unsigned int *) &the_plot->stylecolor2) == 2 )
                  {
                  /************************************************************
                  *
                  * 3d contour and 3d mesh plot types
                  * Read two hexadecimal colors
                  *
                  ************************************************************/
                  the_plot->styleflags = 6;
                  }

               else if ( sscanf(&context->line[10], " 0x%x %c", (unsigned int *) &the_plot->stylecolor1, &the_plot->stylechar2) == 2 ||
                         sscanf(&context->line[10], " 0X%x %c", (unsigned int *) &the_plot->stylecolor1, &the_plot->stylechar2) == 2 )
                  {
                  /************************************************************
                  *
                  * 3d contour and 3d mesh plot types
                  * Read hexadecimal color and color character
                  *
                  ************************************************************/
                  the_plot->styleflags = 5;
                  }

               else if ( sscanf(&context->line[10], " %c 0x%x", &the_plot->stylechar1, (unsigned int *) &the_plot->stylecolor2) == 2 ||
                         sscanf(&context->line[10], " %c 0X%x", &the_plot->stylechar1, (unsigned int *) &the_plot->stylecolor2) == 2 )
                  {
                  /************************************************************
                  *
                  * 3d contour and 3d mesh plot types
                  * Read color character and hexadecimal color
                  *
                  ************************************************************/
                  the_plot->styleflags = 4;
                  }

               else if ( sscanf(&context->line[10], " %c %c", &the_plot->stylechar1, &the_plot->stylechar2) == 2 )
                  {
                  /************************************************************
                  *
                  * 3d contour and 3d mesh plot types
                  * Read two color characters
                  *
                  ************************************************************/
                  the_plot->styleflags = 2;
                  }
               }

            /* Increment index */
	    idx++;
            }

         else if ( strncmp(context->line, "#####", 5) == 0 )
            break;
         }
      }


   /* Read remainder of plot-parameter file */
   ibinwidth = 0;
   ibinvalue = 0;
   ibinref = 0;
   ininterp = 0;
   istemflag = 0;
   imeshcolor = 0;
   icontourcolor = 0;
   index_bin_values = 0;
   index_bin_refs = 0;
   index_stemflags = 0;
   rewind(fptr);
   while ( fgets(context->line, context->maxline, fptr) != NULL )
      {
      if ( strncmp(context->line, "axis_scale", 10) == 0 )
         {
         size = strlen(&context->line[10]);
         if ( (string = get_string (context, &context->line[10], &i1_str, &i2_str, &size, 0)) != NULL )
            strncpy(&context->plot_param.axis_scale[0], string, 5);
         }

      else if ( strncmp(context->line, "axis_limits", 11) == 0 )
         {
         size = strlen(context->line);
         i = 11;
         for ( j=1; j<=6; j++ )
            {
            while ( context->line[i] == ' ' || context->line[i] == '\t' )
               i++;

            if ( sscanf(&context->line[i], "%lf", &context->plot_param.axis_limits[j-1]) == 1 )
               context->axis_limits[j-1] = TRUE;
            else if ( sscanf(&context->line[i], " %c", &character ) == 1 )
               {
               if ( character == '#' )
                  break;
               else if ( character != '*' )
                  {
                  g_set_error_literal (&context->err, context->domain, 0, error_str[4]);
                  longjmp (context->finish, 1);
                  }
               }

            if ( (pchar = strchr(&context->line[i], ' ')) == NULL )
               break;

            i = i + pchar - &context->line[i];
            }
         }

      else if ( strncmp(context->line, "view3d", 6) == 0 )
         {
         if ( sscanf(&context->line[6], "%lf %lf", &context->plot_param_3d.phi, &context->plot_param_3d.theta) != 2 )
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[5]);
            longjmp (context->finish, 1);
            }
         }

      else if ( strncmp(context->line, "minor_ticks", 11) == 0 )
         {
         size = strlen(&context->line[11]);
         if ( (string = get_string (context, &context->line[11], &i1_str, &i2_str, &size, 0)) != NULL )
            {
            strncpy(&context->plot_param.minor_ticks[0], string, 3);
            if ( strcmp(string, "on") == 0 )
               context->minor_ticks_flag = 1;
            }
         }

      else if ( strncmp(context->line, "grid", 4) == 0 )
         {
         size = strlen(&context->line[4]);
         if ( (string = get_string (context, &context->line[4], &i1_str, &i2_str, &size, 0)) != NULL &&
              strcmp(string, "off") == 0 )
            {
            strncpy(&context->plot_param.grid[0], string, 3);
            }
         else if ( sscanf(&context->line[4], " %c 0x%x", &context->gridchar1, &context->gridcolor) == 2 ||
                   sscanf(&context->line[4], " %c 0X%x", &context->gridchar1, &context->gridcolor) == 2 )
            {
            strcpy(&context->plot_param.grid[0], "on2");
            }
         else if ( sscanf(&context->line[4], " %c %c", &context->gridchar1, &context->gridchar2) == 2 &&
                   (pchar = strchr(color_string, context->gridchar2)) != NULL )
            {
            index = pchar - &color_string[0];   /* get index to color character */
            context->gridcolor = context->color_rgba[index];      /* set specified color */
            strcpy(&context->plot_param.grid[0], "on1");
            }
         else
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[6]);
            longjmp (context->finish, 1);
            }
         }

      else if ( strncmp(context->line, "xlabel", 6) == 0 )
         {
         size = strlen(&context->line[6]);
         if ( (string = get_string (context, &context->line[6], &i1_str, &i2_str, &size, 1)) != NULL )
            {
	    context->xlabel = strdup (string);
            }
         }

      else if ( strncmp(context->line, "ylabel", 6) == 0 )
         {
         size = strlen(&context->line[6]);
         if ( (string = get_string (context, &context->line[6], &i1_str, &i2_str, &size, 1)) != NULL )
            {
	      context->ylabel = xmalloc(size + 1);
            strcpy(context->ylabel, string);
            }
         }

      else if ( strncmp(context->line, "zlabel", 6) == 0 )
         {
         size = strlen(&context->line[6]);
         if ( (string = get_string (context, &context->line[6], &i1_str, &i2_str, &size, 1)) != NULL )
            {
	      context->zlabel = xmalloc(size + 1);
            strcpy(context->zlabel, string);
            }
         }

      else if ( strncmp(context->line, "title", 5) == 0 )
         {
         size = strlen(&context->line[5]);
         if ( (string = get_string (context, &context->line[5], &i1_str, &i2_str, &size, 1)) != NULL )
            {
            context->title = xmalloc(size + 1);
            strcpy(context->title, string);
            }
         }

      else if ( strncmp(context->line, "bin_width", 9) == 0 )
         {
         ibinwidth++;
         if ( ibinwidth <= nplots )
            if ( sscanf(&context->line[9], "%lf", &context->plot_parameters[ibinwidth-1].bin_widths) != 1 )
               {
               g_set_error_literal (&context->err, context->domain, 0, error_str[7]);
               longjmp (context->finish, 1);
               }
         }

      else if ( strncmp(context->line, "bin_value", 9) == 0 )
         {
         ibinvalue++;
         if ( ibinvalue <= nplots )
            {
            size = strlen(&context->line[9]);
            if ( (string = get_string (context, &context->line[9], &i1_str, &i2_str, &size, 0)) != NULL )
               {
		 strncpy(context->plot_parameters[index_bin_values++].bin_values, string, 9);
               }
            }
         }

      else if ( strncmp(context->line, "bin_ref", 7) == 0 )
         {
         ibinref++;
         if ( ibinref <= nplots )
            {
            size = strlen(&context->line[7]);
            if ( (string = get_string (context, &context->line[7], &i1_str, &i2_str, &size, 0)) != NULL )
               {
               strncpy (context->plot_parameters[index_bin_refs].bin_refs, string, 9);
               index_bin_refs++;
               }
            }
         }

      else if ( strncmp(context->line, "ninterp", 7) == 0 )
         {
         ininterp++;
         if ( ininterp <= nplots )
            if ( sscanf(&context->line[7], "%d", &context->plot_parameters[ininterp-1].ninterp) != 1 )
               {
               g_set_error_literal (&context->err, context->domain, 0, error_str[8]);
               longjmp (context->finish, 1);
               }
         }

      else if ( strncmp(context->line, "contours", 8) == 0 )
         {
         if ( sscanf(&context->line[8], "%d", &context->ncontours) != 1 )
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[9]);
            longjmp (context->finish, 1);
            }
         }

      else if ( strncmp(context->line, "contour_color", 13) == 0 )
         {
         icontourcolor++;
         if ( icontourcolor <= nplots )
            {
            if ( sscanf(&context->line[13], " 0x%x", (unsigned int *) &context->plot_parameters[icontourcolor-1].contourcolors) != 1 &&
                 sscanf(&context->line[13], " 0X%x", (unsigned int *) &context->plot_parameters[icontourcolor-1].contourcolors) != 1 )
               {
               if ( sscanf(&context->line[13], " %c", &character) == 1 &&
                    (pchar = strchr(color_string, character)) != NULL )
                  {
                  index = pchar - &color_string[0];                     /* get index to color character */
                  context->plot_parameters[icontourcolor-1].contourcolors = context->color_rgba[index];   /* set specified color */
                  }
               else
                  {
                  g_set_error_literal (&context->err, context->domain, 0, error_str[10]);
                  longjmp (context->finish, 1);
                  }
               }
            }
         }

      else if ( strncmp(context->line, "mesh_color", 10) == 0 )
         {
         imeshcolor++;
         if ( imeshcolor <= nplots )
            {
            if ( sscanf(&context->line[10], " 0x%x", (unsigned int *) &context->plot_parameters[imeshcolor-1].meshcolors) != 1 &&
                 sscanf(&context->line[10], " 0X%x", (unsigned int *) &context->plot_parameters[imeshcolor-1].meshcolors) != 1 )
               {
               if ( sscanf(&context->line[10], " %c", &character) == 1 &&
                    (pchar = strchr(color_string, character)) != NULL )
                  {
                  index = pchar - &color_string[0];               /* get index to color character */
                  context->plot_parameters[imeshcolor-1].meshcolors = context->color_rgba[index];   /* set specified color */
                  }
               else
                  {
                  g_set_error_literal (&context->err, context->domain, 0, error_str[11]);
                  longjmp (context->finish, 1);
                  }
               }
            }
         }

      else if ( strncmp(context->line, "stems", 5) == 0 )
         {
         istemflag++;
         if ( istemflag <= nplots )
            {
            if ( sscanf(&context->line[5], "%lf", &context->plot_parameters[istemflag-1].stemvalues) == 1 )
               {
		 strncpy (context->plot_parameters[index_stemflags].stemflags, "num", 4);
               if ( strcmp(context->plot_param.axis_type, "semilogy") == 0 ||
                    strcmp(context->plot_param.axis_type, "loglog")   == 0 )
                  {
                  if ( context->plot_parameters[istemflag-1].stemvalues != 0.0 )
                     context->plot_parameters[istemflag-1].stemvalues = log10(fabs(context->plot_parameters[istemflag-1].stemvalues));
                  else
                     context->plot_parameters[istemflag-1].stemvalues = log10(DBL_MIN);
                  }
               index_stemflags++;
               }
            else
               {
               size = strlen(&context->line[5]);
               if ( (string = get_string (context, &context->line[5], &i1_str, &i2_str, &size, 0)) != NULL )
                  {
                  strncpy (context->plot_parameters[index_stemflags].stemflags, string, 3);
                  index_stemflags++;
                  }
               }
            }
         }

      else if ( strncmp(context->line, "date_time", 9) == 0 )
         {
         size = strlen(&context->line[9]);
         if ( (string = get_string (context, &context->line[9], &i1_str, &i2_str, &size, 0)) != NULL )
            strncpy(&context->plot_param.date_time_anchor[0], string, 9);
         }

      else if ( strncmp(context->line, "save_close", 10) == 0 )
         {
         size = strlen(&context->line[10]);
         if ( (string = get_string (context, &context->line[10], &i1_str, &i2_str, &size, 0)) != NULL )
            {
	      parser->save_filename  = strdup (string);
	      parser->close_after_save = true;
            }
         }

      else if ( strncmp(context->line, "save", 4) == 0 )
         {
         size = strlen(&context->line[4]);
         if ( (string = get_string (context, &context->line[4], &i1_str, &i2_str, &size, 0)) != NULL )
            {
	      parser->save_filename  = strdup (string);
	      parser->close_after_save = false;
            }
         }

      else if ( strncmp(context->line, "window_size", 11) == 0 )
         {
         if ( sscanf(&context->line[11], "%d %d", &context->window_width, &context->window_height ) != 2 )
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[12]);
            longjmp (context->finish, 1);
            }
         }

      else if ( strncmp(context->line, "plot_box", 8) == 0 )
         {
         size = strlen(&context->line[8]);
         if ( (string = get_string (context, &context->line[8], &i1_str, &i2_str, &size, 0)) != NULL )
            strncpy(&context->plot_param.plot_box[0], string, 3);
         }

      else if ( strncmp(context->line, "x_tick_marks", 12) == 0 )
         {
         size = strlen(&context->line[12]);
         if ( (string = get_string (context, &context->line[12], &i1_str, &i2_str, &size, 0)) != NULL )
            strncpy(&context->plot_param.x_tick_marks[0], string, 3);
         }

      else if ( strncmp(context->line, "y_tick_marks", 12) == 0 )
         {
         size = strlen(&context->line[12]);
         if ( (string = get_string (context, &context->line[12], &i1_str, &i2_str, &size, 0)) != NULL )
            strncpy(&context->plot_param.y_tick_marks[0], string, 3);
         }

      else if ( strncmp(context->line, "z_tick_marks", 12) == 0 )
         {
         size = strlen(&context->line[12]);
         if ( (string = get_string (context, &context->line[12], &i1_str, &i2_str, &size, 0)) != NULL )
            strncpy(&context->plot_param.z_tick_marks[0], string, 3);
         }

      else if ( strncmp(context->line, "x_tick_labels", 13) == 0 )
         {
         size = strlen(&context->line[13]);
         if ( (string = get_string (context, &context->line[13], &i1_str, &i2_str, &size, 0)) != NULL )
            strncpy(&context->plot_param.x_tick_labels[0], string, 3);
         }

      else if ( strncmp(context->line, "y_tick_labels", 13) == 0 )
         {
         size = strlen(&context->line[13]);
         if ( (string = get_string (context, &context->line[13], &i1_str, &i2_str, &size, 0)) != NULL )
            strncpy(&context->plot_param.y_tick_labels[0], string, 3);
         }

      else if ( strncmp(context->line, "z_tick_labels", 13) == 0 )
         {
         size = strlen(&context->line[13]);
         if ( (string = get_string (context, &context->line[13], &i1_str, &i2_str, &size, 0)) != NULL )
            strncpy(&context->plot_param.z_tick_labels[0], string, 3);
         }

      else if ( strncmp(context->line, "font_name", 9) == 0 )
         {
         size = strlen(&context->line[9]);
         if ( (string = get_string (context, &context->line[9], &i1_str, &i2_str, &size, 0)) != NULL )
            {
            free(context->font_name);
            context->font_name = xmalloc(size + 1);
            strcpy(context->font_name, string);
            }
         }

      else if ( strncmp(context->line, "font_size_date_time", 19) == 0 )
         {
         if ( sscanf(&context->line[19], "%lf", &context->font_size_date_time) != 1 )
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[13]);
            longjmp (context->finish, 1);
            }
         }

      else if ( strncmp(context->line, "font_size_legend", 16) == 0 )
         {
         if ( sscanf(&context->line[16], "%lf", &context->font_size_legend) != 1 )
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[14]);
            longjmp (context->finish, 1);
            }
         }

      else if ( strncmp(context->line, "font_size_text", 14) == 0 )
         {
         if ( sscanf(&context->line[14], "%lf", &context->font_size_text) != 1 )
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[15]);
            longjmp (context->finish, 1);
            }
         }

      else if ( strncmp(context->line, "font_size_tick_labels", 21) == 0 )
         {
         if ( sscanf(&context->line[21], "%lf", &context->font_size_tick_labels) != 1 )
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[16]);
            longjmp (context->finish, 1);
            }
         }

      else if ( strncmp(context->line, "font_size_axis_labels", 21) == 0 )
         {
         if ( sscanf(&context->line[21], "%lf", &context->font_size_axis_labels) != 1 )
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[17]);
            longjmp (context->finish, 1);
            }
         }

      else if ( strncmp(context->line, "font_size_title", 15) == 0 )
         {
         if ( sscanf(&context->line[15], "%lf", &context->font_size_title) != 1 )
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[18]);
            longjmp (context->finish, 1);
            }
         }

      else if ( strncmp(context->line, "#####", 5) == 0 )
         break;
      }

   fclose(fptr);


   /* Modify data for logarithmic axes */
   if ( strcmp(context->plot_param.axis_type, "semilogx") == 0 )
      {
      for ( i=1; i<=2; i++ )
         if ( context->axis_limits[i-1])
            {
            if ( context->plot_param.axis_limits[i-1] <= 0.0 )
               {
               g_set_error_literal (&context->err, context->domain, 0, error_str[19]);
               longjmp (context->finish, 1);
               }
            context->plot_param.axis_limits[i-1] = log10(fabs(context->plot_param.axis_limits[i-1]));
            }
      }

   else if ( strcmp(context->plot_param.axis_type, "semilogy") == 0 )
      {
      for ( i=3; i<=4; i++ )
         if ( context->axis_limits[i-1])
            {
            if ( context->plot_param.axis_limits[i-1] <= 0.0 )
               {
               g_set_error_literal (&context->err, context->domain, 0, error_str[20]);
               longjmp (context->finish, 1);
               }
            context->plot_param.axis_limits[i-1] = log10(fabs(context->plot_param.axis_limits[i-1]));
            }
      }

   else if ( strcmp(context->plot_param.axis_type, "loglog") == 0 )
      {
      for ( i=1; i<=4; i++ )
         if ( context->axis_limits[i-1])
            {
            if ( context->plot_param.axis_limits[i-1] <= 0.0 )
               {
               g_set_error_literal (&context->err, context->domain, 0, error_str[21]);
               longjmp (context->finish, 1);
               }
            context->plot_param.axis_limits[i-1] = log10(fabs(context->plot_param.axis_limits[i-1]));
            }
      }


   /* Set font family */
   pango_font_description_set_family(context->font_date_time,   context->font_name);
   pango_font_description_set_family(context->font_legend,      context->font_name);
   pango_font_description_set_family(context->font_text,        context->font_name);
   pango_font_description_set_family(context->font_tick_labels, context->font_name);
   pango_font_description_set_family(context->font_axis_labels, context->font_name);
   pango_font_description_set_family(context->font_title,       context->font_name);


   /* Set font sizes */
   pango_font_description_set_absolute_size(context->font_date_time,   context->font_size_date_time*PANGO_SCALE);
   pango_font_description_set_absolute_size(context->font_legend,      context->font_size_legend*PANGO_SCALE);
   pango_font_description_set_absolute_size(context->font_text,        context->font_size_text*PANGO_SCALE);
   pango_font_description_set_absolute_size(context->font_tick_labels, context->font_size_tick_labels*PANGO_SCALE);
   pango_font_description_set_absolute_size(context->font_axis_labels, context->font_size_axis_labels*PANGO_SCALE);
   pango_font_description_set_absolute_size(context->font_title,       context->font_size_title*PANGO_SCALE);

   return;
   }

