/*******************************************************************************
*
* PolarPlot.c
*
* Calculates polar plot of input data.
*
* Copyright  2008, 2009, 2010, 2011, 2012 Spencer A. Buckner
* http://savannah.gnu.org/projects/gsegrafix
*
* This file is part of GSEGrafix, a scientific and engineering plotting program.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*******************************************************************************/

#include "PolarPlot.h"

#include "DrawSymbols.h"
#include "DrawTickMarks.h"
#include "PlotLines.h"
#include "PlotSymbols.h"
#include "Misc.h"
#include "DrawLines.h"

#include <math.h>
#include <string.h>
#include "gsegraf.h"

#include "gse-cairo.h"

static void     DrawLineSegmentsPolar (struct gse_ctx *context, int iplot, int index, int npts,
                                 double xorigin, double yorigin, double rmin, double rmax,
                                 double rscale, int linechar );


static void     DrawDashedCircle (struct gse_ctx *context, double radius, guint32 fill_color_rgba );
static void     DrawDottedCircle (struct gse_ctx *context, double radius, guint32 fill_color_rgba );


void PolarPlot (struct gse_ctx *context)
   {
   /* Declare variables */
   int i, j, ifunc, nyvalues, nrvalues, iplot, nplots, npts, index, 
       anchor, window_width, window_height;
   guint32 fill_color, outline_color;
   double rmin, rmax, xorigin, yorigin, radius, radius_grid, rscale,
          x, y, x1, y1, x2, y2, xtext1, xtext2, ytext1, ytext2,
          theta, theta_minor, r, r1, dr, width_xtick_label;
   char string[21], *pchar;

   GnomeCanvasPoints *points;
   GnomeCanvasItem *text;


   /* Specify plot-circle location and radius */
   window_width  = context->window_width;
   window_height = context->window_height;
   xorigin = 0.375*window_width;
   yorigin = 0.500*window_height;
   if ( window_width >= window_height )
      radius  = 0.375*window_height;
   else
      radius  = 0.375*window_width;


   /* Save plot-box data */
   context->plot_box_data.xmin = xorigin - radius;
   context->plot_box_data.xmax = xorigin + radius;
   context->plot_box_data.ymin = yorigin - radius;
   context->plot_box_data.ymax = yorigin + radius;


   /* Specify axis minimum and maximum values */
   nyvalues = context->ytick_labels.nvalues;
   nrvalues = nyvalues;
   rmin = context->ytick_labels.values[0];
   rmax = context->ytick_labels.values[nyvalues-1];
   rmin = rmin - context->ytick_labels.offset1;
   rmax = rmax + context->ytick_labels.offset2;
   rscale = radius/(rmax - rmin);


   /* Draw grid lines */
   if ( strcmp(context->plot_param.plot_box, "on") == 0 &&
        (strcmp(context->plot_param.grid, "on1") == 0 ||
         strcmp(context->plot_param.grid, "on2") == 0 ) )
      {
      /* Draw constant-theta grid lines */
      if ( strcmp(context->plot_param.x_tick_marks, "on") == 0 )
         {
         points = gnome_canvas_points_new(2);
         for ( i=1; i<=12; i++ )
            {
            theta = (i - 1.0)*30.0*deg2rad;
            points->coords[0] = xorigin;
            points->coords[1] = yorigin;
            points->coords[2] = xorigin + radius*cos(theta);
            points->coords[3] = yorigin - radius*sin(theta);

            if ( context->gridchar1 == 'l' )
	      DrawLine(context, points, 2, context->gridcolor, 1);
            else if ( context->gridchar1 == 'd' )
	      DrawDashedLine(context, points, context->gridcolor, 1);
            else if ( context->gridchar1 == '.' )
	      DrawDottedLine(context, points, context->gridcolor, 1);
            }
         gnome_canvas_points_unref(points);
         }


      /* Draw constant-r grid lines */
      if ( strcmp(context->plot_param.y_tick_marks, "on") == 0 )
         {
	   for ( i=1; i<=nrvalues; i++ )
	     {
	       double x1 = - (context->ytick_labels.values[i-1] - rmin) * rscale;
	       double x2 =   (context->ytick_labels.values[i-1] - rmin) * rscale;
	       double y1 = - (context->ytick_labels.values[i-1] - rmin) * rscale;
	       double y2 =   (context->ytick_labels.values[i-1] - rmin) * rscale;

	       cairo_save (context->cr);

	       cairo_translate (context->cr, xorigin, yorigin);
	       
	       if ( context->gridchar1 == 'l' )
		 {
		   gse_cairo_render_ellipse (context->cr,
					     x1, y1,
					     x2 - x1,
					     y2 - y1,
					     1,
					     context->gridcolor,
					     0xFFFFFF00);
		 }
	       else if ( context->gridchar1 == 'd' )
		 {
		   radius_grid = (context->ytick_labels.values[i-1] - rmin)*rscale;
		   DrawDashedCircle(context, radius_grid, context->gridcolor);

		 }
	       else if ( context->gridchar1 == '.' )
		 {
		   radius_grid = (context->ytick_labels.values[i-1] - rmin)*rscale;
		   DrawDottedCircle(context, radius_grid, context->gridcolor);
		 }
	       cairo_restore (context->cr);	    
	     }
         }
      }


   /* Draw theta tick marks */
   if ( strcmp(context->plot_param.plot_box, "on") == 0 &&
        strcmp(context->plot_param.x_tick_marks, "on") == 0 )
      {
      points = gnome_canvas_points_new(2);
      for ( i=1; i<=12; i++ )
         {
         theta = (i - 1)*30.0*deg2rad;
         points->coords[0] = xorigin + radius*cos(theta);
         points->coords[1] = yorigin - radius*sin(theta);
         points->coords[2] = xorigin + (radius - context->tick_major)*cos(theta);
         points->coords[3] = yorigin - (radius - context->tick_major)*sin(theta);

	 gse_cairo_render_line (context->cr, points, 2, 1, context->canvas_fg_color);

         if ( context->minor_ticks_flag == 1 )
            {
            for ( j=1; j<=5; j++ )
               {
               theta_minor = ((i - 1)*30.0 + j*5.0)*deg2rad;
               points->coords[0] = xorigin + radius*cos(theta_minor);
               points->coords[1] = yorigin - radius*sin(theta_minor);
               points->coords[2] = xorigin + (radius - context->tick_minor)*cos(theta_minor);
               points->coords[3] = yorigin - (radius - context->tick_minor)*sin(theta_minor);

	       gse_cairo_render_line (context->cr, points, 2, 1, context->canvas_fg_color);
               }
            }
         }
      gnome_canvas_points_unref(points);
      }


   /* Draw r axis and tick marks */
   if ( strcmp(context->plot_param.plot_box, "on") == 0 &&
        strcmp(context->plot_param.y_tick_marks, "on") == 0 )
      {
      points = gnome_canvas_points_new(2);

      theta = 45.0*deg2rad;
      points->coords[0] = xorigin;
      points->coords[1] = yorigin;
      points->coords[2] = xorigin + radius*cos(theta);
      points->coords[3] = yorigin - radius*sin(theta);

      gse_cairo_render_line (context->cr, points, 2, 1, context->canvas_fg_color);

      DrawTickMarks(context, "linear", context->minor_ticks_flag, 1,
                    xorigin, yorigin, xorigin + radius*cos(45.0*deg2rad), yorigin - radius*sin(theta),
                    context->ytick_labels.nvalues, &context->ytick_labels.values[0],
                    context->ytick_labels.offset1, context->ytick_labels.offset2,
                    135.0*deg2rad);

      gnome_canvas_points_unref(points);
      }


   /* Draw plot circle */
   if ( strcmp(context->plot_param.plot_box, "on") == 0 )
     {
       gse_cairo_render_ellipse (context->cr,
				 xorigin - radius,
				 yorigin - radius,
				 2 * radius,
				 2 * radius,
				 2,
				 context->canvas_fg_color,
				 0xFFFFFF00);
     }


   /* Draw theta tick-mark labels */
   if ( strcmp(context->plot_param.plot_box, "on") == 0 &&
        strcmp(context->plot_param.x_tick_marks, "on") == 0 &&
        strcmp(context->plot_param.x_tick_labels, "on") == 0 )
      {
      width_xtick_label = 0.0;
      for ( i=1; i<=12; i++ )
         {
         theta = (i - 1.0)*30.0*deg2rad;
         x = xorigin + (radius + 8.0)*cos(theta);
         y = yorigin - (radius + 8.0)*sin(theta);
         memset(string, 0, sizeof(string));
         snprintf(string, sizeof(string), "%d", (i - 1)*30);

         if ( i == 1 )
            anchor = GSE_ANCHOR_WEST;
         else if ( i == 2  || i == 3 )
            anchor = GSE_ANCHOR_SOUTH_WEST;
         else if ( i == 4 )
            anchor = GSE_ANCHOR_SOUTH;
         else if ( i == 5  || i == 6 )
            anchor = GSE_ANCHOR_SOUTH_EAST;
         else if ( i == 7 )
            anchor = GSE_ANCHOR_EAST;
         else if ( i == 8  || i == 9 )
            anchor = GSE_ANCHOR_NORTH_EAST;
         else if ( i == 10 )
            anchor = GSE_ANCHOR_NORTH;
         else if ( i == 11  || i == 12 )
            anchor = GSE_ANCHOR_NORTH_WEST;

	 gse_cairo_render_text_with_extents (context->cr, string, x, y, anchor,
					     context->canvas_fg_color,
					     context->font_tick_labels, 0,
					     &x1, &y1, &x2, &y2);

         if ( i == 1 )
            {
	      width_xtick_label = x2 - x1;
            }
         }
      }


   /* Draw theta-axis label */
   if ( context->xlabel != NULL )
     {
       if ( strcmp(context->plot_param.plot_box, "on") == 0 )
         {
	   double label_offset = 20.0;
	   double theta_max = 45.0/(radius + label_offset + width_xtick_label);

	   {
	     /* Draw the arrowhead */
	     cairo_save (context->cr);
	     cairo_translate (context->cr, xorigin, yorigin);

	     cairo_translate (context->cr, 
			      +(radius + label_offset) * cos (theta_max),
			      -(radius + label_offset) * sin (theta_max));

	     cairo_rotate (context->cr,  -theta_max);

	     gse_cairo_render_arrowhead (context->cr, 25.0,
					 context->canvas_fg_color);
	     cairo_restore (context->cr);
	   }
	 
	   {
	     /* Draw the arc */
	     cairo_new_path (context->cr);
	     cairo_set_line_width (context->cr, 1);
	     cairo_arc (context->cr,  xorigin, yorigin, radius + label_offset, -theta_max, 0);
	     gse_cairo_set_source_rgba (context->cr, context->canvas_fg_color);
	     cairo_stroke (context->cr);
	   }
         }

       x = xorigin + radius + 28.0 + width_xtick_label;
       y = yorigin;
       gse_cairo_render_text (context->cr, context->xlabel, x, y, GSE_ANCHOR_WEST,
			      context->canvas_fg_color,
			      context->font_axis_labels);
     }


   /* Draw r-axis label */
   if ( context->ylabel != NULL )
     {
       theta = 45.0*deg2rad;
       if ( strcmp(context->plot_param.plot_box, "on") == 0 )
         {
	   {
	     /* Draw the arrowhead */
	     cairo_save (context->cr);
	     cairo_translate (context->cr, xorigin, yorigin);

	     cairo_translate (context->cr,
			      + (radius + 60.0)*cos(theta),
			      - (radius + 60.0)*sin(theta));

	     cairo_rotate (context->cr, theta);
	     
	     gse_cairo_render_arrowhead (context->cr, 25.0,
					 context->canvas_fg_color);
	     cairo_restore (context->cr);
	   }
	   
	   {
	     GnomeCanvasPoints *points = gnome_canvas_points_new(2);

	     points->coords[0] = xorigin + (radius + 15.0)*cos(theta);
	     points->coords[1] = yorigin - (radius + 15.0)*sin(theta);
	     points->coords[2] = xorigin + (radius + 60.0)*cos(theta);
	     points->coords[3] = yorigin - (radius + 60.0)*sin(theta);

	     gse_cairo_render_line (context->cr, points, 2, 1, context->canvas_fg_color);
	 
	     gnome_canvas_points_unref(points);
	   }
         }

       x = xorigin + (radius + 70.0)*cos(theta);
       y = yorigin - (radius + 70.0)*sin(theta);
       gse_cairo_render_text (context->cr, context->ylabel, x, y, GSE_ANCHOR_WEST,
			      context->canvas_fg_color, context->font_axis_labels);
     }


   /* Draw plot title */
   if ( context->title != NULL )
      {
      x = xorigin;
      if ( strcmp(context->plot_param.plot_box, "on") == 0 &&
           strcmp(context->plot_param.x_tick_marks, "on") == 0 &&
           strcmp(context->plot_param.x_tick_labels, "on") == 0 )
         y = yorigin - radius - 8.0 - context->font_size_tick_labels - 8.0;
      else
         y = yorigin - radius - 8.0;
      gse_cairo_render_text (context->cr, context->title, x, y, GSE_ANCHOR_SOUTH,
			     context->canvas_fg_color, context->font_title);
      }


   /* Plot data */
   nplots = context->plot_param.nplots;
   index = 0;
   for ( iplot=1; iplot<=nplots; iplot++ )
      {
	const struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];


      npts = the_plot->ndata;

      /* Draw stem lines */
      if ( strncmp (the_plot->stemflags,  "on", 9) == 0 ||
           strncmp (the_plot->stemflags, "num", 9) == 0 )
         {
         points = gnome_canvas_points_new(2);

         /* Calculate r coordinate of stem point 1 */
         if ( strncmp (the_plot->stemflags, "on", 9) == 0 )
            r1 = rmin;
         else if ( strncmp (the_plot->stemflags, "num", 9) == 0 )
            {
            if ( rmin <= the_plot->stemvalues && the_plot->stemvalues <= rmax )
               r1 = the_plot->stemvalues;
            else if ( the_plot->stemvalues < rmin )
               r1 = rmin;
            else if ( the_plot->stemvalues > rmax )
               r1 = rmax;
            }

         for ( i=1; i<=npts; i++ )
            {
            theta = the_plot->samples[i-1].x;
            r     = the_plot->samples[i-1].y;
            if ( r <= rmax )
               {
               /* Calculate coordinates of stem point 1 */
               x = xorigin + (r1 - rmin)*cos(theta)*rscale;
               y = yorigin - (r1 - rmin)*sin(theta)*rscale;
               points->coords[0] = x;
               points->coords[1] = y;

               /* Calculate coordinates of stem point 2 */
               if ( r < rmin )
                  r = rmin;
               x = xorigin + (r - rmin)*cos(theta)*rscale;
               y = yorigin - (r - rmin)*sin(theta)*rscale;
               points->coords[2] = x;
               points->coords[3] = y;
               DrawLine(context, points, 2, the_plot->outline_colors_rgba, 1);
               }
            }

         gnome_canvas_points_unref(points);
         }


      /* Draw lines */
      if ( the_plot->stylechar1 == 'l' )
	DrawLineSegmentsPolar(context, iplot, index, npts, xorigin, yorigin, rmin, rmax, rscale, 'l');


      /* Draw dashed lines */
      else if ( the_plot->stylechar1 == 'd' )
	DrawLineSegmentsPolar(context, iplot, index, npts, xorigin, yorigin, rmin, rmax, rscale, 'd');


      /* Draw dashed lines */
      else if ( the_plot->stylechar1 == '.' )
	DrawLineSegmentsPolar(context, iplot, index, npts, xorigin, yorigin, rmin, rmax, rscale, '.');


      /* Draw symbols in symbol_string1 ("cCtTsSiIpPhH") */
      else if ( (pchar = strchr(symbol_string1, the_plot->stylechar1)) != NULL )
         {
         ifunc = pchar - symbol_string1;
         for ( i=1; i<=npts; i++ )
            {
            theta = the_plot->samples[i-1].x;
            r     = the_plot->samples[i-1].y;
            if ( rmin <= r && r <= rmax )
               {
               x = xorigin + (r - rmin)*cos(theta)*rscale;
               y = yorigin - (r - rmin)*sin(theta)*rscale;
               context->symbol_func1[ifunc](context, x, y,
				   the_plot->fill_colors_rgba,
				   the_plot->outline_colors_rgba,
                                   the_plot->stylesizes);
               }
            }
         }


      /* Draw symbols in symbol_string2 ("+xra") */
      else if ( (pchar = strchr(symbol_string2, the_plot->stylechar1)) != NULL )
         {
         ifunc = pchar - symbol_string2;
         for ( i=1; i<=npts; i++ )
            {
            theta = the_plot->samples[i-1].x;
            r     = the_plot->samples[i-1].y;
            if ( rmin <= r && r <= rmax )
               {
               x = xorigin + (r - rmin)*cos(theta)*rscale;
               y = yorigin - (r - rmin)*sin(theta)*rscale;
               context->symbol_func2[ifunc](context, x, y, the_plot->fill_colors_rgba, the_plot->stylesizes);
               }
            }
         }

      index = index + the_plot->ndata;
      }


   /* Plot specified lines and symbols */
   PlotLines(context);
   PlotSymbols(context);


   /* Draw r tick-mark labels on translucent rectangles */
   if ( strcmp(context->plot_param.plot_box, "on")      == 0 &&
        strcmp(context->plot_param.y_tick_marks, "on")  == 0 &&
        strcmp(context->plot_param.y_tick_labels, "on") == 0 )
      {
      dr = (rmax - rmin)/(nrvalues - 1);
      for ( i=1; i<=nrvalues; i++ )
         {
         theta = 45.0*deg2rad;
         x = xorigin + (context->ytick_labels.values[i-1] - rmin)*rscale*cos(theta);
         y = yorigin - (context->ytick_labels.values[i-1] - rmin)*rscale*sin(theta);

         memset(string, 0, sizeof(string));
         if ( fabs(context->ytick_labels.values[i-1]) < 0.01*dr )
            snprintf(string, sizeof(string), "%1.0f", 0.0);
         else
            snprintf(string, sizeof(string), "%g", context->ytick_labels.values[i-1]);

	 gse_cairo_render_text_with_extents (context->cr, string, x - 10.0, y, GSE_ANCHOR_EAST,
					     context->canvas_fg_color, context->font_tick_labels, 0,
				&xtext1, &ytext1, &xtext2, &ytext2);

         fill_color = context->canvas_bg_color - 0xFF + 0xC0;
         outline_color = context->canvas_bg_color - 0xFF + 0xC0;
         gnome_canvas_item_new(context->group,
                               GNOME_TYPE_CANVAS_RECT,
                               "x1", xtext1,
                               "x2", xtext2,
                               "y1", ytext1,
                               "y2", ytext2,
                               "fill_color_rgba", fill_color,
                               "outline_color_rgba", outline_color,
                               "width_pixels", 1,
                               NULL);

         gnome_canvas_item_raise_to_top(text);
         }
      }

   return;
   }


void DrawLineSegmentsPolar (struct gse_ctx *context, int iplot, int index, int npts,
                             double xorigin, double yorigin, double rmin, double rmax,
                             double rscale, int linechar )
   {
   /* Declare variables */
   int idraw, iseg, nseg, npts_seg, iseg1;
   char linetype[7];

   const struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];

   /* Create linetype string */
   memset(linetype, 0, sizeof(linetype));
   if ( linechar == 'l' )
      strcpy(linetype, "solid");
   else if ( linechar == 'd' )
      strcpy(linetype, "dashed");
   else if ( linechar == '.' )
      strcpy(linetype, "dotted");


   /* Draw all line segments except last */
   idraw = 0;
   nseg = the_plot->nlinebreaks;
   iseg1 = index;
   for (iseg=1; iseg<=nseg; iseg++ )
      {
      if ( index < the_plot->nlinebreak[iseg-1] &&
	   the_plot->nlinebreak[iseg-1] < index + npts )
         {
         idraw++;
         npts_seg = the_plot->nlinebreak[iseg-1] - iseg1;
         DrawLinesPolar(context, npts_seg,
			&the_plot->samples[iseg1 - index],
			xorigin, yorigin, rmin, rmax, rscale,
                        the_plot->fill_colors_rgba, the_plot->stylesizes, linetype);
         iseg1 = the_plot->nlinebreak[iseg-1];
         }
      }


   /* Draw last line segment */
   if ( idraw > 0 )
      {
      npts_seg = index + npts - iseg1;
      DrawLinesPolar(context, npts_seg,
		     &the_plot->samples[iseg1 - index],
		     xorigin, yorigin, rmin, rmax, rscale,
                     the_plot->fill_colors_rgba, the_plot->stylesizes, linetype);
      }

   /* Draw continuous line */
   else
     DrawLinesPolar(context, npts,
		    &the_plot->samples[0],
		    xorigin, yorigin, rmin, rmax, rscale,
                     the_plot->fill_colors_rgba, the_plot->stylesizes, linetype);

   return;
   }


static void
DrawDashedCircle (struct gse_ctx *context, double radius, guint32 fill_color_rgba)
{
  /* Declare variables */
  int i, j;

  /* Check radius */
  if ( radius == 0.0 )
    return;
  
  /* Draw dashed circle */
  int ndashes = roundint(2.0*M_PI*radius/(context->dash + context->space_dash));
  double dtheta1 = 2.0*M_PI/ndashes;
  double dtheta2 = 0.5*M_PI*context->dash/(ndashes*(context->dash + context->space_dash));

  GnomeCanvasPoints *points = gnome_canvas_points_new(5);

  for ( i=1; i<=ndashes; i++ )
    {
      double theta1 = (i - 1)*dtheta1;
      points->coords[0] = radius*cos(theta1);
      points->coords[1] = -radius*sin(theta1);
      for ( j=1; j<=4; j++ )
	{
	  double theta2 = theta1 + j*dtheta2;
	  points->coords[2*j]   = radius*cos(theta2);
	  points->coords[2*j+1] = -radius*sin(theta2);
	}

      gse_cairo_render_line (context->cr, points, 5, 1, fill_color_rgba);
    }

  gnome_canvas_points_unref(points);
}


static void
DrawDottedCircle (struct gse_ctx *context, double radius, guint32 fill_color_rgba)
   {
   /* Declare variables */
   int i, npoints, ndots;
   double theta[361], length[361], length_total, inc, inc_calc, length_interp, theta_interp;


   /* Check radius */
   if ( radius == 0 )
      return;


   /* Calculate circle length */
   npoints = 361;
   for ( i=1; i<=npoints; i++ )
      {
      theta[i-1] = (i - 1)*M_PI/180.0;
      length[i-1] = radius*theta[i-1];
      }
   length_total = length[npoints-1];


   /* Calculate number of dots */
   inc = context->space_dot + 1.0;
   ndots = roundint(length_total/inc + 1.0);
   inc_calc = length_total/(ndots - 1);


   /* Interpolate and draw dots */
   for ( i=1; i<ndots; i++ )
      {
      length_interp = length[0] + (i - 1)*inc_calc;
      interp1(npoints, 1, length, theta, &length_interp, &theta_interp);
      DrawCircle(context, radius*cos(theta_interp), radius*sin(theta_interp),
                 fill_color_rgba, fill_color_rgba, 1);
      }
   }

