/*******************************************************************************
*
* PlotSymbols.c
*
* Plots symbols.
*
* Copyright  2008, 2009, 2010, 2011, 2012 Spencer A. Buckner
* http://savannah.gnu.org/projects/gsegrafix
*
* This file is part of GSEGrafix, a scientific and engineering plotting program.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*******************************************************************************/

#include "PlotSymbols.h"
#include "Misc.h"

#include <ctype.h>
#include <math.h>
#include "gsegraf.h"

#include <string.h>

void PlotSymbols (struct gse_ctx *context)
   {
   /* Declare variables */
   int i, ifunc, nx, ny, nz, ncoords, window_width, window_height;
   unsigned int index, symbol_size;
   guint32 symbol_color;
   double x1_box, x2_box, y1_box, y2_box,
          xmin, xmax, ymin, ymax, zmin, zmax, rmin, rmax,
          xscale, yscale, zscale, xscalesq, yscalesq, zscalesq,
          rscale, xorigin, yorigin, radius, theta, phi,
          axis_length, *p, origin[3], Ry[9], Rz[9], Ryz[9], r[3],
          symbol_coords[3] = { 0.0, 0.0, 0.0 }, x, y;
   char *pchar, symbol_char, color_char;

   const char *error_str[] =
      { "Invalid or missing symbol coordinates.",
        "Invalid symbol color character.",
        "Invalid or missing symbol or color specification.",
        "Invalid symbol character." };
   FILE *fptr;


   /* Return if no symbols to plot */
   fptr = fopen(context->p_param_file, "r");
   i = 0;
   while ( fgets(context->line, context->maxline, fptr) != NULL )
      if ( strncmp(context->line, "symbol_coords", 13) == 0 )
         i++;
   fclose(fptr);
   if ( i == 0 )
      return;


   /* Get minimum and maximum axis values */
   if ( strcmp(context->plot_param.axis_type, "linear")    == 0 )
      {
      nx = context->xtick_labels.nvalues;
      xmin = context->xtick_labels.values[0];
      xmax = context->xtick_labels.values[nx-1];
      xmin = xmin - context->xtick_labels.offset1;
      xmax = xmax + context->xtick_labels.offset2;
      ny = context->ytick_labels.nvalues;
      ymin = context->ytick_labels.values[0];
      ymax = context->ytick_labels.values[ny-1];
      ymin = ymin - context->ytick_labels.offset1;
      ymax = ymax + context->ytick_labels.offset2;
      }

   else if ( strcmp(context->plot_param.axis_type, "semilogx") == 0 )
      {
      nx = context->xtick_labels.nvalues;
      xmin = floor(context->xtick_labels.values[0]);
      xmax = ceil(context->xtick_labels.values[nx-1]);
      nx = roundint(xmax - xmin + 1.0);
      ny = context->ytick_labels.nvalues;
      ymin = context->ytick_labels.values[0];
      ymax = context->ytick_labels.values[ny-1];
      ymin = ymin - context->ytick_labels.offset1;
      ymax = ymax + context->ytick_labels.offset2;
      }

   else if ( strcmp(context->plot_param.axis_type, "semilogy") == 0 )
      {
      nx = context->xtick_labels.nvalues;
      xmin = context->xtick_labels.values[0];
      xmax = context->xtick_labels.values[nx-1];
      xmin = xmin - context->xtick_labels.offset1;
      xmax = xmax + context->xtick_labels.offset2;
      ny = context->ytick_labels.nvalues;
      ymin = floor(context->ytick_labels.values[0]);
      ymax = ceil(context->ytick_labels.values[ny-1]);
      ny = roundint(ymax - ymin + 1.0);
      }

   else if ( strcmp(context->plot_param.axis_type, "loglog") == 0 )
      {
      nx = context->xtick_labels.nvalues;
      xmin = floor(context->xtick_labels.values[0]);
      xmax = ceil(context->xtick_labels.values[nx-1]);
      nx = roundint(xmax - xmin + 1.0);
      ny = context->ytick_labels.nvalues;
      ymin = floor(context->ytick_labels.values[0]);
      ymax = ceil(context->ytick_labels.values[ny-1]);
      ny = roundint(ymax - ymin + 1.0);
      }

   else if ( strcmp(context->plot_param.axis_type, "polar") == 0 )
      {
      ny = context->ytick_labels.nvalues;
      rmin = context->ytick_labels.values[0];
      rmax = context->ytick_labels.values[ny-1];
      rmin = rmin - context->ytick_labels.offset1;
      rmax = rmax + context->ytick_labels.offset2;
      }

   else if ( strcmp(context->plot_param.axis_type, "3d") == 0 )
      {
      nx = context->xtick_labels.nvalues;
      ny = context->ytick_labels.nvalues;
      nz = context->ztick_labels.nvalues;
      xmin = context->xtick_labels.values[0];
      xmax = context->xtick_labels.values[nx-1];
      ymin = context->ytick_labels.values[0];
      ymax = context->ytick_labels.values[ny-1];
      zmin = context->ztick_labels.values[0];
      zmax = context->ztick_labels.values[nz-1];
      xmin = xmin - context->xtick_labels.offset1;
      xmax = xmax + context->xtick_labels.offset2;
      ymin = ymin - context->ytick_labels.offset1;
      ymax = ymax + context->ytick_labels.offset2;
      zmin = zmin - context->ztick_labels.offset1;
      zmax = zmax + context->ztick_labels.offset2;
      }


   /* Calculate axis scale factors */
   if ( strcmp(context->plot_param.axis_type, "linear")    == 0 ||
        strcmp(context->plot_param.axis_type, "semilogx")  == 0 ||
        strcmp(context->plot_param.axis_type, "semilogy")  == 0 ||
        strcmp(context->plot_param.axis_type, "loglog")    == 0 )
      {
      x1_box = context->plot_box_data.xmin;
      x2_box = context->plot_box_data.xmax;
      y1_box = context->plot_box_data.ymin;
      y2_box = context->plot_box_data.ymax;
      xscale = (x2_box - x1_box)/(xmax - xmin);
      yscale = (y2_box - y1_box)/(ymax - ymin);
      xscalesq = xscale*xscale;
      yscalesq = yscale*yscale;
      }

   else if ( strcmp(context->plot_param.axis_type, "polar") == 0 )
      {
      window_width  = context->window_width;
      window_height = context->window_height;
      xorigin = 0.375*window_width;
      yorigin = 0.500*window_height;
      if ( window_width >= window_height )
         radius = 0.375*window_height;
      else
         radius = 0.375*window_width;
      rscale = radius/(rmax - rmin);
      }

   else if ( strcmp(context->plot_param.axis_type, "3d") == 0 )
      {
      /* Get view angles */
      phi = context->plot_param_3d.phi;
      theta = context->plot_param_3d.theta;

      /* Get origin */
      if ( phi >= 0.0 && phi < 90.0 )
         {
         origin[0] = context->plot_param_3d.origin[0];
         origin[1] = context->plot_param_3d.origin[1];
         origin[2] = context->plot_param_3d.origin[2];
         }

      else if ( phi >= 90.0 && phi < 180.0 )
         {
         origin[0] = context->plot_param_3d.origin[0] + context->plot_param_3d.axis2[0];
         origin[1] = context->plot_param_3d.origin[1] + context->plot_param_3d.axis2[1];
         origin[2] = context->plot_param_3d.origin[2] - context->plot_param_3d.axis2[2];
         }

      else if ( phi >= 180.0 && phi < 270.0 )
         {
         origin[0] = context->plot_param_3d.origin[0] + context->plot_param_3d.axis1[0] + context->plot_param_3d.axis2[0];
         origin[1] = context->plot_param_3d.origin[1] + context->plot_param_3d.axis1[1] + context->plot_param_3d.axis2[1];
         origin[2] = context->plot_param_3d.origin[2] - context->plot_param_3d.axis1[2] - context->plot_param_3d.axis2[2];
         }

      else if ( phi >= 270.0 && phi < 360.0 )
         {
         origin[0] = context->plot_param_3d.origin[0] + context->plot_param_3d.axis1[0];
         origin[1] = context->plot_param_3d.origin[1] + context->plot_param_3d.axis1[1];
         origin[2] = context->plot_param_3d.origin[2] - context->plot_param_3d.axis1[2];
         }

      /* Calculate rotation matrices */
      phi = phi*deg2rad;
      theta = theta*deg2rad;

      Ry[0] = cos(-theta);
      Ry[1] = 0.0;
      Ry[2] = -sin(-theta);
      Ry[3] = 0.0;
      Ry[4] = 1.0;
      Ry[5] = 0.0;
      Ry[6] = sin(-theta);
      Ry[7] = 0.0;
      Ry[8] = cos(-theta);

      Rz[0] = cos(phi);
      Rz[1] = sin(phi);
      Rz[2] = 0.0;
      Rz[3] = -sin(phi);
      Rz[4] = cos(phi);
      Rz[5] = 0.0;
      Rz[6] = 0.0;
      Rz[7] = 0.0;
      Rz[8] = 1.0;

      p = multiply_mm(Ry, Rz);
      for ( i=1; i<=9; i++, p++ )
         Ryz[i-1] = *p;

      /* Get axis length */
      axis_length = context->plot_param_3d.axis_length;

      /* Calculate axis scale factors */
      xscale = axis_length/(xmax - xmin);
      yscale = axis_length/(ymax - ymin);
      zscale = axis_length/(zmax - zmin);
      xscalesq = xscale*xscale;
      yscalesq = yscale*yscale;
      zscalesq = zscale*zscale;
      }


   /* Specify number of symbol coordinates */
   if ( strcmp(context->plot_param.axis_type, "linear")    == 0 ||
        strcmp(context->plot_param.axis_type, "semilogx")  == 0 ||
        strcmp(context->plot_param.axis_type, "semilogy")  == 0 ||
        strcmp(context->plot_param.axis_type, "loglog")    == 0 ||
        strcmp(context->plot_param.axis_type, "polar")     == 0 )
      ncoords = 2;

   else if ( strcmp(context->plot_param.axis_type, "3d") == 0 )
      ncoords = 3;


   /* Draw symbols */
   fptr = fopen(context->p_param_file, "r");
   while ( fgets(context->line, context->maxline, fptr) != NULL )
      {
      if ( strncmp(context->line, "symbol_coords", 13) == 0 )
         {
         if ( ncoords == 2 )
            {
            if ( sscanf(&context->line[13], " %lf %lf",
                        &symbol_coords[0], &symbol_coords[1]) != 2 )
               {
               g_set_error_literal (&context->err, context->domain, 0, error_str[0]);
               longjmp (context->finish, 1);
               }
            }

         else if ( ncoords == 3 )
            {
            if ( sscanf(&context->line[13], " %lf %lf %lf",
                        &symbol_coords[0], &symbol_coords[1], &symbol_coords[2]) != 3 )
               {
               g_set_error_literal (&context->err, context->domain, 0, error_str[0]);
               longjmp (context->finish, 1);
               }
            }

         /* Modify symbol coordinates for logarithmic and polar axes */
         if ( strcmp(context->plot_param.axis_type, "semilogx") == 0 )
            symbol_coords[0] = log10(fabs(symbol_coords[0]));

         else if ( strcmp(context->plot_param.axis_type, "semilogy") == 0 )
            symbol_coords[1] = log10(fabs(symbol_coords[1]));

         else if ( strcmp(context->plot_param.axis_type, "loglog") == 0 )
            {
            symbol_coords[0] = log10(fabs(symbol_coords[0]));
            symbol_coords[1] = log10(fabs(symbol_coords[1]));
            }

         else if ( strcmp(context->plot_param.axis_type, "polar") == 0 )
            symbol_coords[0] = symbol_coords[0]*deg2rad;

         /* Get symbol type and color */
         symbol_size = 6;
         if ( fgets(context->line, context->maxline, fptr) != NULL )
            if ( strncmp(context->line, "symbol_style", 12) == 0 )
               {
               if ( sscanf(&context->line[12], " %c 0x%x", &symbol_char, (unsigned int *) &symbol_color) == 2 ||
                    sscanf(&context->line[12], " %c 0X%x", &symbol_char, (unsigned int *) &symbol_color) == 2 )
                  sscanf(&context->line[12], " %*c %*s %u", &symbol_size);

               else if ( sscanf(&context->line[12], " %c %c", &symbol_char, &color_char) == 2 )
                  {
                  if ( (pchar = strchr(color_string, color_char)) == NULL )
                     {
                     g_set_error_literal (&context->err, context->domain, 0, error_str[1]);
                     longjmp (context->finish, 1);
                     }

                  else
                     {
                     index = pchar - &color_string[0];   /* get index to color character */
                     symbol_color = context->color_rgba[index];
                     sscanf(&context->line[12], " %*c %*c %u", &symbol_size);
                     }
                  }

                else
                  {
                  g_set_error_literal (&context->err, context->domain, 0, error_str[2]);
                  longjmp (context->finish, 1);
                  }

               /* Check symbol type */
               if ( (pchar = strchr("cCtTsSiIpPhH+xra", symbol_char)) == NULL )
                  {
                  g_set_error_literal (&context->err, context->domain, 0, error_str[3]);
                  longjmp (context->finish, 1);
                  }
               }

         /* Draw symbol */
         if ( strcmp(context->plot_param.axis_type, "linear")    == 0 ||
              strcmp(context->plot_param.axis_type, "semilogx")  == 0 ||
              strcmp(context->plot_param.axis_type, "semilogy")  == 0 ||
              strcmp(context->plot_param.axis_type, "loglog")    == 0 )
            {
            if ( xmin <= symbol_coords[0] && symbol_coords[0] <= xmax &&
                 ymin <= symbol_coords[1] && symbol_coords[1] <= ymax )
               {
               x = x1_box + (symbol_coords[0] - xmin)*xscale;
               y = y2_box - (symbol_coords[1] - ymin)*yscale;
               }
            else
               continue;
            }

         else if ( strcmp(context->plot_param.axis_type, "polar") == 0 )
            {
            if ( rmin <= symbol_coords[1] && symbol_coords[1] <= rmax )
               {
               x = xorigin + (symbol_coords[1] - rmin)*cos(symbol_coords[0])*rscale;
               y = yorigin - (symbol_coords[1] - rmin)*sin(symbol_coords[0])*rscale;
               }
            else
               continue;
            }

         else if ( strcmp(context->plot_param.axis_type, "3d") == 0 )
            {
            if ( xmin <= symbol_coords[0] && symbol_coords[0] <= xmax &&
                 ymin <= symbol_coords[1] && symbol_coords[1] <= ymax &&
                 zmin <= symbol_coords[2] && symbol_coords[2] <= zmax )
               {
               r[0] = (symbol_coords[0] - xmin)*xscale;
               r[1] = (symbol_coords[1] - ymin)*yscale;
               r[2] = (symbol_coords[2] - zmin)*zscale;

               p = multiply_mv(Ryz, r);
               for ( i=1; i<=3; i++, p++ )
                  r[i-1] = *p;

               x = origin[1] + r[1];
               y = origin[2] - r[2];
               }
            else
               continue;
            }

         /* Draw symbols in symbol_string1 ("cCtTsSiIpPhH") */
         if ( (pchar = strchr(symbol_string1, symbol_char)) != NULL )
            {
            ifunc = pchar - symbol_string1;
            if ( isupper(*pchar) == 0 )
	      context->symbol_func1[ifunc](context, x, y, context->color_rgba[3], symbol_color, symbol_size);
            else
	      context->symbol_func1[ifunc](context, x, y, symbol_color, symbol_color, symbol_size);
            }

         /* Draw symbols in symbol_string2 ("+xra") */
         else if ( (pchar = strchr(symbol_string2, symbol_char)) != NULL )
            {
            ifunc = pchar - symbol_string2;
            context->symbol_func2[ifunc](context, x, y, symbol_color, symbol_size);
            }
         }

      else if ( strncmp(context->line, "#####", 5) == 0 )
         break;
      }

   fclose(fptr);

   return;
   }
