/*******************************************************************************
*
* PlotPoints3d.c
*
* Plots a two-dimensional projection of three-dimensional points data.
*
* Copyright  2008, 2009, 2010, 2011, 2012 Spencer A. Buckner
* http://savannah.gnu.org/projects/gsegrafix
*
* This file is part of GSEGrafix, a scientific and engineering plotting program.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*******************************************************************************/

#include "PlotPoints3d.h"
#include "Misc.h"
#include "DrawLines.h"
#include "DrawSymbols.h"

#include <math.h>
#include "gsegraf.h"

#include <string.h>

static void DrawLineSegments3d (struct gse_ctx *context,
				int iplot, int npts, double *origin, double *Ryz,
                          double xmin, double xmax, double ymin, double ymax, double zmin, double zmax,
				double xscale, double yscale, double zscale, int linechar );


void PlotPoints3d (struct gse_ctx *context, int iplot)
   {
   /* Declare variables */
     int i, j, ifunc, nxvalues, nyvalues, nzvalues;
   double phi, theta, axis_length, origin[3], Ry[9], Rz[9], Ryz[9], r[3], rstem[3], *p,
          xmin, xmax, ymin, ymax, zmin, zmax, xscale, yscale, zscale, x, y;
   char *pchar;
   GnomeCanvasPoints *points;

   const struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];
   int npts = the_plot->ndata;

   /* Specify view angles */
   phi = context->plot_param_3d.phi;       /* view-direction azimuth (deg) from x axis in x-y plane */
   theta = context->plot_param_3d.theta;   /* view-direction elevation (deg) from x-y plane */


   /* Specify origin */
   if ( phi >= 0.0 && phi < 90.0 )
      {
      origin[0] = context->plot_param_3d.origin[0];
      origin[1] = context->plot_param_3d.origin[1];
      origin[2] = context->plot_param_3d.origin[2];
      }

   else if ( phi >= 90.0 && phi < 180.0 )
      {
      origin[0] = context->plot_param_3d.origin[0] + context->plot_param_3d.axis2[0];
      origin[1] = context->plot_param_3d.origin[1] + context->plot_param_3d.axis2[1];
      origin[2] = context->plot_param_3d.origin[2] - context->plot_param_3d.axis2[2];
      }

   else if ( phi >= 180.0 && phi < 270.0 )
      {
      origin[0] = context->plot_param_3d.origin[0] + context->plot_param_3d.axis1[0] + context->plot_param_3d.axis2[0];
      origin[1] = context->plot_param_3d.origin[1] + context->plot_param_3d.axis1[1] + context->plot_param_3d.axis2[1];
      origin[2] = context->plot_param_3d.origin[2] - context->plot_param_3d.axis1[2] - context->plot_param_3d.axis2[2];
      }

   else if ( phi >= 270.0 && phi < 360.0 )
      {
      origin[0] = context->plot_param_3d.origin[0] + context->plot_param_3d.axis1[0];
      origin[1] = context->plot_param_3d.origin[1] + context->plot_param_3d.axis1[1];
      origin[2] = context->plot_param_3d.origin[2] - context->plot_param_3d.axis1[2];
      }


   /* Calculate rotation matrices */
   phi = phi*deg2rad;
   theta = theta*deg2rad;

   Ry[0] = cos(-theta);
   Ry[1] = 0.0;
   Ry[2] = -sin(-theta);
   Ry[3] = 0.0;
   Ry[4] = 1.0;
   Ry[5] = 0.0;
   Ry[6] = sin(-theta);
   Ry[7] = 0.0;
   Ry[8] = cos(-theta);

   Rz[0] = cos(phi);
   Rz[1] = sin(phi);
   Rz[2] = 0.0;
   Rz[3] = -sin(phi);
   Rz[4] = cos(phi);
   Rz[5] = 0.0;
   Rz[6] = 0.0;
   Rz[7] = 0.0;
   Rz[8] = 1.0;

   p = multiply_mm(Ry, Rz);
   for ( i=1; i<=9; i++, p++ )
      Ryz[i-1] = *p;


   /* Specify axis length */
   axis_length = context->plot_param_3d.axis_length;


   /* Specify minimum and maximum axis values */
   nxvalues = context->xtick_labels.nvalues;
   nyvalues = context->ytick_labels.nvalues;
   nzvalues = context->ztick_labels.nvalues;
   xmin = context->xtick_labels.values[0];
   xmax = context->xtick_labels.values[nxvalues-1];
   ymin = context->ytick_labels.values[0];
   ymax = context->ytick_labels.values[nyvalues-1];
   zmin = context->ztick_labels.values[0];
   zmax = context->ztick_labels.values[nzvalues-1];
   xmin = xmin - context->xtick_labels.offset1;
   xmax = xmax + context->xtick_labels.offset2;
   ymin = ymin - context->ytick_labels.offset1;
   ymax = ymax + context->ytick_labels.offset2;
   zmin = zmin - context->ztick_labels.offset1;
   zmax = zmax + context->ztick_labels.offset2;


   /* Calculate axis scale factors */
   xscale = axis_length/(xmax - xmin);
   yscale = axis_length/(ymax - ymin);
   zscale = axis_length/(zmax - zmin);


   /* Draw stem lines */
   if ( strncmp (the_plot->stemflags,  "on", 9) == 0 || 
        strncmp (the_plot->stemflags, "num", 9) == 0 )
      {
      points = gnome_canvas_points_new(2);

      /* Calculate z coordinate of stem point 1 */
      if ( strncmp (the_plot->stemflags, "on", 9) == 0 )
         rstem[2] = 0.0;
      else if ( strncmp (the_plot->stemflags, "num", 9) == 0 )
         {
         if ( zmin <= the_plot->stemvalues && the_plot->stemvalues <= zmax )
            rstem[2] = (the_plot->stemvalues - zmin)*zscale;
         else if ( the_plot->stemvalues < zmin )
            rstem[2] = 0.0;
         else if ( the_plot->stemvalues > zmax )
            rstem[2] = axis_length;
         }

      for ( i=1; i<=npts; i++ )
         if ( xmin <= the_plot->samples[i-1].x && the_plot->samples[i-1].x <= xmax &&
              ymin <= the_plot->samples[i-1].y && the_plot->samples[i-1].y <= ymax &&
              zmin <= the_plot->samples[i-1].z && the_plot->samples[i-1].z <= zmax )
            {
            /* Calculate coordinates of stem point 1 */
            rstem[0] = (the_plot->samples[i-1].x - xmin)*xscale;
            rstem[1] = (the_plot->samples[i-1].y - ymin)*yscale;

            p = multiply_mv(Ryz, rstem);
            for ( j=1; j<=3; j++, p++ )
               r[j-1] = *p;

            x = origin[1] + r[1];
            y = origin[2] - r[2];
            points->coords[0] = x;
            points->coords[1] = y;

            /* Calculate coordinates of stem point 2 */
            r[0] = (the_plot->samples[i-1].x - xmin)*xscale;
            r[1] = (the_plot->samples[i-1].y - ymin)*yscale;
            r[2] = (the_plot->samples[i-1].z - zmin)*zscale;
            p = multiply_mv(Ryz, r);
            for ( j=1; j<=3; j++, p++ )
               r[j-1] = *p;

            x = origin[1] + r[1];
            y = origin[2] - r[2];
            points->coords[2] = x;
            points->coords[3] = y;

            DrawLine(context, points, 2, the_plot->outline_colors_rgba, 1);
            }

      gnome_canvas_points_unref(points);
      }


   /* Draw lines */
   if ( the_plot->stylechar1 == 'l' )
     DrawLineSegments3d(context, iplot, npts, &origin[0], &Ryz[0],
                         xmin, xmax, ymin, ymax, zmin, zmax,
                         xscale, yscale, zscale, 'l');


   /* Draw dashed lines */
   else if ( the_plot->stylechar1 == 'd' )
     DrawLineSegments3d(context, iplot, npts, &origin[0], &Ryz[0],
                         xmin, xmax, ymin, ymax, zmin, zmax,
                         xscale, yscale, zscale, 'd');


   /* Draw dotted lines */
   else if ( the_plot->stylechar1 == '.' )
     DrawLineSegments3d(context, iplot, npts, &origin[0], &Ryz[0],
                         xmin, xmax, ymin, ymax, zmin, zmax,
                         xscale, yscale, zscale, '.');


   /* Draw symbols in symbol_string1 ("cCtTsSiIpPhH") */
   else if ( (pchar = strchr(symbol_string1, the_plot->stylechar1)) != NULL )
      {
      ifunc = pchar - symbol_string1;
      for ( i=1; i<=npts; i++ )
         if ( xmin <= the_plot->samples[i-1].x && the_plot->samples[i-1].x <= xmax &&
              ymin <= the_plot->samples[i-1].y && the_plot->samples[i-1].y <= ymax &&
              zmin <= the_plot->samples[i-1].z && the_plot->samples[i-1].z <= zmax )
            {
            r[0] = (the_plot->samples[i-1].x - xmin)*xscale;
            r[1] = (the_plot->samples[i-1].y - ymin)*yscale;
            r[2] = (the_plot->samples[i-1].z - zmin)*zscale;

            p = multiply_mv(Ryz, r);
            for ( j=1; j<=3; j++, p++ )
               r[j-1] = *p;

            x = origin[1] + r[1];
            y = origin[2] - r[2];
            context->symbol_func1[ifunc](context, x, y, the_plot->fill_colors_rgba, the_plot->outline_colors_rgba,
                                the_plot->stylesizes);
            }
      }


   /* Draw symbols in symbol_string2 ("+xra") */
   else if ( (pchar = strchr(symbol_string2, the_plot->stylechar1)) != NULL )
      {
      ifunc = pchar - symbol_string2;
      for ( i=1; i<=npts; i++ )
         if ( xmin <= the_plot->samples[i-1].x && the_plot->samples[i-1].x <= xmax &&
              ymin <= the_plot->samples[i-1].y && the_plot->samples[i-1].y <= ymax &&
              zmin <= the_plot->samples[i-1].z && the_plot->samples[i-1].z <= zmax )
            {
            r[0] = (the_plot->samples[i-1].x - xmin)*xscale;
            r[1] = (the_plot->samples[i-1].y - ymin)*yscale;
            r[2] = (the_plot->samples[i-1].z - zmin)*zscale;

            p = multiply_mv(Ryz, r);
            for ( j=1; j<=3; j++, p++ )
               r[j-1] = *p;

            x = origin[1] + r[1];
            y = origin[2] - r[2];
            context->symbol_func2[ifunc](context, x, y, the_plot->fill_colors_rgba, the_plot->stylesizes);
            }
      }

   return;
   }


static void
DrawLineSegments3d (struct gse_ctx *context,
		    int iplot, int npts, double *origin, double *Ryz,
		    double xmin, double xmax, double ymin, double ymax, double zmin, double zmax,
		    double xscale, double yscale, double zscale, int linechar )
   {
   /* Declare variables */
   int idraw, iseg, npts_seg, iseg1;
   char linetype[7];


   /* Create linetype string */
   memset(linetype, 0, sizeof(linetype));
   if ( linechar == 'l' )
      strcpy(linetype, "solid");
   else if ( linechar == 'd' )
      strcpy(linetype, "dashed");
   else if ( linechar == '.' )
      strcpy(linetype, "dotted");


   /* Draw all line segments except last */
   idraw = 0;
   iseg1 = 0;
   const struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];
   int nseg = the_plot->nlinebreaks;
   for (iseg=1; iseg<=nseg; iseg++ )
     {
       int nlb = the_plot->nlinebreak[iseg-1];
       idraw++;
       npts_seg = nlb - iseg1;
       DrawLines3d(context, npts_seg,
		   &the_plot->samples[iseg1],
		   origin, Ryz,
		   xmin, xmax, ymin, ymax, zmin, zmax, xscale, yscale, zscale,
		   the_plot->fill_colors_rgba, the_plot->stylesizes, linetype);
       iseg1 = nlb;
     }


   /* Draw last line segment */
   if ( idraw > 0 )
      {
	npts_seg = npts - iseg1;
	DrawLines3d(context, npts_seg,
		    &the_plot->samples[iseg1],
		    origin, Ryz,
		    xmin, xmax, ymin, ymax, zmin, zmax, xscale, yscale, zscale,
		    the_plot->fill_colors_rgba, the_plot->stylesizes, linetype);
      }

   /* Draw continuous line */
   else
     DrawLines3d(context, npts,
		 &the_plot->samples[0],
		 origin, Ryz,
                  xmin, xmax, ymin, ymax, zmin, zmax, xscale, yscale, zscale,
                  the_plot->fill_colors_rgba, the_plot->stylesizes, linetype);
   }
