/*******************************************************************************
*
* PlotEllipses.c
*
* Plots ellipses.
*
* Copyright © 2008, 2009, 2010, 2011, 2012 Spencer A. Buckner
* http://savannah.gnu.org/projects/gsegrafix
*
* This file is part of GSEGrafix, a scientific and engineering plotting program.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*******************************************************************************/

#include "PlotEllipses.h"

#include "DrawLines2d.h"

#include <math.h>
#include "gsegraf.h"

#include <string.h>


void PlotEllipses (struct gse_ctx *context)
   {
   /* Declare variables */
   int i, nx, ny, npts;
   unsigned int index, line_width;
   guint32 line_color;
   double x1_box, x2_box, y1_box, y2_box,
          xmin, xmax, ymin, ymax, xscale, yscale,
          x0, y0, width, height, angle, a, b, t,
     R[4];

   struct datum data[361];
   
   char *pchar, line_char, color_char;
   const char *error_str[] =
      { "Invalid or missing ellipse coordinates.",
        "Invalid ellipse color character.",
        "Invalid or missing ellipse line or color specification.",
        "Invalid ellipse line character." };
   FILE *fptr;


   /* Return if no ellipses to plot */
   fptr = fopen(context->p_param_file, "r");
   i = 0;
   while ( fgets(context->line, context->maxline, fptr) != NULL )
      if ( strncmp(context->line, "ellipse_coords", 14) == 0 )
         i++;
   fclose(fptr);
   if ( i == 0 )
      return;


   /* Get plot box minimum and maximum values */
   if ( strcmp(context->plot_param.axis_type, "linear") == 0 )
      {
      x1_box = context->plot_box_data.xmin;
      x2_box = context->plot_box_data.xmax;
      y1_box = context->plot_box_data.ymin;
      y2_box = context->plot_box_data.ymax;
      }


   /* Get minimum and maximum axis values */
   if ( strcmp(context->plot_param.axis_type, "linear") == 0 )
      {
      nx = context->xtick_labels.nvalues;
      xmin = context->xtick_labels.values[0];
      xmax = context->xtick_labels.values[nx-1];
      ny = context->ytick_labels.nvalues;
      ymin = context->ytick_labels.values[0];
      ymax = context->ytick_labels.values[ny-1];
      xmin = xmin - context->xtick_labels.offset1;
      xmax = xmax + context->xtick_labels.offset2;
      ymin = ymin - context->ytick_labels.offset1;
      ymax = ymax + context->ytick_labels.offset2;
      }


   /* Calculate axis scale factors */
   if ( strcmp(context->plot_param.axis_type, "linear") == 0 )
      {
      xscale = (x2_box - x1_box)/(xmax - xmin);
      yscale = (y2_box - y1_box)/(ymax - ymin);
      }


   /* Draw ellipses */
   fptr = fopen(context->p_param_file, "r");
   while ( fgets(context->line, context->maxline, fptr) != NULL )
      {
      /* Get ellipse shape parameters */
      if ( strncmp(context->line, "ellipse_coords", 14) == 0 )
         {
         if ( sscanf(&context->line[14], "%lf %lf %lf %lf %lf",
                     &x0, &y0, &width, &height, &angle) != 5 )
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[0]);
            longjmp (context->finish, 1);
            }

         line_width = 1;
         if ( fgets(context->line, context->maxline, fptr) != NULL )
            if ( strncmp(context->line, "ellipse_style", 13) == 0 )
               {
               if ( sscanf(&context->line[13], " %c 0x%x", &line_char, (unsigned int *) &line_color) == 2 ||
                    sscanf(&context->line[13], " %c 0X%x", &line_char, (unsigned int *) &line_color) == 2 )
                  sscanf(&context->line[13], " %*c %*s %u", &line_width);

               else if ( sscanf(&context->line[13], " %c %c", &line_char, &color_char) == 2 )
                  {
                  if ( (pchar = strchr(color_string, color_char)) == NULL )
                     {
                     g_set_error_literal (&context->err, context->domain, 0, error_str[1]);
                     longjmp (context->finish, 1);
                     }

                  else
                     {
                     index = pchar - &color_string[0];   /* get index to color character */
                     line_color = context->color_rgba[index];
                     sscanf(&context->line[13], " %*c %*c %u", &line_width);
                     }
                  }

               else
                  {
                  g_set_error_literal (&context->err, context->domain, 0, error_str[2]);
                  longjmp (context->finish, 1);
                  }

               /* Check line character */
               if ( line_char != 'l' && line_char != 'd' && line_char != '.' )
                  {
                  g_set_error_literal (&context->err, context->domain, 0, error_str[3]);
                  longjmp (context->finish, 1);
                  }
               }

         /* Calculate ellipse centered at offset point;   */
         /* equations of ellipse:                         */
         /*    x^2/a^2 + y^2/b^2 = 1                      */
         /*    x = a*cos(t), y = b*sin(t), 0 <= t <= 2*pi */
         a = width/2.0;    /* semi-major axis */
         b = height/2.0;   /* semi-minor axis */

         npts = 361;
         if ( angle == 0.0 )
            {
            /* Calculate unrotated ellipse */
            for ( i=1; i<=npts; i++ )
               {
               t = 2.0*M_PI*(i - 1)/(npts - 1);
               data[i-1].x = x0 + a*cos(t);
               data[i-1].y = y0 - b*sin(t);
               }
            }
         else
            {
            /* Calculate rotation-matrix elements; */
            /* R elements labeled: [ 0 1; 2 3 ]    */
            angle = angle*deg2rad;
            R[0] =  cos(angle);
            R[1] =  sin(angle);
            R[2] = -sin(angle);
            R[3] =  cos(angle);

            /* Calculate rotated ellipse */
            for ( i=1; i<=npts; i++ )
               {
               t = 2.0*M_PI*(i - 1)/(npts - 1);
               data[i-1].x = x0 + (R[0]*a*cos(t) + R[1]*b*sin(t));
               data[i-1].y = y0 - (R[2]*a*cos(t) + R[3]*b*sin(t));
               }
            }

         /* Draw ellipse */
         if ( line_char == 'l' )
	   DrawLines2d(context, npts,
		       //       &x[0],  &y[0],
		       data,
		       xmin, xmax, ymin, ymax, xscale, yscale,
                        line_color, line_width, "solid");
         else if ( line_char == 'd' )
	   DrawLines2d(context, npts,
		       //&x[0],  &y[0],
		       data,
		       xmin, xmax, ymin, ymax, xscale, yscale,
                        line_color, line_width, "dashed");
         else if ( line_char == '.' )
	   DrawLines2d(context, npts,
		       //&x[0],  &y[0],
		       data,
		       xmin, xmax, ymin, ymax, xscale, yscale,
		       line_color, line_width, "dotted");
         }

      else if ( strncmp(context->line, "#####", 5) == 0 )
         break;
      }

   fclose(fptr);

   return;
   }

