/*******************************************************************************
*
* PlotData2d.c
*
* Plots data points.
*
* Copyright  2008, 2009, 2010, 2011, 2012 Spencer A. Buckner
* http://savannah.gnu.org/projects/gsegrafix
*
* This file is part of GSEGrafix, a scientific and engineering plotting program.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*******************************************************************************/

#include "PlotData2d.h"
#include "Misc.h"
#include "DrawSymbols.h"
#include "Histogram.h"
#include "ContourPlot2d.h"
#include "ColorPlot2d.h"
#include "DrawLines2d.h"

#include <math.h>
#include "gsegraf.h"

#include <string.h>

static void     DrawLineSegments (struct gse_ctx *, int iplot, int index, int npts,
                            double xmin, double xmax, double ymin, double ymax,
                            double xscale, double yscale, int linechar );


void PlotData2d (struct gse_ctx *context)
   {
   /* Declare variables */
   int i, ifunc, nx, ny, iplot, nplots, npts,
       index,
       ipoints, ihist, icontour, icolor,
       nxcontourplot, nycontourplot, xindex_contour, yindex_contour, zindex_contour,
       nxcolorplot, nycolorplot, xindex_color, yindex_color, zindex_color;
   double x1_box, x2_box, y1_box, y2_box, xmin, xmax, ymin, ymax,
          xscale, yscale, x, y;
   char *pchar;


   
   /* Get plot box minimum and maximum values */
   x1_box = context->plot_box_data.xmin;
   x2_box = context->plot_box_data.xmax;
   y1_box = context->plot_box_data.ymin;
   y2_box = context->plot_box_data.ymax;


   /* Get minimum and maximum axis values */
   if ( strcmp(context->plot_param.axis_type, "linear") == 0 )
      {
      nx = context->xtick_labels.nvalues;
      xmin = context->xtick_labels.values[0];
      xmax = context->xtick_labels.values[nx-1];
      ny = context->ytick_labels.nvalues;
      ymin = context->ytick_labels.values[0];
      ymax = context->ytick_labels.values[ny-1];
      xmin = xmin - context->xtick_labels.offset1;
      xmax = xmax + context->xtick_labels.offset2;
      ymin = ymin - context->ytick_labels.offset1;
      ymax = ymax + context->ytick_labels.offset2;
      }

   else if ( strcmp(context->plot_param.axis_type, "semilogx") == 0 )
      {
      nx = context->xtick_labels.nvalues;
      xmin = floor(context->xtick_labels.values[0]);
      xmax = ceil(context->xtick_labels.values[nx-1]);
      nx = roundint(xmax - xmin + 1.0);
      ny = context->ytick_labels.nvalues;
      ymin = context->ytick_labels.values[0];
      ymax = context->ytick_labels.values[ny-1];
      ymin = ymin - context->ytick_labels.offset1;
      ymax = ymax + context->ytick_labels.offset2;
      }

   else if ( strcmp(context->plot_param.axis_type, "semilogy") == 0 )
      {
      nx = context->xtick_labels.nvalues;
      xmin = context->xtick_labels.values[0];
      xmax = context->xtick_labels.values[nx-1];
      ny = context->ytick_labels.nvalues;
      ymin = floor(context->ytick_labels.values[0]);
      ymax = ceil(context->ytick_labels.values[ny-1]);
      ny = roundint(ymax - ymin + 1.0);
      xmin = xmin - context->xtick_labels.offset1;
      xmax = xmax + context->xtick_labels.offset2;
      }

   else if ( strcmp(context->plot_param.axis_type, "loglog") == 0 )
      {
      nx = context->xtick_labels.nvalues;
      xmin = floor(context->xtick_labels.values[0]);
      xmax = ceil(context->xtick_labels.values[nx-1]);
      nx = roundint(xmax - xmin + 1.0);
      ny = context->ytick_labels.nvalues;
      ymin = floor(context->ytick_labels.values[0]);
      ymax = ceil(context->ytick_labels.values[ny-1]);
      ny = roundint(ymax - ymin + 1.0);
      }


   /* Calculate axis scale factors */
   xscale = (x2_box - x1_box)/(xmax - xmin);
   yscale = (y2_box - y1_box)/(ymax - ymin);


   /* Plot data */
   nplots = context->plot_param.nplots;
   index = 0;
   ipoints = 0;
   ihist = 0;
   icontour = 0;
   icolor = 0;
   xindex_contour = 0;
   yindex_contour = 0;
   zindex_contour = 0;
   xindex_color = 0;
   yindex_color = 0;
   zindex_color = 0;
   for ( iplot=1; iplot<=nplots; iplot++ )
      {
   const struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];

	
      if ( strncmp (the_plot->plot_types, "points", 10) == 0 )
         {
         ipoints++;
         npts = the_plot->ndata;

         /* Draw stem lines */
         if ( strncmp (the_plot->stemflags,  "on", 9) == 0 ||
              strncmp (the_plot->stemflags, "num", 9) == 0 )
            {
	      GnomeCanvasPoints *points;
            points = gnome_canvas_points_new(2);

            /* Calculate y coordinate of stem point 1 */
            if ( strncmp (the_plot->stemflags, "on", 9) == 0 )
               y = y2_box;
            else if ( strncmp (the_plot->stemflags, "num", 9) == 0 )
               {
               if ( ymin <= the_plot->stemvalues && the_plot->stemvalues <= ymax )
                  y = y2_box - (the_plot->stemvalues - ymin)*yscale;
               else if ( the_plot->stemvalues < ymin )
                  y = y2_box;
               else if ( the_plot->stemvalues > ymax )
                  y = y1_box;
               }
            points->coords[1] = y;

            for ( i=1; i<=npts; i++ )
               if ( xmin <= the_plot->samples[i-1].x && the_plot->samples[i-1].x <= xmax &&
                    ymin <= the_plot->samples[i-1].y && the_plot->samples[i-1].y <= ymax )
                  {
                  /* Calculate x coordinate of stem point 1 */
                  x = x1_box + (the_plot->samples[i-1].x - xmin)*xscale;
                  points->coords[0] = x;

                  /* Calculate x and y coordinates of stem point 2 */
                  x = x1_box + (the_plot->samples[i-1].x - xmin)*xscale;
                  y = y2_box - (the_plot->samples[i-1].y - ymin)*yscale;
                  points->coords[2] = x;
                  points->coords[3] = y;

                  DrawLine(context, points, 2, the_plot->outline_colors_rgba, 1);
                  }

            gnome_canvas_points_unref(points);
            }


         /* Draw lines */
         if ( the_plot->stylechar1 == 'l' )
	   DrawLineSegments(context, iplot, index, npts, xmin, xmax, ymin, ymax,
                             xscale, yscale, 'l');


         /* Draw dashed lines */
         else if ( the_plot->stylechar1 == 'd' )
	   DrawLineSegments(context, iplot, index, npts, xmin, xmax, ymin, ymax,
                             xscale, yscale, 'd');


         /* Draw dotted lines */
         else if ( the_plot->stylechar1 == '.' )
	   DrawLineSegments(context, iplot, index, npts, xmin, xmax, ymin, ymax,
                             xscale, yscale, '.');


         /* Draw symbols in symbol_string1 ("cCtTsSiIpPhH") */
         else if ( (pchar = strchr(symbol_string1, the_plot->stylechar1)) != NULL )
            {
            ifunc = pchar - symbol_string1;
            for ( i=1; i<=npts; i++ )
               if ( xmin <= the_plot->samples[i-1].x && the_plot->samples[i-1].x <= xmax &&
                    ymin <= the_plot->samples[i-1].y && the_plot->samples[i-1].y <= ymax )
                  {
                  x = x1_box + (the_plot->samples[i-1].x - xmin)*xscale;
                  y = y2_box - (the_plot->samples[i-1].y - ymin)*yscale;
                  context->symbol_func1[ifunc](context, x, y, the_plot->fill_colors_rgba, the_plot->outline_colors_rgba,
                                      the_plot->stylesizes);
                  }
            }


         /* Draw symbols in symbol_string2 ("+xra") */
         else if ( (pchar = strchr(symbol_string2, the_plot->stylechar1)) != NULL )
            {
            ifunc = pchar - symbol_string2;
            for ( i=1; i<=npts; i++ )
               if ( xmin <= the_plot->samples[i-1].x && the_plot->samples[i-1].x <= xmax &&
                    ymin <= the_plot->samples[i-1].y && the_plot->samples[i-1].y <= ymax )
                  {
                  x = x1_box + (the_plot->samples[i-1].x - xmin)*xscale;
                  y = y2_box - (the_plot->samples[i-1].y - ymin)*yscale;
                  context->symbol_func2[ifunc](context, x, y, the_plot->fill_colors_rgba, the_plot->stylesizes);
                  }
            }

         index = index + context->plot_parameters[ipoints+ihist-1].ndata;
         }


      else if ( strncmp (the_plot->plot_types, "histogram", 10) == 0 )
         {
         ihist++;
         PlotHistogram (context, iplot);
         index = index + context->plot_parameters[ipoints+ihist-1].ndata;
         }


      else if ( strncmp (the_plot->plot_types, "contour", 10) == 0 )
         {
         icontour++;
	 nxcontourplot = the_plot->samples3d.nx;
         nycontourplot = the_plot->samples3d.ny;
	 
         ContourPlot2d(context, iplot, icontour, xindex_contour, yindex_contour, zindex_contour, nxcontourplot, nycontourplot);
         xindex_contour = xindex_contour + nxcontourplot;
         yindex_contour = yindex_contour + nycontourplot;
         zindex_contour = zindex_contour + nxcontourplot*nycontourplot;
         }


      else if ( strncmp (the_plot->plot_types, "color", 10)  == 0 )
         {
         icolor++;
	 nxcolorplot = the_plot->samples3d.nx;
         nycolorplot = the_plot->samples3d.ny;
         ColorPlot2d(context, iplot, icolor, xindex_color, yindex_color, zindex_color, nxcolorplot, nycolorplot);
         xindex_color = xindex_color + nxcolorplot;
         yindex_color = yindex_color + nycolorplot;
         zindex_color = zindex_color + nxcolorplot*nycolorplot;
         }
      }

   return;
   }


static void DrawLineSegments (struct gse_ctx *context, int iplot, int index, int npts,
                        double xmin, double xmax, double ymin, double ymax,
                        double xscale, double yscale, int linechar )
   {
   /* Declare variables */
   int idraw, iseg, nseg, npts_seg, iseg1;
   char linetype[7];

   const struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];


   /* Create linetype string */
   memset(linetype, 0, sizeof(linetype));
   if ( linechar == 'l' )
      strcpy(linetype, "solid");
   else if ( linechar == 'd' )
      strcpy(linetype, "dashed");
   else if ( linechar == '.' )
      strcpy(linetype, "dotted");


   /* Draw all line segments except last */
   idraw = 0;
   nseg = the_plot->nlinebreaks;
   iseg1 = index;
   for (iseg=1; iseg<=nseg; iseg++ )
      {
      if ( index < the_plot->nlinebreak[iseg-1] &&
	   the_plot->nlinebreak[iseg-1] < index + npts )
         {
         idraw++;
         npts_seg = the_plot->nlinebreak[iseg-1] - iseg1;
         DrawLines2d(context, npts_seg,
		     &the_plot->samples[iseg1 - index],
		     xmin, xmax, ymin, ymax,
                     xscale, yscale, the_plot->fill_colors_rgba, the_plot->stylesizes,
                     linetype);
         iseg1 = the_plot->nlinebreak[iseg-1];
         }
      }


   /* Draw last line segment */
   if ( idraw > 0 )
      {
      npts_seg = index + npts - iseg1;
      DrawLines2d(context, npts_seg,
		  &the_plot->samples[iseg1 - index],
		  xmin, xmax, ymin, ymax,
                  xscale, yscale, the_plot->fill_colors_rgba, the_plot->stylesizes,
                  linetype);
      }

   /* Draw continuous line */
   else
     DrawLines2d(context, npts,
		 the_plot->samples,
		 xmin, xmax, ymin, ymax,
		 xscale, yscale, the_plot->fill_colors_rgba, the_plot->stylesizes,
		 linetype);

   return;
   }
