/*******************************************************************************
*
* InitializePlot.c
*
* Contains functions:
*    InitializePlot
*    PlotBox
*
* Initializes plot variables, reads plot-parameter file, checks plot-parameter
* values, calculates various plot variables, and analyzes various plot
* variables.
*
* Copyright  2008, 2009, 2010, 2011, 2012 Spencer A. Buckner
* http://savannah.gnu.org/projects/gsegrafix
*
* This file is part of GSEGrafix, a scientific and engineering plotting program.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*******************************************************************************/

#include "InitializePlot.h"

#include "ReadParamFile.h"
#include "CheckParamData.h"
#include "FileRead.h"
#include "FileRead3d.h"
#include "DataMinMax.h"
#include "DataMinMax3d.h"
#include "AxisLimits.h"
#include "AxesEqual.h"
#include "Misc.h"


#include <math.h>
#include "gsegraf.h"

#include <string.h>
#include <stdlib.h>

static void
destroy_parser (struct data_parser *p)
{
  free (p->formats);
  free (p->formats_mod);
  free (p->nformats);
  free (p->filenames);
  free (p->nfilenames);
  free (p->save_filename);
}

void InitializePlot (struct gse_ctx *context)
   {
   /* Declare variables */
   double window_width, window_height, max_dimension;
  /* Declare axis_type flags */
  int flag_linear, flag_logx, flag_logy, flag_loglog, flag_polar, flag_3d,
    flag_2d, flag_2d_rect;


  struct data_parser data_parser = {0};

   /* Read plot-parameter file */
   ReadParamFile(context, &data_parser);


   /* Check plot-parameter values */
   CheckParamData(context, &data_parser);


   /* Set axis_type flags */
   flag_linear = 0;
   flag_logx = 0;
   flag_logy = 0;
   flag_loglog = 0;
   flag_polar = 0;
   flag_3d = 0;
   flag_2d = 0;
   flag_2d_rect = 0;
   if ( strcmp(context->plot_param.axis_type, "linear") == 0 )
      flag_linear = 1;
   else if ( strcmp(context->plot_param.axis_type, "semilogx") == 0 )
      flag_logx = 1;
   else if ( strcmp(context->plot_param.axis_type, "semilogy") == 0 )
      flag_logy = 1;
   else if ( strcmp(context->plot_param.axis_type, "loglog") == 0 )
      flag_loglog = 1;
   else if ( strcmp(context->plot_param.axis_type, "polar") == 0 )
      flag_polar = 1;
   else if ( strcmp(context->plot_param.axis_type, "3d") == 0 )
      flag_3d = 1;

   if ( flag_linear == 1 ||
        flag_logx   == 1 ||
        flag_logy   == 1 ||
        flag_loglog == 1 ||
        flag_polar  == 1 )
      flag_2d = 1;

   if ( flag_linear == 1 ||
        flag_logx   == 1 ||
        flag_logy   == 1 ||
        flag_loglog == 1 )
      flag_2d_rect = 1;


   /* Calculate tick-mark and dashed-line variables */
   window_width  = context->window_width;
   window_height = context->window_height;
   if ( window_width >= window_height )
      max_dimension = window_width;
   else
      max_dimension = window_height;
   context->tick_major = (8.0/768.0)*max_dimension;
   context->tick_minor = 0.5*context->tick_major;
   context->dash = context->tick_major;
   context->space_dash = 0.5*context->dash;
   context->space_dot = context->space_dash - 1.0;


   /* Analyze plot data */
   if ( flag_2d_rect == 1 )
      {
      /* Read plot data files */
	FileRead(context, &data_parser);

      /* Find minimum and maximum values of plot data */
      DataMinMax(context);

      /* Calculate axes */
      AxisLimits(context, 1);
      }

   else if ( flag_polar == 1 )
      {
      /* Read plot data files */
	FileRead(context, &data_parser);

      /* Find minimum and maximum values of plot data */
      DataMinMax(context);

      /* Adjust axes */
      AxisLimits(context, 1);
      }

   else if ( flag_3d == 1 )
      {
      /* Read plot data file */
	FileRead3d(context, &data_parser);

      /* Find minimum and maximum values of plot data */
      DataMinMax3d(context);

      /* Adjust axes */
      AxisLimits(context, 1);
      }

   /* Specify plot-box coordinates */
   PlotBox(context);
   
   /* Adjust axes */
   if ( strcmp(context->plot_param.axis_type, "linear") == 0 &&
	strcmp(context->plot_param.axis_scale, "equal") == 0 )
     AxesEqual(context, 1);

   destroy_parser (&data_parser);
   }


void
PlotBox (struct gse_ctx *context)
   {
   /* Declare variables */
   int iplot, nplots, flag;


   /* Specify plot box screen coordinates */
   nplots = context->plot_param.nplots;
   flag = 0;
   if ( strcmp(context->plot_param.axis_type, "linear") == 0 )
      for ( iplot=1; iplot<=nplots; iplot++ )
	{
	   const struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];

         if ( strncmp (the_plot->plot_types, "color", 10) == 0 ||
              (strncmp (the_plot->plot_types, "contour", 10) == 0 &&
               the_plot->styleflags == 7) )
            flag = 1;
	}


   /* Plot types: points, contour, and histogram */
   if ( flag == 0 )
      {
      context->plot_box_data.xmin = 0.15625*context->window_width;
      context->plot_box_data.xmax = 0.90625*context->window_width;
      context->plot_box_data.ymin = 0.09375*context->window_height;
      context->plot_box_data.ymax = 0.84375*context->window_height;
      }

   /* Plot types: points, contour, histogram, and color */
   else if ( flag == 1 )
      {
      context->plot_box_data.xmin = 0.18750*context->window_width;
      context->plot_box_data.xmax = 0.75000*context->window_width;
      context->plot_box_data.ymin = 0.09375*context->window_height;
      context->plot_box_data.ymax = 0.84375*context->window_height;
      }
   }
