/*******************************************************************************
 *
 * Histogram.c
 *
 * Calculates histogram of input data.
 *
 * Copyright  2008, 2009, 2010, 2011, 2012 Spencer A. Buckner
 * http://savannah.gnu.org/projects/gsegrafix
 *
 * This file is part of GSEGrafix, a scientific and engineering plotting program.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *******************************************************************************/

#include "Histogram.h"
#include "Misc.h"
#include "DrawSymbols.h"

#include <math.h>
#include "gsegraf.h"

#include <string.h>
#include <stdlib.h>

void
HistogramCalculate (struct gse_ctx *context, int iplot)
{
  struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];

  /* Calculate data mean and standard deviation */
  double sum = 0.0;
  int i;

  for ( i=1; i<=the_plot->ndata; i++ )
    sum += the_plot->samples[i-1].x;
  double mean = sum/the_plot->ndata;

  double sumsq = 0.0;
  for ( i=1; i<=the_plot->ndata; i++ )
    sumsq +=  (the_plot->samples[i-1].x - mean)*(the_plot->samples[i-1].x - mean);

  /* Calculate data quartile values */
  double data_min = the_plot->samples[0].x;
  double data_max = the_plot->samples[the_plot->ndata-1].x;
  double q[5];
  q[0] = data_min;
  q[4] = data_max;
  for ( i=1; i<=3; i++ )
    {
      int j = roundint(i*0.25*the_plot->ndata);
      q[i] = the_plot->samples[j-1].x;
    }


  /* Estimate optimal histogram bin width */
  if ( the_plot->bin_widths <= 0.0 )
    {
      /* binwidth = 3.49*std/pow((double) the_plot->ndata, 1.0/3.0); */   /* Scott */
      the_plot->bin_widths = 2.0*(q[3] - q[1])/pow((double) the_plot->ndata, 1.0/3.0);   /* Freedman, Diaconis */
    }


  /* Calculate number of bins, context->binmin, and context->binmax */
  if ( strncmp (the_plot->bin_refs, "mean", 9) == 0 )
    {
      double n1 = ceil(((mean - the_plot->bin_widths/2.0) - data_min)/the_plot->bin_widths);
      double n2 = ceil((data_max - (mean + the_plot->bin_widths/2.0))/the_plot->bin_widths);
      the_plot->nbins = n1 + n2 + 1;
      the_plot->binmin = (mean - the_plot->bin_widths/2.0) - n1*the_plot->bin_widths;
      the_plot->binmax = (mean + the_plot->bin_widths/2.0) + n2*the_plot->bin_widths;
    }
  else if ( strncmp (the_plot->bin_refs, "zero", 9) == 0 )
    {
      if ( data_max > 0 )
	{
	  the_plot->nbins = ceil(data_max/the_plot->bin_widths);
	  the_plot->binmin = 0.0;
	  the_plot->binmax = the_plot->nbins*the_plot->bin_widths;
	}
      else
	return;
    }
  else if ( strncmp (the_plot->bin_refs, "integers", 9) == 0 )
    {
      the_plot->bin_widths = 1.0;
      the_plot->nbins = roundint(data_max) - roundint(data_min) + 1.0;
      the_plot->binmin = (double) roundint(data_min) - 0.5;
      the_plot->binmax = (double) roundint(data_max) + 0.5;
    }


  /* Calculate number of samples in each bin */
  the_plot->yhist = xmalloc(the_plot->nbins*sizeof(double));
  for ( i=1; i<=the_plot->nbins; i++ )
    the_plot->yhist[i-1] = 0.0;
  int j = 1;
  double bin2;
  for ( i=1; i<=the_plot->nbins; i++ )
    {
      double bin1 = the_plot->binmin + (i - 1.0)/(the_plot->nbins - 1.0)*(the_plot->binmax - the_plot->binmin - the_plot->bin_widths);
      bin2 = bin1 + the_plot->bin_widths;
      while ( (j <= the_plot->ndata) && (bin1 <= the_plot->samples[j-1].x) && (the_plot->samples[j-1].x < bin2) )
	{
	  the_plot->yhist[i-1] = the_plot->yhist[i-1] + 1.0;
	  j++;
	}
    }
  if ( the_plot->samples[the_plot->ndata-1].x == bin2 )
    the_plot->yhist[the_plot->nbins-1] = the_plot->yhist[the_plot->nbins-1] + 1.0;

  the_plot->histmin = 0.0;

  if ( strncmp(the_plot->bin_values, "number", 9) == 0 )
    the_plot->histmax = max(the_plot->nbins, the_plot->yhist);
  else if ( strncmp(the_plot->bin_values, "fraction", 9) == 0 )
    the_plot->histmax = max(the_plot->nbins, the_plot->yhist)/the_plot->ndata;
  else if ( strncmp(the_plot->bin_values, "percent", 9) == 0 )
    the_plot->histmax = 100.0*max(the_plot->nbins, the_plot->yhist)/the_plot->ndata;
}

void
PlotHistogram (struct gse_ctx *context, int iplot)
{
  GnomeCanvasPoints *points;

  struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];

  /* Specify axis minimum and maximum values */
  int nx = context->xtick_labels.nvalues;
  int ny = context->ytick_labels.nvalues;
  double xmin = context->xtick_labels.values[0];
  double xmax = context->xtick_labels.values[nx-1];
  double ymin = context->ytick_labels.values[0];
  double ymax = context->ytick_labels.values[ny-1];
  xmin -= context->xtick_labels.offset1;
  xmax += context->xtick_labels.offset2;
  ymin -= context->ytick_labels.offset1;
  ymax += context->ytick_labels.offset2;


  /* Get plot box minimum and maximum values */
  double x1_box = context->plot_box_data.xmin;
  double x2_box = context->plot_box_data.xmax;
  double y1_box = context->plot_box_data.ymin;
  double y2_box = context->plot_box_data.ymax;


  /* Calculate axis scale factors */
  double xscale = (x2_box - x1_box)/(xmax - xmin);
  double yscale = (y2_box - y1_box)/(ymax - ymin);

  int i;
  /* Plot histogram */
  points = gnome_canvas_points_new(2);
  for ( i=1; i<=the_plot->nbins; i++ )
    {
      double y2_bar;
      double y1_bar = 0.0;
      if ( strncmp(the_plot->bin_values, "number", 9) == 0 )
	y2_bar = the_plot->yhist[i-1];
      else if ( strncmp(the_plot->bin_values, "fraction", 9) == 0 )
	y2_bar = the_plot->yhist[i-1]/the_plot->ndata;
      else if ( strncmp(the_plot->bin_values, "percent", 9) == 0 )
	y2_bar = 100.0*the_plot->yhist[i-1]/the_plot->ndata;

      if ( y1_bar >= ymax || y2_bar <= ymin )
	continue;
      minimize (&ymin, y1_bar);
      maximize (&ymax, y2_bar);

      double y1 = y2_box - (y1_bar - ymin)*yscale;
      double y2 = y2_box - (y2_bar - ymin)*yscale;

      if ( the_plot->stylechar1 == 'b' || the_plot->stylechar1 == 'B' )
	{
	  /* Draw bars */
	  double x1_bar = the_plot->binmin + (i - 1.0)/(the_plot->nbins - 1.0)*(the_plot->binmax - the_plot->bin_widths - the_plot->binmin);
	  double x2_bar = x1_bar + the_plot->bin_widths;

	  if ( x1_bar >= xmax || x2_bar <= xmin )
            continue;
	  minimize (&xmin, x1_bar);
	  maximize (&xmax, x2_bar);

	  double x1 = x1_box + (x1_bar - xmin)*xscale;
	  double x2 = x1_box + (x2_bar - xmin)*xscale;

	  DrawBar(context, x1+1.0, y1, x2-1.0, y2, the_plot->fill_colors_rgba, the_plot->outline_colors_rgba);
	  DrawBar(context, x1, y1+1.0, x2, y2-1.0, 0xFFFFFF00, context->canvas_bg_color);
	}
      else if ( the_plot->stylechar1 == 'l' )
	{
	  /* Draw lines */
	  double xline = the_plot->binmin + (i - 1.0)/(the_plot->nbins - 1.0)*(the_plot->binmax - the_plot->bin_widths - the_plot->binmin) + the_plot->bin_widths/2.0;
	  if ( xmin < xline && xline < xmax )
            {
	      points->coords[0] = x1_box + (xline - xmin)*xscale;
	      points->coords[1] = y1;
	      points->coords[2] = points->coords[0];
	      points->coords[3] = y2;
	      DrawLine(context, points, 2,
		       the_plot->fill_colors_rgba, the_plot->stylesizes);
            }
	}
    }

  gnome_canvas_points_unref(points);
}
