/*******************************************************************************
*
* FileRead3d.c
*
* Reads 3d data files.
*
* Copyright  2008, 2009, 2010, 2011, 2012 Spencer A. Buckner
* http://savannah.gnu.org/projects/gsegrafix
*
* This file is part of GSEGrafix, a scientific and engineering plotting program.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*******************************************************************************/

#include "FileRead3d.h"
#include "InitializePlot.h"
#include "Misc.h"
#include "gsegraf.h"

#include <string.h>
#include <stdlib.h>

void FileRead3d (struct gse_ctx *context, struct data_parser *parser)
   {
   /* Declare variables */
   int i, j, iplot, ipoints, imesh, icontour, icolor, nplots, npts, npts_total,
       npts_total_xmesh, npts_total_ymesh, npts_total_zmesh,
       npts_total_xcontour, npts_total_ycontour, npts_total_zcontour,
       npts_total_xcolor, npts_total_ycolor, npts_total_zcolor,
       index_filenames, index_formats,
       index, nx, ny, xindex_mesh, yindex_mesh, zindex_mesh,
       xindex_contour, yindex_contour, zindex_contour,
       xindex_color, yindex_color, zindex_color,
     ndouble, nchar;
   unsigned int size;
   double x, y, z;
   char *string = NULL, ast1, ast2, ast3;
   const char *error_str[] =
      { "Error reading mesh data file.",
        "Error reading contour data file.",
        "Not enough memory for data.",
        "Not enough memory for line-break data." };
   FILE *fptr;


   /* Get sizes of data files */
   nplots = context->plot_param.nplots;
   npts_total = 0;
   npts_total_xmesh = 0;
   npts_total_ymesh = 0;
   npts_total_zmesh = 0;
   npts_total_xcontour = 0;
   npts_total_ycontour = 0;
   npts_total_zcontour = 0;
   npts_total_xcolor = 0;
   npts_total_ycolor = 0;
   npts_total_zcolor = 0;
   index_filenames = 0;
   index_formats = 0;
   ipoints = 0;
   imesh = 0;
   icontour = 0;
   icolor = 0;
   for ( iplot=1; iplot<=nplots; iplot++ )
      {
	struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];

	/* Open data file */
      if ( (fptr = fopen( &parser->filenames[index_filenames], "r")) == NULL )
         {
         size = strlen("Cannot open data file:\n") + strlen(&parser->filenames[index_filenames]);
         string = xmalloc(size + 1);
         sprintf(string, "%s%s", "Cannot open data file:\n", &parser->filenames[index_filenames]);
         g_set_error_literal (&context->err, context->domain, 0, string);
         free(string);
         longjmp (context->finish, 1);
         }


      /* Get size of data file */
      if ( strncmp (the_plot->plot_types, "points", 10) == 0 )
         {
         ipoints++;
         i = 0;
         ndouble = 0;
         nchar = 0;
         while ( (ndouble = fscanf(fptr, &parser->formats[index_formats], &x, &y, &z)) == 3 ||
                 ((nchar = fscanf(fptr, &parser->formats_mod[index_formats], &ast1, &ast2, &ast3)) == 3 &&
                   ast1 == '*' && ast2 == '*' && ast3 == '*') )
            {
            if ( ndouble == 3 )
               {
               i++;
               ndouble = 0;
               }
            else if ( nchar == 3 )
               {
	       the_plot->nlinebreaks++;
               nchar = 0;
               }
            }
         index_formats = index_formats + parser->nformats[ipoints-1];
         npts = i;
	 the_plot->ndata = npts;
         npts_total = npts_total + npts;
         }

      else if ( strncmp (the_plot->plot_types, "mesh", 10) == 0 )
         {
         imesh++;
         if ( fscanf(fptr, "%d %d", &nx, &ny) != 2 )
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[0]);
            longjmp (context->finish, 1);
            }
         npts_total_xmesh = npts_total_xmesh + nx;
         npts_total_ymesh = npts_total_ymesh + ny;
         npts_total_zmesh = npts_total_zmesh + nx*ny;
         }

      else if ( strncmp (the_plot->plot_types, "contour", 10) == 0 )
         {
         icontour++;
         if ( fscanf(fptr, "%d %d", &nx, &ny) != 2 )
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[1]);
            longjmp (context->finish, 1);
            }
         npts_total_xcontour = npts_total_xcontour + nx;
         npts_total_ycontour = npts_total_ycontour + ny;
         npts_total_zcontour = npts_total_zcontour + nx*ny;
         }

      else if ( strncmp (the_plot->plot_types, "color", 10) == 0 )
         {
         icolor++;
         if ( fscanf(fptr, "%d %d", &nx, &ny) != 2 )
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[1]);
            longjmp (context->finish, 1);
            }
         npts_total_xcolor = npts_total_xcolor + nx;
         npts_total_ycolor = npts_total_ycolor + ny;
         npts_total_zcolor = npts_total_zcolor + nx*ny;
         }


      /* Increment indices */
      fclose(fptr);
      index_filenames = index_filenames + parser->nfilenames[iplot-1];
      }


   /* Read data files */
   index_filenames = 0;
   index_formats = 0;
   index = 0;
   xindex_mesh = 0;
   yindex_mesh = 0;
   zindex_mesh = 0;
   xindex_contour = 0;
   yindex_contour = 0;
   zindex_contour = 0;
   xindex_color = 0;
   yindex_color = 0;
   zindex_color = 0;
   ipoints = 0;
   imesh = 0;
   icontour = 0;
   icolor = 0;

   for ( iplot=1; iplot<=nplots; iplot++ )
      {
	int ichar = 0;
	struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];

	
	the_plot->nlinebreak =
	  xmalloc (the_plot->nlinebreaks * sizeof (int));

	the_plot->samples = xmalloc (sizeof (*the_plot->samples) * the_plot->ndata);
	
      /* Open data file */
      fptr = fopen( &parser->filenames[index_filenames], "r");

      /* Read data file */
      if ( strncmp (the_plot->plot_types, "points", 10) == 0 )
         {
         ipoints++;
	 npts = the_plot->ndata;
         for ( i=1; i<=npts; i++ )
	   {
	     int ndouble = fscanf(fptr, &parser->formats[index_formats],
				  &the_plot->samples[i-1].x,
				  &the_plot->samples[i-1].y,
				  &the_plot->samples[i-1].z);

	     if (ndouble != 3)
	       {
		 int nchar = fscanf(fptr, &parser->formats_mod[index_formats],
				    &ast1, &ast2, &ast3);
		 
		 if (nchar == 3)
		   {
		     ichar++;
		     the_plot->nlinebreak[ichar-1] = index + i - 1;
		     i--;
		   }
	       }
	   }
         index_formats = index_formats + parser->nformats[ipoints-1];
         index = index + context->plot_parameters[ipoints-1].ndata;
         }
      
      else if (
	       ( strncmp (the_plot->plot_types, "contour", 10) == 0 )
	       ||
	       ( strncmp (the_plot->plot_types, "color", 10) == 0 )
	       ||
	       ( strncmp (the_plot->plot_types, "mesh", 10) == 0 )
	       )
	{
         icontour++;
	 int nx, ny;
         if ( fscanf(fptr, "%d %d", &nx, &ny) != 2 )
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[1]);
            longjmp (context->finish, 1);
            }
	 
	 the_plot->samples3d.nx = nx;
	 the_plot->samples3d.ny = ny;
	 the_plot->samples3d.x = xmalloc (nx * sizeof (double));
	 the_plot->samples3d.y = xmalloc (ny * sizeof (double));
	 the_plot->samples3d.z = xmalloc (ny * nx * sizeof (double));

         for ( i=1; i<=nx; i++ )
	   {
	     double x;
	     if ( fscanf(fptr, "%lf", &x) != 1 )
               {
		 g_set_error_literal (&context->err, context->domain, 0, error_str[1]);
		 longjmp (context->finish, 1);
               }
	     the_plot->samples3d.x[i-1] = x;
	   }

         for ( j=1; j<=ny; j++ )
	   {
	     double y;
            if ( fscanf(fptr, "%lf", &y) != 1 )
	      {
		g_set_error_literal (&context->err, context->domain, 0, error_str[1]);
		longjmp (context->finish, 1);
	      }
	    the_plot->samples3d.y [j-1] = y;
	   }

         for ( i=1; i<=nx; i++ )
            for ( j=1; j<=ny; j++ )
	      {
		double z;
               if ( fscanf(fptr, "%lf", &z) != 1 )
                  {
		    g_set_error_literal (&context->err, context->domain, 0, error_str[1]);
		    longjmp (context->finish, 1);
                  }
	       the_plot->samples3d.z[ny*(i-1)+j-1] = z;
	      }

         xindex_contour = xindex_contour + nx;
         yindex_contour = yindex_contour + ny;
         zindex_contour = zindex_contour + nx*ny;
	}

      /* Increment indices */
      fclose(fptr);
      index_filenames = index_filenames + parser->nfilenames[iplot-1];
      }

   return;
   }
