/*******************************************************************************
*
* FileRead.c
*
* Contains functions:
*    FileRead
*    compare
*
* Reads 2d data files.
*
* Copyright  2008, 2009, 2010, 2011, 2012 Spencer A. Buckner
* http://savannah.gnu.org/projects/gsegrafix
*
* This file is part of GSEGrafix, a scientific and engineering plotting program.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*******************************************************************************/

#include "FileRead.h"
#include "InitializePlot.h"
#include "Misc.h"
#include <math.h>
#include "gsegraf.h"

#include <string.h> // for strlen
#include <stdlib.h>

static int      compare_datum_by_x ( const void *p1, const void *p2 );


void FileRead (struct gse_ctx *context, struct data_parser *parser)
   {
   /* Declare variables */
   int i, j, iplot, ipoints, ihist, icolor, nplots, npts, npts_total,
       npts_total_xcolor, npts_total_ycolor, npts_total_zcolor,
       index_filenames, index_formats, index,
       nx, ny, 
       xindex_color, yindex_color, zindex_color,
     ndouble, nchar;
   unsigned int size;
   double x, y;
   char *string = NULL, ast1, ast2;
   const char *error_str[] =
      { "Error reading contour-plot data file.",
        "Error reading color-plot data file.",
        "Not enough memory for data.",
        "Not enough memory for line-break data." };
   /* Get sizes of data files */
   nplots = context->plot_param.nplots;
   npts_total = 0;
   npts_total_xcolor = 0;
   npts_total_ycolor = 0;
   npts_total_zcolor = 0;
   index_filenames = 0;
   index_formats = 0;
   ipoints = 0;
   ihist = 0;
   icolor = 0;
   for ( iplot=1; iplot<=nplots; iplot++ )
      {
	struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];

	FILE *fptr;
   
	/* Open data file */
      if ( (fptr = fopen( &parser->filenames[index_filenames], "r")) == NULL )
         {
         size = strlen("Cannot open data file:\n") + strlen(&parser->filenames[index_filenames]);
         string = xmalloc(size + 1);
         sprintf(string, "%s%s", "Cannot open data file:\n", &parser->filenames[index_filenames]);
         g_set_error_literal (&context->err, context->domain, 0, string);
         free(string);
         longjmp (context->finish, 1);
         }

      /* Get size of data file */
      i = 0;
      if ( strcmp(context->plot_param.axis_type, "linear")   == 0 ||
           strcmp(context->plot_param.axis_type, "semilogx") == 0 ||
           strcmp(context->plot_param.axis_type, "semilogy") == 0 ||
           strcmp(context->plot_param.axis_type, "loglog")   == 0 ||
           strcmp(context->plot_param.axis_type, "polar")    == 0 )
         {
         if ( strncmp (the_plot->plot_types, "points", 10) == 0 )
            {
            ipoints++;
            ndouble = 0;
            nchar = 0;
            while ( (ndouble = fscanf(fptr, &parser->formats[index_formats], &x, &y)) == 2 ||
                    ((nchar = fscanf(fptr, &parser->formats_mod[index_formats], &ast1, &ast2)) == 2 &&
                      ast1 == '*' && ast2 == '*') )
               {
               if ( ndouble == 2 )
                  {
                  i++;
                  ndouble = 0;
                  }
               else if ( nchar == 2 )
                  {
		  the_plot->nlinebreaks++;
                  nchar = 0;
                  }
               }
            index_formats = index_formats + parser->nformats[ipoints+ihist-1];
            npts = i;
	    the_plot->ndata = npts;
            npts_total = npts_total + npts;
            }

         else if ( strncmp (the_plot->plot_types, "histogram", 10) == 0 )
            {
            ihist++;
            while ( fscanf(fptr, &parser->formats[index_formats], &x) == 1 )
               i++;
            index_formats = index_formats + parser->nformats[ipoints+ihist-1];
            npts = i;
            context->plot_parameters[ipoints+ihist-1].ndata = npts;
            npts_total = npts_total + npts;
            }

         else if ( (strncmp (the_plot->plot_types, "contour", 10) == 0)
		   ||
		   (strncmp (the_plot->plot_types, "color", 10) == 0))
	   
            {
            if ( fscanf(fptr, "%d %d", &nx, &ny) != 2 )
               {
               g_set_error_literal (&context->err, context->domain, 0, error_str[0]);
               longjmp (context->finish, 1);
               }
	    the_plot->samples3d.nx = nx;
	    the_plot->samples3d.ny = ny;
	    the_plot->samples3d.x = xmalloc (nx * sizeof (double));
	    the_plot->samples3d.y = xmalloc (ny * sizeof (double));
	    the_plot->samples3d.z = xmalloc (ny * nx * sizeof (double));
            }
         }

      /* Increment indices */
      fclose(fptr);
      index_filenames = index_filenames + parser->nfilenames[iplot-1];
      }


   /* Read data files */
   index_filenames = 0;
   index_formats = 0;
   index = 0;
   xindex_color = 0;
   yindex_color = 0;
   zindex_color = 0;
   ipoints = 0;
   ihist = 0;
   icolor = 0;

   for ( iplot=1; iplot<=nplots; iplot++ )
      {
	struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];

	int ichar = 0;
	/* Open data file */
	FILE *fptr = fopen( &parser->filenames[index_filenames], "r");

	the_plot->nlinebreak =
	  xmalloc (the_plot->nlinebreaks * sizeof (int));

	the_plot->samples =
	  xmalloc (the_plot->ndata * sizeof (*the_plot->samples));

	
      /* Read data file */
      if ( strcmp(context->plot_param.axis_type, "linear")   == 0 ||
           strcmp(context->plot_param.axis_type, "semilogx") == 0 ||
           strcmp(context->plot_param.axis_type, "semilogy") == 0 ||
           strcmp(context->plot_param.axis_type, "loglog")   == 0 ||
           strcmp(context->plot_param.axis_type, "polar")    == 0 )
         {
	   if ( strncmp (the_plot->plot_types, "points", 10) == 0 )
	     {
	       ipoints++;
	       npts = the_plot->ndata;
	       nchar = 0;
	       for ( i=1; i<=npts; i++ )
		 {
		   int ndouble = fscanf (fptr, &parser->formats[index_formats],
					 &the_plot->samples[i-1].x,
					 &the_plot->samples[i-1].y);

		   if (ndouble != 2)
		     {
		       int nchar = fscanf (fptr,
					   &parser->formats_mod[index_formats],
					   &ast1, &ast2);

		       if (nchar == 2)
			 {
			   ichar++;
			   the_plot->nlinebreak[ichar-1] = index + i - 1;
			   i--;
			 }
		     }
		 }
	       if (ichar < the_plot->nlinebreaks)
		 the_plot->nlinebreak[ichar] = index + i - 1;

	       {
		 /* Modify data for logarithmic axes */
		 if ( strcmp(context->plot_param.axis_type, "semilogx") == 0 ||
		      strcmp(context->plot_param.axis_type, "loglog")   == 0 )
		   for ( i=1; i<=npts; i++ )
		     {
		       if ( the_plot->samples[i-1].x != 0.0 )
			 the_plot->samples[i-1].x = log10(fabs(the_plot->samples[i-1].x));
		       else
			 the_plot->samples[i-1].x = log10(DBL_MIN);
		     }

		 if ( strcmp(context->plot_param.axis_type, "semilogy") == 0 ||
		      strcmp(context->plot_param.axis_type, "loglog")   == 0 )
		   for ( i=1; i<=npts; i++ )
		     {
		       if ( the_plot->samples[i-1].y != 0.0 )
			 the_plot->samples[i-1].y = log10(fabs(the_plot->samples[i-1].y));
		       else
			 the_plot->samples[i-1].y = log10(DBL_MIN);
		     }
	       }

	       index_formats = index_formats + parser->nformats[ipoints+ihist-1];
	       index = index + context->plot_parameters[ipoints+ihist-1].ndata;
	     }

         else if ( strncmp (the_plot->plot_types, "histogram", 10) == 0 )
            {
            ihist++;
	    npts = the_plot->ndata;
            for ( i=1; i<=npts; i++ )
               {
		 fscanf(fptr, &parser->formats[index_formats],
			&the_plot->samples[i-1].x);
		 the_plot->samples[i-1].y = 0.0;
               }

	    qsort(the_plot->samples, npts, sizeof(*the_plot->samples), compare_datum_by_x);
            index_formats = index_formats + parser->nformats[ipoints+ihist-1];
            index = index + context->plot_parameters[ipoints+ihist-1].ndata;
            }

         else if (( strncmp (the_plot->plot_types, "contour", 10) == 0 )
		  ||
		  ( strncmp (the_plot->plot_types, "color", 10) == 0 )
		  )
            {
	    int nx, ny;
            if ( fscanf(fptr, "%d %d", &nx, &ny) != 2 )
               {
               g_set_error_literal (&context->err, context->domain, 0, error_str[0]);
               longjmp (context->finish, 1);
               }
	    g_assert (nx == the_plot->samples3d.nx);
	    g_assert (ny == the_plot->samples3d.ny);
	    
            for ( i=1; i<=nx; i++ )
	      {
		double x;
               if ( fscanf(fptr, "%lf", &x) != 1 )
                  {
                  g_set_error_literal (&context->err, context->domain, 0, error_str[0]);
                  longjmp (context->finish, 1);
                  }
	       the_plot->samples3d.x[i-1] = x;
	      }

            for ( j=1; j<=ny; j++ )
	      {
		double y;
               if ( fscanf(fptr, "%lf", &y) != 1 )
                  {
                  g_set_error_literal (&context->err, context->domain, 0, error_str[0]);
                  longjmp (context->finish, 1);
                  }
	       the_plot->samples3d.y[j-1] = y;
	      }

            for ( i=1; i<=nx; i++ )
               for ( j=1; j<=ny; j++ )
		 {
		   double z;
                  if ( fscanf(fptr, "%lf", &z) != 1 )
                     {
                     g_set_error_literal (&context->err, context->domain, 0, error_str[0]);
                     longjmp (context->finish, 1);
                     }
		  the_plot->samples3d.z[ny*(i-1)+j-1] = z;
		 }
            }
	 }

      /* Increment indices */
      fclose(fptr);
      index_filenames = index_filenames + parser->nfilenames[iplot-1];
      }

   return;
   }

static int
compare_datum_by_x (const void *p1_, const void *p2_)
{
  const struct datum *p1 = p1_;
  const struct datum *p2 = p2_;

  if (p1->x < p2->x)
    return -1;

  return (p1->x > p2->x);
}
