/*******************************************************************************
*
* DrawLegend.c
*
* Contains functions:
*    DrawLegend
*    background_legend
*
* Draws legend.
*
* Copyright  2008, 2009, 2010, 2011, 2012 Spencer A. Buckner
* http://savannah.gnu.org/projects/gsegrafix
*
* This file is part of GSEGrafix, a scientific and engineering plotting program.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*******************************************************************************/

#include "DrawLegend.h"
#include "Misc.h"
#include "GetWindowCoords.h"
#include "DrawSymbols.h"

#include "gse-cairo.h"
#include "gsegraf.h"

#include <string.h>
#include <stdlib.h>

static void
background_legend (struct gse_ctx *context, char *legend_str, double height_lines, double x1, double y1 );


void DrawLegend (struct gse_ctx *context)
   {
   /* Declare variables */
   int i, iplot, ifunc, nplots, nlines, imesh, icontour, index, anchor_text;
   unsigned int i1_str, i2_str, size, nchar;
   long int file_position, file_position_1;
   double x, y, x1, y1, x2, y2, x1_box, y1_box, x2_box, y2_box,
          xlegend, ylegend, zlegend, xanchor, yanchor,
          xanchor_text, yanchor_text, xanchor_symbol, yanchor_symbol,
          width_text, height_text, width_legend, height_legend,
          height_lines, dx_text, dx_symbol, dy_symbol,
          yinc1, yinc2, plot_coords[3], window_coords[2];
   double dy_bar[] = { 0.0, 6.0, 0.0 };
   const char *string;
   char *legend_str=NULL, anchor[10], *pchar,
        legend_coords_flag[4];

   const char *error_str[] =
      { "Invalid or incomplete legend coordinates.",
        "Legend anchor not found.",
        "Legend coordinates or legend anchor not found.",
        "Invalid legend anchor." };
   GnomeCanvasPoints *points;
   GnomeCanvasItem *text;
   FILE *fptr;

   /* Get plot box minimum and maximum values */
   x1_box = context->plot_box_data.xmin;
   x2_box = context->plot_box_data.xmax;
   y1_box = context->plot_box_data.ymin;
   y2_box = context->plot_box_data.ymax;

   /* Get number of legend lines */
   nlines = 0;
   fptr = fopen(context->p_param_file, "r");
   file_position = ftell(fptr);
   while ( fgets(context->line, context->maxline, fptr) != NULL )
      {
      if ( strncmp(context->line, "legend_string", 13) == 0 )
         {
         nlines++;
         file_position_1 = file_position;
         while ( fgets(context->line, context->maxline, fptr) != NULL )
            {
            if ( strncmp(context->line, "legend_string", 13) == 0 )
               nlines++;
            else
               break;
            }
         break;
         }
      else if ( strncmp(context->line, "#####", 5) == 0 )
          break;
      else
         file_position = ftell(fptr);
      }
   if ( nlines == 0 )
      return;


   /* Get legend-line sizes */
   fseek(fptr, file_position_1, SEEK_SET);
   nchar = 0;
   for ( i=1; i<= nlines; i++ )
      {
      fgets(context->line, context->maxline, fptr);
      size = strlen(&context->line[13]);
      if ( (string = get_string(context, &context->line[13], &i1_str, &i2_str, &size, 1)) != NULL )
         nchar = nchar + size + 1;
      }


   /* Read legend lines */
   fseek(fptr, file_position_1, SEEK_SET);
   legend_str = xzalloc(nchar);
   index = 0;
   for ( i=1; i<= nlines; i++ )
      {
      fgets(context->line, context->maxline, fptr);
      size = strlen(&context->line[13]);
      if ( (string = get_string(context, &context->line[13], &i1_str, &i2_str, &size, 1)) != NULL )
         {
         strcpy(&legend_str[index], string);
         index = index + size + 1;
         legend_str[index-1] = '\n';
         }
      }
   legend_str[index-1] = '\0';


   /* Get legend coordinates and anchor */
   memset(legend_coords_flag, 0, sizeof(legend_coords_flag));
   for ( i=1; i<=2; i++ )
      {
      if ( fgets(context->line, context->maxline, fptr) != NULL )
         {
         if ( strncmp(context->line, "legend_coords_abs", 17) == 0 ||
              strncmp(context->line, "legend_coords_rel", 17) == 0 )
            {
            if ( (strcmp(context->plot_param.axis_type, "3d") != 0 &&
                  strncmp(context->line, "legend_coords_abs", 17) == 0 &&
                  sscanf(&context->line[17], "%lf %lf", &xlegend, &ylegend) == 2) ||
                 (strcmp(context->plot_param.axis_type, "3d") == 0 &&
                  strncmp(context->line, "legend_coords_abs", 17) == 0 &&
                  sscanf(&context->line[17], "%lf %lf %lf", &xlegend, &ylegend, &zlegend) == 3) ||
                 (strncmp(context->line, "legend_coords_rel", 17) == 0 &&
                  sscanf(&context->line[17], "%lf %lf", &xlegend, &ylegend) == 2) )
               strncpy(legend_coords_flag, &context->line[14], 3);

            else
               {
               g_set_error_literal (&context->err, context->domain, 0, error_str[0]);
               free(legend_str);
               fclose(fptr);
               longjmp (context->finish, 1);
               }
            }

         else if ( strncmp(context->line, "legend_anchor", 13) == 0 )
            {
            memset(anchor, 0, sizeof(anchor));
            size = strlen(&context->line[13]);
            if ( (string = get_string(context, &context->line[13], &i1_str, &i2_str, &size, 0)) != NULL )
               strncpy(&anchor[0], string, 9);
            else
               {
               g_set_error_literal (&context->err, context->domain, 0, error_str[1]);
               free(legend_str);
               fclose(fptr);
               longjmp (context->finish, 1);
               }
            }

         else
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[2]);
            free(legend_str);
            fclose(fptr);
            longjmp (context->finish, 1);
            }
         }
      }


   PangoRectangle rect;
   PangoLayout *layout =
     gse_cairo_create_layout (context->cr, legend_str, context->font_legend, &rect);
   
   width_text = rect.width / PANGO_SCALE;
   height_text = rect.height / PANGO_SCALE;
   width_legend = 70.0 + width_text;
   height_legend = height_text;
   if ( strcmp(anchor, "center") == 0 )
      {
      anchor_text = GSE_ANCHOR_CENTER;
      dx_text = (width_legend - width_text)/2.0;
      dx_symbol = -width_legend/2.0;
      dy_symbol = -height_legend/2.0;
      }
   else if ( strcmp(anchor, "north") == 0 )
      {
      anchor_text = GSE_ANCHOR_NORTH;
      dx_text = (width_legend - width_text)/2.0;
      dx_symbol = -width_legend/2.0;
      dy_symbol = 0.0;
      }
   else if ( strcmp(anchor, "northeast") == 0 )
      {
      anchor_text = GSE_ANCHOR_NORTH_EAST;
      dx_text = 0.0;
      dx_symbol = -width_legend;
      dy_symbol = 0.0;
      }
   else if ( strcmp(anchor, "east") == 0 )
      {
      anchor_text = GSE_ANCHOR_EAST;
      dx_text = 0.0;
      dx_symbol = -width_legend;
      dy_symbol = -height_legend/2.0;
      }
   else if ( strcmp(anchor, "southeast") == 0 )
      {
      anchor_text = GSE_ANCHOR_SOUTH_EAST;
      dx_text = 0.0;
      dx_symbol = -width_legend;
      dy_symbol = -height_legend;
      }
   else if ( strcmp(anchor, "south") == 0 )
      {
      anchor_text = GSE_ANCHOR_SOUTH;
      dx_text = (width_legend - width_text)/2.0;
      dx_symbol = -width_legend/2.0;
      dy_symbol = -height_legend;
      }
   else if ( strcmp(anchor, "southwest") == 0 )
      {
      anchor_text = GSE_ANCHOR_SOUTH_WEST;
      dx_text = width_legend - width_text;
      dx_symbol = 0.0;
      dy_symbol = -height_legend;
      }
   else if ( strcmp(anchor, "west") == 0 )
      {
      anchor_text = GSE_ANCHOR_WEST;
      dx_text = width_legend - width_text;
      dx_symbol = 0.0;
      dy_symbol = -height_legend/2.0;
      }
   else if ( strcmp(anchor, "northwest") == 0 )
      {
      anchor_text = GSE_ANCHOR_NORTH_WEST;
      dx_text = width_legend - width_text;
      dx_symbol = 0.0;
      dy_symbol = 0.0;
      }
   else
      {
      g_set_error_literal (&context->err, context->domain, 0, error_str[3]);
      free(legend_str);
      fclose(fptr);
      longjmp (context->finish, 1);
      }

   if ( strcmp(legend_coords_flag, "abs") == 0 )
      {
      if ( strcmp(context->plot_param.axis_type, "3d") == 0 )
         {
         plot_coords[0] = xlegend;
         plot_coords[1] = ylegend;
         plot_coords[2] = zlegend;
         }
      else
         {
         plot_coords[0] = xlegend;
         plot_coords[1] = ylegend;
         }
      GetWindowCoords(context, plot_coords, window_coords);
      xanchor = window_coords[0];
      yanchor = window_coords[1];
      }
   else if ( strcmp(legend_coords_flag, "rel") == 0 )
      {
      xanchor = (1.0 - xlegend)*x1_box + xlegend*x2_box;
      yanchor = (1.0 - ylegend)*y2_box + ylegend*y1_box;
      }


   /* Check legend is within plot box for absolute coordinates */
   if ( strcmp(legend_coords_flag, "abs") == 0 &&
        (xanchor < 0 || yanchor < 0) )
      {
      free(legend_str);
      fclose(fptr);
      return;
      }

   xanchor_text = xanchor + dx_text;
   yanchor_text = yanchor;
   gnome_canvas_item_set(text,
                         "x", xanchor_text,
                         "y", yanchor_text,
                         "anchor", anchor_text,
                         NULL);

   gse_cairo_render_layout (context->cr, layout,
			    &rect,
			    xanchor_text,
			    yanchor_text,
			    anchor_text,
			    context->canvas_fg_color);
   g_object_unref (layout);

   /* Draw legend background rectangles */
   gnome_canvas_item_get_bounds(text, &x1, &y1, &x2, &y2);
   height_lines = (y2 - y1)/nlines;
   //   background_legend(context, legend_str, height_lines, x1 - 70.0, y1);

   /* Specify legend symbol coordinate parameters */
   xanchor_symbol = xanchor + dx_symbol;
   yanchor_symbol = yanchor + dy_symbol;
   yinc1 = height_legend/nlines;
   yinc2 = yinc1/2.0;


   /* Draw symbols */
   nplots = context->plot_param.nplots;
   imesh = 0;
   icontour = 0;
   for ( iplot=1; iplot<=nplots; iplot++ )
      {
	const struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];

      if ( strncmp (the_plot->plot_types, "points", 10)    == 0 ||
           strncmp (the_plot->plot_types, "histogram", 10) == 0 )
         {
         /* Draw line */
         if ( the_plot->stylechar1 == 'l' )
            {
            points = gnome_canvas_points_new(2);
            points->coords[0] = xanchor_symbol;
            points->coords[1] = yanchor_symbol + yinc1*(iplot - 1) + yinc2;
            points->coords[2] = xanchor_symbol + 60.0;
            points->coords[3] = yanchor_symbol + yinc1*(iplot - 1) + yinc2;
            DrawLine(context, points, 2, the_plot->fill_colors_rgba, the_plot->stylesizes);
            gnome_canvas_points_unref(points);
            }


         /* Draw dashed line */
         else if ( the_plot->stylechar1 == 'd' )
            {
            points = gnome_canvas_points_new(2);
            points->coords[0] = xanchor_symbol;
            points->coords[1] = yanchor_symbol + yinc1*(iplot - 1) + yinc2;
            points->coords[2] = xanchor_symbol + 60.0;
            points->coords[3] = yanchor_symbol + yinc1*(iplot - 1) + yinc2;
            DrawDashedLine(context, points, the_plot->fill_colors_rgba, the_plot->stylesizes);
            gnome_canvas_points_unref(points);
            }


         /* Draw dotted line */
         else if ( the_plot->stylechar1 == '.' )
            {
            points = gnome_canvas_points_new(2);
            points->coords[0] = xanchor_symbol;
            points->coords[1] = yanchor_symbol + yinc1*(iplot - 1) + yinc2;
            points->coords[2] = xanchor_symbol + 60.0;
            points->coords[3] = yanchor_symbol + yinc1*(iplot - 1) + yinc2;
            DrawDottedLine(context, points, the_plot->fill_colors_rgba, the_plot->stylesizes);
            gnome_canvas_points_unref(points);
            }


         /* Draw symbols in symbol_string1 ("cCtTsSiIpPhH") */
         else if ( (pchar = strchr(symbol_string1, the_plot->stylechar1)) != NULL )
            {
            ifunc = pchar - symbol_string1;
            for ( i=1; i<=3; i++ )
               {
               x = xanchor_symbol + 10.0 + (i-1)*20.0;
               y = yanchor_symbol + yinc1*(iplot - 1) + yinc2;
               context->symbol_func1[ifunc](context, x, y, the_plot->fill_colors_rgba, the_plot->outline_colors_rgba,
                                   the_plot->stylesizes);
               }
            }


         /* Draw symbols in symbol_string2 ("+xra") */
         else if ( (pchar = strchr(symbol_string2, the_plot->stylechar1)) != NULL )
            {
            ifunc = pchar - symbol_string2;
            for ( i=1; i<=3; i++ )
               {
               x = xanchor_symbol + 10.0 + (i-1)*20.0;
               y = yanchor_symbol + yinc1*(iplot - 1) + yinc2;
               context->symbol_func2[ifunc](context, x, y, the_plot->fill_colors_rgba, the_plot->stylesizes);
               }
            }


         /* Draw bars */
         else if ( the_plot->stylechar1 == 'b' || the_plot->stylechar1 == 'B' )
            {
            for ( i=1; i<=3; i++ )
               {
               x = xanchor_symbol + 10.0 + (i-1)*20.0;
               y = yanchor_symbol + yinc1*(iplot - 1) + yinc2;

               x1 = x - 10.0;
               x2 = x + 10.0;
               y1 = y + 6.0;
               y2 = y - dy_bar[i-1];

               DrawBar(context, x1+1.0, y1, x2-1.0, y2, the_plot->fill_colors_rgba, the_plot->outline_colors_rgba);
               DrawBar(context, x1, y1+1.0, x2, y2-1.0, 0xFFFFFF00, context->canvas_bg_color);
               }
            }
         }


      else if ( strncmp (the_plot->plot_types, "mesh", 10) == 0 )
         {
         imesh++;
         x1 = xanchor_symbol;
         y1 = yanchor_symbol + yinc1*(iplot - 1) + yinc2;
         x2 = xanchor_symbol + 60.0;
         y2 = yanchor_symbol + yinc1*(iplot - 1) + yinc2;

         if ( the_plot->styleflags == 2 || the_plot->styleflags == 4 )
            {
            if ( (pchar = strchr(color_string, the_plot->stylechar1)) != NULL )
               {
               index = pchar - &color_string[0];
               DrawMesh(context, x1, y1, x2, y2, context->color_rgba[index], context->plot_parameters[imesh-1].meshcolors, the_plot->styleflags);
               }
            }

         else if ( the_plot->styleflags == 5 || the_plot->styleflags == 6 )
	   DrawMesh(context, x1, y1, x2, y2, the_plot->stylecolor1, context->plot_parameters[imesh-1].meshcolors, the_plot->styleflags);

         else if ( the_plot->styleflags == 7 )
	   DrawMesh(context, x1, y1, x2, y2, the_plot->alphacolor, context->plot_parameters[imesh-1].meshcolors, the_plot->styleflags);
         }


      else if ( strncmp (the_plot->plot_types, "contour", 10) == 0 )
         {
         icontour++;
         x1 = xanchor_symbol;
         y1 = yanchor_symbol + yinc1*(iplot - 1) + yinc2;
         x2 = xanchor_symbol + 60.0;
         y2 = yanchor_symbol + yinc1*(iplot - 1) + yinc2;

         if ( the_plot->styleflags == 1 )
            {
            /* 2d contour plot */
            if ( (pchar = strchr(color_string, the_plot->stylechar1)) != NULL )
               {
               index = pchar - &color_string[0];
               DrawContour(context, x1, y1, x2, y2, context->color_rgba[index], 0xFFFFFF00, the_plot->styleflags);
               }
            }

         else if ( the_plot->styleflags == 3 )
            {
            /* 2d contour plot */
	      DrawContour(context, x1, y1, x2, y2, the_plot->stylecolor1, 0xFFFFFF00, the_plot->styleflags);
            }

         else if ( the_plot->styleflags == 7 )
            {
            /* 2d contour plot */
	      DrawContour(context, x1, y1, x2, y2, context->color_rgba[3], 0xFFFFFF00, the_plot->styleflags);
            }

         else if ( the_plot->styleflags == 2 || the_plot->styleflags == 4 )
            {
            /* 3d contour plot */
            if ( (pchar = strchr(color_string, the_plot->stylechar1)) != NULL )
               {
               index = pchar - &color_string[0];
               DrawContour(context, x1, y1, x2, y2, context->color_rgba[index], context->plot_parameters[icontour-1].contourcolors, the_plot->styleflags);
               }
            }

         else if ( the_plot->styleflags == 5 || the_plot->styleflags == 6 )
            {
            /* 3d contour plot */
	      DrawContour(context, x1, y1, x2, y2, the_plot->stylecolor1, context->plot_parameters[icontour-1].contourcolors, the_plot->styleflags);
            }
         }


      else if ( strncmp (the_plot->plot_types, "color", 10) == 0 )
         {
         x1 = xanchor_symbol;
         y1 = yanchor_symbol + yinc1*(iplot - 1) + yinc2;
         x2 = xanchor_symbol + 60.0;
         y2 = yanchor_symbol + yinc1*(iplot - 1) + yinc2;
         DrawColorPlot(context, x1, y1, x2, y2);                                            /* 2d color plot */
         }
      }

   free(legend_str);
   fclose(fptr);

   return;
   }



static void
background_legend (struct gse_ctx *context, char *legend_str, double height_lines, double x1, double y1 )
   {
   /* Declare variables */
   int i, i1, nline;
   double x, y, x1rec, y1rec, x2rec, y2rec,
          x11, y11, x22, y22;
   char ch;
   GnomeCanvasItem *text, *rectangle;


   /* Get lengths of text lines and draw text background */
   x = context->window_width/2.0;
   y = 10.0;
   i = 0;
   i1 = 0;
   nline = 0;
   while ( 1 )
      {
      if ( (ch = legend_str[i]) == '\n' || (ch = legend_str[i]) == '\0' )
         {
         nline++;
         legend_str[i] = '\0';

         if ( strlen(&legend_str[i1]) > 0 )
            {
            text = gnome_canvas_item_new(context->group,
                                         GNOME_TYPE_CANVAS_TEXT,
                                         "text", &legend_str[i1],
                                         "x", x,
                                         "y", y,
                                         "anchor", GSE_ANCHOR_NORTH,
                                         "font-desc", context->font_legend,
                                         "fill_color_rgba", 0xFFFFFF00,
                                         NULL);
            gnome_canvas_item_get_bounds(text, &x1rec, &y1rec, &x2rec, &y2rec);
            g_object_unref (text);

            x11 = x1 - 1.0;
            x22 = x1 + 70.0 + x2rec - x1rec + 1.0;
            y11 = y1 + (nline - 1)*height_lines;
            y22 = y11 + height_lines + 1.0;
            rectangle = gnome_canvas_item_new(context->group,
                                              GNOME_TYPE_CANVAS_RECT,
                                              "x1", x11,
                                              "x2", x22,
                                              "y1", y11,
                                              "y2", y22,
                                              "fill_color_rgba", context->canvas_bg_color,
                                              "outline_color_rgba", context->canvas_bg_color,
                                              "width_pixels", 1,
                                              NULL);
            }

         if ( ch == '\0' )
            return;

         i++;
         i1 = i;
         }

      else
         i++;
      }
   }
