/*******************************************************************************
*
* DataMinMax3d.c
*
* Finds minimum and maximum values of 3d-plot data.
*
* Copyright  2008, 2009, 2010, 2011, 2012 Spencer A. Buckner
* http://savannah.gnu.org/projects/gsegrafix
*
* This file is part of GSEGrafix, a scientific and engineering plotting program.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*******************************************************************************/

#include "DataMinMax3d.h"

#include "gsegraf.h"

#include <string.h> // for strncmp

#include "Misc.h"

void DataMinMax3d (struct gse_ctx *context)
   {
   /* Declare variables */
   int i, j, iplot, nplots,
       nx, ny, index, 
       imesh, xindex_mesh, yindex_mesh, zindex_mesh,
       icontour, xindex_contour, yindex_contour, zindex_contour,
       icolor, xindex_color, yindex_color, zindex_color,
       nlines, nsymbols;
   double xmin, xmax, ymin, ymax, zmin, zmax, line_coords[6], symbol_coords[3];
   const char *error_str[] =
      { "Invalid line coordinates.",
        "Invalid symbol coordinates." };
   FILE *fptr;

   /* Initialize min-max data */
   xmin =  DBL_MAX;
   xmax = -DBL_MAX;
   ymin =  DBL_MAX;
   ymax = -DBL_MAX;
   zmin =  DBL_MAX;
   zmax = -DBL_MAX;

   /* Find minimum and maximum values of data files */
   nplots = context->plot_param.nplots;
   index = 0;
   xindex_mesh = 0;
   yindex_mesh = 0;
   zindex_mesh = 0;
   xindex_contour = 0;
   yindex_contour = 0;
   zindex_contour = 0;
   xindex_color = 0;
   yindex_color = 0;
   zindex_color = 0;
   imesh = 0;
   icontour = 0;
   icolor = 0;
   for ( iplot=1; iplot<=nplots; iplot++ )
      {
	const struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];

      if ( strncmp (the_plot->plot_types, "points", 10) == 0 )
         {
         for ( i=1; i<=the_plot->ndata; i++ )
            {
	      minimize (&xmin, the_plot->samples[i-1].x);
	      maximize (&xmax, the_plot->samples[i-1].x);
	      minimize (&ymin, the_plot->samples[i-1].y);
	      maximize (&ymax, the_plot->samples[i-1].y);
	      minimize (&zmin, the_plot->samples[i-1].z);
	      maximize (&zmax, the_plot->samples[i-1].z);
            }

         index = index + the_plot->ndata;
         }

      else if (
	       ( strncmp (the_plot->plot_types, "contour", 10) == 0 )
	       ||
	       ( strncmp (the_plot->plot_types, "color", 10) == 0 )
	       ||
	       ( strncmp (the_plot->plot_types, "mesh", 10) == 0 )
	       )
         {
         icontour++;

	 nx = the_plot->samples3d.nx;
	 minimize (&xmin, the_plot->samples3d.x[0]);
	 maximize (&xmax, the_plot->samples3d.x[0+nx-1]);

	 ny = the_plot->samples3d.ny;
	 minimize (&ymin, the_plot->samples3d.y[0]);
	 maximize (&ymax, the_plot->samples3d.y[0+ny-1]);

         for ( i=1; i<=nx; i++ )
            for ( j=1; j<=ny; j++ )
               {
                 minimize (&zmin, the_plot->samples3d.z[0+ny*(i-1)+j-1]);
                 maximize (&zmax, the_plot->samples3d.z[0+ny*(i-1)+j-1]);
               }

         xindex_contour = xindex_contour + nx;
         yindex_contour = yindex_contour + ny;
         zindex_contour = zindex_contour + nx*ny;
         }
      }


   /* Find minimum and maximum values of line data */
   fptr = fopen(context->p_param_file, "r");
   nlines = 0;
   while ( fgets(context->line, context->maxline, fptr) != NULL )
     {
       /* Get line coordinates */
       if ( strncmp(context->line, "line_coords", 11) == 0 )
         {
	   nlines++;

	   if ( sscanf(context->line, "%*s %lf %lf %lf %lf %lf %lf",
		       &line_coords[0], &line_coords[3],
		       &line_coords[1], &line_coords[4],
		       &line_coords[2], &line_coords[5]) != 6 )
	     {
	       g_set_error_literal (&context->err, context->domain, 0, error_str[0]);
	       longjmp (context->finish, 1);
	     }

	   /* Modify axis minimum and maximum values */
	   minimize (&xmin, line_coords[0]);
	   maximize (&xmax, line_coords[0]);
	   minimize (&xmin, line_coords[3]);
	   maximize (&xmax, line_coords[3]);

	   minimize (&ymin, line_coords[1]);
	   maximize (&ymax, line_coords[1]);
	   minimize (&ymin, line_coords[4]);
	   maximize (&ymax, line_coords[4]);

	   minimize (&zmin, line_coords[2]);
	   maximize (&zmax, line_coords[2]);
	   minimize (&zmin, line_coords[5]);
	   maximize (&zmax, line_coords[5]);
         }

       else if ( strncmp(context->line, "#####", 5) == 0 )
         break;
     }
   fclose(fptr);


   /* Find minimum and maximum values of symbol data */
   fptr = fopen(context->p_param_file, "r");
   nsymbols = 0;
   while ( fgets(context->line, context->maxline, fptr) != NULL )
      {
      /* Get symbol coordinates */
      if ( strncmp(context->line, "symbol_coords", 13) == 0 )
         {
         nsymbols++;

         if ( sscanf(context->line, "%*s %lf %lf %lf",
                     &symbol_coords[0], &symbol_coords[1], &symbol_coords[2]) != 3 )
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[1]);
            longjmp (context->finish, 1);
            }

         /* Modify axis minimum and maximum values */
	 minimize (&xmin, symbol_coords[0]);
	 maximize (&xmax, symbol_coords[0]);

	 minimize (&ymin, symbol_coords[1]);
	 maximize (&ymax, symbol_coords[1]);

	 minimize (&zmin, symbol_coords[2]);
	 maximize (&zmax, symbol_coords[2]);
         }

      else if ( strncmp(context->line, "#####", 5) == 0 )
         break;
      }
   fclose(fptr);


   /* Save data */
   if ( nplots > 0 || nlines > 0 || nsymbols > 0 )
      {
      context->data_min_max.xmin = xmin;
      context->data_min_max.xmax = xmax;
      context->data_min_max.ymin = ymin;
      context->data_min_max.ymax = ymax;
      context->data_min_max.zmin = zmin;
      context->data_min_max.zmax = zmax;
      }

   return;
   }
