/*******************************************************************************
*
* DataMinMax.c
*
* Finds minimum and maximum values of 2d-plot data.
*
* Copyright  2008, 2009, 2010, 2011, 2012 Spencer A. Buckner
* http://savannah.gnu.org/projects/gsegrafix
*
* This file is part of GSEGrafix, a scientific and engineering plotting program.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*******************************************************************************/

#include "DataMinMax.h"
#include "Histogram.h"
#include <float.h>
#include <math.h>
#include "gsegraf.h"

#include <string.h> // for strncmp

#include "Misc.h"

void DataMinMax (struct gse_ctx *context)
   {
   /* Declare variables */
     int i, j, iplot, nplots, npts,
       ipoints, ihist, icontour, icolor,
       nx, ny, xindex_contour, yindex_contour, zindex_contour,
       xindex_color, yindex_color, zindex_color,
       nlines, nsymbols;
   double xmin, xmax, ymin, ymax, zmin, zmax,
          line_coords[4], symbol_coords[2];
   const char *error_str[] =
      { "Invalid line coordinates.",
        "Invalid symbol coordinates." };
   FILE *fptr;


   /* Initialize min-max data */
   xmin =  DBL_MAX;
   xmax = -DBL_MAX;
   ymin =  DBL_MAX;
   ymax = -DBL_MAX;
   zmin =  DBL_MAX;
   zmax = -DBL_MAX;


   /* Find minimum and maximum values of data files */
   nplots = context->plot_param.nplots;
   ipoints = 0;
   ihist = 0;
   icontour = 0;
   icolor = 0;
   xindex_contour = 0;
   yindex_contour = 0;
   zindex_contour = 0;
   xindex_color = 0;
   yindex_color = 0;
   zindex_color = 0;
   for ( iplot=1; iplot<=nplots; iplot++ )
      {
	const struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];

      if ( strncmp (the_plot->plot_types, "points", 10) == 0 )
         {
         ipoints++;
         npts = context->plot_parameters[ipoints + ihist -1].ndata;
         for ( i=1; i<=npts; i++ )
            {
	      minimize (&xmin, the_plot->samples[i-1].x);
	      maximize (&xmax, the_plot->samples[i-1].x);

	      minimize (&ymin, the_plot->samples[i-1].y);
	      maximize (&ymax, the_plot->samples[i-1].y);
            }

         }

      else if ( strncmp (the_plot->plot_types, "histogram", 10) == 0 )
         {
         ihist++;
         HistogramCalculate(context, iplot);

	 minimize (&xmin, the_plot->binmin);
	 maximize (&xmax, the_plot->binmax);

	 minimize (&ymin, the_plot->histmin);
	 maximize (&ymax, the_plot->histmax);

         }

      else if ( (strncmp (the_plot->plot_types, "contour", 10) == 0)
		||
		(strncmp (the_plot->plot_types, "color", 10) == 0)
		)
         {
         icontour++;

	 nx = the_plot->samples3d.nx;
	 ny = the_plot->samples3d.ny;

	 
	 minimize (&xmin, the_plot->samples3d.x[0]);
	 maximize (&xmax, the_plot->samples3d.x[nx-1]);

	 minimize (&ymin, the_plot->samples3d.y[0]);
	 maximize (&ymax, the_plot->samples3d.y[ny-1]);

         for ( i=1; i<=nx; i++ )
            for ( j=1; j<=ny; j++ )
               {
		 minimize (&zmin, the_plot->samples3d.z[ny*(i-1)+j-1]);
		 maximize (&zmax, the_plot->samples3d.z[ny*(i-1)+j-1]);
               }

         xindex_contour = xindex_contour + nx;
         yindex_contour = yindex_contour + ny;
         zindex_contour = zindex_contour + nx*ny;
         }

      }


   /* Find minimum and maximum values of line data */
   fptr = fopen(context->p_param_file, "r");
   nlines = 0;
   while ( fgets(context->line, context->maxline, fptr) != NULL )
      {
      /* Get line coordinates */
      if ( strncmp(context->line, "line_coords", 11) == 0 )
         {
         nlines++;

         if ( sscanf(context->line, "%*s %lf %lf %lf %lf",
                     &line_coords[0], &line_coords[2],
                     &line_coords[1], &line_coords[3]) != 4 )
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[0]);
            longjmp (context->finish, 1);
            }

         /* Modify line coordinates for logarithmic and polar axes */
         if ( strcmp(context->plot_param.axis_type, "semilogx") == 0 )
            for ( i=1; i<=3; i=i+2 )
               line_coords[i-1] = log10(fabs(line_coords[i-1]));

         else if ( strcmp(context->plot_param.axis_type, "semilogy") == 0 )
            for ( i=2; i<=4; i=i+2 )
               line_coords[i-1] = log10(fabs(line_coords[i-1]));

         else if ( strcmp(context->plot_param.axis_type, "loglog") == 0 )
            for ( i=1; i<=4; i++ )
               line_coords[i-1] = log10(fabs(line_coords[i-1]));

         else if ( strcmp(context->plot_param.axis_type, "polar") == 0 )
            for ( i=1; i<=3; i=i+2 )
               line_coords[i-1] = line_coords[i-1]*deg2rad;

         /* Modify axis minimum and maximum values */
	   minimize (&xmin, line_coords[0]);
	   maximize (&xmax, line_coords[0]);
	   minimize (&xmin, line_coords[2]);
	   maximize (&xmax, line_coords[2]);

	   minimize (&ymin, line_coords[1]);
	   maximize (&ymax, line_coords[1]);
	   minimize (&ymin, line_coords[3]);
	   maximize (&ymax, line_coords[3]);
         }

      else if ( strncmp(context->line, "#####", 5) == 0 )
         break;
      }
   fclose(fptr);


   /* Find minimum and maximum values of symbol data */
   fptr = fopen(context->p_param_file, "r");
   nsymbols = 0;
   while ( fgets(context->line, context->maxline, fptr) != NULL )
      {
      /* Get symbol coordinates */
      if ( strncmp(context->line, "symbol_coords", 13) == 0 )
         {
         nsymbols++;

         if ( sscanf(context->line, "%*s %lf %lf",
                     &symbol_coords[0], &symbol_coords[1]) != 2 )
            {
            g_set_error_literal (&context->err, context->domain, 0, error_str[1]);
            longjmp (context->finish, 1);
            }

         /* Modify symbol coordinates for logarithmic and polar axes */
         if ( strcmp(context->plot_param.axis_type, "semilogx") == 0 )
            symbol_coords[0] = log10(fabs(symbol_coords[0]));

         else if ( strcmp(context->plot_param.axis_type, "semilogy") == 0 )
            symbol_coords[1] = log10(fabs(symbol_coords[1]));

         else if ( strcmp(context->plot_param.axis_type, "loglog") == 0 )
            {
            symbol_coords[0] = log10(fabs(symbol_coords[0]));
            symbol_coords[1] = log10(fabs(symbol_coords[1]));
            }

         else if ( strcmp(context->plot_param.axis_type, "polar") == 0 )
            symbol_coords[0] = symbol_coords[0]*deg2rad;

         /* Modify axis minimum and maximum values */
	 minimize (&xmin, symbol_coords[0]);
	 maximize (&xmax, symbol_coords[0]);
	 minimize (&ymin, symbol_coords[1]);
	 maximize (&ymax, symbol_coords[1]);
         }

      else if ( strncmp(context->line, "#####", 5) == 0 )
         break;
      }
   fclose(fptr);


   /* Save data */
   if ( nplots > 0 || nlines > 0 || nsymbols > 0 )
      {
      context->data_min_max.xmin = xmin;
      context->data_min_max.xmax = xmax;
      context->data_min_max.ymin = ymin;
      context->data_min_max.ymax = ymax;
      }

   if ( icontour > 0 || icolor > 0 )
      {
      context->data_min_max.zmin = zmin;
      context->data_min_max.zmax = zmax;
      }

   return;
   }
