/*******************************************************************************
*
* ColorPlot2d.c
*
* Plots two-dimensional samples3d information as a function of color.
*
* Copyright  2008, 2009, 2010, 2011, 2012 Spencer A. Buckner
* http://savannah.gnu.org/projects/gsegrafix
*
* This file is part of GSEGrafix, a scientific and engineering plotting program.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*******************************************************************************/

#include "ColorPlot2d.h"
#include "Misc.h"
#include "DrawTickMarks.h"
#include "DrawGrid2d.h"
#include "DrawGridLog.h"
#include <math.h>
#include "gsegraf.h"

#include <string.h>
#include <stdlib.h>  // for free

#include "gse-cairo.h"

void ColorPlot2d (struct gse_ctx *context, int iplot, int icolor, int xindex, int yindex, int zindex, int nx, int ny )
   {
   /* Declare variables */
   int i, j, i1, j1, in, jn, nxvalues, nyvalues, nzvalues, width, height;
   guint32 color;
   double x1_box, x2_box, y1_box, y2_box, xmin, xmax, ymin, ymax, zmin, zmax,
          xscale, yscale, zscale, dz, width_bar, height_bar,
          x1_bar, x2_bar, y1_bar, y2_bar, width_ztick_labels,
          x1, x2, y1, y2, fraction, dx1, dx2, dy1, dy2,
          *xi, *yi, *zi, x, y, z;
   char string[21];
   GdkPixbuf *pixbuf_colorbar, *pixbuf_color;
   GnomeCanvasPoints *points;

   const struct plot_parameters *the_plot = &context->plot_parameters[iplot-1];


   /* Get plot box minimum and maximum values */
   x1_box = context->plot_box_data.xmin;
   x2_box = context->plot_box_data.xmax;
   y1_box = context->plot_box_data.ymin;
   y2_box = context->plot_box_data.ymax;


   /* Get minimum and maximum axis values */
   nxvalues = context->xtick_labels.nvalues;
   nyvalues = context->ytick_labels.nvalues;
   nzvalues = context->ztick_labels.nvalues;
   xmin = context->xtick_labels.values[0];
   xmax = context->xtick_labels.values[nxvalues-1];
   ymin = context->ytick_labels.values[0];
   ymax = context->ytick_labels.values[nyvalues-1];
   zmin = context->ztick_labels.values[0];
   zmax = context->ztick_labels.values[nzvalues-1];
   xmin = xmin - context->xtick_labels.offset1;
   xmax = xmax + context->xtick_labels.offset2;
   ymin = ymin - context->ytick_labels.offset1;
   ymax = ymax + context->ytick_labels.offset2;
   zmin = zmin - context->ztick_labels.offset1;
   zmax = zmax + context->ztick_labels.offset2;


   /* Calculate axis scale factors */
   xscale = (x2_box - x1_box)/(xmax - xmin);
   yscale = (y2_box - y1_box)/(ymax - ymin);


   /* Draw color-bar pixbuf */
   if ( icolor == 1 )
      {
      width_bar = 20.0;
      height_bar = 0.875*(y2_box - y1_box);
      x1_bar = x2_box + 20.0;
      x2_bar = x1_bar + width_bar;
      y1_bar = (y1_box + y2_box)/2.0 + height_bar/2.0;
      y2_bar = (y1_box + y2_box)/2.0 - height_bar/2.0;
      if ( strcmp(context->plot_param.plot_box, "on") == 0 )
         {
         width  = roundint(width_bar);
         height = roundint(height_bar);
         pixbuf_colorbar = gdk_pixbuf_new(GDK_COLORSPACE_RGB, TRUE, 8, width, height);
         gdk_pixbuf_fill(pixbuf_colorbar, 0xFFFFFF00);

         for ( j=height; j>=1; j-- )
            {
            fraction = (j - 1.0)/(height - 1.0);
            color = interp_color_1(context, fraction);
            for ( i=1; i<=width; i++ )
               put_pixel(pixbuf_colorbar, i-1, height-j, color);
            }

	 gse_cairo_render_pixbuf (context->cr, pixbuf_colorbar,
				  x1_bar, y1_bar, GSE_ANCHOR_SOUTH_WEST);

         g_object_unref(pixbuf_colorbar);
         }

      /* Draw and label color-bar tick marks */
      if ( strcmp(context->plot_param.plot_box, "on") == 0 )
         {
         /* Draw vertical line */
         points = gnome_canvas_points_new(4);
         points->coords[0] = x2_bar + 8.0;
         points->coords[1] = y1_bar;
         points->coords[2] = x2_bar + 8.0 + context->tick_major;
         points->coords[3] = y1_bar;
         points->coords[4] = x2_bar + 8.0 + context->tick_major;
         points->coords[5] = y2_bar;
         points->coords[6] = x2_bar + 8.0;
         points->coords[7] = y2_bar;

	 gse_cairo_render_line (context->cr, points, 4, 2, context->canvas_fg_color);
				
	 
         gnome_canvas_points_unref(points);

         /* Draw tick marks and tick-mark labels */
         if ( strcmp(context->plot_param.z_tick_marks, "on") == 0 )
            {
	      DrawTickMarks(context, "linear", context->minor_ticks_flag, 0,
                          x2_bar + 8.0 + context->tick_major, y1_bar, x2_bar + 8.0 + context->tick_major, y2_bar,
                          context->ztick_labels.nvalues, &context->ztick_labels.values[0],
                          context->ztick_labels.offset1, context->ztick_labels.offset2,
                          180*deg2rad);

            if ( strcmp(context->plot_param.z_tick_labels, "on") == 0 )
               {
               zscale = height_bar/(zmax - zmin);
               dz = (zmax - zmin)/(nzvalues - 1);
               width_ztick_labels = 0.0;
               for ( i=1; i<=nzvalues; i++ )
                  {
                  memset(string, 0, sizeof(string));
                  if ( fabs(context->ztick_labels.values[i-1]) < 0.01*dz )
                     snprintf(string, sizeof(string), "%1.0f", 0.0);
                  else
                     snprintf(string, sizeof(string), "%g", context->ztick_labels.values[i-1]);

		  gse_cairo_render_text_with_extents (context->cr, string,
						      x2_bar + 8.0 + context->tick_major + 8.0,
						      y1_bar - (context->ztick_labels.values[i-1] - zmin)*zscale,
						      GSE_ANCHOR_WEST,
						      context->canvas_fg_color,
						      context->font_tick_labels, 0,
						      &x1, &y1, &x2, &y2);

		  
                  if ( x2 - x1 > width_ztick_labels )
                     width_ztick_labels = x2 - x1;
                  }
               }
            }
         }
      }


   /* Label color-bar */
   if ( strcmp(context->plot_param.plot_box, "on") == 0 &&
        icolor == 1 && context->zlabel != NULL)
      {
      /* Draw z-axis label */
      y = (y1_box + y2_box)/2.0;
      if ( strcmp(context->plot_param.z_tick_marks, "on") == 0 &&
           strcmp(context->plot_param.z_tick_labels, "on") == 0 )
         x = x2_bar + 8.0 + context->tick_major + 8.0 + width_ztick_labels + 8.0;
      else
         x = x2_bar + 8.0 + context->tick_major + 8.0;



      cairo_save (context->cr);
      cairo_translate (context->cr, x, y);
      cairo_rotate (context->cr, -M_PI / 2.0);
      gse_cairo_render_text (context->cr, context->zlabel, 0, 0,
			     GSE_ANCHOR_WEST,
			     context->canvas_fg_color,
			     context->font_axis_labels);
      cairo_restore (context->cr);
      }


   /* Calculate pixbuf width and height */
   dx1 = 0.0;
   dx2 = 0.0;
   dy1 = 0.0;
   dy2 = 0.0;

   if ( xmin >= the_plot->samples3d.x[0] )
      x1 = xmin;
   else
      {
      x1 = the_plot->samples3d.x[0];
      dx1 = the_plot->samples3d.x[0] - xmin;
      }

   if ( xmax <= the_plot->samples3d.x[0+nx-1] )
      x2 = xmax;
   else
      {
      x2 = the_plot->samples3d.x[0+nx-1];
      dx2 = xmax - the_plot->samples3d.x[0+nx-1];
      }

   if ( ymin >= the_plot->samples3d.y[0] )
      y1 = ymin;
   else
      {
      y1 = the_plot->samples3d.y[0];
      dy1 = the_plot->samples3d.y[0] - ymin;
      }

   if ( ymax <= the_plot->samples3d.y[0+ny-1] )
      y2 = ymax;
   else
      {
      y2 = the_plot->samples3d.y[0+ny-1];
      dy2 = ymax - the_plot->samples3d.y[0+ny-1];
      }

   width  = roundint(x2_box - x1_box - (dx1 + dx2)*xscale);
   height = roundint(y2_box - y1_box - (dy1 + dy2)*yscale);


   /* Check pixbuf width and height */
   if ( width <= 0 || height <= 0 )
      return;


   /* Get interpolated values of x and y */
   xi = xmalloc(width*sizeof(double));
   yi = xmalloc(height*sizeof(double));
   zi = xmalloc(width*height*sizeof(double));
   for ( i=1; i<=width; i++ )
      xi[i-1] = x1 + (i - 1)*(x2 - x1)/(width - 1);
   for ( j=1; j<=height; j++ )
      yi[j-1] = y1 + (j - 1)*(y2 - y1)/(height - 1);


   /* Get interpolated values of z (bilinear interpolation) */
   if ( the_plot->styleflags == 8 )
      for ( i=1; i<=width; i++ )
         for ( j=1; j<=height; j++ )
            interp2(nx, ny, 1, &the_plot->samples3d.x[0], &the_plot->samples3d.y[0], &the_plot->samples3d.z[0],
                    &xi[i-1], &yi[j-1], &zi[height*(i-1)+(j-1)]);


   /* Get interpolated values of z (nearest-neighbor interpolation) */
   else if ( the_plot->styleflags == 9 )
      for ( i=1; i<=width; i++ )
         {
         if ( xi[i-1] <= the_plot->samples3d.x[0] )
            i1 = 0;
         else if ( xi[i-1] >= the_plot->samples3d.x[0+nx-1] )
            i1 = nx - 2;
         else
            i1 = find_indices(0, nx-1, &the_plot->samples3d.x[0], xi[i-1]);

         for ( j=1; j<=height; j++ )
            {
            if ( yi[j-1] <= the_plot->samples3d.y[0] )
               j1 = 0;
            else if ( yi[j-1] >= the_plot->samples3d.y[0+ny-1] )
               j1 = ny - 2;
            else
               j1 = find_indices(0, ny-1, &the_plot->samples3d.y[0], yi[j-1]);

            if ( fabs(xi[i-1] - the_plot->samples3d.x[0+i1]) < fabs(xi[i-1] - the_plot->samples3d.x[0+i1+1]) )
               in = i1;
            else
               in = i1 + 1;

            if ( fabs(yi[j-1] - the_plot->samples3d.y[0+j1]) < fabs(yi[j-1] - the_plot->samples3d.y[0+j1+1]) )
               jn = j1;
            else
               jn = j1 + 1;

            zi[height*(i-1)+(j-1)] = the_plot->samples3d.z[0+ny*in+jn];
            }
         }


   /* Create pixbuf */
   pixbuf_color = gdk_pixbuf_new(GDK_COLORSPACE_RGB, TRUE, 8, width, height);
   gdk_pixbuf_fill(pixbuf_color, 0xFFFFFF00);


   /* Draw color plot */
   zscale = 1.0/(zmax - zmin);
   for ( i=1; i<=width; i++ )
      for ( j=height; j>=1; j-- )
         {
         z = zi[height*(i-1)+(j-1)];
         fraction = (z - zmin)*zscale;
         if ( fraction < 0.0 )
            fraction = 0.0;
         else if ( fraction > 1.0 )
            fraction = 1.0;
         if ( z < the_plot->zblack )
            color = 0x000000FF;
         else if ( z > the_plot->zwhite )
            color = 0xFFFFFFFF;
         else
	   color = interp_color_1(context, fraction);
         put_pixel(pixbuf_color, i-1, height-j, color);
         }
   

   gse_cairo_render_pixbuf (context->cr, pixbuf_color,
		     (x1_box + x2_box + (dx1 - dx2)*xscale)/2.0,
		     (y2_box + y1_box + (dy2 - dy1)*yscale)/2.0,
		     GSE_ANCHOR_CENTER);
		     
   
   g_object_unref(pixbuf_color);


   /* Draw grid */
   DrawGrid2d(context);
   DrawGridLog(context);


   /* Free memory */
   free(xi);
   free(yi);
   free(zi);

   return;
   }
