/*
 *  grotto - a roguelike RPG
 *  Copyright (C) 2006 Tobias Heinzen
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
 
#include "headers/character.hpp"

void CHARACTER::create_bar( struct SKILL skill,char bar[] )
{
	memset( bar,' ',BAR_WIDTH ); bar[BAR_WIDTH-1] = '\0';
	for( int i=0;i<(int)(skill.level*(BAR_WIDTH-3)/skill.learned);i++ )
		bar[1+i] = '=';
}

char *CHARACTER::skill_out( struct SKILL skill )
{
	char *buffer = new char[255]; char bar[BAR_WIDTH]; this->create_bar( skill,bar );
	sprintf( buffer,"%d/%d\t[%s]",skill.level,skill.learned,bar );
	return buffer;
}

char *CHARACTER::mp_out()
{
	return this->skill_out( this->mp );	
}

char *CHARACTER::hp_out()
{
	return this->skill_out( this->hp );	
}

void CHARACTER::show()
{
	struct TUPLE max; getmaxyx( stdscr,max.y,max.x ); BOX box; 
	WINDOW *win = box.create( CWINDOW_WIDTH,CWINDOW_HEIGHT,max.x/2-CWINDOW_WIDTH/2,max.y/2-CWINDOW_HEIGHT/2 );
		
	char health[BAR_WIDTH]; this->create_bar( this->hp,health );
	char mana[BAR_WIDTH];   this->create_bar( this->mp,mana );
	mvwprintw( win,1,2,"HP: %s",this->hp_out() );
	mvwprintw( win,2,2,"MP: %s",this->mp_out() );	
	mvwprintw( win,4,2,"Attributes:" );
	mvwprintw( win,5,3,"Strength:\t\t%d",this->strength.level );
	mvwprintw( win,6,3,"Intelligence:\t%d",this->intelligence.level );
	mvwprintw( win,8,2,"Skills:" );
	mvwprintw( win,9,3,"Attack:\t\t%d (+%d)",this->attack.level,this->attack.modifier );
	mvwprintw( win,10,3,"Offensive Magic:\t%d (+%d)",this->magic_offensive.level,this->magic_offensive.modifier );
	mvwprintw( win,11,3,"Defensive Magic:\t%d (+%d)",this->magic_defensive.level,this->magic_defensive.modifier );
	mvwprintw( win,CWINDOW_HEIGHT-2,10,"<< press space to continue >>" );

	box.show();
}

void CHARACTER::init()
{
	this->hp.level = 10; this->hp.learned = 10; this->hp.modifier = 0;
	this->mp.level = 10; this->mp.learned = 10; this->mp.modifier = 0;
	this->strength.level = 5; this->strength.learned = 5;
	this->intelligence.level = 5; this->intelligence.learned = 5;
	this->attack.level = 1; this->attack.learned = 1; this->attack.modifier = 0;
	this->magic_offensive.level = 1; this->magic_offensive.learned = 1; this->magic_offensive.modifier = 0;
	this->magic_defensive.level = 1; this->magic_defensive.learned = 1; this->magic_defensive.modifier = 0;
	this->dungeon_level = 1;
	this->inventory.init();
}

void CHARACTER::recalibrate_hp_and_mp()
{
	this->hp.learned = 10+(this->intelligence.level-5)*2+(this->strength.level-5)*7 + this->hp.modifier;
	this->mp.learned = 10+(this->intelligence.level-5)*5+(this->strength.level-5)*0 + this->mp.modifier;	
}

void CHARACTER::learn_skill( int skill )
{
	struct SKILL *learning = 0;
	switch( skill )
	{
		case SKILL_ATTACK:
			learning = &(this->attack); break;
		case SKILL_STRENGTH:
			learning = &(this->strength); break;
		case SKILL_INTELLIGENCE:
			learning = &(this->intelligence); break;
		case SKILL_MAGIC_OFF:
			learning = &(this->magic_offensive); break;
		case SKILL_MAGIC_DEF:
			learning = &(this->magic_defensive); break;
		default:;
	}	
	
	if( learning != NULL )
	{
		learning->learned--;
		if( learning->learned <= 0 )
		{
			learning->level++; learning->learned = learning->level;	
		}
	}
	this->recalibrate_hp_and_mp();
}

void CHARACTER::update_life( int modifier )
{
	this->hp.level += modifier;	
	if( this->hp.level < 0 ) this->hp.level = 0;
	if( this->hp.level > this->hp.learned ) this->hp.level = this->hp.learned;
}

void CHARACTER::update_mana( int modifier )
{
	this->mp.level += modifier;	
	if( this->mp.level < 0 ) this->mp.level = 0;
	if( this->mp.level > this->mp.learned ) this->mp.level = this->mp.learned;
}

void CHARACTER::fill_hp()
{
	this->hp.level = this->hp.learned;	
}

void CHARACTER::fill_mp()
{
	this->mp.level = this->mp.learned;	
}

int CHARACTER::physical_dmg()
{
	return RANDOM( 1,(this->attack.level+BASE_ATTACK) ) + this->attack.modifier;	
}

int CHARACTER::magical_dmg()
{
	return RANDOM( 1,(this->magic_offensive.level+BASE_MAGIC) ) + this->magic_offensive.modifier;	
}

int CHARACTER::healing()
{
	return RANDOM( 1,(this->magic_defensive.level+BASE_HEAL) ) + this->magic_defensive.modifier;	
}

bool CHARACTER::has_enough_mana( int magic_action )
{
	if( magic_action == ATTACK && this->mp.level >= this->mana_cost( ATTACK ) )
		return true;
	if( magic_action == HEAL && this->mp.level >= this->mana_cost( HEAL ) )
		return true;
	return false;
}

int CHARACTER::mana_cost( int action )
{
	if( action == ATTACK )
		return (COST_MAGIC+this->magic_offensive.level/3);
	if( action == HEAL )
		return (COST_HEAL+this->magic_defensive.level/4);
	return 0;	
}

int CHARACTER::powerindex()
{
	return this->strength.level*100000 + this->intelligence.level;	
}

bool CHARACTER::is_dead()
{
	if( this->hp.level <= 0 ) 
		return true;	
	return false;
}
