/* outputs.c

   Written by Frederic Bois
   22 June 2014

   Copyright (c) 2014 Frederic Bois.

   This code is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   See the GNU General Public License at <http://www.gnu.org/licenses/> 

   -- Revisions -----
     Logfile:  %F%
    Revision:  %I%
        Date:  %G%
     Modtime:  %U%
      Author:  @a
   -- SCCS  ---------

   Output routines.
*/


/* ----------------------------------------------------------------------------
   Inclusions
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "matrices.h"
#include "outputs.h"
#include "graph_sampler.h"


/* ----------------------------------------------------------------------------
   CloseOutputs

   Close output files.
*/
void CloseOutputs (char *const prefixname)
{

  if (bsave_the_chain) {
    fclose(pChainFile);
    if (prefixname)
      printf ("MCMC chain written to %s_results_mcmc.bin.\n\n",
              prefixname);
    else
      printf ("MCMC chain written to results_mcmc.bin.\n\n");
  }

  if (bsave_the_chain && bTempered) {
    fclose(pInvTemperatureFile);
    if (prefixname)
      printf ("Inverse temperatures written to"
              " %s_inverse_temperatures.out.\n\n",
              prefixname);
    else
      printf ("Inverse temperatures  written to inverse_temperatures.out.\n\n");
  }

  if (bsave_the_edge_probabilities) {
    fclose(pEdgeFile);

    if (bTempered && n_at_targetT == 0) {
      printf("No sample at target temperature, "
             "edge probabilities not estimated.\n");
      if (prefixname)
        printf("File %s_edge_p.out is empty.\n\n", prefixname);
      else
        printf("File edge_p.out is empty.\n\n");
    }
    else {
      if (prefixname)
        printf("Edge probabilities written to %s_edge_p.out",
               prefixname);
      else
        printf("Edge probabilities written to edge_p.out");

      if (bTempered)
        printf("\n(with %ld samples at target temperature).\n\n", n_at_targetT);
      else
        printf(".\n\n");
    }
  }

  if (bsave_some_graphs) {
    fclose(pGraphsFile);
    if (prefixname)
      printf("%d Random graphs written to %s_graph_samples.out.\n\n",
             n_saved_adjacency, prefixname);
    else
      printf("%d Random graphs written to graph_samples.out.\n\n",
             n_saved_adjacency);
  }

  if (bsave_best_graph) {
    fclose(pBestFile);
    if (prefixname)
      printf ("Best graph written to %s_best_graph.out.\n\n",
              prefixname);
    else
      printf ("Best graph written to best_graph.out.\n\n");
  }

  if (bsave_the_degree_counts) {
    fclose(pDegreeFile);
    if (prefixname)
      printf("Degree counts written to %s_degree_count.out.\n\n",
             prefixname);
    else
      printf("Degree counts written to degree_count.out.\n\n");
  }

  if (bsave_the_motifs_probabilities) {
    fclose(pMotifFile);
    if (prefixname)
      printf("Motifs frequencies written to %s_motifs_count.out.\n\n", 
             prefixname);
    else
      printf("Motifs frequencies written to motifs_count.out.\n\n");
  }

  if (bNAData) {
    fclose(pImputedFile);
    if (prefixname)
      printf("Imputed data written to %s_missing_data.out.\n\n", 
             prefixname);
    else
      printf("Imputed data written to missing_data.out.\n\n");
  }

} /* CloseOutputs */


/* ----------------------------------------------------------------------------
   CreateFileName

*/
void CreateFileName (char *const filename, char *const prefixname,
                     char *const suffixname)
{
  strcpy(filename, "");
  if (prefixname) {
    strcat(filename, prefixname);
    strcat(filename, "_");
  }
  strcat(filename, suffixname);

} /* CreateFileName */


/* ----------------------------------------------------------------------------
   InitOutputs

   Initialize various output files.
*/
void InitOutputs (char *const prefixname)
{
  char filename[MAXFILENAME+20];
   
  /* initialize an output file for the best (maximum probability) adjacency 
     matrix */
  if (bsave_best_graph) {
    CreateFileName(filename, prefixname, "best_graph.out");
    if (!(pBestFile = fopen(filename, "w"))) {
      printf ("Error: cannot open %s - Exiting.\n\n", filename);
      exit (0);
    }
  }

  // open chain output file
  if (bsave_the_chain) {
    CreateFileName(filename, prefixname, "results_mcmc.bin");
    if (!(pChainFile = fopen(filename, "wb"))) {
      printf ("Error: cannot open %s - Exiting.\n\n", filename);
      exit (0);
    }

    // first output the number of nodes to it
    fwrite (&nNodes, sizeof(int), 1, pChainFile);
  }

  // open temperature index output file
  if (bsave_the_chain && bTempered) {
    CreateFileName(filename, prefixname, "inverse_temperatures.out");
    if (!(pInvTemperatureFile = fopen(filename, "w"))) {
      printf ("Error: cannot open %s - Exiting.\n\n", filename);
      exit (0);
    }
  }

  // open graph samples output file
  if (bsave_some_graphs) {
    CreateFileName(filename, prefixname, "graph_samples.out");
    if (!(pGraphsFile = fopen(filename, "w"))) {
      printf ("Error: cannot open %s - Exiting.\n\n", filename);
      exit (0);
    }

    // init the print count
    if (n_saved_adjacency < (nRuns - nBurnin)) {
      dPrintStep = (nRuns - nBurnin) / n_saved_adjacency;
    }
    else {
      dPrintStep = 1;
      n_saved_adjacency = nRuns - nBurnin;
    }
    dNextPrint = dPrintStep + nBurnin - 1;
  }

  // open edge probabilities output file
  if (bsave_the_edge_probabilities) {
    CreateFileName(filename, prefixname, "edge_p.out");
    if (!(pEdgeFile = fopen(filename, "w"))) {
      printf ("Error: cannot open %s - Exiting.\n\n", filename);
      exit (0);
    }
  }

  // open degrees output file
  if (bsave_the_degree_counts) {
    CreateFileName(filename, prefixname, "degree_count.out");
    if (!(pDegreeFile = fopen(filename, "w"))) {
      printf ("Error: cannot open %s - Exiting.\n\n", filename);
      exit (0);
    }

    // print header line
    fprintf(pDegreeFile, "degree\tcount\n");
  }

  // open motifs output file
  if (bsave_the_motifs_probabilities) {
    CreateFileName(filename, prefixname, "motifs_count.out");
    if (!(pMotifFile = fopen(filename, "w"))) {
      printf ("Error: cannot open %s - Exiting.\n\n", filename);
      exit (0);
    }
  }

  // open imputed data output file
  if (bNAData) {
    CreateFileName (filename, prefixname, "missing_data.out");
    if (!(pImputedFile = fopen (filename, "w"))) {
      printf ("Error: cannot open %s - Exiting.\n\n", filename);
      exit (0);
    }
    // print header line
    fprintf (pImputedFile, "iter\tparent\tchild\tdata\tllikelihood\n");
  }

} /* InitOutputs */


/* ----------------------------------------------------------------------------
   ReadChain

   Read a saved chain. See in SaveChain how the chain is written. 
   This is provided for convenience and example.
*/
void ReadChain ()
{
  int  j, k, diff;
  long nSamples;
  char szCurrentFile[100];
  int  *adjacency_matrix = NULL;

  printf ("\nName of chainfile ?: ");
  j = scanf ("%s", &szCurrentFile[0]);

  // number of samples to read
  printf ("number of samples to read ? : ");
  j = scanf  ("%ld", &nSamples);

  if (!(pChainFile = fopen(szCurrentFile, "r"))) {
    printf ("Error: cannot open file %s - Exiting\n\n", szCurrentFile);
    exit (0);
  }

  // read the number of nodes
  j = fread (&nNodes, sizeof(int), 1, pChainFile);
  printf("\n\nNumber of nodes: %d.\n\n", nNodes);

  // read the initial adjacency matrix (flat!)
  if (!adjacency_matrix)
    adjacency_matrix = InitiVector(nNodes * nNodes);

  j = fread (adjacency_matrix, sizeof(int), nNodes * nNodes, pChainFile);

  printf("initial adjacency matrix:\n");
  for (j = 0; j < nNodes; j++) {
    for (k = 0; k < nNodes; k++)
      printf("%d ", adjacency_matrix[j+k*nNodes]);
    printf("\n");
  }

  // iterate through the file, matrix by matrix,
  // coded by difference as signed integer 
  // and reconstruct the adjacency matrix at each step
  for (j = 0; j < nSamples; j++) {
    k = fread (&diff, sizeof(int), 1, pChainFile);

    if (diff > 0)
      adjacency_matrix[diff-1]  = adjacency_matrix[diff-1]  + 1;
    if (diff < 0)
      adjacency_matrix[-diff-1] = adjacency_matrix[-diff-1] - 1;
  } /* for j */

  printf("final adjacency matrix:\n");
  for (j = 0; j < nNodes; j++) {
    for (k = 0; k < nNodes; k++)
      printf("%d ", adjacency_matrix[j+k*nNodes]);
    printf("\n");
  }

  free (adjacency_matrix);
  fclose (pChainFile);

} /* ReadChain */


/* ----------------------------------------------------------------------------
   SaveBestGraph

   Just do that.
*/
void SaveBestGraph (void)
{
  if (bsave_best_graph) {
    PrintiMatrix (pBestFile, nNodes, best_adj);
    fprintf(pBestFile, "Log(prior probability):     %.20g\n", dBestPrior);
    if ((bBN || bDBN) && bData) {
      fprintf(pBestFile, "Log(likelihood):            %.20g\n", 
              dBestLikelihood);
      fprintf(pBestFile, "Log(posterior probability): %.20g\n", dBestPosterior);
    }
  }
} /* SaveBestGraph */


/* ----------------------------------------------------------------------------
   SaveChain

   Write the full adjacency matrix at start or just "to_out" (which code for
   the difference and its location) after that.
*/
void SaveChain (int to_out)
{
  int i, j;
  
  if (iter > nBurnin - 1) { // remember that iter starts at zero
    if (iter == nBurnin) {
      /* output the (eventually updated) current adjacency matrix
         going down the columns! */
      for (j = 0; j < nNodes; j++)  // jth column
        for (i = 0; i < nNodes; i++)  // ith line
          fwrite (&current_adj[i][j], sizeof(int), 1, pChainFile);
    }
    
    fwrite (&to_out, sizeof(int), 1, pChainFile);
  }

} /* SaveChain */


/* ----------------------------------------------------------------------------
   SaveDegreeCounts

   Just do that.
*/
void SaveDegreeCounts (void)
{
  long i;

  if (bsave_the_degree_counts) {
    for (i = 0; i < (nNodes+nNodes); i++) {
      fprintf(pDegreeFile, "%ld\t%lg\n", i, cumdegree_count[i]);
    }
  }
} /* SaveDegreeCounts */


/* ----------------------------------------------------------------------------
   SaveEdgeP

   Just do that.
*/
void SaveEdgeP (FILE * outFile)
{
  int i, j;

  if (bsave_the_edge_probabilities) {

    if (!bTempered)
      n_at_targetT = nRuns - nBurnin;
    else {
      if (n_at_targetT > 0) 
        fprintf(outFile, "%ld samples at target temperature.\n\n", 
                         n_at_targetT);
      else {
        fprintf(outFile, "No sample at target temperature, "
                         "edge probabilities not estimated.\n\n");
	return;
      }
    }

    for (i = 0; i < nNodes; i++) {
      for (j = 0; j < nNodes-1; j++) {
        fprintf(outFile, "%lg\t", mat_sum[i][j] / (double) n_at_targetT);
      }
      fprintf(outFile, "%lg\n", mat_sum[i][nNodes-1] / (double) n_at_targetT);
    }

  }

} /* SaveEdgeP */


/* ----------------------------------------------------------------------------
   SaveGraph

   Just do that.
*/
void SaveGraph (void)
{
  if (bsave_some_graphs && (iter == dNextPrint)) {
    PrintiMatrix (pGraphsFile, nNodes, current_adj);

    fprintf(pGraphsFile, "Number of edges: %ld\n", current_edge_count);

    if (bTempered)
      fprintf(pGraphsFile, "Inverse temperature:        %g\n", 
              pInvTemperatures[indexT]);

    fprintf(pGraphsFile, "Log(prior probability):     %.20g\n", 
            current_logprior);

    if ((bBN || bDBN) && bData) {
      fprintf(pGraphsFile, "Log(likelihood):            %.20g\n", 
              current_loglikelihood);
      fprintf(pGraphsFile, "Log(posterior probability): %.20g\n\n\n", 
              current_logposterior);
    }
    else
      fprintf(pGraphsFile, "\n\n");

    dNextPrint += dPrintStep;
  }
} /* SaveGraph */


/* ----------------------------------------------------------------------------
   SaveInverseTemperature

   Saves in fact the index to the current inverse temperature.
*/
void SaveInverseTemperature (void)
{
  if (iter > nBurnin - 1) { // remember that iter starts at zero
    fprintf (pInvTemperatureFile, "%d\n", indexT);
  }
} /* SaveInverseTemperature */


/* ----------------------------------------------------------------------------
   SaveMotifsP

   Just do that.
*/
void SaveMotifsP (FILE * outFile)
{
  if (bsave_the_motifs_probabilities) {
    fprintf(outFile, "Endless/Frustrated triangular loops:\n");
    fprintf(outFile, "E loops count: %lg\n", cum_nEloops);
    fprintf(outFile, "F loops count: %lg\n", cum_nFloops);

    if ((cum_nEloops != 0) || (cum_nFloops != 0))
      fprintf(outFile, "E / (E + F): %lg\n\n",
             cum_nEloops / (cum_nEloops+ cum_nFloops));
    else
      fprintf(outFile, "\n");
  }

} /* SaveMotifsP */


/* end */
