/* outputs.c

   Written by Frederic Bois
   22 June 2014

   Copyright (c) 2014 Frederic Bois.

   This code is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   See the GNU General Public License at <http://www.gnu.org/licenses/> 

   -- Revisions -----
     Logfile:  %F%
    Revision:  %I%
        Date:  %G%
     Modtime:  %U%
      Author:  @a
   -- SCCS  ---------

   Output routines.
*/


/* ----------------------------------------------------------------------------
   Inclusions
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "matrices.h"
#include "outputs.h"


/* ----------------------------------------------------------------------------
   CloseOutputs

   Close output files.
*/
void CloseOutputs (char *const prefixname)
{

  if (bsave_the_chain) {
    fclose(pChainFile);
    if (prefixname)
      printf ("MCMC chain written to %s_results_mcmc.bin.\n\n",
              prefixname);
    else
      printf ("MCMC chain written to results_mcmc.bin.\n\n");
  }

  if (bsave_the_edge_probabilies) {
    fclose(pEdgeFile);
    if (prefixname)
      printf("Edge probabilities written to %s_edge_p.out.\n\n",
             prefixname);
    else
      printf("Edge probabilities written to edge_p.out.\n\n");
  }

  if (bsave_some_graphs) {
    fclose(pGraphsFile);
    if (prefixname)
      printf("%d Random graphs written to %s_graph_samples.out.\n\n",
             n_saved_adjacency, prefixname);
    else
      printf("%d Random graphs written to graph_samples.out.\n\n",
             n_saved_adjacency);
  }

  if (bsave_best_graph) {
    fclose(pBestFile);
    if (prefixname)
      printf ("Best graph written to %s_best_graph.out.\n\n",
              prefixname);
    else
      printf ("Best graph written to best_graph.out.\n\n");
  }

  if (bsave_the_degree_counts) {
    fclose(pDegreeFile);
    if (prefixname)
      printf("Degree counts written to %s_degree_count.out.\n\n",
             prefixname);
    else
      printf("Degree counts written to degree_count.out.\n\n");
  }

  if (bsave_the_motifs_probabilies) {
    fclose(pMotifFile);
    if (prefixname)
      printf("Motifs frequencies written to %s_motifs_count.out.\n\n", 
             prefixname);
    else
      printf("Motifs frequencies written to motifs_count.out.\n\n");
  }

} /* CloseOutputs */


/* ----------------------------------------------------------------------------
   CreateFileName

*/
void CreateFileName (char *const filename, char *const prefixname,
                     char *const suffixname)
{
  strcpy(filename, "");
  if (prefixname) {
    strcat(filename, prefixname);
    strcat(filename, "_");
  }
  strcat(filename, suffixname);

} /* CreateFileName */


/* ----------------------------------------------------------------------------
   InitOutputs

   Initialize various output files.
*/
void InitOutputs (char *const prefixname)
{
  char filename[MAXFILENAME+20];
   
  /* initialize an output file for the best (maximum probability) adjacency 
     matrix */
  if (bsave_best_graph) {
    CreateFileName(filename, prefixname, "best_graph.out");
    if (!(pBestFile = fopen(filename, "w"))) {
      printf ("Error: cannot open %s - Exiting.\n\n", filename);
      exit (0);
    }
  }

  // open chain output file
  if (bsave_the_chain) {
    CreateFileName(filename, prefixname, "results_mcmc.bin");
    if (!(pChainFile = fopen(filename, "wb"))) {
      printf ("Error: cannot open %s - Exiting.\n\n", filename);
      exit (0);
    }

    // first output the number of nodes to it
    fwrite (&nNodes, sizeof(int), 1, pChainFile);
  }

  // open graph samples output file
  if (bsave_some_graphs) {
    CreateFileName(filename, prefixname, "graph_samples.out");
    if (!(pGraphsFile = fopen(filename, "w"))) {
      printf ("Error: cannot open %s - Exiting.\n\n", filename);
      exit (0);
    }

    // init the print count
    if (n_saved_adjacency < (nRuns - nBurnin)) {
      dPrintStep = (nRuns - nBurnin) / n_saved_adjacency;
    }
    else {
      dPrintStep = 1;
    }
    dNextPrint = dPrintStep + nBurnin - 1;
  }

  // open edge probabilities output file
  if (bsave_the_edge_probabilies) {
    CreateFileName(filename, prefixname, "edge_p.out");
    if (!(pEdgeFile = fopen(filename, "w"))) {
      printf ("Error: cannot open %s - Exiting.\n\n", filename);
      exit (0);
    }
  }

  // open degrees output file
  if (bsave_the_degree_counts) {
    CreateFileName(filename, prefixname, "degree_count.out");
    if (!(pDegreeFile = fopen(filename, "w"))) {
      printf ("Error: cannot open %s - Exiting.\n\n", filename);
      exit (0);
    }

    // print header line
    fprintf(pDegreeFile, "degree\tcount\n");
  }

  // open motifs output file
  if (bsave_the_motifs_probabilies) {
    CreateFileName(filename, prefixname, "motifs_count.out");
    if (!(pMotifFile = fopen(filename, "w"))) {
      printf ("Error: cannot open %s - Exiting.\n\n", filename);
      exit (0);
    }
  }

} /* InitOutputs */


/* ----------------------------------------------------------------------------
   SaveBestGraph

   Just do that.
*/
void SaveBestGraph (void)
{
  if (bsave_best_graph) {
    PrintiMatrix (pBestFile, nNodes, best_adj);
    fprintf(pBestFile, "Log(prior probability):     %g\n", dBestPrior);
    if (bBN && bData) {
      fprintf(pBestFile, "Log(likelihood):            %g\n", dBestLikelihood);
      fprintf(pBestFile, "Log(posterior probability): %g\n", dBestPosterior);
    }
  }
} /* SaveBestGraph */


/* ----------------------------------------------------------------------------
   SaveChain

   Write the full adjacency matrix at start or just "to_out" (which code for
   the difference and its location) after that.
*/
void SaveChain (int to_out)
{
  int i, j;
  
  if (iter > nBurnin - 1) {
    if (iter == nBurnin) {
      /* output the (eventually updated) current adjacency matrix
         going down the columns! */
      for (j = 0; j < nNodes; j++)  // jth column
        for (i = 0; i < nNodes; i++)  // ith line
          fwrite (&current_adj[i][j], sizeof(int), 1, pChainFile);
    }
    
    fwrite (&to_out, sizeof(int), 1, pChainFile);
  }

} /* SaveChain */


/* ----------------------------------------------------------------------------
   SaveDegreeCounts

   Just do that.
*/
void SaveDegreeCounts (void)
{
  long i;

  if (bsave_the_degree_counts) {
    for (i = 0; i < (nNodes+nNodes); i++) {
      fprintf(pDegreeFile, "%ld\t%lg\n", i, cumdegree_count[i]);
    }
  }
} /* SaveDegreeCounts */


/* ----------------------------------------------------------------------------
   SaveEdgeP

   Just do that.
*/
void SaveEdgeP (FILE * outFile)
{
  int i, j;

  if (bsave_the_edge_probabilies) {

    // print header line
    for (i = 0; i < nNodes-1; i++) {
      fprintf(outFile, "%d\t", i+1);
    }
    fprintf(outFile, "%d\n", nNodes);

    for (i = 0; i < nNodes; i++) {
      for (j = 0; j < nNodes-1; j++) {
        fprintf(outFile, "%lg\t", mat_sum[i][j] /
                                  (double) (nRuns - nBurnin));
      }
      fprintf(outFile, "%lg\n", mat_sum[i][nNodes-1] /
                                (double) (nRuns - nBurnin));
    }
  }

} /* SaveEdgeP */


/* ----------------------------------------------------------------------------
   SaveGraph

   Just do that.
*/
void SaveGraph (void)
{
  if (bsave_some_graphs && (iter == dNextPrint)) {
    PrintiMatrix (pGraphsFile, nNodes, current_adj);
    fprintf(pGraphsFile, "Log(prior probability):     %g\n", 
            current_logprior);
    if (bBN && bData) {
      fprintf(pGraphsFile, "Log(likelihood):            %g\n", 
              current_loglikelihood);
      fprintf(pGraphsFile, "Log(posterior probability): %g\n\n", 
              current_logposterior);
    }

    dNextPrint += dPrintStep;
  }
} /* SaveGraph */


/* ----------------------------------------------------------------------------
   SaveMotifsP

   Just do that.
*/
void SaveMotifsP (FILE * outFile)
{
  if (bsave_the_motifs_probabilies) {
    fprintf(outFile, "Endless/Frustrated triangular loops:\n");
    fprintf(outFile, "E loops count: %lg\n", cum_nEloops);
    fprintf(outFile, "F loops count: %lg\n", cum_nFloops);

    if ((cum_nEloops != 0) || (cum_nFloops != 0))
      fprintf(outFile, "E / (E + F): %lg\n\n",
             cum_nEloops / (cum_nEloops+ cum_nFloops));
    else
      fprintf(outFile, "\n");
  }

} /* SaveMotifsP */


/* end */
