/*
 *  Graphical Turing Machine
 *
 *  Copyright (C) 2008 by Eric Hutchins
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **/

#include <cmath>

void drawLine(SDL_Surface* dest, int x1, int y1, int x2, int y2, int r, int g, int b)
{
  int minx = x1;
  int miny = y1;
  int maxx = x2;
  int maxy = y2;
  if(minx > maxx) {
    int temp = minx;
    minx = maxx;
    maxx = temp;
    temp = miny;
    miny = maxy;
    maxy = temp;
  }
  if(x1 == x2)
  {
    if(miny > maxy) {
      int temp = minx;
      minx = maxx;
      maxx = temp;
      temp = miny;
      miny = maxy;
      maxy = temp;
    }
    for(int i = miny; i <= maxy; i++)
    {
      SDL_Rect rect = {x1, i, 1, 1};
      SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, r, g, b));
    }
    return;
  }
  else if(y1 == y2)
  {
    for(int i = minx; i <= maxx; i++)
    {
      SDL_Rect rect = {i, y1, 1, 1};
      SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, r, g, b));
    }
    return;
  }
  double slope = ((double)maxy - (double)miny) / ((double)maxx - (double)minx);
  if(slope >= -1 && slope <= 1)
  {
    for(int i = minx; i <= maxx; i++)
    {
      SDL_Rect rect;
      rect.w = rect.h = 1;
      rect.x = i;
      rect.y = (int)(miny + (i - minx) * slope);
      SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, r, g, b));
    }
  }
  else
  {
    if(miny > maxy) {
      int temp = minx;
      minx = maxx;
      maxx = temp;
      temp = miny;
      miny = maxy;
      maxy = temp;
    }
    slope = ((double)maxx - (double)minx) / ((double)maxy - (double)miny);
    for(int i = miny; i <= maxy; i++)
    {
      SDL_Rect rect;
      rect.w = rect.h = 1;
      rect.y = i;
      rect.x = (int)(minx + (i - miny) * slope);
      SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, r, g, b));
    }
  }
}

void drawArrow(SDL_Surface* dest, int x1, int y1, int x2, int y2, int r, int g, int b, double thickness)
{
  int minx = x1;
  int miny = y1;
  int maxx = x2;
  int maxy = y2;
  if(minx > maxx) {
    int temp = minx;
    minx = maxx;
    maxx = temp;
    temp = miny;
    miny = maxy;
    maxy = temp;
  }
  if(x1 == x2)
  {
    if(miny > maxy) {
      int temp = minx;
      minx = maxx;
      maxx = temp;
      temp = miny;
      miny = maxy;
      maxy = temp;
    }
    for(int i = miny; i <= maxy; i++)
    {
      SDL_Rect rect = {x1, i, 1, 1};
      SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, r, g, b));
    }
    double starty = y2 * 0.75 + y1 * 0.25;
    drawLine(dest, (int)(x1 - thickness), (int)starty, x2, y2, 0, 0, 0);
    drawLine(dest, (int)(x1 + thickness), (int)starty, x2, y2, 0, 0, 0);
    drawLine(dest, (int)(x1 - thickness), (int)starty, 
                   (int)(x1 + thickness), (int)starty, 0, 0, 0);
    return;
  }
  else if(y1 == y2)
  {
    for(int i = minx; i <= maxx; i++)
    {
      SDL_Rect rect = {i, y1, 1, 1};
      SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, r, g, b));
      double startx = x2 * 0.75 + x1 * 0.25;
      drawLine(dest, (int)startx, (int)(y1 - thickness), x2, y2, 0, 0, 0);
      drawLine(dest, (int)startx, (int)(y1 + thickness), x2, y2, 0, 0, 0);
      drawLine(dest, (int)startx, (int)(y1 - thickness), 
                     (int)startx, (int)(y1 + thickness), 0, 0, 0);
    }
    return;
  }
  SDL_Rect rect;
  rect.w = rect.h = 1;
  double slope = ((double)maxy - (double)miny) / ((double)maxx - (double)minx);
  double inverseslope = -((double)maxx - (double)minx) / 
                         ((double)maxy - (double)miny);
  if(slope >= -1 && slope <= 1)
  {
    double d = sqrt(inverseslope * inverseslope + 1);
    double unity = inverseslope / d;
    double unitx = 1.0 / d;
    double startx = x1 * 0.25 + x2 * 0.75;
    double starty = y1 * 0.25 + y2 * 0.75;
    double cross1x = startx + unitx * thickness;
    double cross1y = starty + unity * thickness;
    double cross2x = startx - unitx * thickness;
    double cross2y = starty - unity * thickness;
    drawLine(dest, (int)cross1x, (int)cross1y, x2, y2, 0, 0, 0);
    drawLine(dest, (int)cross2x, (int)cross2y, x2, y2, 0, 0, 0);
    drawLine(dest, (int)cross1x, (int)cross1y, 
                   (int)cross2x, (int)cross2y, 0, 0, 0);
    for(int i = minx; i <= maxx; i++)
    {
      rect.x = i;
      rect.y = (int)(miny + (i - minx) * slope);
      SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, r, g, b));
    }
  }
  else
  {
    if(miny > maxy) {
      int temp = miny;
      miny = maxy;
      maxy = temp;
      temp = minx;
      minx = maxx;
      maxx = temp;
    }
    inverseslope = -((double)maxy - (double)miny) / 
                    ((double)maxx - (double)minx);
    double d = sqrt(inverseslope * inverseslope + 1);
    double unitx = inverseslope / d;
    double unity = 1.0 / d;
    double startx = x1 * 0.25 + x2 * 0.75;
    double starty = y1 * 0.25 + y2 * 0.75;
    double cross1x = startx + unitx * thickness;
    double cross1y = starty + unity * thickness;
    double cross2x = startx - unitx * thickness;
    double cross2y = starty - unity * thickness;
    drawLine(dest, (int)cross1x, (int)cross1y, x2, y2, 0, 0, 0);
    drawLine(dest, (int)cross2x, (int)cross2y, x2, y2, 0, 0, 0);
    drawLine(dest, (int)cross1x, (int)cross1y, 
                   (int)cross2x, (int)cross2y, 0, 0, 0);
    slope = ((double)maxx - (double)minx) / ((double)maxy - (double)miny);
    for(int i = miny; i < maxy; i++)
    {
      rect.y = i;
      rect.x = (int)(minx + (i - miny) * slope);
      SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, r, g, b));
    }
  }
}

void drawSemiCircle(SDL_Surface* dest, int x1, int y1, int x2, int y2, int r, int g, int b)
{
  double d = sqrt((x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2));
  double nx = (x2 - x1) / d;
  double ny = (y2 - y1) / d;
  double invnx = -ny;
  double invny = -nx;
  for(double angle = 0; angle < 3.14159265; angle += 0.1)
  {
    double x = x1 + cos(angle) * nx + sin(angle) * invnx;
    double y = y1 + cos(angle) * ny + sin(angle) * invny;
    double newx = x1 + cos(angle+0.1) * nx + sin(angle+0.1) * invnx;
    double newy = y1 + cos(angle+0.1) * ny + sin(angle+0.1) * invny;
    drawLine(dest, (int)x,(int)y,(int)newx,(int)newy,r,g,b);
  }
}


