/*
 *  Graphical Turing Machine
 *
 *  Copyright (C) 2008 by Eric Hutchins
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **/

#include "tm.h"

TM::TM()
{
}

TM::TM(string tm_file, int tape_count)
{
  m_accept = false;
  m_reject = false;
  for(int i = 0; i < tape_count; i++)
  {
    m_tape_drives.push_back(new TapeDrive());
  }
  m_state_set = new StateSet();
  loadTM(tm_file);
}

TM::~TM()
{
}

void TM::loadTM(string filename)
{
  ifstream file(filename.c_str());
  string buffer;
  getline(file,buffer);
  stringstream ss(buffer);
  string start_state_name;
  ss >> start_state_name;
  m_start_state = m_state_set->getState(start_state_name);

  m_current_state = m_start_state;

  int final_state_num;
  file >> final_state_num;
  for(int i = 0; i < final_state_num; i++)
  {
    string final_state_name;
    file >> final_state_name;

    m_final_state_set[final_state_name] = 
      m_state_set->getState(final_state_name);
  }
  getline(file, buffer);
  while(!file.eof())
  {
    string begin_state_name;
    vector<string> input_symbols;
    vector<string> output_symbols;
    vector<int> movement;
    string end_state_name;
    file >> begin_state_name;
    if(begin_state_name.compare("") == 0) break;
    if(begin_state_name[0] == '/' && begin_state_name[1] == '/') continue;
    for(int i = 0; i < m_tape_drives.size(); i++)
    {
      string buffer;
      file >> buffer;
      input_symbols.push_back(buffer);
    }
    for(int i = 0; i < m_tape_drives.size(); i++)
    {
      string buffer;
      file >> buffer;
      output_symbols.push_back(buffer);
    }
    for(int i = 0; i < m_tape_drives.size(); i++)
    {
      int buffer;
      file >> buffer;
      movement.push_back(buffer);
    }
    file >> end_state_name;
    State* begin_state = m_state_set->getState(begin_state_name);
    State* end_state = m_state_set->getState(end_state_name);
    string input = begin_state_name + " ";
    for(int i = 0; i < input_symbols.size(); i++)
    {
      input += input_symbols[i];
      if(i < input_symbols.size() - 1) input += " ";
    }
    m_delta[input] = new Transition(begin_state, input_symbols, 
                                    output_symbols, movement, end_state);
  }
  file.close();
}

void TM::loadInput(string filename)
{
  m_tape_drives[0]->clear();
  ifstream file(filename.c_str());
  while(!file.eof())
  {
    string input_symbol;
    file >> input_symbol;
    if(input_symbol.length() < 1) break;
    m_tape_drives[0]->write(input_symbol);
    m_tape_drives[0]->shift(1);
  }
  m_tape_drives[0]->reset();
  file.close();
}

void TM::loadInput(vector<string> filenames)
{
  for(int i = 0; i < filenames.size(); i++)
  {
    while(m_tape_drives.size() <= i) m_tape_drives.push_back(new TapeDrive());
    m_tape_drives[i]->clear();
    ifstream file(filenames[i].c_str());
    while(!file.eof())
    {
      string input_symbol;
      file >> input_symbol;
      if(input_symbol.length() < 1) break;
      m_tape_drives[i]->write(input_symbol);
      m_tape_drives[i]->shift(1);
    }
    m_tape_drives[i]->reset();
    file.close();
  }
}

void TM::print()
{
  if(m_tape_drives.size() == 1)
  {
    vector<string> head_tail = m_tape_drives[0]->getHeadTail();
    cout << head_tail[0];
    if(head_tail[0].length() > 0) cout << " ";
    cout << m_current_state->getName() << "> " << head_tail[1] << endl;
    return;
  }
  for(int i = 0; i < m_tape_drives.size(); i++)
  {
    cout << "Tape " << i << ": ";
    vector<string> head_tail = m_tape_drives[i]->getHeadTail();
    cout << head_tail[0];
    if(head_tail[0].length() > 0) cout << " ";
    cout << m_current_state->getName() << "> " << head_tail[1] << endl;
  }
  cout << endl;
}

void TM::clearAcceptReject()
{
  m_accept = false;
  m_reject = false;
}

bool TM::cycle()
{
  string input = "" + m_current_state->getName() + " ";
  for(int i = 0; i < m_tape_drives.size(); i++)
  {
    input += m_tape_drives[i]->read();
    if(i < m_tape_drives.size() - 1) input += " ";
  }
  Transition* t = m_delta[input];
  if(t == NULL)
  {
    m_reject = true;
    return false;
  }
  vector<string> output_symbols = t->getOutputSymbols();
  vector<int> movement = t->getMovement();
  for(int i = 0; i < m_tape_drives.size(); i++)
  {
    m_tape_drives[i]->write(output_symbols[i]);
    m_tape_drives[i]->shift(movement[i]);
  }
  m_current_state = t->getEndState();
  if(isInFinalState()) m_accept = true;
  return true;
}

void TM::printInfo()
{
  cout << "State-transition map: {";
  for(map<string,Transition*>::iterator m = m_delta.begin(); 
      m != m_delta.end(); 
      m++)
  {
    Transition* t = m->second;
    vector<string> input_symbols = t->getInputSymbols();
    vector<string> output_symbols = t->getOutputSymbols();
    vector<int> movement = t->getMovement();
    cout << "( " << t->getBeginState()->getName() << " ";
    for(int i = 0; i < input_symbols.size(); i++)
    {
      cout << input_symbols[i];
      if(i < input_symbols.size() - 1) cout << " ";
    }
    cout << " )=( ";
    for(int i = 0; i < output_symbols.size(); i++)
    {
      cout << output_symbols[i] << " ";
    }
    for(int i = 0; i < movement.size(); i++)
    {
      cout << movement[i] << " ";
    }
    cout << t->getEndState()->getName() << " )";
    map<string,Transition*>::iterator next = m;
    next++;
    if(next != m_delta.end()) cout << ", ";
    else cout << "}" << endl;
  }
  cout << "Initial state: " << m_start_state->getName() << endl;
  cout << "Final state set: [";
  for(map<string,State*>::iterator m = m_final_state_set.begin(); 
      m != m_final_state_set.end(); 
      m++)
  {
    cout << m->second->getName();
    map<string,State*>::iterator next = m;
    next++;
    if(next != m_final_state_set.end()) cout << ", ";
    else cout << "]" << endl;
  }
  if(m_tape_drives.size() == 1)
  {
    cout << "Initial tape: " << m_tape_drives[0]->getHeadTail()[1] << endl << endl;
  }
  else
  {
    cout << "Initial tapes: " << endl;
    for(int i = 0; i < m_tape_drives.size(); i++)
    {
      cout << m_tape_drives[i]->getHeadTail()[1] << endl;
    }
  }
}

