/*
 *  Graphical Turing Machine
 *
 *  Copyright (C) 2008 by Eric Hutchins
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **/

#include <iostream>
#include <sstream>
#include <string>
#include <cstdio>
using namespace std;

#include "SDL.h"
#include "SDL_ttf.h"

#include "graphicaltm.h"

// checks to see if the expected files exist and returns false if any 
// do not
bool check_files(string tm_file, vector<string> tape_files)
{
  if(!ifstream(tm_file.c_str()).is_open())
  {
    cout << "\"" << tm_file << "\" not found!" << endl;
    return false;
  }
  for(int i = 0; i < tape_files.size(); i++)
  {
    if(!ifstream(tape_files[i].c_str()).is_open())
    {
      cout << "\"" << tape_files[i] << "\" not found!" << endl;
      return false;
    }
  }
  return true;
}

int main(int argc, char** argv) {
  if(argc < 3)
  {
    cout << "usage: graphicaltm tmFile numberOfTapes tapeFile1 [tapeFile2] [tapeFile3] . . ." << endl;
    return 1;
  }

  int tape_count;
  if(argc >= 4)
  {
    stringstream ss(string(argv[2]),stringstream::in|stringstream::out);
    ss >> tape_count;
  }

  //cout << "there " << ((tape_count == 1)?("is"):("are")) << tape_count << "tape" << ((tape_count == 1)?(""):("s")) << endl;

  // get the filenames from the arguments
  string tm_file = argv[1];
  vector<string> tape_files;
  for(int i = 0; i < tape_count; i++)
  {
    tape_files.push_back(string(argv[3+i]));
  }

  // check to see that all the files exist
  if(!check_files(tm_file, tape_files)) return false;

  // initialize SDL
  if(SDL_Init(SDL_INIT_VIDEO) < 0)
  {
    cout << "Failed to initialize SDL" << endl;
    exit(1);
  }

  // initialize SDL_ttf
  if(TTF_Init() < 0)
  {
    cout << "Failed to initialize SDL_ttf" << endl;
    exit(1);
  }

  // create the font FreeSans to use for state labels, the tape, etc.
  string font_location = string(PREFIX) + "/share/graphicaltm/FreeSans.ttf";
  TTF_Font* font = TTF_OpenFont(font_location.c_str(), 16);
  if(!font)
  {
    cout << "Failed to open font: " << TTF_GetError() << endl;
  }

  // set the video mode and get the screen surface
  SDL_Surface* screen = SDL_SetVideoMode(640,480,0,SDL_DOUBLEBUF);
  
  // initialize the random number generator for randomizing 
  // the state positions
  srand(SDL_GetTicks());

  // set the "held" state to NULL
  // the "held" state is the state being held by the mouse.  
  // The user may click and drag states to obtain a more
  // attractive state layout
  GraphicalState* heldState = NULL;

  // create the Turing Machine from the file
  GraphicalTM* tm = new GraphicalTM(tm_file, tape_count);

  // load the tape into the machine
  tm->loadInput(tape_files);

  // print the info about the states, transitions, etc.
  tm->printInfo();

  // start_time is the initial pause before starting to run
  // it allows the user to see the layout for a bit before
  // things start moving
  int start_time = 1000;

  // frame_time is the time between each frame
  int frame_time = 50;
  
  // next_frame starts as the current time plus the start time
  long next_frame = SDL_GetTicks() + start_time;
  
  // draw the Turing Machine to the screen using the premade font
  tm->draw(screen, font);
  
  // initialize running to true and halted to false
  bool running = true;
  bool halted = false;

  // main loop
  while(running)
  {
    // if it did not halt yet, run
    if(!halted)
    {
      // check that it isn't in a final state
      if(!tm->isInFinalState())
      {
        // wait until the time is right to draw the next frame
        while(SDL_GetTicks() < next_frame);

	// set the next frame to occur at the right time
        next_frame = SDL_GetTicks() + frame_time;

	// print the tm's terminal output
        tm->print();

	// cycle
        if(!tm->cycle())
        {
          // there was no transition for the current state and input

          // set halted to true and write the terminal output
          halted = true;
          cout << "balk" << endl;
        }
      }
      else
      {
        // tm is in a final state - accepted

        // set halted to true
        halted = true;

        // print to terminal
        tm->print();
      }
    }

    // draw the tm
    tm->draw(screen, font);

    // event loop
    SDL_Event event;
    // while there is an event on the queue
    while(SDL_PollEvent(&event))
    {
      switch(event.type)
      {
      case SDL_KEYDOWN:
        // press ESC to exit
        if(event.key.keysym.sym == SDLK_ESCAPE) running = false;
        
        // press R to replay
        if(event.key.keysym.sym == SDLK_r)
        {
          // set halted to false
          halted = false;
          // tm hasn't accepted or rejected
	  tm->clearAcceptReject();
          // reload the tape
          tm->loadInput(tape_files);
          // put the state back to the start
          tm->moveToStartState();
        }
        break;
      case SDL_QUIT:
        // exit for standard quit events
	// (pressing the X on the window, key shortcuts, etc.)
        running = false;
        break;
      case SDL_MOUSEBUTTONDOWN:
        // if the left mouse button is pressed
        if(event.button.button == SDL_BUTTON_LEFT)
        {
          // set the held state to the state at the position
          heldState = tm->getStateAtPos(event.button.x,event.button.y);
        }
        break;
      case SDL_MOUSEBUTTONUP:
        // if the left mouse button is released
        if(event.button.button == SDL_BUTTON_LEFT)
        {
          // free the held state
          heldState = NULL;
        }
        break;
      case SDL_MOUSEMOTION:
        // if a state is being held then move it to the current mouse pos
        if(heldState != NULL)
          heldState->moveTo(event.motion.x, event.motion.y);
        break;
      }
    }
  }

  // quit if running becomes false and we get here
  SDL_Quit();
}

