/*
 *  Graphical Turing Machine
 *
 *  Copyright (C) 2008 by Eric Hutchins
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **/

#include "graphicaltm.h"
#include "utilities.h"

GraphicalTM::GraphicalTM(string tm_file, int tape_count):TM()
{
  m_accept = false;
  m_reject = false;
  for(int i = 0; i < tape_count; i++)
  {
    m_tape_drives.push_back(new TapeDrive());
  }
  m_state_set = new GraphicalStateSet();
  loadTM(tm_file);
}

GraphicalTM::~GraphicalTM()
{
}

void GraphicalTM::loadTM(string filename)
{
  ifstream file(filename.c_str());
  string buffer;
  getline(file,buffer);
  stringstream ss(buffer);
  string start_state_name;
  ss >> start_state_name;
  m_start_state = ((GraphicalStateSet*)m_state_set)->getGraphicalState(start_state_name);

  m_current_state = m_start_state;

  int final_state_num;
  file >> final_state_num;
  for(int i = 0; i < final_state_num; i++)
  {
    string final_state_name;
    file >> final_state_name;

    m_final_state_set[final_state_name] = 
      ((GraphicalStateSet*)m_state_set)->getGraphicalState(final_state_name);
  }
  getline(file, buffer);
  while(!file.eof())
  {
    string begin_state_name;
    vector<string> input_symbols;
    vector<string> output_symbols;
    vector<int> movement;
    string end_state_name;
    file >> begin_state_name;
    if(begin_state_name.compare("") == 0) break;
    if(begin_state_name[0] == '/' && begin_state_name[1] == '/') continue;

    for(int i = 0; i < m_tape_drives.size(); i++)
    {
      string buffer;
      file >> buffer;
      input_symbols.push_back(buffer);
    }

    for(int i = 0; i < m_tape_drives.size(); i++)
    {
      string buffer;
      file >> buffer;
      output_symbols.push_back(buffer);
    }

    for(int i = 0; i < m_tape_drives.size(); i++)
    {
      int buffer;
      file >> buffer;
      movement.push_back(buffer);
    }

    file >> end_state_name;

    State* begin_state = 
        ((GraphicalStateSet*)m_state_set)->getGraphicalState(begin_state_name);

    State* end_state = 
        ((GraphicalStateSet*)m_state_set)->getGraphicalState(end_state_name);

    string input = begin_state_name + " ";
    for(int i = 0; i < input_symbols.size(); i++)
    {
      input += input_symbols[i];
      if(i < input_symbols.size() - 1) input += " ";
    }

    m_delta[input] = new Transition((State*)begin_state, input_symbols, 
                                    output_symbols, movement, 
                                    (State*)end_state);
  }
  file.close();
}

void GraphicalTM::loadInput(string filename)
{
  TM::loadInput(filename);
}

void GraphicalTM::loadInput(vector<string> filenames)
{
  TM::loadInput(filenames);
}

void GraphicalTM::print()
{
  TM::print();
}

GraphicalState* GraphicalTM::getStateAtPos(int x, int y)
{
  return ((GraphicalStateSet*)m_state_set)->getStateAtPos(x,y);
}

void GraphicalTM::draw(SDL_Surface* dest, TTF_Font* font)
{
  // clear screen to light gray
  SDL_FillRect(dest,NULL,SDL_MapRGB(dest->format,192,192,192));

  // draw light blue rectangle on top
  SDL_Rect tape_rect = {0,0,640,80};
  tape_rect.h = 20 + 50 * m_tape_drives.size();
  SDL_FillRect(dest,&tape_rect,SDL_MapRGB(dest->format,160,160,192));

  // draw the states
  map<string,State*> state_set = m_state_set->getStates();
  for(map<string,State*>::iterator m = state_set.begin(); m != state_set.end(); m++)
  {
    ((GraphicalState*)m->second)->draw(dest, font);
  }

  // draw the transitions
  for(map<string,Transition*>::iterator m = m_delta.begin(); m != m_delta.end(); m++)
  {
    Transition* t = m->second;
    if(t == NULL) continue;
    GraphicalState* b = (GraphicalState*)t->getBeginState();
    GraphicalState* e = (GraphicalState*)t->getEndState();
    double dist = sqrt((b->getX() - e->getX()) * (b->getX() - e->getX())
                       + (b->getY() - e->getY()) * (b->getY() - e->getY()));
    double nx = (e->getX() - b->getX()) / dist;
    double ny = (e->getY() - b->getY()) / dist;
    drawArrow(dest, (int)(b->getX() + nx * b->getSize() / 2), 
                    (int)(b->getY() + ny * b->getSize() / 2), 
                    (int)(e->getX() - nx * e->getSize() / 2), 
                    (int)(e->getY() - ny * e->getSize() / 2), 0, 0, 0, 10);
    double newx = b->getX() * 0.25 + e->getX() * 0.75;
    double newy = b->getY() * 0.25 + e->getY() * 0.75;
    newx = b->getX() * 0.5 + e->getX() * 0.5;
    newy = b->getY() * 0.5 + e->getY() * 0.5;
    string strLabel = "";
    vector<string> input_symbols = t->getInputSymbols();
    vector<string> output_symbols = t->getOutputSymbols();
    vector<int> movement = t->getMovement();
    for(int i = 0; i < input_symbols.size(); i++)
    {
      strLabel += input_symbols[i] + " ";
      //if(i < input_symbols.size() - 1) strLabel += " ";
    }
    strLabel += "->";
    for(int i = 0; i < output_symbols.size(); i++)
    {
      strLabel += output_symbols[i];
      if(i < output_symbols.size() - 1) strLabel += " ";
    }
    strLabel += ",";
    for(int i = 0; i < movement.size(); i++)
    {
      stringstream ss(stringstream::in|stringstream::out);
      ss << movement[i];
      strLabel += ss.str();
      if(i < movement.size() - 1) strLabel += " ";
    }
    SDL_Color color = {0,0,0};
    SDL_Surface* label = TTF_RenderText_Blended(font, strLabel.c_str(), color);
    if(label)
    {
      SDL_Rect rect = {(int)newx, (int)newy, 0, 0};
      SDL_BlitSurface(label, NULL, dest, &rect);
      SDL_FreeSurface(label);
    }
  }
  ((GraphicalState*)m_current_state)->drawLit(dest, font);

  // Draw Tape Drives
  for(int k = 0; k < m_tape_drives.size(); k++)
  {
    vector<string> tape = m_tape_drives[k]->getTape();
    for(int i = 0; i < tape.size(); i++)
    {
      for(int j = 0; j < 8; j++)
      {
        SDL_Rect rect;
        rect.x = 20 + i * 24 - j;
        rect.y = 20 - j + 50 * k;
        rect.w = 18;
        rect.h = 18;
        SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, j * 32, j * 31, j * 24));
      }
      SDL_Color color = {0,0,0};
      SDL_Surface* label = TTF_RenderText_Blended(font, tape[i].c_str(), color);
      if(label)
      {
        SDL_Rect rect = {24 + i * 24 - 7, 16 - 7 + 50*k, 0, 0};
        SDL_BlitSurface(label, NULL, dest, &rect);
        SDL_FreeSurface(label);
      }
    }
  }

  // draw the arrow to mark the tape heads
  for(int i = 0; i < m_tape_drives.size(); i++)
  {
    drawArrow(dest, 22 + m_tape_drives[i]->getPos() * 24, 60 + i*50, 
                    22 + m_tape_drives[i]->getPos() * 24, 40 + i*50, 0, 0, 0, 5);
  }

  SDL_Rect rect = {500,400,25,25};
  SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, 64, 64, 64));
  SDL_Color color = {0,0,0};
  SDL_Surface* label = TTF_RenderText_Blended(font, "Accept", color);
  if(label)
  {
    SDL_Rect rect = {500, 375, 0, 0};
    SDL_BlitSurface(label, NULL, dest, &rect);
    SDL_FreeSurface(label);
  }

  rect.x += 80;
  SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, 64, 64, 64));
  label = TTF_RenderText_Blended(font, "Reject", color);
  if(label)
  {
    SDL_Rect rect = {580, 375, 0, 0};
    SDL_BlitSurface(label, NULL, dest, &rect);
    SDL_FreeSurface(label);
  }

  if(m_accept)
  {
    rect.x = 505;
    rect.y = 405;
    rect.w = 15;
    rect.h = 15;
    SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, 64, 224, 64));
  }

  if(m_reject)
  {
    rect.x = 585;
    rect.y = 405;
    rect.w = 15;
    rect.h = 15;
    SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, 224, 64, 64));
  }

  SDL_Flip(dest);
}

bool GraphicalTM::cycle()
{
  TM::cycle();
}

