/*
 *  Graphical Turing Machine
 *
 *  Copyright (C) 2008 by Eric Hutchins
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **/

#include "tm.h"

TM::TM()
{
}

TM::TM(string tm_file)
{
  m_accept = false;
  m_reject = false;
  m_tape_drive = new TapeDrive();
  m_state_set = new StateSet();
  loadTM(tm_file);
}

TM::~TM()
{
}

void TM::loadTM(string filename)
{
  ifstream file(filename.c_str());
  string buffer;
  getline(file,buffer);
  stringstream ss(buffer);
  string start_state_name;
  ss >> start_state_name;
  m_start_state = m_state_set->getState(start_state_name);

  m_current_state = m_start_state;

  int final_state_num;
  file >> final_state_num;
  for(int i = 0; i < final_state_num; i++)
  {
    string final_state_name;
    file >> final_state_name;

    m_final_state_set[final_state_name] = 
      m_state_set->getState(final_state_name);
  }
  getline(file, buffer);
  while(!file.eof())
  {
    string begin_state_name;
    string input_symbol;
    string output_symbol;
    int movement;
    string end_state_name;
    file >> begin_state_name;
    if(begin_state_name.compare("") == 0) break;
    if(begin_state_name[0] == '/' && begin_state_name[1] == '/') continue;
    file >> input_symbol;
    file >> output_symbol;
    file >> movement;
    file >> end_state_name;
    State* begin_state = m_state_set->getState(begin_state_name);
    State* end_state = m_state_set->getState(end_state_name);
    string input = begin_state_name + " " + input_symbol;
    m_delta[input] = new Transition(begin_state, input_symbol, 
                                    output_symbol, movement, end_state);
  }
  file.close();
}

void TM::loadInput(string filename)
{
  m_tape_drive->clear();
  ifstream file(filename.c_str());
  while(!file.eof())
  {
    string input_symbol;
    file >> input_symbol;
    if(input_symbol.length() < 1) break;
    m_tape_drive->write(input_symbol);
    m_tape_drive->shift(1);
  }
  m_tape_drive->reset();
  file.close();
}

void TM::print()
{
  vector<string> head_tail = m_tape_drive->getHeadTail();
  cout << head_tail[0];
  if(head_tail[0].length() > 0) cout << " ";
  cout << m_current_state->getName() << "> " << head_tail[1] << endl;
}

void TM::clearAcceptReject()
{
  m_accept = false;
  m_reject = false;
}

bool TM::cycle()
{
  string input = "" + m_current_state->getName() + " " + m_tape_drive->read();
  Transition* t = m_delta[input];
  if(t == NULL)
  {
    m_reject = true;
    return false;
  }
  m_tape_drive->write(t->getOutputSymbol());
  m_tape_drive->shift(t->getMovement());
  m_current_state = t->getEndState();
  if(isInFinalState()) m_accept = true;
  return true;
}

void TM::printInfo()
{
  cout << "State-transition map: {";
  for(map<string,Transition*>::iterator m = m_delta.begin(); 
      m != m_delta.end(); 
      m++)
  {
    Transition* t = m->second;
    cout << "( " << t->getBeginState()->getName() << " " << 
	    t->getInputSymbol() << " )=( " << t->getOutputSymbol() << " " << 
	    t->getMovement() << " " << t->getEndState()->getName() << " )";
    map<string,Transition*>::iterator next = m;
    next++;
    if(next != m_delta.end()) cout << ", ";
    else cout << "}" << endl;
  }
  cout << "Initial state: " << m_start_state->getName() << endl;
  cout << "Final state set: [";
  for(map<string,State*>::iterator m = m_final_state_set.begin(); 
      m != m_final_state_set.end(); 
      m++)
  {
    cout << m->second->getName();
    map<string,State*>::iterator next = m;
    next++;
    if(next != m_final_state_set.end()) cout << ", ";
    else cout << "]" << endl;
  }
  cout << "Initial tape: " << m_tape_drive->getHeadTail()[1] << endl << endl;
}

