/*
 *  Graphical Turing Machine
 *
 *  Copyright (C) 2008 by Eric Hutchins
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **/

#include "graphicaltm.h"
#include "utilities.h"

GraphicalTM::GraphicalTM(string tm_file):TM()
{
  m_accept = false;
  m_reject = false;
  m_tape_drive = new TapeDrive();
  m_state_set = new GraphicalStateSet();
  loadTM(tm_file);
}

GraphicalTM::~GraphicalTM()
{
}

void GraphicalTM::loadTM(string filename)
{
  ifstream file(filename.c_str());
  string buffer;
  getline(file,buffer);
  stringstream ss(buffer);
  string start_state_name;
  ss >> start_state_name;
  m_start_state = ((GraphicalStateSet*)m_state_set)->getGraphicalState(start_state_name);

  m_current_state = m_start_state;

  int final_state_num;
  file >> final_state_num;
  for(int i = 0; i < final_state_num; i++)
  {
    string final_state_name;
    file >> final_state_name;

    m_final_state_set[final_state_name] = 
      ((GraphicalStateSet*)m_state_set)->getGraphicalState(final_state_name);
  }
  getline(file, buffer);
  while(!file.eof())
  {
    string begin_state_name;
    string input_symbol;
    string output_symbol;
    int movement;
    string end_state_name;
    file >> begin_state_name;
    if(begin_state_name.compare("") == 0) break;
    if(begin_state_name[0] == '/' && begin_state_name[1] == '/') continue;
    file >> input_symbol;
    file >> output_symbol;
    file >> movement;
    file >> end_state_name;
    State* begin_state = ((GraphicalStateSet*)m_state_set)->getGraphicalState(begin_state_name);
    State* end_state = ((GraphicalStateSet*)m_state_set)->getGraphicalState(end_state_name);
    string input = begin_state_name + " " + input_symbol;
    m_delta[input] = new Transition((State*)begin_state, input_symbol, 
                                    output_symbol, movement, (State*)end_state);
  }
  file.close();
}

void GraphicalTM::loadInput(string filename)
{
  TM::loadInput(filename);
}

void GraphicalTM::print()
{
  TM::print();
}

GraphicalState* GraphicalTM::getStateAtPos(int x, int y)
{
  return ((GraphicalStateSet*)m_state_set)->getStateAtPos(x,y);
}

void GraphicalTM::draw(SDL_Surface* dest, TTF_Font* font)
{
  // clear screen to light gray
  SDL_FillRect(dest,NULL,SDL_MapRGB(dest->format,192,192,192));

  // draw light blue rectangle on top
  SDL_Rect tape_rect = {0,0,640,80};
  SDL_FillRect(dest,&tape_rect,SDL_MapRGB(dest->format,160,160,192));

  // draw the states
  map<string,State*> state_set = m_state_set->getStates();
  for(map<string,State*>::iterator m = state_set.begin(); m != state_set.end(); m++)
  {
    ((GraphicalState*)m->second)->draw(dest, font);
  }

  // draw the transitions
  for(map<string,Transition*>::iterator m = m_delta.begin(); m != m_delta.end(); m++)
  {
    Transition* t = m->second;
    if(t == NULL) continue;
    GraphicalState* b = (GraphicalState*)t->getBeginState();
    GraphicalState* e = (GraphicalState*)t->getEndState();
    double dist = sqrt((b->getX() - e->getX()) * (b->getX() - e->getX())
                       + (b->getY() - e->getY()) * (b->getY() - e->getY()));
    double nx = (e->getX() - b->getX()) / dist;
    double ny = (e->getY() - b->getY()) / dist;
    drawArrow(dest, (int)(b->getX() + nx * b->getSize() / 2), 
                    (int)(b->getY() + ny * b->getSize() / 2), 
                    (int)(e->getX() - nx * e->getSize() / 2), 
                    (int)(e->getY() - ny * e->getSize() / 2), 0, 0, 0, 10);
    double newx = b->getX() * 0.25 + e->getX() * 0.75;
    double newy = b->getY() * 0.25 + e->getY() * 0.75;
    newx = b->getX() * 0.5 + e->getX() * 0.5;
    newy = b->getY() * 0.5 + e->getY() * 0.5;
    string strLabel = "";
    strLabel += t->getInputSymbol();
    strLabel += "->";
    strLabel += t->getOutputSymbol();
    strLabel += ",";
    stringstream ss(stringstream::in|stringstream::out);
    ss << t->getMovement();
    strLabel += ss.str();
    SDL_Color color = {0,0,0};
    SDL_Surface* label = TTF_RenderText_Blended(font, strLabel.c_str(), color);
    if(label)
    {
      SDL_Rect rect = {(int)newx, (int)newy, 0, 0};
      SDL_BlitSurface(label, NULL, dest, &rect);
      SDL_FreeSurface(label);
    }
  }
  ((GraphicalState*)m_current_state)->drawLit(dest, font);

  // Draw Tape Drive
  vector<string> tape = m_tape_drive->getTape();
  for(int i = 0; i < tape.size(); i++)
  {
    for(int j = 0; j < 8; j++)
    {
      SDL_Rect rect;
      rect.x = 20 + i * 24 - j;
      rect.y = 20 - j;
      rect.w = 18;
      rect.h = 18;
      SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, j * 32, j * 31, j * 24));
    }
    SDL_Color color = {0,0,0};
    SDL_Surface* label = TTF_RenderText_Blended(font, tape[i].c_str(), color);
    if(label)
    {
      SDL_Rect rect = {24 + i * 24 - 7, 16 - 7, 0, 0};
      SDL_BlitSurface(label, NULL, dest, &rect);
      SDL_FreeSurface(label);
    }
  }

  // draw the arrow to mark the tape head
  drawArrow(dest, 22 + m_tape_drive->getPos() * 24, 70, 
                  22 + m_tape_drive->getPos() * 24, 40, 0, 0, 0, 5);

  SDL_Rect rect = {500,400,25,25};
  SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, 64, 64, 64));
  SDL_Color color = {0,0,0};
  SDL_Surface* label = TTF_RenderText_Blended(font, "Accept", color);
  if(label)
  {
    SDL_Rect rect = {500, 375, 0, 0};
    SDL_BlitSurface(label, NULL, dest, &rect);
    SDL_FreeSurface(label);
  }

  rect.x += 80;
  SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, 64, 64, 64));
  label = TTF_RenderText_Blended(font, "Reject", color);
  if(label)
  {
    SDL_Rect rect = {580, 375, 0, 0};
    SDL_BlitSurface(label, NULL, dest, &rect);
    SDL_FreeSurface(label);
  }

  if(m_accept)
  {
    rect.x = 505;
    rect.y = 405;
    rect.w = 15;
    rect.h = 15;
    SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, 64, 224, 64));
  }

  if(m_reject)
  {
    rect.x = 585;
    rect.y = 405;
    rect.w = 15;
    rect.h = 15;
    SDL_FillRect(dest, &rect, SDL_MapRGB(dest->format, 224, 64, 64));
  }

  SDL_Flip(dest);
}

bool GraphicalTM::cycle()
{
  TM::cycle();
}

