/*
 * Parser.cs
 *
 * This work is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * As a special exception, the copyright holders of this library give
 * you permission to link this library with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also meet,
 * for each linked independent module, the terms and conditions of the
 * license of that module. An independent module is a module which is
 * not derived from or based on this library. If you modify this
 * library, you may extend this exception to your version of the
 * library, but you are not obligated to do so. If you do not wish to
 * do so, delete this exception statement from your version.
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

using System.Collections;

namespace PerCederberg.Grammatica.Parser {

    /**
     * A base parser class. This class provides the standard parser 
     * interface, as well as token handling.
     *
     * @author   Per Cederberg, <per at percederberg dot net>
     * @version  1.0
     */
    public abstract class Parser {

        /**
         * The tokenizer to use.
         */
        private Tokenizer tokenizer;

        /**
         * The analyzer to use for callbacks.
         */
        private Analyzer analyzer;

        /**
         * The start production pattern id. This will be set to the
         * first production pattern added.
         */
        private int start = 0;

        /**
         * The map with all production patterns. The production
         * patterns are indexed by their id.
         */
        private Hashtable patterns = new Hashtable();

        /**
         * The list of buffered tokens. This list will contain tokens
         * that have been read from the tokenizer, but not yet
         * consumed.
         */
        private ArrayList tokens = new ArrayList();

        /**
         * Creates a new parser.
         * 
         * @param tokenizer       the tokenizer to use
         */
        internal Parser(Tokenizer tokenizer) 
            : this(tokenizer, new Analyzer()) {
        }

        /**
         * Creates a new parser.
         * 
         * @param tokenizer       the tokenizer to use
         * @param analyzer        the analyzer callback to use
         */
        internal Parser(Tokenizer tokenizer, Analyzer analyzer) {
            this.tokenizer = tokenizer;
            this.analyzer = analyzer;
        }

        /**
         * Adds a new production pattern to the parser. The first pattern 
         * added is assumed to be the starting point in the grammar. The 
         * patterns added may be validated to some extent.
         * 
         * @param pattern        the pattern to add
         * 
         * @throws ParserCreationException if the pattern couldn't be 
         *             added correctly to the parser
         */
        public virtual void AddPattern(ProductionPattern pattern) {
            if (pattern.GetAlternativeCount() <= 0) {
                throw new ParserCreationException(
                    ParserCreationException.ErrorType.INVALID_PRODUCTION,
                    pattern.GetName(),
                    "no production alternatives are present (must have at " +
                    "least one)");
            }
            if (patterns.Count == 0) {
                start = pattern.GetId();
            }
            if (patterns.ContainsKey(pattern.GetId())) {
                throw new ParserCreationException(
                    ParserCreationException.ErrorType.INVALID_PRODUCTION,
                    pattern.GetName(),
                    "another pattern with the same id (" + pattern.GetId() + 
                    ") has already been added");
            }
            patterns.Add(pattern.GetId(), pattern);
        }

        /**
         * Initializes the parser. All the added production patterns
         * will be analyzed for ambiguities and errors. This method
         * also initializes internal data structures used during the
         * parsing.
         * 
         * @throws ParserCreationException if the parser couldn't be 
         *             initialized correctly 
         */
        public virtual void Prepare() {
            IDictionaryEnumerator  e;
            
            if (patterns.Count == 0) {
                throw new ParserCreationException(
                    ParserCreationException.ErrorType.INVALID_PARSER,
                    "no production patterns have been added");
            }
            e = patterns.GetEnumerator();
            while (e.MoveNext()) {
                CheckPattern((ProductionPattern) e.Value);
            }
        }

        /**
         * Checks a production pattern for completeness. If some rule
         * in the pattern referenced an production pattern not added
         * to this parser, a parser creation exception will be thrown.
         * 
         * @param pattern        the production pattern to check
         * 
         * @throws ParserCreationException if the pattern referenced a 
         *             pattern not added to this parser
         */
        private void CheckPattern(ProductionPattern pattern) {
            for (int i = 0; i < pattern.GetAlternativeCount(); i++) {
                CheckRule(pattern.GetName(), pattern.GetAlternative(i));     
            }
        }

        /**
         * Checks a production pattern rule for completeness. If some
         * element in the rule referenced an production pattern not
         * added to this parser, a parser creation exception will be
         * thrown.
         *
         * @param name           the name of the pattern being checked 
         * @param rule           the production pattern rule to check
         * 
         * @throws ParserCreationException if the rule referenced a 
         *             pattern not added to this parser
         */
        private void CheckRule(string name, 
                               ProductionPatternAlternative rule) {
            
            for (int i = 0; i < rule.GetElementCount(); i++) {
                CheckElement(name, rule.GetElement(i));
            }
        }

        /**
         * Checks a production pattern element for completeness. If
         * the element references a production pattern not added to
         * this parser, a parser creation exception will be thrown.
         * 
         * @param name           the name of the pattern being checked 
         * @param elem           the production pattern element to check
         * 
         * @throws ParserCreationException if the element referenced a
         *             pattern not added to this parser
         */
        private void CheckElement(string name, 
                                  ProductionPatternElement elem) {

            if (elem.IsProduction() && GetPattern(elem.GetId()) == null) {
                throw new ParserCreationException(
                    ParserCreationException.ErrorType.INVALID_PRODUCTION,
                    name,
                    "an undefined production pattern id (" + elem.GetId() +
                    ") is referenced");
            }
        }

        /**
         * Parses the token stream and returns a parse tree. This method
         * will call prepare() if not previously called.
         * 
         * @return the parse tree
         * 
         * @throws ParserCreationException if the parser couldn't be
         *             initialized correctly
         * @throws ParseException if the input couldn't be parsed 
         *             correctly
         * 
         * @see #prepare
         */
        public abstract Node Parse();

        /**
         * Returns the production pattern with the specified id.
         *  
         * @param id             the production pattern id
         * 
         * @return the production pattern found, or
         *         null if non-existent
         */
        internal ProductionPattern GetPattern(int id) {
            return (ProductionPattern) patterns[id];
        }

        /**
         * Returns the production pattern for the starting production.
         *  
         * @return the start production pattern, or
         *         null if no patterns have been added
         */
        internal ProductionPattern GetStartPattern() {
            return GetPattern(start);
        }

        /**
         * Returns the unordered set of production patterns.
         * 
         * @return the unordered set of production patterns
         */
        internal ICollection GetPatterns() {
            return patterns.Values;
        }

        /**
         * Handles the parser entering a production. This method calls
         * the appropriate analyzer callback if the node is not
         * hidden.
         * 
         * @param node            the parse tree node
         * 
         * @throws ParseException if the node couldn't be handled 
         *             correctly
         */
        internal void EnterNode(Node node) {
            if (!node.IsHidden()) {
                analyzer.Enter(node);
            }
        }
    
        /**
         * Handles the parser leaving a production. This method calls
         * the appropriate analyzer callback if the node is not
         * hidden, and returns the result.
         * 
         * @param node           the parse tree node
         * 
         * @return the parse tree node, or
         *         null if no parse tree should be created
         * 
         * @throws ParseException if the node couldn't be handled 
         *             correctly
         */
        internal Node ExitNode(Node node) {
            if (node.IsHidden()) {
                return node;
            } else {
                return analyzer.Exit(node);
            }
        }

        /**
         * Handles the parser adding a child node to a production.
         * This method calls the appropriate analyzer callback.
         * 
         * @param node           the parent parse tree node
         * @param child          the child parse tree node, or null
         * 
         * @throws ParseException if the node couldn't be handled 
         *             correctly
         */
        internal void AddNode(Production node, Node child) {
            if (node.IsHidden()) {
                node.AddChild(child);
            } else if (child != null && child.IsHidden()) {
                for (int i = 0; i < child.GetChildCount(); i++) {
                    AddNode(node, child.GetChildAt(i));
                }
            } else {
                analyzer.Child(node, child);
            }
        }

        /**
         * Reads and consumes the next token in the queue. If no token
         * was available for consumation, a parse error will be
         * thrown.
         * 
         * @return the token consumed
         * 
         * @throws ParseException if the input stream couldn't be read or
         *             parsed correctly
         */
        internal Token NextToken() {
            Token  token = PeekToken(0);
        
            if (token != null) {
                tokens.RemoveAt(0);
                return token;
            } else {
                throw new ParseException(
                    ParseException.ErrorType.UNEXPECTED_EOF,
                    null,
                    tokenizer.GetCurrentLine(),
                    tokenizer.GetCurrentColumn());
            }
        }
    
        /**
         * Reads and consumes the next token in the queue. If no token was
         * available for consumation, a parse error will be thrown. A 
         * parse error will also be thrown if the token id didn't match 
         * the specified one. 
         *
         * @param id             the expected token id
         *  
         * @return the token consumed
         * 
         * @throws ParseException if the input stream couldn't be parsed
         *             correctly, or if the token wasn't expected
         */
        internal Token NextToken(int id) {
            Token      token = NextToken();
            ArrayList  list;
            
            if (token.GetId() == id) {
                return token;
            } else {
                list = new ArrayList(1);
                list.Add(tokenizer.GetPatternDescription(id));
                throw new ParseException(
                    ParseException.ErrorType.UNEXPECTED_TOKEN,
                    token.ToShortString(),
                    list,
                    token.GetStartLine(),
                    token.GetStartColumn());
            }
        }

        /**
         * Returns a token from the queue. This method is used to
         * check coming tokens before they have been consumed. Any
         * number of tokens forward can be checked.
         * 
         * @param steps          the token queue number, zero (0) for first
         * 
         * @return the token in the queue, or
         *         null if no more tokens in the queue
         * 
         * @throws ParseException if the input stream couldn't be read or
         *             parsed correctly
         */
        internal Token PeekToken(int steps) {
            Token  token;

            while (steps >= tokens.Count) {
                token = tokenizer.Next();
                if (token == null) {
                    return null;
                } else {
                    tokens.Add(token);
                }
            }
            return (Token) tokens[steps];
        }

        /**
         * Returns a token description for a specified token.
         * 
         * @param token          the token to describe
         * 
         * @return the token description
         */
        internal string GetTokenDescription(int token) {
            if (tokenizer == null) {
                return "";
            } else {
                return tokenizer.GetPatternDescription(token);
            }
        }
    }
}
