/*
 * TokenMatch.cs
 *
 * This work is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * As a special exception, the copyright holders of this library give
 * you permission to link this library with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also meet,
 * for each linked independent module, the terms and conditions of the
 * license of that module. An independent module is a module which is
 * not derived from or based on this library. If you modify this
 * library, you may extend this exception to your version of the
 * library, but you are not obligated to do so. If you do not wish to
 * do so, delete this exception statement from your version.
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

namespace Gnu.Grammar {

    /**
     * A token pattern match. This class contains the details of a string
     * match to a token pattern. There are two types of matches allowed, 
     * either the exact or the possible matches. The exact match contains
     * a string that matched the token pattern. A possible match is only
     * a flag that is set when the input string matched the beginning of
     * the pattern, but more characters are needed for an exact match. 
     * Both type of matches can be used at the same time, implying that
     * there may be an even longer match if more input is provided.   
     *
     * @author   Per Cederberg, <per at percederberg dot net>
     * @version  1.0
     */
    public class TokenMatch {

        /**
         * The token pattern matched.
         */
        private TokenPattern pattern;
    
        /**
         * The string matching the pattern. If this value is set to null,
         * no exact match could be made (only a possible one).
         */
        private string match;
    
        /**
         * The start position for the match.
         */
        private int position;

        /**
         * The possible match flag. This flag is set when a string matched
         * the beginning of the pattern, but more characters are needed 
         * for an exact match. This flag can be set together with an exact
         * match to notify the possibility of an even longer match.
         */
        private bool possible;

        /**
         * Creates a new token match.
         * 
         * @param Pattern        the token pattern
         * @param Match          the string matching the pattern
         */
        public TokenMatch(TokenPattern Pattern, string Match) 
            : this(Pattern, Match, 0, false) {
        }
    
        /**
         * Creates a new token match. This constructor can be used even
         * if no exact match was made, but only a possible match.
         * 
         * @param Pattern        the token pattern
         * @param Match          the string matching the pattern, or null
         * @param Position       the match string position
         * @param Possible       the possible match flag
         */
        public TokenMatch(TokenPattern Pattern, 
                          string Match, 
                          int Position,
                          bool Possible) {

            pattern = Pattern;
            match = Match;
            position = Position;
            possible = Possible;
        }
    
        /**
         * Checks if this is an exact match.
         * 
         * @return true if this is an exact token pattern match, or
         *         false otherwise
         */
        public bool IsExactMatch() {
            return match != null;
        }
    
        /**
         * Checks if this is a possible match.
         * 
         * @return true if this is a possible token pattern match, or
         *         false otherwise
         */
        public bool IsPossibleMatch() {
            return possible;
        }

        /**
         * Returns the length of an exact match.
         * 
         * @return the length of the string exactly matching the pattern
         */
        public int Length() {
            if (match == null) {
                return 0;
            } else {
                return match.Length;
            }
        }
    
        /**
         * Returns the start position of the match. 
         * 
         * @return the start position of the match
         */
        public int GetPosition() {
            return position;
        }
    
        /**
         * Sets the start position of the match.
         * 
         * @param Position       the new start position
         */
        internal void SetPosition(int Position) {
            position = Position;
        }

        /**
         * Sets the possible match flag.
         * 
         * @param Possible       the new value of the possible match flag
         */
        internal void SetPossible(bool Possible) {
            possible = Possible;
        }

        /**
         * Creates a token from the pattern match.
         * 
         * @param Line           the starting line number 
         * @param Column         the starting column number
         * 
         * @return a new token
         */
        internal Token CreateToken(int Line, int Column) {
            return new Token(pattern, match, Line, Column);
        }
    }
}
