/*
 * TestMain.cs
 *
 * This work is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * As a special exception, the copyright holders of this library give
 * you permission to link this library with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also meet,
 * for each linked independent module, the terms and conditions of the
 * license of that module. An independent module is a module which is
 * not derived from or based on this library. If you modify this
 * library, you may extend this exception to your version of the
 * library, but you are not obligated to do so. If you do not wish to
 * do so, delete this exception statement from your version.
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

using System;
using System.IO;
using Gnu.Grammar;

class TestMain {

    public static void Main(String[] Args) {
        Tokenizer  tokenizer;
        Object     obj;
    
        // Check command-line arguments
        if (Args.Length != 1) {
            Console.Error.WriteLine("ERROR: Expected grammar file.");
            Environment.Exit(1);
        }

        // Initialize tokenizer
        tokenizer = new Tokenizer(new StreamReader(Args[0]));
        createPatterns(tokenizer);
        
        // Read tokens
        try {
            Console.WriteLine("Reading tokens...");
            while ((obj = tokenizer.Next()) != null) {
                Console.WriteLine(obj.ToString());
            }
        } catch (ParseException e) {
            Console.Error.WriteLine(e.ToString());
            Environment.Exit(1);
        }
    }

    private static void createPatterns(Tokenizer tokenizer) {
        TokenPattern  pattern;

        pattern = new TokenPattern(1,
                                   "HEADER",
                                   TokenPattern.PatternType.String,
                                   "%header%");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(2,
                                   "TOKENS",
                                   TokenPattern.PatternType.String,
                                   "%tokens%");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(3,
                                   "PRODUCTIONS",
                                   TokenPattern.PatternType.String,
                                   "%productions%");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(4,
                                   "IGNORE",
                                   TokenPattern.PatternType.RegExp,
                                   "%ignore([^%]*)%");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(5,
                                   "ERROR",
                                   TokenPattern.PatternType.RegExp,
                                   "%error([^%]*)%");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(6,
                                   "EQUALS",
                                   TokenPattern.PatternType.String,
                                   "=");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(7,
                                   "LPAREN",
                                   TokenPattern.PatternType.String,
                                   "(");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(8,
                                   "RPAREN",
                                   TokenPattern.PatternType.String,
                                   ")");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(9,
                                   "LBRACE",
                                   TokenPattern.PatternType.String,
                                   "{");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(10,
                                   "RBRACE",
                                   TokenPattern.PatternType.String,
                                   "}");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(11,
                                   "LBRACKET",
                                   TokenPattern.PatternType.String,
                                   "[");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(12,
                                   "RBRACKET",
                                   TokenPattern.PatternType.String,
                                   "]");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(13,
                                   "PLUS_SIGN",
                                   TokenPattern.PatternType.String,
                                   "+");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(14,
                                   "ASTERISK",
                                   TokenPattern.PatternType.String,
                                   "*");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(15,
                                   "VERTICAL_BAR",
                                   TokenPattern.PatternType.String,
                                   "|");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(16,
                                   "SEMICOLON",
                                   TokenPattern.PatternType.String,
                                   ";");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(17,
                                   "IDENTIFIER",
                                   TokenPattern.PatternType.RegExp,
                                   "[A-Za-z][A-Za-z0-9-_]*");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(18,
                                   "QUOTED_STRING",
                                   TokenPattern.PatternType.RegExp,
                                   "(\"[^\"]*\")|('[^']*')");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(19,
                                   "REGEXP",
                                   TokenPattern.PatternType.RegExp,
                                   "<<([^\\\\>]|(\\\\.)|(>[^>]))*>>");
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(20,
                                   "SINGLE_LINE_COMMENT",
                                   TokenPattern.PatternType.RegExp,
                                   "//.*");
        pattern.SetIgnore();
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(21,
                                   "MULTI_LINE_COMMENT",
                                   TokenPattern.PatternType.RegExp,
                                   "/\\*([^*]|\\*[^/])*\\*/");
        pattern.SetIgnore();
        tokenizer.AddPattern(pattern);

        pattern = new TokenPattern(22,
                                   "WHITESPACE",
                                   TokenPattern.PatternType.RegExp,
                                   "[ \\t\\n\\r]+");
        pattern.SetIgnore();
        tokenizer.AddPattern(pattern);
    }
}
