/*
 * ParseException.cs
 *
 * This work is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * As a special exception, the copyright holders of this library give
 * you permission to link this library with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also meet,
 * for each linked independent module, the terms and conditions of the
 * license of that module. An independent module is a module which is
 * not derived from or based on this library. If you modify this
 * library, you may extend this exception to your version of the
 * library, but you are not obligated to do so. If you do not wish to
 * do so, delete this exception statement from your version.
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

using System;
using System.Text;

namespace Gnu.Grammar {

    /**
     * A parse exception.
     *
     * @author   Per Cederberg, <per at percederberg dot net>
     * @version  1.0
     */
    public class ParseException : Exception {

        /**
         * The error type enumeration.
         */
        public enum ErrorType {
            /**
             * The internal error type is only used to signal an error that 
             * is a result of a bug in the parser or tokenizer code.
             */
            Internal, 

            /**
             * The I/O error type is used for stream I/O errors. 
             */
            IO,
    
            /**
             * The unexpected end of file error type is used when end of file 
             * is encountered instead of a valid token.
             */
            UnexpectedEndOfFile,

            /**
             * The unexpected character error type is used when a character is 
             * read that isn't handled by one of the token patterns.
             */
            UnexpectedCharacter,

            /**
             * The unexpected token error type is used when another token than 
             * the expected one is encountered.
             */
            UnexpectedToken,
    
            /**
             * The invalid token error type is used when a token pattern with 
             * an error message is matched. The additional information provided 
             * should contain the error message. 
             */
            InvalidToken
        }

        /**
         * The error type.
         */
        private ErrorType type;
    
        /**
         * The additional information string.
         */
        private string info;

        /**
         * The line number.
         */
        private int line;
    
        /**
         * The column number.
         */
        private int column;

        /**
         * Creates a new parse exception
         * 
         * @param Type           the parse error type
         * @param Info           the additional information
         * @param Line           the line number, or -1 for unknown
         * @param Column         the column number, or -1 for unknown
         */
        public ParseException(ErrorType Type, 
                              string Info, 
                              int Line, 
                              int Column) {

            type = Type;
            info = Info;
            line = Line;
            column = Column;
        }

        /**
         * The message property. This property contains the detailed
         * exception error message.
         */
        public override string Message {
            get{
                return GetMessage(); 
            }   
        }

        /**
         * Returns the line number where the error occured.
         * 
         * @return the line number of the error, or 
         *         -1 if unknown
         */
        public int GetLine() {
            return line;
        }
    
        /**
         * Returns the column number where the error occured.
         * 
         * @return the column number of the error, or 
         *         -1 if unknown
         */
        public int GetColumn() {
            return column;
        }
    
        /**
         * Returns a default error message.
         * 
         * @return a default error message
         */
        private string GetMessage() {
            StringBuilder  buffer = new StringBuilder();

            // Add type and info
            switch (type) {
            case ErrorType.IO:
                buffer.Append("I/O error: ");
                buffer.Append(info);
                break;
            case ErrorType.UnexpectedEndOfFile:
                buffer.Append("unexpected end of file");
                break;
            case ErrorType.UnexpectedCharacter:
                buffer.Append("unexpected character: '");
                buffer.Append(info);
                buffer.Append("'");
                break;
            case ErrorType.UnexpectedToken:
                buffer.Append("unexpected token: '");
                buffer.Append(info);
                buffer.Append("'");
                break;
            case ErrorType.InvalidToken:
                buffer.Append(info);
                break;
            default:
                buffer.Append("internal error");
                break;
            }

            // Add line and column
            if (line > 0 && column > 0) {
                buffer.Append(", on line: ");
                buffer.Append(line);
                buffer.Append(" column: ");
                buffer.Append(column);
            }

            return buffer.ToString();
        }
    }
}
