/*
 * Element.java
 *
 * This work is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * As a special exception, the copyright holders of this library give
 * you permission to link this library with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also meet,
 * for each linked independent module, the terms and conditions of the
 * license of that module. An independent module is a module which is
 * not derived from or based on this library. If you modify this
 * library, you may extend this exception to your version of the
 * library, but you are not obligated to do so. If you do not wish to
 * do so, delete this exception statement from your version.
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.grammatica.parser.re;

import java.io.PrintStream;
import java.io.PrintWriter;

/**
 * A regular expression element. This is the common base class for all
 * regular expression elements, i.e. the parts of the regular 
 * expression.
 *
 * @author   Per Cederberg, <per at percederberg dot net>
 * @version  1.0
 */
abstract class Element implements Cloneable {

    /**
     * Creates a copy of this element. The copy will be an instance
     * of the same class matching the same strings. Copies of elements
     * are necessary to allow elements to cache intermediate results
     * while matching strings without interfering with other threads.
     * 
     * @return a copy of this element
     */
    public abstract Object clone();

    /**
     * Returns the length of a matching string starting at the 
     * specified position. The number of matches to skip can also be
     * specified, but numbers higher than zero (0) cause a failed 
     * match for any element that doesn't attempt to combine other 
     * elements.
     *
     * @param m              the matcher being used 
     * @param str            the string to match
     * @param start          the starting position
     * @param skip           the number of matches to skip
     * 
     * @return the length of the longest matching string, or
     *         -1 if no match was found
     */
    public abstract int match(Matcher m, 
                              CharBuffer str, 
                              int start, 
                              int skip);
    
    /**
     * Prints this element to the specified output stream.
     * 
     * @param out            the output stream to use
     * @param indent         the current indentation
     */
    public final void printTo(PrintStream out, String indent) {
        PrintWriter  writer = new PrintWriter(out);
        
        printTo(writer, indent);
        writer.flush();
    }

    /**
     * Prints this element to the specified output stream.
     * 
     * @param out            the output stream to use
     * @param indent         the current indentation
     */
    public abstract void printTo(PrintWriter out, String indent);
    
}
