/*
 * RecursiveDescentParser.java
 *
 * This work is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * As a special exception, the copyright holders of this library give
 * you permission to link this library with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also meet,
 * for each linked independent module, the terms and conditions of the
 * license of that module. An independent module is a module which is
 * not derived from or based on this library. If you modify this
 * library, you may extend this exception to your version of the
 * library, but you are not obligated to do so. If you do not wish to
 * do so, delete this exception statement from your version.
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.grammatica.parser;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

/**
 * A recursive descent parser. This parser handles LL(n) grammars, 
 * selecting the appropriate pattern to parse based on the next few
 * tokens. The parser is more efficient the fewer look-ahead tokens
 * that is has to consider.
 *
 * @author   Per Cederberg, <per at percederberg dot net>
 * @version  1.0
 */
public class RecursiveDescentParser extends Parser {

    /**
     * The parser initialization flag.
     */
    private boolean initialized = false;

    /**
     * The map of pattern look-ahead sets. The map is indexed by the
     * production pattern object. 
     */
    private HashMap lookAheads = new HashMap();

    /**
     * Creates a new parser.
     * 
     * @param tokenizer      the tokenizer to use
     */
    public RecursiveDescentParser(Tokenizer tokenizer) {
        super(tokenizer);
    }

    /**
     * Creates a new parser.
     * 
     * @param tokenizer      the tokenizer to use
     * @param analyzer       the analyzer callback to use
     */
    public RecursiveDescentParser(Tokenizer tokenizer, Analyzer analyzer) {
        super(tokenizer, analyzer);
    }

    /**
     * Adds a new production pattern to the parser. The pattern will 
     * be added last in the list. The first pattern added is assumed 
     * to be the starting point in the grammar. The pattern will be
     * validated against the grammar type to some extent.
     * 
     * @param pattern        the pattern to add
     * 
     * @throws ParserCreationException if the pattern couldn't be 
     *             added correctly to the parser
     */
    public void addPattern(ProductionPattern pattern) 
        throws ParserCreationException {

        // Check for empty matches
        if (pattern.isMatchingEmpty()) {
            throw new ParserCreationException(
                "couldn't add production pattern " + pattern.getName() +
                " , as it can match zero elements (minimum is one)");
        }
                
        // Check for left-recusive patterns
        if (pattern.isLeftRecursive()) {
            throw new ParserCreationException(
                "couldn't add production pattern " + pattern.getName() +
                " , as it is left recursive");
        }

        // Add pattern 
        super.addPattern(pattern);
        lookAheads.put(pattern, new PatternLookAhead(pattern));
        initialized = false;
    }

    /**
     * Initializes the parser. All the added production patterns will
     * be analyzed for ambiguities and errors. This method also 
     * initializes the internal data structures used during the 
     * parsing. 
     * 
     * @throws ParserCreationException if the parser couldn't be 
     *             initialized correctly 
     */
    public void prepare() throws ParserCreationException {
        Iterator           iter;

        // Performs production pattern checks
        super.prepare();

        // Calculate production look-ahead sets
        iter = getPatterns().iterator();
        while (iter.hasNext()) {
            calculateLookAhead((ProductionPattern) iter.next());
        }

        // Set initialized flag
        initialized = true;
    }

    /**
     * Parses the input stream and creates a parse tree.
     * 
     * @return the parse tree
     * 
     * @throws ParserCreationException if the parser couldn't be
     *             initialized correctly
     * @throws ParseException if the input couldn't be parsed 
     *             correctly
     * 
     * @see #prepare
     */
    public Node parse() throws ParserCreationException, ParseException {
        Token  token;
        Node   node;
        
        if (!initialized) {
            prepare();
        }
        node = parsePattern(getStartPattern());
        token = peekToken(0); 
        if (token != null) {
            throw new ParseException(
                ParseException.UNEXPECTED_TOKEN_ERROR,
                token.getImage(),
                token.getStartLine(),
                token.getStartColumn());
        }
        return node;
    }

    /**
     * Parses a production pattern. A parse tree node may or may not
     * be created depending on the analyzer callbacks.
     * 
     * @param pattern        the production pattern to parse
     * 
     * @return the parse tree node created, or null
     * 
     * @throws ParseException if the input couldn't be parsed 
     *             correctly
     */
    private Node parsePattern(ProductionPattern pattern) 
        throws ParseException {

        Token                  token = peekToken(0);
        PatternLookAhead       look;
        ProductionPatternRule  rule;

        look = (PatternLookAhead) lookAheads.get(pattern);
        for (int i = 0; i < pattern.getRuleCount(); i++) {
            rule = pattern.getRule(i);
            if (look.getFallback() != rule && look.isNext(rule)) {
                return parseRule(rule);
            }
        }
        if (look.getFallback() != null) {
            return parseRule(look.getFallback());
        } else {
            throw new ParseException(
                ParseException.UNEXPECTED_TOKEN_ERROR,
                token.getImage(),
                token.getStartLine(),
                token.getStartColumn());
        }
    }
    
    /**
     * Parses a production pattern rule. A parse tree node may or may 
     * not be created depending on the analyzer callbacks.
     * 
     * @param rule           the production pattern rule to parse
     * 
     * @return the parse tree node created, or null
     * 
     * @throws ParseException if the input couldn't be parsed 
     *             correctly
     */
    private Node parseRule(ProductionPatternRule rule) 
        throws ParseException {

        Production  node;
        
        node = new Production(rule.getPattern());
        enterNode(node);
        for (int i = 0; i < rule.getElementCount(); i++) {
            parseElement(node, rule.getElement(i));
        }
        return exitNode(node);
    }

    /**
     * Parses a production pattern element. All nodes parsed may or
     * may not be added to the parse tree node specified, depending
     * on the analyzer callbacks.
     *
     * @param node           the production parse tree node  
     * @param elem           the production pattern element to parse
     * 
     * @throws ParseException if the input couldn't be parsed 
     *             correctly
     */
    private void parseElement(Production node, 
                              ProductionPatternElement elem)
        throws ParseException {

        Token              token;
        ProductionPattern  pattern;
        PatternLookAhead   look;
        int                i;

        if (elem.isToken()) {
            for (i = 0; i < elem.getMaxCount(); i++) {
                token = peekToken(0);
                if (i < elem.getMinCount() || elem.isMatch(token)) {
                    token = nextToken(elem.getId());
                    enterNode(token);
                    addNode(node, exitNode(token));
                } else {
                    break;
                }
            }
        } else {
            pattern = getPattern(elem.getId());
            look = (PatternLookAhead) lookAheads.get(pattern);
            for (i = 0; i < elem.getMaxCount(); i++) {
                if (i < elem.getMinCount() || look.isNext()) {
                    addNode(node, parsePattern(pattern));
                } else {
                    break;
                }
            }
        }
    }
    
    /**
     * Calculates the look-ahead needed for the specified production 
     * pattern. This method attempts to resolve any conflicts and 
     * stores the results in the pattern look-ahead object. 
     * 
     * @param pattern        the production pattern
     * 
     * @throws ParserCreationException if the look-ahead set couldn't
     *             be determined due to inherent ambiguities
     */
    private void calculateLookAhead(ProductionPattern pattern) 
        throws ParserCreationException {

        PatternLookAhead       look;
        ProductionPatternRule  rule;
        LookAhead              result;
        LookAhead              ruleResults[];
        LookAhead              conflicts;
        int                    length = 1;
        int                    i;

        // Calculate simple look-ahead
        look = (PatternLookAhead) lookAheads.get(pattern);
        result = new LookAhead(1);
        ruleResults = new LookAhead[pattern.getRuleCount()];
        for (i = 0; i < pattern.getRuleCount(); i++) {
            ruleResults[i] = findLookAhead(pattern.getRule(i), 1);
            result.addAll(ruleResults[i]);
        }
        if (look.getLookAhead() == null) {
            look.setLookAhead(result);
        }
        conflicts = findConflicts(1, ruleResults);

        // Resolve conflicts
        while (conflicts.size() > 0) {
            length++;
            for (i = 0; i < pattern.getRuleCount(); i++) {
                rule = pattern.getRule(i);
                if (ruleResults[i].intersects(conflicts)) {
                    ruleResults[i] = findLookAhead(rule, length);
                    look.setRuleLookAhead(rule, ruleResults[i]);
                }
                if (ruleResults[i].intersects(conflicts)) {
                    if (look.getFallback() == null) {
                        look.setFallback(rule);
                    } else if (look.getFallback() != rule) {
                        throw new ParserCreationException(
                            "inherent ambiguity in " + pattern.getName());
                    }
                }
            }
            conflicts = findConflicts(length, ruleResults);
        }
        
        // Resolve conflicts inside rules
        for (i = 0; i < pattern.getRuleCount(); i++) {
            calculateLookAhead(pattern.getRule(i), 0);
        }
    }

    /**
     * Calculates the look-aheads needed for the specified pattern 
     * rule. This method attempts to resolve any conflicts in optional 
     * elements by recalculating look-aheads for referenced 
     * productions. 
     * 
     * @param rule           the production pattern rule
     * @param pos            the pattern element position
     * 
     * @throws ParserCreationException if the look-ahead set couldn't
     *             be determined due to inherent ambiguities
     */
    private void calculateLookAhead(ProductionPatternRule rule,
                                    int pos)
        throws ParserCreationException {

        ProductionPattern         pattern;
        PatternLookAhead          look;
        ProductionPatternElement  elem;
        LookAhead                 first;
        LookAhead                 follow;
        LookAhead                 conflicts;
        int                       length = 1;

        // Check trivial cases
        if (pos >= rule.getElementCount()) {
            return;
        }

        // Check for non-optional element
        pattern = rule.getPattern();
        elem = rule.getElement(pos);
        if (elem.getMinCount() == elem.getMaxCount()) {
            calculateLookAhead(rule, pos + 1);
            return;
        }

        // Calculate simple look-aheads
        if (elem.isToken()) {
            first = new LookAhead(1);
            first.add(elem.getId());
        } else {
            first = findLookAhead(getPattern(elem.getId()), 1);
        }
        follow = findLookAhead(rule, 1, pos + 1); 
        
        // Resolve conflicts
        conflicts = first.createIntersection(follow);
        if (conflicts.size() > 0 && elem.isToken()) {
            throw new ParserCreationException(
                "inherent ambiguity in " + pattern.getName() + 
                " in element " + (pos + 1));
        }
        while (conflicts.size() > 0) {
            length++;
            first = findLookAhead(getPattern(elem.getId()), length);
            follow = findLookAhead(rule, length, pos + 1); 
            if (first.intersects(conflicts)) {
                throw new ParserCreationException(
                    "inherent ambiguity in " + pattern.getName() + 
                    " in element " + (pos + 1));
            }
            conflicts = first.createIntersection(follow);
        }

        // Change sub-element look-ahead
        if (length > 1) {
            pattern = getPattern(elem.getId());
            look = (PatternLookAhead) lookAheads.get(pattern);
            look.setLookAhead(first);
        }

        // Check remaining elements
        calculateLookAhead(rule, pos + 1);
    }

    /**
     * Finds the look-ahead set for a production pattern. The maximum 
     * look-ahead length must be specified. This method will cache 
     * some results in the pattern look-ahead wrapper for improved 
     * performance.
     * 
     * @param pattern        the production pattern
     * @param length         the maximum look-ahead length
     * 
     * @return the look-ahead set for the production pattern
     */
    private LookAhead findLookAhead(ProductionPattern pattern, 
                                    int length) {

        PatternLookAhead  look;
        LookAhead         result;

        // Check for cached result
        look = (PatternLookAhead) lookAheads.get(pattern);
        result = look.getLookAhead();
        if (result != null) {
            if (length == result.getMaxLength()) {
                return result;
            } else if (length < result.getMaxLength()) {
                return new LookAhead(length, result);
            }
        }

        // Find pattern look-ahead
        result = new LookAhead(length);
        for (int i = 0; i < pattern.getRuleCount(); i++) {
            result.addAll(findLookAhead(pattern.getRule(i), length));
        }
        if (look.getLookAhead() == null) {
            look.setLookAhead(result);
        }

        return result;
    }

    /**
     * Finds the look-ahead set for a production pattern rule. The 
     * maximum look-ahead length must be specified. This method will 
     * cache some results in the pattern look-ahead wrapper for 
     * improved performance.
     * 
     * @param rule           the production pattern rule
     * @param length         the maximum look-ahead length
     * 
     * @return the look-ahead set for the production pattern rule
     */
    private LookAhead findLookAhead(ProductionPatternRule rule, 
                                    int length) {

        PatternLookAhead   look;
        LookAhead          result;
                                
        // Check for cached result
        look = (PatternLookAhead) lookAheads.get(rule.getPattern());
        result = look.getRuleLookAhead(rule);
        if (result != null) {
            if (length == result.getMaxLength()) {
                return result;
            } else if (length < result.getMaxLength()) {
                return new LookAhead(length, result); 
            }
        }

        // Find rule look-ahead
        result = findLookAhead(rule, length, 0);
        if (look.getRuleLookAhead(rule) == null) {
            look.setRuleLookAhead(rule, result);
        }

        return result;
    }


    /**
     * Finds the look-ahead set for a production pattern rule. The 
     * pattern position and maximum look-ahead length must be 
     * specified.
     * 
     * @param rule           the production pattern rule
     * @param length         the maximum look-ahead length
     * @param pos            the pattern element position
     * 
     * @return the look-ahead set for the production pattern rule
     */
    private LookAhead findLookAhead(ProductionPatternRule rule,
                                    int length,
                                    int pos) {

        ProductionPatternElement  elem;
        LookAhead                 result;
        LookAhead                 next;
        int                       max;

        // Check trivial cases
        if (length <= 0 || pos >= rule.getElementCount()) {
            return new LookAhead(0);
        }

        // Find look-ahead for this element
        elem = rule.getElement(pos);
        if (elem.isToken()) {
            next = new LookAhead(length);
            next.add(elem.getId());
        } else {
            next = findLookAhead(getPattern(elem.getId()), length);
        }

        // Handle element repetitions
        result = new LookAhead(length);
        result.addAll(next);
        if (elem.getMinCount() == 0) {
            result.addEmpty();
        }
        max = Math.min(length, elem.getMaxCount());
        for (int i = 1; i < max; i++) {
            next = next.createCombination(next);
            result.addAll(next);
        }
        
        // Find remaining look-ahead
        max = length - result.getMinLength();
        if (max <= 0) {
            return result;
        } else {
            next = findLookAhead(rule, max, pos + 1); 
            return result.createCombination(next); 
        }
    }

    /**
     * Returns a look-ahead set with all conflics between a number of
     * look-ahead sets.
     * 
     * @param maxLength      the maximum token sequence length
     * @param sets           the look-ahead sets to check
     * 
     * @return a look-ahead set with the conflicts found
     */
    private LookAhead findConflicts(int maxLength, LookAhead[] sets) {
        LookAhead  result = new LookAhead(maxLength);

        for (int i = 0; i < sets.length; i++) {
            for (int j = 0; j < i; j++) {
                result.addAll(sets[i].createIntersection(sets[j]));
            }
        }
        return result;
    }


    /**
     * A production patterns look-ahead set. This class is a wrapper 
     * to add the required look-ahead sets to a production pattern.
     */
    private class PatternLookAhead extends Object {

        /**
         * The production pattern.
         */
        private ProductionPattern pattern;

        /**
         * The fallback pattern rule. If no fallback has been defined,
         * null will be used.
         */
        private ProductionPatternRule fallback = null;

        /**
         * The pattern look-ahead set.
         */
        private LookAhead lookAhead = null;

        /**
         * The map of pattern rule look-ahead set. This map is indexed 
         * by the pattern rule objects.
         */
        private HashMap ruleLookAheads = new HashMap();
        
        /**
         * Creates a new production pattern look-ahead set.
         * 
         * @param pattern        the first pattern to add 
         */
        public PatternLookAhead(ProductionPattern pattern) {
            this.pattern = pattern;
        }
        
        /**
         * Checks if the next tokens matches the look-ahead set for
         * this production pattern.
         * 
         * @return true if the next tokens match, or
         *         false otherwise
         * 
         * @throws ParseException if the next token couldn't be read 
         *            or parsed correctly
         */
        public boolean isNext() throws ParseException {
            if (lookAhead == null) {
                return false;
            } else {
                return lookAhead.isNext();
            }
        }

        /**
         * Checks if the next tokens matches the look-ahead set for
         * the specified production pattern rule.
         *
         * @param rule           the pattern rule to check
         *  
         * @return true if the next tokens match, or
         *         false otherwise
         * 
         * @throws ParseException if the next token couldn't be read 
         *            or parsed correctly
         */
        public boolean isNext(ProductionPatternRule rule) 
            throws ParseException {

            LookAhead  set = getRuleLookAhead(rule); 

            if (set == null) {
                return false;
            } else {
                return set.isNext();
            }
        }

        /**
         * Returns the fallback pattern rule. The fallback rule is 
         * only used when no other rule look-ahead set match. It is 
         * used as a means to compensate for rules that are unable to
         * lengthen their look-ahead sets to avoid ambiguities.  
         * 
         * @return the fallback pattern rule, or 
         *         null if none has been set
         */
        public ProductionPatternRule getFallback() {
            return fallback;
        }
        
        /**
         * Sets the fallback pattern rule. The fallback rule is only 
         * used when no other rule look-ahead set match. It is used as 
         * a means to compensate for rules that are unable to lengthen 
         * their look-ahead sets to avoid ambiguities.
         *   
         * @param rule           the new fallback pattern rule
         */
        public void setFallback(ProductionPatternRule rule) {
            this.fallback = rule;
        }

        /**
         * Returns the pattern look-ahead set.
         * 
         * @return the pattern look-ahead set
         */
        public LookAhead getLookAhead() {
            return lookAhead;
        }

        /**
         * Sets the pattern look-ahead.
         * 
         * @param set            the new pattern look-ahead set
         */
        public void setLookAhead(LookAhead set) {
            this.lookAhead = set;
        }

        /**
         * Returns the pattern rule look-ahead set.
         * 
         * @param rule           the production pattern rule
         * 
         * @return the pattern rule look-ahead set, or
         *         null if it hasn't been set
         */
        public LookAhead getRuleLookAhead(ProductionPatternRule rule) {
            return (LookAhead) ruleLookAheads.get(rule);
        }

        /**
         * Sets the pattern rule look-ahead set.
         * 
         * @param rule           the production pattern rule
         * @param set            the new look-ahead set
         */
        public void setRuleLookAhead(ProductionPatternRule rule, 
                                     LookAhead set) {

            ruleLookAheads.put(rule, set);
        }
    }

    
    /**
     * A token look-ahead set. This set consists of token id sequences
     * with no single sequence being longer than a maximum value. The
     * set also guarantees that no single token id sequence is present
     * more than once.
     */
    private class LookAhead {

        /**
         * The set of token look-ahead sequences. Each sequence in 
         * turn is represented by an ArrayList with Integers for the
         * token id:s.
         */
        private ArrayList elements = new ArrayList();
        
        /**
         * The maximum length of any look-ahead sequence.
         */
        private int maxLength;

        /**
         * Creates a new look-ahead set with the specified maximum
         * length.
         * 
         * @param maxLength      the maximum token sequence length
         */
        public LookAhead(int maxLength) {
            this.maxLength = maxLength;
        }

        /**
         * Creates a duplicate look-ahead set with a different maximum
         * length. 
         * 
         * @param maxLength      the maximum token sequence length
         * @param src            the look-ahead set to copy
         */
        public LookAhead(int maxLength, LookAhead src) {
            this(maxLength);
            for (int i = 0; i < src.elements.size(); i++) {
                add((ArrayList) src.elements.get(i));
            }
        }

        /**
         * Returns the size of this look-ahead set.
         * 
         * @return the number of token sequences in the set
         */
        public int size() {
            return elements.size();
        }

        /**
         * Returns the length of the shortest token sequence in this
         * set. This method will return zero (0) if the set is empty.
         * 
         * @return the length of the shortest token sequence
         */
        public int getMinLength() {
            ArrayList  list;
            int        min = -1;
            
            for (int i = 0; i < elements.size(); i++) {
                list = (ArrayList) elements.get(i);
                if (min < 0 || list.size() < min) {
                    min = list.size();
                }
            }
            return (min < 0) ? 0 : min;
        }

        /**
         * Returns the length of the longest token sequence in this
         * set. This method will return zero (0) if the set is empty.
         * 
         * @return the length of the longest token sequence
         */
        public int getMaxLength() {
            ArrayList  list;
            int        max = 0;
            
            for (int i = 0; i < elements.size(); i++) {
                list = (ArrayList) elements.get(i);
                if (list.size() > max) {
                    max = list.size();
                }
            }
            return max;
        }

        /**
         * Checks if the next token(s) match any token sequence in 
         * this set.
         * 
         * @return true if the next tokens are in the set, or
         *         false otherwise
         * 
         * @throws ParseException if the next token couldn't be read 
         *            or parsed correctly
         */
        public boolean isNext() throws ParseException {
            for (int i = 0; i < elements.size(); i++) {
                if (isNext((ArrayList) elements.get(i))) {
                    return true;
                }
            }
            return false;
        }
        
        /**
         * Checks if the next token(s) match a specified token 
         * sequence.
         * 
         * @param list           the token sequence to check
         * 
         * @return true if the next tokens are in the list, or
         *         false otherwise
         * 
         * @throws ParseException if the next token couldn't be read 
         *            or parsed correctly
         */
        private boolean isNext(ArrayList list) throws ParseException {
            Token    token;
            Integer  id;
            
            for (int i = 0; i < list.size(); i++) {
                token = peekToken(i);
                id = (Integer) list.get(i);
                if (token == null || token.getId() != id.intValue()) {
                    return false;
                }
            }
            return true;
        }
        
        /**
         * Checks if the specified token sequence is present in the
         * set.
         * 
         * @param elem           the look-ahead list to check
         * 
         * @return true if the list is present in this set, or
         *         false otherwise
         */
        private boolean contains(ArrayList elem) {
            for (int i = 0; i < elements.size(); i++) {
                if (elements.get(i).equals(elem)) {
                    return true;
                }
            }
            return false;
        }

        /**
         * Checks if some token sequence is present in both this set 
         * and a specified one.
         * 
         * @param set            the look-ahead set to compare with
         * 
         * @return true if the look-ahead sets intersect, or
         *         false otherwise 
         */
        public boolean intersects(LookAhead set) {
            for (int i = 0; i < set.elements.size(); i++) {
                if (contains((ArrayList) set.elements.get(i))) {
                    return true;
                }
            }
            return false;
        }

        /**
         * Adds a token sequence list to this set. The list will only 
         * be added if it is not already in the set. Also, if the list 
         * is longer than the allowed maximum, it will be truncated.
         * 
         * @param list           the token sequence list to add
         */
        private void add(ArrayList list) {
            if (list.size() > maxLength) {
                list = (ArrayList) list.clone();
                while (list.size() > maxLength) {
                    list.remove(list.size() - 1);
                }
            }
            if (!contains(list)) {
                elements.add(list);
            }
        }

        /**
         * Adds a new token sequence list with a single token to this 
         * set. The list will only be added if it is not already in 
         * the set.
         * 
         * @param token          the token to add
         */
        public void add(int token) {
            ArrayList  list = new ArrayList(1);
            
            list.add(new Integer(token));
            add(list);
        }

        /**
         * Adds an empty token sequence list to this set. The list 
         * will only be added if it is not already in the set. 
         */
        public void addEmpty() {
            add(new ArrayList(0));
        }

        /**
         * Adds all the token sequence lists from a specified set. 
         * Only lists not already in this set will be added.
         * 
         * @param set            the set to add from
         */
        public void addAll(LookAhead set) {
            for (int i = 0; i < set.elements.size(); i++) {
                add((ArrayList) set.elements.get(i));
            }
        }

        /**
         * Creates a new look-ahead set that is the intersection of
         * this set with another set.
         * 
         * @param set            the set to intersect with
         * 
         * @return a new look-ahead set containing the intersection
         */
        public LookAhead createIntersection(LookAhead set) {
            LookAhead  result = new LookAhead(maxLength);
            ArrayList  list;

            for (int i = 0; i < elements.size(); i++) {
                list = (ArrayList) elements.get(i);
                if (set.contains(list)) {
                    result.add(list);
                }
            }
            return result;
        }

        /**
         * Creates a new look-ahead set that is the combination of 
         * this set with another set. The combination is created by 
         * creating new token sequence lists that consist of appending
         * all elements from the specified set onto all elements in
         * this set. This is sometimes referred to as the cartesian
         * product.
         * 
         * @param set            the set to combine with
         * 
         * @return a new look-ahead set containing the combination
         */
        public LookAhead createCombination(LookAhead set) {
            LookAhead  result = new LookAhead(maxLength);
            ArrayList  first;
            ArrayList  second;
            ArrayList  temp;
            
            // Handle special cases
            if (this.size() <= 0) {
                return set;
            } else if (set.size() <= 0) {
                return this;
            }

            // Create combinations
            for (int i = 0; i < elements.size(); i++) {
                first = (ArrayList) elements.get(i);
                if (first.size() >= maxLength) {
                    result.add(first);
                } else if (first.size() <= 0) {
                    result.addAll(set);  
                } else {
                    for (int j = 0; j < set.elements.size(); j++) {
                        second = (ArrayList) set.elements.get(j);
                        temp = new ArrayList(first.size() + second.size());
                        temp.addAll(first);
                        temp.addAll(second);
                        result.add(temp);
                    }
                }
            }
            return result;
        }

        /**
         * Returns a string representation of this object.
         * 
         * @return a string representation of this object
         */
        public String toString() {
            StringBuffer  buffer = new StringBuffer();

            buffer.append("{");
            for (int i = 0; i < elements.size(); i++) {
                buffer.append("\n  ");
                buffer.append(elements.get(i));
            }
            buffer.append("\n}");
            return buffer.toString();
        }
    }
}
