/*
 * ProductionPatternRule.java
 *
 * This work is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * As a special exception, the copyright holders of this library give
 * you permission to link this library with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also meet,
 * for each linked independent module, the terms and conditions of the
 * license of that module. An independent module is a module which is
 * not derived from or based on this library. If you modify this
 * library, you may extend this exception to your version of the
 * library, but you are not obligated to do so. If you do not wish to
 * do so, delete this exception statement from your version.
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.grammatica.parser;

import java.util.Vector;

/**
 * A production pattern rule. This class represents a list of 
 * production pattern elements. In order to provide alternatives that
 * cannot be represented with the element occurance counters, multiple 
 * rules must be created and added to the same production pattern. A
 * production pattern rule is always contained within a production 
 * pattern.  
 *
 * @author   Per Cederberg, <per at percederberg dot net>
 * @version  1.0
 */
public class ProductionPatternRule {

    /**
     * The production pattern.
     */
    private ProductionPattern pattern;

    /**
     * The rule elements.
     */
    private Vector elements = new Vector();

    /**
     * Creates a new production pattern rule.  
     */
    public ProductionPatternRule() {
    }

    /**
     * Checks if this rule is recursive on the left-hand side. This
     * method checks all the possible left side elements and returns 
     * true if the pattern itself is among them.
     *  
     * @return true if the rule is left side recursive, or
     *         false otherwise
     */
    public boolean isLeftRecursive() {
        ProductionPatternElement  elem;

        for (int i = 0; i < elements.size(); i++) {
            elem = (ProductionPatternElement) elements.get(i);
            if (elem.getId() == pattern.getId()) {
                return true;
            } else if (elem.getMinCount() > 0) {
                break;
            }
        }
        return false;
    }
    
    /**
     * Checks if this rule is recursive on the right-hand side. This 
     * method checks all the possible right side elements and returns 
     * true if the pattern itself is among them.
     *  
     * @return true if the rule is right side recursive, or
     *         false otherwise
     */
    public boolean isRightRecursive() {
        ProductionPatternElement  elem;

        for (int i = elements.size() - 1; i >= 0; i++) {
            elem = (ProductionPatternElement) elements.get(i);
            if (elem.getId() == pattern.getId()) {
                return true;
            } else if (elem.getMinCount() > 0) {
                break;
            }
        }
        return false;
    }

    /**
     * Checks if this rule would match an empty stream of tokens. This
     * check is equivalent of getMinElementCount() returning zero (0).
     *  
     * @return true if the rule can match an empty token stream, or
     *         false otherwise
     */
    public boolean isMatchingEmpty() {
        return getMinElementCount() == 0;
    }

    /**
     * Returns the production pattern containing this rule.
     * 
     * @return the production pattern for this rule
     */
    public ProductionPattern getPattern() {
        return pattern;
    }

    /**
     * Changes the production pattern containing this rule. This 
     * method should only be called by the production pattern class.
     * 
     * @param pattern        the new production pattern
     */
    void setPattern(ProductionPattern pattern) {
        this.pattern = pattern;
    }

    /**
     * Returns the number of elements in this rule.
     *  
     * @return the number of elements in this rule
     */
    public int getElementCount() {
        return elements.size();
    }

    /**
     * Returns the minimum number of elements needed to satisfy this
     * rule. The value returned is the sum of all the elements 
     * minimum count.
     * 
     * @return the minimum number of elements
     */
    public int getMinElementCount() {
        ProductionPatternElement  elem;
        int                       min = 0;
        
        for (int i = 0; i < elements.size(); i++) {
            elem = (ProductionPatternElement) elements.get(i);
            min += elem.getMinCount();
        }
        return min;
    }
    
    /**
     * Returns the maximum number of elements needed to satisfy this
     * rule. The value returned is the sum of all the elements 
     * maximum count.  
     * 
     * @return the maximum number of elements
     */
    public int getMaxElementCount() {
        ProductionPatternElement  elem;
        int                       max = 0;
        
        for (int i = 0; i < elements.size(); i++) {
            elem = (ProductionPatternElement) elements.get(i);
            if (elem.getMaxCount() >= Integer.MAX_VALUE) {
                return Integer.MAX_VALUE;
            } else {
                max += elem.getMaxCount();
            }
        }
        return max;
    }

    /**
     * Returns an element in this rule.
     * 
     * @param pos            the element position, 0 <= pos < count
     * 
     * @return the element found
     */
    public ProductionPatternElement getElement(int pos) {
        return (ProductionPatternElement) elements.get(pos);
    }

    /**
     * Adds a token to this rule. The token is appended to the end 
     * of the element list. The multiplicity values specified define 
     * if the token is optional or required, and if it can be 
     * repeated. 
     * 
     * @param id             the token (pattern) id
     * @param min            the minimum number of occurancies
     * @param max            the maximum number of occurancies, or
     *                       -1 for infinite 
     */
    public void addToken(int id, int min, int max) {
        addElement(new ProductionPatternElement(true, id, min, max));
    }
    
    /**
     * Adds a production to this rule. The production is appended to 
     * the end of the element list. The multiplicity values specified 
     * define if the production is optional or  required, and if it 
     * can be repeated. 
     * 
     * @param id             the production (pattern) id
     * @param min            the minimum number of occurancies
     * @param max            the maximum number of occurancies, or
     *                       -1 for infinite 
     */
    public void addProduction(int id, int min, int max) {
        addElement(new ProductionPatternElement(false, id, min, max));
    }

    /**
     * Adds a production pattern element to this rule. The element is 
     * appended to the end of the element list.
     * 
     * @param elem           the production pattern element
     */
    public void addElement(ProductionPatternElement elem) {
        elements.add(elem);
    }

    /**
     * Adds a production pattern element to this rule. The 
     * multiplicity values in the element will be overridden with the
     * specified values. The element is appended to the end of the 
     * element list.
     * 
     * @param elem           the production pattern element
     * @param min            the minimum number of occurancies
     * @param max            the maximum number of occurancies, or
     *                       -1 for infinite 
     */
    public void addElement(ProductionPatternElement elem, 
                           int min, 
                           int max) {

        if (elem.isToken()) {
            addToken(elem.getId(), min, max);
        } else {
            addProduction(elem.getId(), min, max);
        }
    }

    /**
     * Returns a string representation of this object.
     *
     * @return a token string representation
     */
    public String toString() {
        StringBuffer  buffer = new StringBuffer();

        for (int i = 0; i < elements.size(); i++) {
            if (i > 0) {
                buffer.append(" ");
            }
            buffer.append(elements.get(i));
        }
        return buffer.toString();
    }
}
