/*
 * Grammatica.java
 *
 * This work is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * As a special exception, the copyright holders of this library give
 * you permission to link this library with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also meet,
 * for each linked independent module, the terms and conditions of the
 * license of that module. An independent module is a module which is
 * not derived from or based on this library. If you modify this
 * library, you may extend this exception to your version of the
 * library, but you are not obligated to do so. If you do not wish to
 * do so, delete this exception statement from your version.
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.grammatica;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;

import net.percederberg.grammatica.output.CSharpParserGenerator;
import net.percederberg.grammatica.output.JavaParserGenerator;
import net.percederberg.grammatica.parser.ParseException;
import net.percederberg.grammatica.parser.Parser;
import net.percederberg.grammatica.parser.ParserCreationException;
import net.percederberg.grammatica.parser.Token;
import net.percederberg.grammatica.parser.Tokenizer;

/**
 * The GNU Parse main application. This class provides the
 * command-line interface for invoking the application. See separate
 * documentation for information on usage and command-line parameters.
 *
 * @author   Per Cederberg, <per at percederberg dot net>
 * @version  1.0
 */
public class Grammatica extends Object {

    /**
     * The command-line help output.
     */
    private static final String COMMAND_HELP =
        "Syntax: Grammatica <grammarfile> <action> [<options>]\n" +
        "\n" +
        "Actions:\n" +
        "  --tokenize <file>\n" +
        "      Debugs the grammar by using it to tokenize the specified\n" +
        "      file. No code has to be generated for this.\n" +
        "  --parse <file>\n" +
        "      Debugs the grammar by using it to parse the specified\n" +
        "      file. No code has to be generated for this.\n" +
        "  --javaoutput <dir>\n" +
        "      Creates a Java parser for the grammar (in source code).\n" +
        "      The specified directory with be used as base directory\n" +
        "      when generating source code files.\n" +
        "  --csoutput <dir>\n" +
        "      Creates a C# parser for the grammar (in source code).\n" +
        "      The specified directory with be used as base directory\n" +
        "      when generating source code files.\n" +
        "\n" +
        "Options:\n" +
        "  --javapackage <package>\n" +
        "      Sets the Java package to use in generated source code files.\n" +
        "  --javaclassname <name>\n" +
        "      Sets the Java class name prefix to use in generated source\n" +
        "      code files.\n" +
        "  --javapublic\n" +
        "      Sets public access for all Java classes and interfaces\n" +
        "      generated.\n" +
        "  --csnamespace <package>\n" +
        "      Sets the C# namespace to use in generated source code files.\n" +
        "  --csclassname <name>\n" +
        "      Sets the C# class name prefix to use in generated source\n" +
        "      code files.";

    /**
     * The application entry point.
     *
     * @param args           the command-line parameters
     */
    public static void main(String[] args) {
        Grammar  grammar = null;

        // Parse command-line arguments
        if (args.length == 1 && args[0].equals("--help")) {
            printHelp(null);
            System.exit(1);
        }
        if (args.length < 3 || args.length > 7) {
            printHelp("Wrong number of arguments");
            System.exit(1);
        }

        // Read grammar file
        try {
            grammar = new Grammar(new File(args[0]));
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            System.exit(1);
        } catch (GrammarException e) {
            e.printStackTrace();
            System.exit(1);
        }

        // Check action parameter
        if (args[1].equals("--tokenize")) {
            tokenize(grammar, new File(args[2]));
        } else if (args[1].equals("--parse")) {
            parse(grammar, new File(args[2]));
        } else if (args[1].equals("--javaoutput")) {
            writeJavaCode(args, grammar);
        } else if (args[1].equals("--csoutput")) {
            writeCSharpCode(args, grammar);
        } else {
            printHelp("unrecognized option: " + args[1]);
            System.exit(1);
        }
    }

    /**
     * Prints command-line help information.
     *
     * @param error          an optional error message, or null
     */
    private static void printHelp(String error) {
        System.err.println(COMMAND_HELP);
        System.err.println();
        if (error != null) {
            System.err.print("Error: ");
            System.err.println(error);
            System.err.println();
        }
    }

    /**
     * Tokenizes the specified file with the token patterns from the
     * grammar.
     * 
     * @param grammar        the grammar to use
     * @param file           the file to parse
     */
    private static void tokenize(Grammar grammar, File file) {
        Tokenizer  tokenizer;
        Token      token;
        
        try {
            tokenizer = grammar.createTokenizer(new FileReader(file));
            System.out.println("Tokens from " + file + ":");
            while ((token = tokenizer.next()) != null) {
                System.out.println(token);
            }
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            System.exit(1);
        } catch (ParserCreationException e) {
            e.printStackTrace();
            System.exit(1);
        } catch (ParseException e) {
            e.printStackTrace();
            System.exit(1);
        }
    }
    
    /**
     * Parses the specified file with the grammar.
     * 
     * @param grammar        the grammar to use
     * @param file           the file to parse
     */
    private static void parse(Grammar grammar, File file) {
        Tokenizer  tokenizer;
        Parser     parser;
        Token      token;
        
        try {
            tokenizer = grammar.createTokenizer(new FileReader(file));
            parser = grammar.createParser(tokenizer);
            System.out.println("Parse tree from " + file + ":");
            parser.parse().printTo(System.out);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            System.exit(1);
        } catch (ParserCreationException e) {
            e.printStackTrace();
            System.exit(1);
        } catch (ParseException e) {
            e.printStackTrace();
            System.exit(1);
        }
    }

    /**
     * Parses the command-line arguments and generates the Java source 
     * code for a parser. 
     * 
     * @param args           the command-line arguments
     * @param grammar        the grammar to use
     */
    private static void writeJavaCode(String[] args, Grammar grammar) {
        JavaParserGenerator gen = new JavaParserGenerator(grammar);

        // Read command-line arguments
        for (int i = 1; i < args.length; i++) {
            if (args[i].equals("--javaoutput")) {
                gen.setBaseDir(new File(args[++i]));
            } else if (args[i].equals("--javapackage")) {
                gen.setBasePackage(args[++i]);
            } else if (args[i].equals("--javaclassname")) {
                gen.setBaseName(args[++i]);
            } else if (args[i].equals("--javapublic")) {
                gen.setPublicAccess(true);
            } else {
                printHelp("unrecognized option: " + args[i]);
                System.exit(1);
            }
        }
        
        // Write parser source code
        try {
            System.out.println("Writing Java parser source code...");
            gen.write();
            System.out.println("Done.");
        } catch (IOException e) {
            e.printStackTrace();
            System.exit(1);
        }
    }

    /**
     * Parses the command-line arguments and generates the C# source 
     * code for a parser. 
     * 
     * @param args           the command-line arguments
     * @param grammar        the grammar to use
     */
    private static void writeCSharpCode(String[] args, Grammar grammar) {
        CSharpParserGenerator gen = new CSharpParserGenerator(grammar);

        // Read command-line arguments
        for (int i = 1; i < args.length; i++) {
            if (args[i].equals("--csoutput")) {
                gen.setBaseDir(new File(args[++i]));
            } else if (args[i].equals("--csnamespace")) {
                gen.setNamespace(args[++i]);
            } else if (args[i].equals("--csclassname")) {
                gen.setBaseName(args[++i]);
            } else {
                printHelp("unrecognized option: " + args[i]);
                System.exit(1);
            }
        }

        // Write parser source code
        try {
            System.out.println("Writing C# parser source code...");
            gen.write();
            System.out.println("Done.");
        } catch (IOException e) {
            e.printStackTrace();
            System.exit(1);
        }
    }
}
