#!/usr/bin/perl -w  # -*- cperl -*- #
#
#  playlist.pm - Allow users to create their own custom playlists.
#
#  GNU MP3D - A portable(ish) MP3 server.
#
# Homepage:
#   http://www.gnump3d.org/
#
# Author:
#  Steve Kemp <steve@steve.org.uk>
#
# Version:
#  $Id: playlist.pm,v 1.3 2003/10/27 19:29:36 skx Exp $
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
#
#  Steve Kemp
#  ---
#  http://www.steve.org.uk/
#
#


use gnump3d::files;

#
#  Count of songs and directories we've displayed.
#
my $count     = 0;
my $songCount = 0;

#
#  Return the author of this plugin.
#
sub getAuthor()
{
    return( 'Steve Kemp <steve@steve.org.uk>' );
}


#
#  Return the version of this plugin.
#
sub getVersion()
{
    my $REVISION      = '$Id: playlist.pm,v 1.3 2003/10/27 19:29:36 skx Exp $';
    my $VERSION       = "";
    $VERSION = join (' ', (split (' ', $REVISION))[1..3]);
    $VERSION =~ s/,v\b//;
    $VERSION =~ s/(\S+)$/($1)/;

    return( $VERSION );
}



#
#  Handle requests to this plugin.
#
sub handlePath( )
{
    #
    # Should we play the results of the form?
    #
    if ( defined $ARGUMENTS{ "play" }  )
    {
	my $playlist = "";
	foreach my $option ( keys %ARGUMENTS )
	{
	    if ( $option =~ /^dir/ )
	    {
		my $dir = $ARGUMENTS{ $option };
		my $dirPlaylist = &getPlaylist( $dir . "/all.m3u" );
		$playlist .= $dirPlaylist;
	    }
	    if ( $option =~ /^song/ )
	    {
		my $song        = $ARGUMENTS{ $option };
		$playlist .= "http://$host/$song\n";
	    }

	}

	#
	#  Remove the play argument from the cookies we send back
	# in our header.
	#
	#  If we don't do this subsequent requests to the /play/
	# URL will not view the form.
	#
	#  Also default to only seeing the directory names.
	# (This is substantially faster).
	#
	delete( $ARGUMENTS{ "play" } );
	delete( $ARGUMENTS{ "playlist_show_files" } );

	my $header   = &getHTTPHeader( 200, "audio/x-mpegurl" );
	&sendData($data, $header );
	&sendData($data, $playlist );
	close($data);
	exit;
    }

    delete( $ARGUMENTS{ "play" } );
    delete( $ARGUMENTS{ "playlist_show_files" } );

    my $header   = &getHTTPHeader( 200, "text/html" );
    &sendData( $data, $header );
    my $text = &showCustomPlaylistForm( $ARGUMENTS{"theme"});
    &sendData( $data, $text );
    close( $data );
    exit;
}


#
#  Show the list of all the top level directories within the music
# archive and allow people to choose between them.
#
sub showCustomPlaylistForm( $ )
{
    my ($theme) = (@_);
    my $text  ="";

    my @template = &getThemeFile( $theme, "playlist.html" );
    foreach my $line (@template )
    {
	#
	# Make global substitutions.
	#
        $line =~ s/\$HEADER//g;
	$line =~ s/\$HOSTNAME/$host/g;
	$line =~ s/\$VERSION/$VERSION/g;
	$line =~ s/\$RELEASE/$RELEASE/g;
	$line =~ s/\$DIRECTORY/\/playlist\//g;
	$line =~ s/\$HEADING/Custom Playlist Generation/g;
	$line =~ s/\$TITLE/Custom Playlist Generation/g;

	#
	# Now handle the special sections.
	#
	if ( $line =~ /(.*)\$BANNER(.*)/ )
	{
	    # Insert banner;
	    my $pre  = $1;
	    my $post = $2;

	    $text .= $pre;
	    $text .= &getBanner( "/playlist/" );
	    $text .= $post;
	}
	elsif ( $line =~ /(.*)\$TEXT(.*)/ )
	{
	    my $pre  = $1;
	    my $post = $2;

	    if ( defined( $ARGUMENTS{ "playlist_show_files"} ) and
		 ( $ARGUMENTS{"playlist_show_files" } == 1 ) )
	      {
		  $text .= "<p>&nbsp;If you would prefer to just see the names of directories, which would be much faster to process then please visit <a href='?playlist_show_files=0'>this link instead</a></p>\n";
	      }
	    else
	      {
		  $text .= "<p>&nbsp;If you would like to be able to choose individual songs, then please visit <a href='?playlist_show_files=1'>this link instead</a>.  (If you have a large music archive this will be slow)</p>\n";
	      }


	    $text .= "<form action=\"/playlist/\">\n";
	    $text .= "<input type=\"hidden\" name=\"play\" value=\"1\">\n";

	    $text .= $pre;
	    $text .= &getDirectoryContents( $ROOT );

	    $text .= "<input type=\"reset\" value=\"Clear All\"> <input type=\"submit\" value=\"Play\">\n";
	    $text .= "</form>\n";
	    $text .= $post;
	}
	else
	{
	    $text .= $line;
	}
    }

    return( $text );
}


#
#  build up a list of files and directories which may be selected
# from beneath the directory root.
#
sub getDirectoryContents( $ )
{
    my ( $dir ) = ( @_ );

    my $text = "<ul>\n";

    my @subdirs = ( );
    my @files   = ( );

    if ( -d $dir )
    {
	@subdirs = &dirsInDir( $dir );
    }

    my $showFilesOnly = 0;
    my $hideOGG       = 0;
    my $hideMP3       = 0;
    my $extension     = "";

    #
    # See if we should be including the names of all files,
    # and if so if we should hide particular filetypes.
    #
    if  ( (defined( $ARGUMENTS{"playlist_show_files" } ) ) and
	  ( $ARGUMENTS{ "playlist_show_files" } == 1 ) )
    {
	$showFilesOnly = 1;

	if ( defined( $ARGUMENTS{ "hide_mp3" } ) and
	     $ARGUMENTS{ "hide_mp3" } )
	{
	    $hideMP3 = 1;
	}

	if ( defined( $ARGUMENTS{ "hide_ogg" } ) and
	     $ARGUMENTS{ "hide_ogg" } )
	{
	    $hideOGG = 1;
	}

	if ( &getConfig( "always_stream", 0 ) )
	{
	    $extension = ".m3u";
	}
    }

    #
    # Per-plugin format string.
    #
    my $format = &getConfig( "plugin_playlist_song_format",
			     '$ARTIST - $SONGNAME' );

    #
    # Read in the files within the directory
    # if we should do this.
    #
    if ( $showFilesOnly )
    {
	if ( -d $dir )
	{
	    @files = &filesInDir( $dir );
	    push( @subdirs, @files );
	    @subdirs = sort( @subdirs );
	}
    }

    #
    # Process each file or song
    #
    foreach my $content ( @subdirs )
    {
	my $checked = "";

	#
	# Make the path fully qualified without the directory
	# root in the way.
	#
	my $file = $dir . "/" . $content;
	$file    = substr( $file, length( $ROOT ) );
	if ( ! $file =~ /\/.*/ )
	{
	    $file = "/" . $file;
	}

	if ( -d ( $dir . "/" . $content ) )
	{
	    if ( defined( $ARGUMENTS{"dir$count" } ) )
	    {
		$checked = "checked";
	    }

	    if ( $showFilesOnly )
	    {
		$text .= "<li> <a href=\"$file\">$content</a></li>";
	    }
	    else
	    {
		$text .= "<li><input type='checkbox' name='dir$count' value='$file' $checked> <a href=\"$file\">$content</a></li>";
		$count ++;
	    }
	}
	else
	{
	    if ( isAudio( $dir . "/" . $content ) )
	    {
		if ( defined( $ARGUMENTS{"song$songCount" } ) )
		{
		    $checked = "checked";
		}

		$content = &getSongDisplay( "$dir/$content", $format );

		$text .= "<li><input type='checkbox' name='song$songCount' value='$file' $checked> <a href=\"$file$extension\">$content</a></li>";
		$songCount++;
	    }
	}

	my $subText = &getDirectoryContents( $dir . "/" . $content );

	if ( length( $subText ) )
	{
	    $text .= $subText ;
	}
    }

    $text .= "</ul>\n";
    return( $text );
}
