;; -*- coding: utf-8 -*-
;;
;;   gnowsys-mode.el -- A major mode of Emacs for knowledge networking
;;   with GNOWSYS server (http://www.gnu.org/software/gnowsys/).
;;
;;   Author: Divya <divyas15@gmail.com>
;; 
;;   This file is part of gnowsys-mode for GNOWSYS: Gnowledge Networking 
;;   and Organizing System.
;;
;;   gnowsys-mode is free software; you can redistribute it and/or modify
;;   it under the terms of the GNU General Public License as
;;   published by the Free Software Foundation; either version 3 of
;;   the License, or (at your option) any later version.
;;
;;   gnowsys-mode is distributed in the hope that it will be useful, but
;;   WITHOUT ANY WARRANTY; without even the implied warranty of
;;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;   GNU General Public License for more details.
;;
;;   You should have received a copy of the GNU General Public
;;   License along with Gnome-Gnowser (gpl.txt); if not, write to the
;;   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
;;   Boston, MA  02110-1301  USA59 Temple Place, Suite 330,

(require 'org)
(require 'xml-rpc)

(eval-and-compile
  (require 'url-http))

(defvar gnowsys-buffer "*gnowsys*"
  "Buffer for gnowsys-mode")

(defconst gnow-message "\n gnowsys-mode is a an Emacs extension package for knowledge\n networking and organization using [[http://www.gnu.org/software/gnowsys/][GNOWSYS]] ([[http://lab.gnowledge.org/About/about-gnowledge.org/][Gnowledge]] Networking and\n Organizing SYStem) as a major mode. It is an extension package\n for collaborative knowledge management using [[http://www.gnu.org/software/gnowsys/][GNOWSYS]] server.\n\n You need a GNOWSYS server either locally or on the Internet.\n Read for details in 'README.txt' or 'README.html' found the\n gnwosys-mode distribution directory.\n\n You can use the function 'gnowsys-connect' at any time to\n initiate new sessions.\n\n Get gnowing!\n\n\t\t")

(defvar gnow-window nil)
(defvar url nil)
(defvar search-str nil)
(defvar display-symbol "*")
(defvar node-symbol "*")
(defconst gnowsys-nodes (list "object" "objecttype" "relation" "relationtype" "attribute" "attributetype" "metatype" "user" "usertype" "process" "processtype"))
(defvar image-url nil)
(defvar portal-url nil)
(defvar gnow-link-info nil)
(defvar link-type nil)
(defvar gnow-link-message nil)
(defvar start-content nil)
(defvar gnowsys-insert-status nil)

;;(setq org-drawers '("PROPERTIES" "CLOCK" "RELATIONS" "ATTRIBUTES" "SNAPSHOT" "VERSION" "VIEW")) 
(setq org-drawers '("PROPERTIES" "CLOCK" "RELATIONS" "ATTRIBUTES" "SNAPSHOTMETADATA" "VIEW")) 
(setq org-link-types '("http" "https" "ftp" "mailto" "file" "news" "shell" "elisp" "obj" "otype" "rel" "rtype" "attr" "atype" "proc" "ptype" "usr" "utype" "mtype" "func" "class" "flow" "gnow" "hasVersion"))

(autoload 'n3-mode "n3-mode" "Major mode for OWL or N3 files" t)
;; Turn on font lock when in n3 mode
(add-hook 'n3-mode-hook
          'turn-on-font-lock)

(setq auto-mode-alist
      (append
       (list
        '("\\.n3" . n3-mode)
        '("\\.owl" . n3-mode))
       auto-mode-alist))

(defvar gnowsys-gnowsys-map)

(define-derived-mode gnowsys-mode org-mode "Org GNOWSYS"
  "Major mode for collaboratively building knowledge networks using GNOWSYS server. It create a buffer *gnowsys* & changes the mode to gnowsys.
Commands:\\{gnowsys-mode-map}"
  
  (setq buffer-read-only nil)
  (setq start-content (buffer-string))
  
  (rename-buffer gnowsys-buffer)
  (erase-buffer)
  (setq truncate-lines nil word-wrap t)
  (insert gnow-message)
  
  (setq gnowsys-mode-map (current-local-map))
  (define-key gnowsys-mode-map [?\C-c ?\C-c] 'gnowsys-commit)
  (define-key gnowsys-mode-map [?\C-n ?\C-t] 'generate-template)
  (define-key gnowsys-mode-map [?\C-n ?\C-s] 'gnowsys-search)
  (define-key gnowsys-mode-map [?\C-n ?\C-c] 'gnowsys-connect)
  (define-key gnowsys-mode-map [?\C-n ?\C-g] 'gnowsys-getnodes)
  (define-key gnowsys-mode-map [?\C-n ?\C-n ?o] 'gnowsys-getnodes-object)
  (define-key gnowsys-mode-map [?\C-n ?\C-n ?O] 'gnowsys-getnodes-objecttype)
  (define-key gnowsys-mode-map [?\C-n ?\C-n ?p] 'gnowsys-getnodes-process)
  (define-key gnowsys-mode-map [?\C-n ?\C-n ?P] 'gnowsys-getnodes-processtype)
  (define-key gnowsys-mode-map [?\C-n ?\C-n ?a] 'gnowsys-getnodes-attribute)
  (define-key gnowsys-mode-map [?\C-n ?\C-n ?A] 'gnowsys-getnodes-attributetype)
  (define-key gnowsys-mode-map [?\C-n ?\C-n ?r] 'gnowsys-getnodes-relation)
  (define-key gnowsys-mode-map [?\C-n ?\C-n ?R] 'gnowsys-getnodes-relationtype)
  (define-key gnowsys-mode-map [?\C-n ?\C-n ?u] 'gnowsys-getnodes-user)
  (define-key gnowsys-mode-map [?\C-n ?\C-n ?U] 'gnowsys-getnodes-usertype)
  (define-key gnowsys-mode-map [?\C-n ?\C-n ?m] 'gnowsys-getnodes-metatype)
  (define-key gnowsys-mode-map [?\r] 'drawer-template)
  (define-key gnowsys-mode-map " " 'template-autolist)
  (define-key gnowsys-mode-map [(meta return)] 'gnowsys-insert)
  (define-key gnowsys-mode-map [?\C-n ?o] 'gnowsys-insert-object)
  (define-key gnowsys-mode-map [?\C-n ?O] 'gnowsys-insert-objecttype)
  (define-key gnowsys-mode-map [?\C-n ?a] 'gnowsys-insert-attribute)
  (define-key gnowsys-mode-map [?\C-n ?A] 'gnowsys-insert-attributetype)
  (define-key gnowsys-mode-map [?\C-n ?r] 'gnowsys-insert-relation)
  (define-key gnowsys-mode-map [?\C-n ?R] 'gnowsys-insert-relationtype)
  (define-key gnowsys-mode-map [?\C-n ?p] 'gnowsys-insert-process)
  (define-key gnowsys-mode-map [?\C-n ?P] 'gnowsys-insert-processtype)
  (define-key gnowsys-mode-map [?\C-n ?m] 'gnowsys-insert-metatype)
  (define-key gnowsys-mode-map [?\C-n ?\C-o] 'gnowsys-add-ontology)
  (define-key gnowsys-mode-map [?\C-n ?\C-c] 'gnowsys-ontologies)
  (define-key gnowsys-mode-map [?\C-n ?\C-e] 'add-to-gnowsys-ontology)
  (define-key gnowsys-mode-map [?\C-n ?\C-n ?e] 'gnowsys-export)
  (define-key gnowsys-mode-map [?\C-n ?d] 'gnowsys-n3-export)
  (define-key gnowsys-mode-map [?\C-n ?v] 'gnowsys-getversions)
  (define-key gnowsys-mode-map [?\C-n ?b] 'gnowsys-get-buffer)

  (easy-menu-define gnowsys-gnowsys-menu gnowsys-mode-map "gnowsys menu"
  '("Gnowsys"
    ["Search" gnowsys-search]
    ["HTML Export" gnowsys-export]
    ["N3 export" gnowsys-n3-export]
    ["Buffer Structure" gnowsys-get-buffer]
    ("Ontology"
     ["Add" gnowsys-add-ontology]
     ["Change Context" gnowsys-ontologies]
     )
    ["Add to Ontology" add-to-gnowsys-ontology]
    ("Insert"
     ["Metatype" gnowsys-insert-metatype]
     ["Objecttype" gnowsys-insert-objecttype]
     ["Object" gnowsys-insert-object]
     ["Attributetype" gnowsys-insert-attributetype]
     ["Attribute" gnowsys-insert-attribute]
     ["Relationtype" gnowsys-insert-relationtype]
     ["Relation" gnowsys-insert-relation]
     ["Processtype" gnowsys-insert-processtype]
     ["Process" gnowsys-insert-process]
     )
    ("Get all"
     ["Metatype" gnowsys-getnodes-metatype]
     ["Objecttype" gnowsys-getnodes-objecttype]
     ["Object" gnowsys-getnodes-object]
     ["Attributetype" gnowsys-getnodes-attributetype]
     ["Attribute" gnowsys-getnodes-attribute]
     ["Relationtype" gnowsys-getnodes-relationtype]
     ["Relation" gnowsys-getnodes-relation]
     ["Processtype" gnowsys-getnodes-processtype]
     ["Process" gnowsys-getnodes-process]
     ["Usertype" gnowsys-getnodes-usertype]
     ["User" gnowsys-getnodes-user]
     )
    ))

  (easy-menu-add gnowsys-gnowsys-menu)
  (gnowsys-connect)

;;  (erase-buffer)
;;  (insert start-content)
  
  (message "gnowsys-mode successfully loaded."))

(defvar gnow-ontology nil)
(defvar ontology-list '("None"))
(defvar ontology-ssid nil)
(defvar ontology-structure nil)

(defun gnowsys-connect ()
  "Prompts user to enter GNOWSYS server's URL, username & password. All the credentials being
   provided correctly, it establishes the connection to the specified server. Connection is established
   using xml-rpc-method-call function provided in xml-rpc.el file"

  (interactive)
  (if (not url)
      (setq url "http://sandboxat.gnowledge.org/gnowql"))
  (setq url (read-no-blanks-input "Enter GNOWSYS server URL: " url))
  (setq portal-url (concat (xml-rpc-method-call url 'getPortalUrl) "/gnowql"))
  (insert (concat "\n\n\n You are connected to " url))
  (gnowsys-ontologies)
  (message "You are connected: )")
  (insert "\n\nYou may start by searching the gnowledge base: M-x \"gnowsys-search\".")
  )

(defun gnowsys-ontologies ()
  "About all the ontologies residing in the system."

  (interactive)
  (setq ontology-list (xml-rpc-method-call url 'getOntologies))
  (push (cons "None" "None") ontology-list)
  (setq gnow-ontology (completing-read "Ontology: " ontology-list))
  (if (not (equal "None" gnow-ontology))
      (progn
	(setq ontology-ssid (cdr (assoc gnow-ontology ontology-list)))
	(setq ontology-structure (cdr (assoc "structure" (cdr (assoc "rendered_nbh" (elt (xml-rpc-method-call url 'getAllnbh (list ontology-ssid)) 0))))))
 	))
;;  (message ontology-structure)
  
  )

(defun gnowsys-get-ontologies ()
  "To get the list of ontologies in the buffer."

  (interactive)
  (erase-buffer)
  (insert "\n\n")
  (insert "List of Ontologies:\n\n")
  (setnode-type "otype")
  (setlink-type gnow-node)
  (setq ontology-list (xml-rpc-method-call url 'getOntologies))
  (dolist (i ontology-list)
    (insert "* " node-symbol " [[" link-type ":" (cdr (assoc (elt i 0) ontology-list)) "][" (elt i 0) "]]\n" )
    )
  (org-overview)
  )

(defvar search-result nil
  "This variable holds the search results.")

(defvar search-nodes (list "None" "object" "objecttype" "relation" "relationtype" "attribute" "attributetype" "metatype" "user" "usertype" "process" "processtype"))

(defun gnowsys-search ()
  "It prompts user to enter the search string and then looks into the GNOWSYS storage for matching entries.
   After getting the results it iterates over the resultset & then displays the result using disp function."

  (interactive)
  (setq temp nil)
  (setq search-str (read-from-minibuffer "Search string: "))
  (setq temp (completing-read "Node: " search-nodes))
  (if (equal url nil)
      (gnowsys-connect))
  (if (equal temp nil)
      (setq search-result (xml-rpc-method-call url 'search search-str))
    (progn
      (if (equal temp "process")
	     (setq temp (concat "gb" temp))
	   (setq temp (concat "gb" temp "s")))
      (setq search-result (xml-rpc-method-call url 'search search-str (list temp)))
      ))
  (erase-buffer)
  (insert (concat "\n\n Search result for \"" search-str "\""))
  (dolist (n search-result)
    (disp n))
  (org-overview)
  )


(defun disp (search-list)
  "search-list is an association list provided here. It displays the nid of the list with approriate symbol
   notation depending on its nodetype. Notations followed here are:
   O: --> object, OT: --> objecttype, A: --> attributes, AT: --> attributetypes, R: --> relations, RT: --> relationtypes."

  (setlink-type (cdr (assoc "nodetype" search-list)))
  (insert "\n")
  (org-insert-heading)
  (insert node-symbol " [[" link-type ":" (cdr (assoc "ssid" search-list)) "][" (cdr (assoc "nid" search-list)) "]]")
)

(defvar all-nbh nil)
(defvar result-nbh nil
  "This variable holds the neighbourhood information of the individual search result.")
(defvar search-attr nil
  "It extracts & holds the attributes keys of the individual search result.")
(defvar search-rel nil
  "It extracts & holds the relations keys of the individual search result.")
(defvar nbh-keys nil)
(defvar rel-keys nil)
(defvar nbh-ssid nil)
(defvar nbh-ssid-nid nil)
(defvar nbh-ssid-nodetype nil)
(defvar keys nil)
(defvar rel-drawer-open nil)
(defvar attr-drawer-open nil)
(defvar snap-drawer-open nil)
(defvar ver-drawer-open nil)
(defvar nbh-start-pos nil)
(defvar nbh-end-pos nil)
(defvar content-flag nil)

(defun disp-expand (search-ssid search-nid)
  "It collects & displays the neighbourhood of the selected node.
   Relations in the neighbourhood are represented within RELATIONS drawer & 
   attributes in the neighbourhood are represented within ATTRIBUTES drawer."

  (setq nbh-start-pos (point))
  (setq all-nbh (elt (xml-rpc-method-call url 'getAllnbh (list search-ssid)) 0))
  (setq result-nbh (cdr (assoc "rendered_nbh" all-nbh)))
  (setq nbh-keys (get-keys result-nbh))
  (nbh-getnids result-nbh nbh-keys)
  (if (member "relations" nbh-keys)
      (progn
	(insert "\n\n:RELATIONS:")
	(setq rel-drawer-open t)
	(dolist (k search-rel)
	  (dolist (role (get-keys (cdr (assoc k (cdr (assoc "relations" result-nbh))))))
	    (if (equal role "rightroles")
		(progn
		  (if (equal k "instanceof")
		      (insert "\n:memberof:")
		    (if (equal k "subtypeof")
			(insert "\n:typeof:")
		      (insert (concat "\n:" k ":" ))))
		  
		  (dolist (i (cdr (assoc role (assoc k (cdr (assoc "relations" result-nbh))))))
		    (setlink-type (cdr (assoc i nbh-ssid-nodetype)))
		    (insert (concat " [[" link-type ":" i "][" (cdr (assoc i nbh-ssid-nid)) "]];")))
		  ))))))

  (if (member "relationtypes" nbh-keys)
      (progn
	(if (not rel-drawer-open)
	    (progn
	      (insert "\n\n:RELATIONS:")
		(setq rel-drawer-open t)))
	(insert "\n:RelationType:")
	(setlink-type "gbrelationtypes")
	(dolist (k (cdr (assoc "relationtypes" result-nbh)))
	  (insert (concat " [[" link-type ":" k "][" (cdr (assoc k nbh-ssid-nid)) "]];")))
	))
  
  (if (member "subjecttypes" nbh-keys)
      (progn
	(if (not rel-drawer-open)
	    (progn
	      (insert "\n\n:RELATIONS:")
	      (setq rel-drawer-open t)))
	(insert "\n:SubjectType:")
	(dolist (k (cdr (assoc "subjecttypes" result-nbh)))
	  (setlink-type (cdr (assoc k nbh-ssid-nodetype)))
	  (insert (concat " [[" link-type ":" k "][" (cdr (assoc k nbh-ssid-nid)) "]];")))
	))

  (if (member "priorstatedef" nbh-keys)
      (progn
	(if (not rel-drawer-open)
	    (progn
	      (insert "\n\n:RELATIONS:")
	      (setq rel-drawer-open t)))
	(insert "\n:priorstatedef:")
	(dolist (k (cdr (assoc "priorstatedef" result-nbh)))
	  (setlink-type (cdr (assoc k nbh-ssid-nodetype)))
	  (insert (concat " [[" link-type ":" k "][" (cdr (assoc k nbh-ssid-nid)) "]];")))
	))

  (if (member "poststatedef" nbh-keys)
      (progn
	(if (not rel-drawer-open)
	    (progn
	      (insert "\n\n:RELATIONS:")
	      (setq rel-drawer-open t)))
	(insert "\n:poststatedef:")
	(dolist (k (cdr (assoc "poststatedef" result-nbh)))
	  (setlink-type (cdr (assoc k nbh-ssid-nodetype)))
	  (insert (concat " [[" link-type ":" k "][" (cdr (assoc k nbh-ssid-nid)) "]];")))
	))
  
  (if (member "priorstate" nbh-keys)
      (progn
	(if (not rel-drawer-open)
	    (progn
	      (insert "\n\n:RELATIONS:")
	      (setq rel-drawer-open t)))
	(insert "\n:priorstate:")
	(dolist (k (cdr (assoc "priorstate" result-nbh)))
	  (setlink-type (cdr (assoc k nbh-ssid-nodetype)))
	  (insert (concat " [[" link-type ":" k "][" (cdr (assoc k nbh-ssid-nid)) "]];")))
	))

  (if (member "poststate" nbh-keys)
      (progn
	(if (not rel-drawer-open)
	    (progn
	      (insert "\n\n:RELATIONS:")
	      (setq rel-drawer-open t)))
	(insert "\n:poststate:")
	(dolist (k (cdr (assoc "poststate" result-nbh)))
	  (setlink-type (cdr (assoc k nbh-ssid-nodetype)))
	  (insert (concat " [[" link-type ":" k "][" (cdr (assoc k nbh-ssid-nid)) "]];")))
	))

  (if rel-drawer-open
      (progn
	(setq rel-drawer-open nil)
	(insert "\n:END:")))
  
  (setq nbh-end-pos (point))
  (indent-rigidly nbh-start-pos nbh-end-pos 2)

  (if (member"attributes" nbh-keys)
      (progn
	(insert "\n\n  :ATTRIBUTES:")
	(setq attr-drawer-open t)
	(setq search-attr (get-keys (cdr (assoc "attributes" result-nbh))))
	(setlink-type "gbattributes")
	(dolist (k search-attr)
	  (if (equal k "content")
		(setq content-flag t)
	    (if (not (member k '("email" "password")))
		(if (elt (cdr (assoc k (cdr (assoc "attributes" result-nbh)))) 0)
		    (if (listp (elt (cdr (assoc k (cdr (assoc "attributes" result-nbh)))) 1))
			(progn
			  (insert "\n  :" k ":")
			  (dolist (temp (elt (cdr (assoc k (cdr (assoc "attributes" result-nbh)))) 1))
			    (insert " " temp ";")))
		      (insert (concat "\n  :" k ": " (elt (cdr (assoc k (cdr (assoc "attributes" result-nbh)))) 1))))))))))

  (if (member "attributetypes" nbh-keys)
      (progn
	(if (not attr-drawer-open)
	    (progn
	      (insert "\n\n  :ATTRIBUTES:")
	      (setq attr-drawer-open t)))
	(insert "\n  :AttributeType:")
	(setlink-type "gbattributetypes")
	(dolist (k (cdr (assoc "attributetypes" result-nbh)))
	  (insert (concat " [[" link-type ":" k "][" (cdr (assoc k nbh-ssid-nid)) "]];")))
	))

  (if (member "inversename" nbh-keys)
      (progn
	(if (not attr-drawer-open)
	    (progn
	      (insert "\n\n  :ATTRIBUTES:")
	      (setq attr-drawer-open t)))
	(insert "\n  :inversename:")
	(if (cdr (assoc "inversename" result-nbh))
	    (insert " " (cdr (assoc "inversename" result-nbh))))
	))

  (if (member "structure" nbh-keys)
      (progn
	(if (not attr-drawer-open)
	    (progn
	      (insert "\n\n  :ATTRIBUTES:")
	      (setq attr-drawer-open t)))
	(insert "\n  :Structure:")
	(if (listp (cdr (assoc "structure" result-nbh)))
	    (dolist (k (cdr (assoc "structure" result-nbh)))
	      (insert  " " k  ";"))
	  (insert " " (cdr (assoc "structure" result-nbh)) ";"))
	))

  (if attr-drawer-open
      (progn
	(setq attr-drawer-open nil)
	(insert "\n  :END:")))

  (setq nbh-start-pos (point))
  
;;  (insert "\n\n:SNAPSHOT:")
  (insert "\n\n:SNAPSHOTMETADATA:")
  (insert "\n:ssid: " (number-to-string (cdr (assoc "ssid" all-nbh))))
  (insert "\n:inid: " (number-to-string (cdr (assoc "inid" all-nbh))))
  (insert "\n:status: " (cdr (assoc "status" all-nbh)))
  (insert "\n:uid: " (number-to-string (cdr (assoc "uid" all-nbh))))
  (get-unique-values (xml-rpc-method-call url 'getContributors search-ssid))
  (insert "\n:contributors:")
  (dolist (i val-list)
    (insert " " i ";"))
;;  (insert "\n:END:")
  
;;  (insert "\n\n:VERSION:")
  (insert "\n:noofchanges: " (number-to-string (cdr (assoc "noofchanges" all-nbh))))
  (insert "\n:noofcommits: " (number-to-string (cdr (assoc "noofcommits" all-nbh))))
  (insert "\n:noofchangesaftercommit: " (number-to-string (cdr (assoc "noofchangesaftercommit" all-nbh))))
  (insert "\n:changetype:" )
  (dolist (i (cdr (assoc "changetype" all-nbh)))
    (insert " " (number-to-string i) ";"))
  (if (equal (buffer-name) gnowsys-buffer)
      (progn
	(insert "\n:history:" )
	(dolist (i (cdr (assoc "history" all-nbh)))
	  (if (not (equal i 0))
	      (insert " [[hasVersion:" (number-to-string i) "][" (number-to-string i) "]];")))
	))
  (insert "\n:fieldschanged:" )
  (dolist (i (cdr (assoc "fieldschanged" all-nbh)))
    (insert " " i ";"))
  (insert "\n:timestamp: " (cdr (assoc "gbtimestamp" all-nbh)))
  (insert "\n:END:")

  (insert "\n\n:VIEW:")
  (insert "\n:MAP: [[" portal-url "/genimage?id=" search-ssid "][Graph]]")
  (insert "\n:RDF: [[" portal-url "/genN3?ssid=" search-ssid "&system=gnowsys][N3]]")
  (insert "\n:END:")

  (setq nbh-end-pos (point))
  (indent-rigidly nbh-start-pos nbh-end-pos 2)

  (if content-flag
	(progn
	  (if (elt (cdr (assoc "content" (cdr (assoc "attributes" result-nbh)))) 1)
	      (progn
		(insert "\n\n")
		(insert (elt (cdr (assoc "content" (cdr (assoc "attributes" result-nbh)))) 1))))
	  (setq content-flag nil)))
  )

(defvar val-list nil)

(defun get-unique-values (a-list)
  "To get unique values from the association list."

  (setq val-list nil)
  (dolist (i (get-keys a-list))
    (if (not (member (cdr (assoc i a-list)) val-list))
	(push (cdr (assoc i a-list)) val-list))))

(defvar rel-keys nil)

(defun nbh-getnids (nbh keys)
  "To get ssid-nids pair of all the ssid in the nbh of a node."

  (setq nbh-ssid nil)
  (if (member "attributetypes" keys)
      (setq nbh-ssid (append (cdr (assoc "attributetypes" nbh)) nbh-ssid)))
  (if (member "relationtypes" keys)
      (setq nbh-ssid (append (cdr (assoc "relationtypes" nbh)) nbh-ssid)))
  (if (member "subjecttypes" keys)
      (setq nbh-ssid (append (cdr (assoc "subjecttypes" nbh)) nbh-ssid)))
  (if (member "priorstatedef" keys)
      (setq nbh-ssid (append (cdr (assoc "priorstatedef" nbh)) nbh-ssid)))
  (if (member "poststatedef" keys)
      (setq nbh-ssid (append (cdr (assoc "poststatedef" nbh)) nbh-ssid)))
  (if (member "priorstate" keys)
      (setq nbh-ssid (append (cdr (assoc "priorstate" nbh)) nbh-ssid)))
  (if (member "poststate" keys)
      (setq nbh-ssid (append (cdr (assoc "poststate" nbh)) nbh-ssid)))
  (if (member "relations" keys)
      (progn
	(setq search-rel (get-keys (cdr (assoc "relations" nbh))))
	(dolist (i search-rel)
	  (setq rel-keys (get-keys (cdr (assoc i (cdr (assoc "relations" nbh))))))
	  (dolist (k rel-keys)
	    (setq nbh-ssid (append (cdr (assoc k  (cdr (assoc i (cdr (assoc "relations" nbh)))))) nbh-ssid))))))
  (if nbh-ssid
      (progn
	(setq nbh-ssid-nid (xml-rpc-method-call url 'getnids nbh-ssid))
	(setq nbh-ssid-nodetype (xml-rpc-method-call url 'getNodetypes nbh-ssid)))))

(defvar add-node nil)
(defvar node-nid nil)
(defvar gnowsys-ots nil)
(defvar gnowsys-ats nil)
(defvar gnowsys-os nil)
(defvar gnowsys-rts nil)
(defvar gnowsys-pts nil)
(defvar gnowsys-meta nil)
(defvar gnowsys-nodetype (list "types" "tokens"))
(defvar gnowsys-type-list (list "metatype" "objecttype" "relationtype" "attributetype"))
(defvar at-selectiontype nil)
(defvar gnow-list nil)
(defvar gnow-reg nil)
(defvar at-regex nil)
(defvar at-listid nil)
(defvar at-length nil)
(defvar at-range nil)
(defvar at-precision nil)
(defvar at-restriction nil)
(defvar meta-sub nil)
(defvar a-ins nil)
(defvar rel-subjects nil)
(defvar rel-rt nil)
(defvar a-value nil)
(defvar rt-rev nil)
(defvar username nil)
(defvar password nil)
(defvar email nil)
(defvar rt-nodetype nil)
(defvar gnowsys-set-value nil)
(defvar gnow-title nil)
(defvar gnow-desc nil)
(defvar gnow-content nil)
(defvar gnow-datatypes nil)
(defvar next-char nil)
(defvar temp-list nil)
(defvar dict-list nil)
(defvar datatype nil)
(defvar list-val nil)
(defvar attr-ins-list nil)
(defconst at-restrict-precision-datatype '("int2" "int4" "int8" "int2[]" "int4[]" "int8[]" "float4" "float8" "float4[]" "float8[]"))
(defvar selected-nodetype nil)

(defun gnowsys-addnode ()
  "Prompts to add node to Gnowsys. Depending upon the nodetype selected, 
it prompts for the required fields to be be filled in. Triple mode representation
is followed while adding the node. ;; on any line means termination of that particular
property or relation."

  (interactive)
  (erase-buffer)
  (setq gnowsys-set-value nil)
  (setq add-node (completing-read "Node: " gnowsys-nodes))
;;  (erase-buffer)
  (insert (concat "Add " add-node "\n\n") )
  
  (cond ((equal add-node "object")
	 (progn
	 (push (cons "nodetype" add-node) gnowsys-set-value) 
	 (push (cons "status" "public") gnowsys-set-value) 
	 (push (cons "language" "english") gnowsys-set-value) 
	 (setlink-type "gbobjects") 
	 (org-insert-heading)
	 (insert node-symbol " ")
	 (setq node-nid (read-from-minibuffer "Nid: "))
	 (nid-check node-nid)
	 (insert "[[" link-type ":][" node-nid "]]\n\t")
	 (push (cons "nid" node-nid) gnowsys-set-value) 
	 (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbobjecttypes"))
	 (insert "==member of==") 
	 (setq temp-list nil)
	 (setlink-type "gbobjecttypes") 
	 (gnow-recursive-complete "instanceof" gnowsys-ots 0)
	 (insert "::title:: ")
	 (setq gnow-title (read-from-minibuffer "Title: "))
	 (insert gnow-title ";;\n\t::description:: ")
	 (setq gnow-desc (read-from-minibuffer "Description: "))
	 (insert gnow-desc ";;\n\t::content:: ")
	 (setq gnow-content  (read-from-minibuffer "Content: "))
	 (insert gnow-content ";;\n\t")
	 (setq temp-list nil)
	 (if (not (equal (length gnow-title) 0))
	     (push (vector "title" gnow-title) temp-list))
	 (if (not (equal (length gnow-desc) 0))
	     (push (vector "description" gnow-desc) temp-list))
	 (if (not (equal (length gnow-content) 0))
	     (push (vector "content" gnow-content) temp-list))
	 (if temp-list
	     (push (cons "attributes" (vconcat temp-list)) gnowsys-set-value))
	 (message "Press C-c C-c to commit")))

	((equal add-node "objecttype")
	 (progn
	 (push (cons "nodetype" add-node) gnowsys-set-value) 
	 (push (cons "status" "public") gnowsys-set-value) 
	 (push (cons "language" "english") gnowsys-set-value) 
	 (setlink-type "gbobjecttypes")
         (org-insert-heading)
         (insert node-symbol " ")
	 (setq node-nid (read-from-minibuffer "Nid: "))
	 (nid-check node-nid)
	 (setlink-type "gbobjects")
         (org-insert-heading)
         (insert node-symbol " ")
	 (insert "[[" link-type ":][" node-nid "]]\n\t")
	 (push (cons "nid" node-nid) gnowsys-set-value) 
	 (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbobjecttypes"))
	 (insert "==type of==") 
	 (setq temp-list nil)
	 (setlink-type "gbobjecttypes")
	 (gnow-recursive-complete "subtypeof" gnowsys-ots 0)
	 (setq gnowsys-meta (xml-rpc-method-call url 'getAll "gbmetatypes"))
	 (insert "==member of==") 
	 (setq temp-list nil)
	 (setlink-type "gbmetatypes")
	 (gnow-recursive-complete "instanceof" gnowsys-meta 0)
	 (insert "::title:: ")
	 (setq gnow-title (read-from-minibuffer "Title: "))
	 (insert gnow-title ";;\n\t::description:: ")
	 (setq gnow-desc (read-from-minibuffer "Description: "))
	 (insert gnow-desc ";;\n\t")
	 (setq temp-list nil)
	 (if (not (equal (length gnow-title) 0))
	     (push (vector "title" gnow-title) temp-list))
	 (if (not (equal (length gnow-desc) 0))
	     (push (vector "description" gnow-desc) temp-list))
	 (if temp-list
	     (push (cons "attributes" (vconcat temp-list)) gnowsys-set-value))
	 (message "Press C-c C-c to commit")))

	((equal add-node "process")
	 (progn
	 (push (cons "nodetype" add-node) gnowsys-set-value) 
	 (push (cons "status" "public") gnowsys-set-value) 
	 (push (cons "language" "english") gnowsys-set-value) 
	 (setq node-nid (read-from-minibuffer "Nid: "))
	 (nid-check node-nid)
	 (setlink-type "gprocess")
         (org-insert-heading)
         (insert node-symbol " ")
	 (insert "[[" link-type ":][" node-nid "]]\n\t")
	 (push (cons "nid" node-nid) gnowsys-set-value) 
	 (setq gnowsys-pts (xml-rpc-method-call url 'getAll "gbprocesstypes"))
	 (insert "==member of==") 
	 (setq temp-list nil)
	 (setlink-type "gbprocesstypes")
	 (gnow-recursive-complete "instanceof" gnowsys-pts 0)
	 (setq gnowsys-ots (xml-rpc-method-call url 'getPriorstates (cdr (assoc "instanceof" gnowsys-set-value))))
	 (insert "==prior state==") 
	 (setq temp-list nil)
	 (setlink-type "gbobjects")
	 (gnow-recursive-complete "priorstate" gnowsys-ots 0)
	 (setq gnowsys-ots (xml-rpc-method-call url 'getPoststates (cdr (assoc "instanceof" gnowsys-set-value))))
	 (insert "==post state==") 
	 (setq temp-list nil)
	 (setlink-type "gbobjects")
	 (gnow-recursive-complete "poststate" gnowsys-ots 0)
	 (insert "::title:: ")
	 (setq gnow-title (read-from-minibuffer "Title: "))
	 (insert gnow-title ";;\n\t::description:: ")
	 (setq gnow-desc (read-from-minibuffer "Description: "))
	 (insert gnow-desc ";;\n\t")
	 (setq temp-list nil)
	 (if (not (equal (length gnow-title) 0))
	     (push (vector "title" gnow-title) temp-list))
	 (if (not (equal (length gnow-desc) 0))
	     (push (vector "description" gnow-desc) temp-list))
	 (if temp-list
	     (push (cons "attributes" (vconcat temp-list)) gnowsys-set-value))
	 (message "Press C-c C-c to commit")))

	((equal add-node "processtype")
	 (progn
	 (push (cons "nodetype" add-node) gnowsys-set-value) 
	 (push (cons "status" "public") gnowsys-set-value) 
	 (push (cons "language" "english") gnowsys-set-value) 
	 (setq node-nid (read-from-minibuffer "Nid: "))
	 (nid-check node-nid)
	 (setlink-type "gbprocesstypes")
         (org-insert-heading)
         (insert node-symbol " ")
	 (insert "[[" link-type ":][" node-nid "]]\n\t")
	 (push (cons "nid" node-nid) gnowsys-set-value) 
	 (setq gnowsys-pts (xml-rpc-method-call url 'getAll "gbprocesstypes"))
	 (insert "==type of==") 
	 (setq temp-list nil)
	 (setlink-type "gbprocesstypes")
	 (gnow-recursive-complete "subtypeof" gnowsys-pts 0)
	 (setq gnowsys-meta (xml-rpc-method-call url 'getAll "gbmetatypes"))
	 (insert "==member of==") 
	 (setq temp-list nil)
	 (setlink-type "gbmetatypes")
	 (gnow-recursive-complete "instanceof" gnowsys-meta 0)
	 (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbobjecttypes"))
	 (insert "==prior state def==") 
	 (setq temp-list nil)
	 (setlink-type "gbobjecttypes")
	 (gnow-recursive-complete "priorstatedef" gnowsys-ots 0)
	 (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbobjecttypes"))
	 (insert "==post state def==") 
	 (setq temp-list nil)
	 (setlink-type "gbobjecttypes")
	 (gnow-recursive-complete "poststatedef" gnowsys-ots 0)
	 (insert "::title:: ")
	 (setq gnow-title (read-from-minibuffer "Title: "))
	 (insert gnow-title ";;\n\t::description:: ")
	 (setq gnow-desc (read-from-minibuffer "Description: "))
	 (insert gnow-desc ";;\n\t")
	 (setq temp-list nil)
	 (if (not (equal (length gnow-title) 0))
	     (push (vector "title" gnow-title) temp-list))
	 (if (not (equal (length gnow-desc) 0))
	     (push (vector "description" gnow-desc) temp-list))
	 (if temp-list
	     (push (cons "attributes" (vconcat temp-list)) gnowsys-set-value))
	 (message "Press C-c C-c to commit")))


	((equal add-node "metatype")
	 (progn
	 (push (cons "nodetype" add-node) gnowsys-set-value) 
	 (push (cons "status" "public") gnowsys-set-value) 
	 (push (cons "language" "english") gnowsys-set-value) 
	 (setq node-nid (read-from-minibuffer "Nid: "))
	 (nid-check node-nid)
	 (setlink-type "gbmetatypes")
         (org-insert-heading)
         (insert node-symbol " ")
	 (insert "[[" link-type ":][" node-nid "]]\n\t")
	 (push (cons "nid" node-nid) gnowsys-set-value) 
	 (setq gnowsys-meta (xml-rpc-method-call url 'getAll "gbmetatypes"))
	 (insert "==type of==")
	 (setq temp-list nil)
	 (setlink-type "gbmetatypes")
	 (gnow-recursive-complete "subtypeof" gnowsys-meta 0)
	 (insert "::title:: ")
	 (setq gnow-title (read-from-minibuffer "Title: "))
	 (insert gnow-title ";;\n\t::description:: ")
	 (setq gnow-desc (read-from-minibuffer "Description: "))
	 (insert gnow-desc ";;\n\t")
	 (setq temp-list nil)
	 (if (not (equal (length gnow-title) 0))
	     (push (vector "title" gnow-title) temp-list))
	 (if (not (equal (length gnow-desc) 0))
	     (push (vector "description" gnow-desc) temp-list))
	 (if temp-list
	     (push (cons "attributes" (vconcat temp-list)) gnowsys-set-value))
	 (message "Press C-c C-c to commit")))
	
	((equal add-node "usertype")
	 (progn
	 (push (cons "nodetype" add-node) gnowsys-set-value) 
	 (push (cons "status" "public") gnowsys-set-value) 
	 (push (cons "language" "english") gnowsys-set-value) 
	 (setq node-nid (read-from-minibuffer "Nid: "))
	 (nid-check node-nid)
	 (setlink-type "gbusertypes")
         (org-insert-heading)
         (insert node-symbol " ")
	 (insert "[[" link-type ":][" node-nid "]]\n\t")
	 (push (cons "nid" node-nid) gnowsys-set-value) 
	 (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbusertypes"))
	 (insert "==type of==") 
	 (setq temp-list nil)
	 (setlink-type "gbusertypes")
	 (gnow-recursive-complete "subtypeof" gnowsys-ots 0)
	 (setq gnowsys-meta (xml-rpc-method-call url 'getAll "gbmetatypes"))
	 (insert "==member of==") 
	 (setq temp-list nil)
	 (setlink-type "gbmetatypes")
	 (gnow-recursive-complete "instanceof" gnowsys-meta 0)
	 (insert "::title:: ")
	 (setq gnow-title (read-from-minibuffer "Title: "))
	 (insert gnow-title ";;\n\t::description:: ")
	 (setq gnow-desc (read-from-minibuffer "Description: "))
	 (insert gnow-desc ";;\n\t")
	 (setq temp-list nil)
	 (if (not (equal (length gnow-title) 0))
	     (push (vector "title" gnow-title) temp-list))
	 (if (not (equal (length gnow-desc) 0))
	     (push (vector "description" gnow-desc) temp-list))
	 (if temp-list
	     (push (cons "attributes" (vconcat temp-list)) gnowsys-set-value))
	 (message "Press C-c C-c to commit")))

	((equal add-node "user")
	 (progn
	 (push (cons "nodetype" add-node) gnowsys-set-value) 
	 (push (cons "status" "public") gnowsys-set-value) 
	 (setlink-type "gbusers")
         (org-insert-heading)
         (insert node-symbol " \n\t")
	 (setq gnowsys-uts (xml-rpc-method-call url 'getAll "gbusertypes"))
	 (insert "==member of==") 
	 (setq temp-list nil)
	 (setlink-type "gbusertypes")
	 (gnow-recursive-complete "instanceof" gnowsys-uts 0)
	 (insert "::username:: ")
	 (setq username (read-from-minibuffer "Username: "))
	 (insert username ";;\n\t::password:: ")
	 (setq password (read-passwd "Password: "))
	 (insert (make-string (length password) ?.) ";;\n\t::email:: ")
	 (setq email  (read-no-blanks-input "Email: "))
	 (insert email ";;\n\t")
	 (push (cons "attributes" (vector (vector "username" username) (vector "password" password) (vector "email" email))) gnowsys-set-value) 
	 (message "Press C-c C-c to commit")))

	((equal add-node "attribute")
	 (progn
	 (setq attr-ins-list nil)
	 (push (cons "nodetype" add-node) gnowsys-set-value) 
	 (push (cons "status" "public") gnowsys-set-value) 
	 (push (cons "language" "english") gnowsys-set-value) 
	 (setlink-type "gbattributes")
         (org-insert-heading)
         (insert node-symbol " \n\t")
	 (setq selected-nodetype (completing-read "Select type for subject: " gnowsys-nodes))
	 (if (equal selected-nodetype "process")
	     (setq selected-nodetype (concat "gb" gnow-node))
	   (setq selected-nodetype (concat "gb" gnow-node "s")))
;;	 (setq selected-nodetype (concat "gb" (completing-read "Select type for subject: " gnowsys-nodes) "s"))
	 (setlink-type selected-nodetype)
	 (setq gnowsys-os (xml-rpc-method-call url 'getAll selected-nodetype))
	 (insert "==subject== ")
	 (setq a-subj (completing-read "subject: " gnowsys-os))
	 (insert " [[" link-type ":" (number-to-string (cdr (assoc a-subj gnowsys-os)))"]["a-subj"]];;\n\t")
	 (push (cons "subject" (number-to-string (cdr (assoc a-subj gnowsys-os)))) gnowsys-set-value)
	 (setq gnowsys-ats (xml-rpc-method-call url 'getPossibleAttr (number-to-string (cdr (assoc a-subj gnowsys-os)))))
	 (insert "==member of==") 
	 (setlink-type "gbattributetypes")
	 (setq a-ins (attribute-template gnowsys-ats))
	 (push (cons "instanceof" (vconcat a-ins)) gnowsys-set-value)
	 (message "Press C-c C-c to commit")))

	((equal add-node "attributetype")
	 (progn
	 (push (cons "nodetype" add-node) gnowsys-set-value) 
	 (push (cons "status" "public") gnowsys-set-value) 
	 (setq node-nid (read-from-minibuffer "Nid: "))
	 (nid-check node-nid)
	 (setlink-type "gbattributetypes")
         (org-insert-heading)
         (insert node-symbol " ")
	 (insert "[[" link-type ":][" node-nid "]]\n\t")
	 (push (cons "nid" node-nid) gnowsys-set-value) 
	 (setq gnowsys-meta (xml-rpc-method-call url 'getAll "gbmetatypes"))
	 (insert "==member of==") 
	 (setq temp-list nil)
	 (setlink-type "gbmetatypes")
	 (gnow-recursive-complete "instanceof" gnowsys-meta 0)
	 (setq gnowsys-ats (xml-rpc-method-call url 'getAll "gbattributetypes"))
	 (insert "==type of==") 
	 (setq temp-list nil)
	 (setlink-type "gbattributetypes")
	 (gnow-recursive-complete "subtypeof" gnowsys-ats 0)
	 (setq selected-nodetype (completing-read "Select type for subject: " gnowsys-nodes))
	 (if (equal selected-nodetype "process")
	     (setq selected-nodetype (concat "gb" gnow-node))
	   (setq selected-nodetype (concat "gb" gnow-node "s")))
;;	 (setq selected-nodetype (concat "gb" (completing-read "Select type for subject: " gnowsys-nodes) "s"))
	 (setlink-type selected-nodetype)
	 (setq gnowsys-ots (xml-rpc-method-call url 'getAll selected-nodetype))
	 (insert "==subjecttypes==")
	 (setq temp-list nil)
	 (gnow-recursive-complete "subjecttypes" gnowsys-ots 0)

	 (insert "::applicableto::") 
	 (setq rt-nodetype (completing-read "Applicableto: " gnowsys-nodetype))
	 (insert rt-nodetype ";;\n\t")
	 (if (not (equal (length rt-nodetype) 0))
	     (push (cons "applicablenodetypes" rt-nodetype) gnowsys-set-value) )

	 (insert "::restrictiontype:: ")
	 (setq at-restriction (completing-read "restrictiontype: " (list "valuerestriction" "selectionlist")))
	 (if (equal at-restriction "valuerestriction")
	     (push (cons "restrictiontype" 1) gnowsys-set-value)
	   (push (cons "restrictiontype" 0) gnowsys-set-value))
	 (insert at-restriction ";;\n\t")

	 (if (equal at-restriction "selectionlist")
	     (progn 
	       (insert "::selectiontype:: ")
	       (setq at-selectiontype (completing-read "selectionlist: " (list "multipleselection" "singleselection")))
	       (if (equal at-selectiontype "multipleselection")
		   (push (cons "selectiontype" 1) gnowsys-set-value)
		 (push (cons "selectiontype" 0) gnowsys-set-value))
	       (insert at-restriction ";;\n\t")
	       (setq gnow-list (xml-rpc-method-call url 'getSelectionlist))
	       (insert "::listname:: ")
	       (setq at-listid (completing-read "listname: " (list-from-dict gnow-list "nid" "listid")))
	       (insert at-listid ";;\n\t")
	       (if (not (equal (length at-listid) 0))		   
		   (push (cons "listid" (cdr (assoc at-listid dict-list))) gnowsys-set-value)))
	   (progn
	     (insert "::datatype:: ")
	     (setq gnow-datatypes (xml-rpc-method-call url 'getDatatypes "all"))
	     (setq datatype (completing-read "Datatype: " gnow-datatypes))
	     (if (not (equal (length datatype) 0))		   
		 (push (cons "datatype" datatype) gnowsys-set-value))
	     (insert datatype ";;\n\t")
	     (setq gnow-reg (xml-rpc-method-call url 'getRegex))
	     (insert "::regex:: ")
	     (setq at-regex (completing-read "Regular Expression: " (list-from-dict gnow-reg "exp" "id")))
	     (insert at-regex ";;\n\t")
	     (if (not (equal (length at-regex) 0))		   
		 (push (cons "regex" at-regex) gnowsys-set-value))
	     (insert "::length:: ")
	     (setq at-length (read-from-minibuffer "Length: "))
	     (insert at-length ";;\n\t")
	     (if (not (equal (length at-length) 0))		   
		 (push (cons "length" (string-to-number at-length)) gnowsys-set-value))
	     (if (member datatype at-restrict-precision-datatype)
		 (progn
		   (insert "::range:: ")
		   (setq at-range (read-from-minibuffer "Range: "))
		   (insert at-range ";;\n\t")
		   (if (not (equal (length at-range) 0))		   
		       (push (cons "range" at-range) gnowsys-set-value))
		   (insert "::precision:: ")
		   (setq at-precision (read-from-minibuffer "Precision: "))
		   (insert at-precision ";;\n\t")
		   (if (not (equal (length at-precision) 0))		   
		       (push (cons "precision" (string-to-number at-precision)) gnowsys-set-value))
		   ))))

	 (message "Press C-c C-c to commit")))

	((equal add-node "relation")
	 (progn
	 (push (cons "nodetype" add-node) gnowsys-set-value) 
	 (push (cons "status" "public") gnowsys-set-value) 
	 (setlink-type "gbrelations")
         (org-insert-heading)
         (insert node-symbol " ")
	 (insert "[[" link-type ":][" node-nid "]]\n\t")


	 (setq gnowsys-rts (xml-rpc-method-call url 'getAll "gbrelationtypes"))
	 (setq gnowsys-rts (remove (cons "instanceof" (cdr (assoc "instanceof" gnowsys-rts))) gnowsys-rts))
	 (setq gnowsys-rts (remove (cons "subtypeof" (cdr (assoc "subtypeof" gnowsys-rts))) gnowsys-rts))
	 (setq rel-rt (completing-read "Relationtype: " gnowsys-rts))
	 (setlink-type "gbrelationtypes")
	 (insert "==RT== [[" link-type ":" (cdr (assoc rel-rt gnowsys-rts)) "][" rel-rt "]];;\n\t")
	 (push (cons "relationtypes" rel-rt) gnowsys-set-value) 

	 (setq rel-subjects (xml-rpc-method-call url 'getPossibleRel rel-rt))
	 (if (elt rel-subjects 0)
	     (setlink-type (xml-rpc-method-call url 'getNodetype (cdr (assoc (elt (elt (elt rel-subjects 0) 0) 0) (elt rel-subjects 0))))))
	 (insert "==subject1==") 
	 (gnow-recursive-complete "subject1" (elt rel-subjects 0) 0)
	 (if (elt rel-subjects 0)
	     (setlink-type (xml-rpc-method-call url 'getNodetype (cdr (assoc (elt (elt (elt rel-subjects 1) 0) 0) (elt rel-subjects 1))))))
	 (insert "==subject2== ")
	 (gnow-recursive-complete "subject2" (elt rel-subjects 1) 0)
	 (message "Press C-c C-c to commit")))

	((equal add-node "relationtype")
	 (progn
	 (push (cons "nodetype" add-node) gnowsys-set-value) 
	 (push (cons "status" "public") gnowsys-set-value) 
	 (setq node-nid (read-from-minibuffer "Nid: "))
	 (nid-check node-nid)
	 (setlink-type "gbrelationtypes")
         (org-insert-heading)
         (insert node-symbol " ")
	 (insert "[[" link-type ":][" node-nid "]]\n\t::inversename:: ")
	 (push (cons "nid" node-nid) gnowsys-set-value) 
	 (setq rt-rev (read-from-minibuffer "Inversename: ")) 
	 (insert rt-rev ";;\n\t")
	 (push (cons "inversename" rt-rev) gnowsys-set-value) 
	 (setq gnowsys-meta (xml-rpc-method-call url 'getAll "gbmetatypes"))
	 (insert "==member of==")
	 (setq temp-list nil)
	 (setlink-type "gbmetatypes")
	 (gnow-recursive-complete "instanceof" gnowsys-meta 0)
	 (setq gnowsys-rts (xml-rpc-method-call url 'getAll "gbrelationtypes"))
	 (setq gnowsys-rts (remove (cons "instanceof" (cdr (assoc "instanceof" gnowsys-rts))) gnowsys-rts))
	 (setq gnowsys-rts (remove (cons "subtypeof" (cdr (assoc "subtypeof" gnowsys-rts))) gnowsys-rts))
	 (insert "==type of==")
	 (setq temp-list nil)
	 (setlink-type "gbrelationtypes")
	 (gnow-recursive-complete "subtypeof" gnowsys-rts 0)
	 (setq selected-nodetype (completing-read "Select type for subject: " gnowsys-nodes))
	 (if (equal selected-nodetype "process")
	     (setq selected-nodetype (concat "gb" gnow-node))
	   (setq selected-nodetype (concat "gb" gnow-node "s")))
;;	 (setq selected-nodetype (concat "gb" (completing-read "Select type for subject: " gnowsys-nodes) "s"))
	 (setlink-type selected-nodetype)
	 (setq gnowsys-ots (xml-rpc-method-call url 'getAll selected-nodetype))
	 (insert "==subjecttype1==")
	 (setq temp-list nil)
	 (gnow-recursive-complete "subjecttype1" gnowsys-ots 0)
	 (setq selected-nodetype (completing-read "Select type for subject: " gnowsys-nodes))
	 (if (equal selected-nodetype "process")
	     (setq selected-nodetype (concat "gb" gnow-node))
	   (setq selected-nodetype (concat "gb" gnow-node "s")))
;;	 (setq selected-nodetype (concat "gb" (completing-read "Select type for subject: " gnowsys-nodes) "s"))
	 (setlink-type selected-nodetype)
	 (setq gnowsys-ots (xml-rpc-method-call url 'getAll selected-nodetype))
	 (insert "==subjecttype2==") 
	 (setq temp-list nil)
	 (gnow-recursive-complete "subjecttype2" gnowsys-ots 0)
	 (insert "::applicableto::") 
	 (setq rt-nodetype (completing-read "Applicableto: " gnowsys-nodetype))
	 (insert rt-nodetype ";;")
	 (if (not (equal (length rt-nodetype) 0))
	     (push (cons "applicablenodetypes" rt-nodetype) gnowsys-set-value) )
	 (message "Press C-c C-c to commit")))
	))


(defun nid-check (nid)
  "Tests if nid already exists in the kb.
   If already exists prompts user to re-enter the nid."

  (loop
   (if (or (equal (xml-rpc-method-call url 'nidExists nid) 1) (equal (length nid) 0))
       (setq node-nid (read-from-minibuffer "Re-enter Nid: ")))
   (if (equal (xml-rpc-method-call url 'nidExists node-nid) 0)
       (return))))
       
(defun list-from-dict (dictlist dictkey dictvalue)
  "To make a list from dict."

  (setq dict-list nil)
  (dolist (i dictlist)
    (push (cons (cdr (assoc dictkey i)) (cdr (assoc dictvalue i))) dict-list))
  dict-list)

(defvar regex nil)

(defun gnowsys-setregex ()
  "To set regular expression in GNOWSYS."
  
  (interactive)
  (erase-buffer)
  (insert "Add Regular Expression\n\n$\n\t::regex:: ")
  (setq add-node "regex")
  (setq gnowsys-set-value nil)
  (setq regex (read-no-blanks-input "Regular Expression: "))
  (insert regex ";;\n\t::description:: ")
  (push (cons "regex" regex) gnowsys-set-value) 
  (setq gnow-desc (read-from-minibuffer "Description: "))
  (insert gnow-desc ";;")
  (push (cons "description" gnow-desc) gnowsys-set-value)
  (message "Press C-c C-c to commit"))



(defun gnowsys-setlist ()
  "To set list into GNOWSYS."
  
  (interactive)
  (erase-buffer)
  (insert "Add List\n\n[ ]\n\t::listname:: ")
  (setq temp-list nil)
  (setq add-node "list")
  (setq gnowsys-set-value nil)
  (setq node-nid (read-no-blanks-input "Listname: "))
  (insert node-nid ";;\n\t::datatype:: ")
  (push (cons "nid" node-nid) gnowsys-set-value) 
  (push (cons "status" "public") gnowsys-set-value) 
  (push (cons "language" "english") gnowsys-set-value) 
  (setq gnow-datatypes (xml-rpc-method-call url 'getDatatypes "array"))
  (setq datatype (completing-read "Datatype: " gnow-datatypes))
  (insert datatype ";;\n\t::structure:: ")
  (push (cons "datatype" datatype) gnowsys-set-value)
  (loop
   (setq list-val (read-from-minibuffer "Value: "))
   (insert list-val "; ")
   (push list-val temp-list)
   (message (concat "Type ; to stop adding values."))
   (setq next-char (read-char))
   (if (equal next-char 59)
       (progn
	 (insert ";")
	 (push (cons "structure" (vconcat temp-list)) gnowsys-set-value) 
	 (return))))
  (message "Press C-c C-c to commit"))
  
(defun get-keys (dict)
  "It return the keys of the assoctiation list."

  (setq keys nil)
  (dolist (n dict)
    (push (elt n 0) keys))
  keys)

(defvar temp-name nil)

(defun image-display ( ssid )
  "Calls the function to generate node graph. 
   Downloads the graph from server to local filesystem in ~/.gnowsys.
   Deletes the *Shell Command Outbut* buffer window & inserts the image
   in the current buffer. "

  (xml-rpc-method-call url 'genNBHImage ssid)
  (shell-command (concat "wget -P ~/.gnowsys " image-url ssid ".png"))
  (if (get-buffer-window "*Shell Command Output*")
      (delete-window (get-buffer-window "*Shell Command Output*")))
  (insert "\n\n  ")
  (setq temp-name (concat (expand-file-name "~") "/.gnowsys/" ssid ".png"))
  (insert-image (create-image temp-name))
  )


(defun gnowsys-commit ()
  "Commits data to gnowsys"

  (interactive)
  (if (equal add-node "regex")
      (xml-rpc-method-call url 'setregex gnowsys-set-value)
    (progn
      (if (equal add-node "list")
	  (xml-rpc-method-call url 'setlist gnowsys-set-value)
	(progn
	  (if (not gnowsys-set-value)
	      (progn
		(gnowsys-head)
		(if (equal (xml-rpc-method-call url 'nidExists node-nid) 0)
		    (progn
		      (commit-template)
		      (xml-rpc-method-call url 'setnode gnowsys-set-value)
		      (if (member gnow-node '("gbrelations" "gbattributes"))
			  (progn
			    (goto-char context-start-pos)
			    (beginning-of-line)
			    (delete-region (point) context-end-pos))
			(progn
			  (setq context-ssid (cdr (assoc node-nid (xml-rpc-method-call url 'getlatestssids (list node-nid) "nid"))))
			  (update-presentation context-ssid node-nid))
			)
		      (if (and (not (equal gnow-ontology "None")) (not (equal gnow-ontology nil)) (member gnow-node '("gbobjecttypes" "gbrelationtypes" "gbattributetypes")) (equal ont-flag nil))
			  (ontology-update))
		      (if (equal ont-flag t)
			  (progn 
			    (setq gnow-ontology node-nid)
			    (setq ontology-ssid (cdr (assoc gnow-ontology (xml-rpc-method-call url 'getlatestssids (list gnow-ontology) "nid"))))
			    (setq ontology-structure (cdr (assoc "structure" (cdr (assoc "rendered_nbh" (elt (xml-rpc-method-call url 'getAllnbh (list ontology-ssid)) 0))))))
			    (setq ont-flag nil)
			    ))
		      )
		  (progn
		    (update-template)
		    (xml-rpc-method-call url 'Update gnowsys-set-value)
		    (if (member gnow-node '("gbrelations" "gbattributes"))
			  (progn
			    (goto-char context-start-pos)
			    (beginning-of-line)
			    (delete-region (point) context-end-pos))
		      (progn
			(setq context-ssid (cdr (assoc node-nid (xml-rpc-method-call url 'getlatestssids (list node-nid) "nid"))))
			(update-presentation context-ssid node-nid))
			)
		    )))
	    (xml-rpc-method-call url 'setnode gnowsys-set-value))
	  ))))
  (setq gnowsys-set-value nil)
  (setq gnow-node nil)
  (message "Successfully added !!!"))

(defvar relation-keys nil)
(defvar gnowsys-commit-relations nil)
(defvar attribute-keys nil)
(defvar gnowsys-commit-attributes nil)

(defun update-presentation (ssid nid)
  "Updates presentation in the buffer whenever it is updated in the server."

  (goto-char context-start-pos)
  (beginning-of-line)
  (delete-region (point) context-end-pos)
  (setlink-type gnow-node)
  (org-insert-heading)
  (insert node-symbol " ")
  (insert "[[" link-type ":" ssid "][" nid "]]\n")
  (disp-expand ssid nid)
  (insert "\n")
  (org-cycle-hide-drawers 'all)
  )
  
(defun commit-template ()
  "Create template to commit."

  (setq relation-keys nil)
  (setq gnowsys-commit-relations nil)
  (setq gnowsys-commit-attributes nil)
  
  (push (cons "nodetype" add-node) gnowsys-set-value)

  (if (member gnow-node '("gbobjects" "gbobjecttypes" "gbprocess" "gbprocesstypes" "gbmetatypes" "gbusertypes" "gbattributetypes" "gbrelationtypes"))
      (push (cons "nid" node-nid) gnowsys-set-value))
  (if (member gnow-node '("gbobjects" "gbobjecttypes" "gbprocess" "gbprocesstypes" "gbmetatypes" "gbusertypes" "gbusers" "gbattributes" "gbattributetypes" "gbrelations" "gbrelationtypes"))
      (push (cons "status" "public") gnowsys-set-value))
  (if (member gnow-node '("gbobjects" "gbobjecttypes" "gbprocess" "gbprocesstypes" "gbmetatypes" "gbusertypes" "gbattributes"))
      (push (cons "language" "english") gnowsys-set-value))
  (setq relation-keys (get-keys rel-element-list))
  (dolist (i relation-keys)
    (if (> (length (cdr (assoc i rel-element-list))) 0)
	(cond ((member i '("subjecttype2" "subjecttype1" "subject2" "subject1" "poststatedef" "priorstatedef" "poststate" "priorstate" "subtypeof" "instanceof" "subject" "subjecttypes" "relationtypes"))
	       (push (cons i (cdr (assoc i rel-element-list))) gnowsys-set-value))

	      ((not (member i '("RelationType" "subjecttype2" "subjecttype1" "subject2" "subject1" "poststatedef" "priorstatedef" "poststate" "priorstate" "subtypeof" "instanceof" "subject" "subjecttypes" "relationtypes")))
	       (push (vector i (cdr (assoc i rel-element-list))) gnowsys-commit-relations)))))
  (setq attribute-keys (get-keys attr-element-list))
  (dolist (i attribute-keys)
    (if (> (length (cdr (assoc i attr-element-list))) 0)
	(progn
	  (if (not (member i '("AttributeType" "restrictiontype" "datatype" "regex" "length" "range" "precision" "selectiontype" "listid" "inversename" "applicablenodetypes")))
	     (push (vector i (cdr (assoc i attr-element-list))) gnowsys-commit-attributes))
	  (if (equal i "restrictiontype")
	      (if (equal (cdr (assoc i attr-element-list)) "valuerestriction")
		  (push (cons "restrictiontype" 1) gnowsys-set-value)
		(progn
		  (push (cons "restrictiontype" 0) gnowsys-set-value)
		  (push (cons "selectiontype" 1) gnowsys-set-value)
		  )))
	  (if (member i '("inversename" "datatype" "regex" "length" "range" "precision" "listid" "applicablenodetypes"))
	      (push (cons i (cdr (assoc i attr-element-list))) gnowsys-set-value))
      )))
  (if (equal gnow-node "gbattributetypes") 
      (if (equal (cdr (assoc "restrictiontype" attr-element-list)) "selectionlist")
	  (push (cons "selectiontype" 1) gnowsys-set-value)))
  (if (> (length gnowsys-commit-attributes) 0)
      (if (equal gnow-node "gbattributes")
	  (push (cons "instanceof" (vconcat gnowsys-commit-attributes)) gnowsys-set-value)
	(push (cons "attributes" (vconcat gnowsys-commit-attributes)) gnowsys-set-value)))
  (if (> (length gnowsys-commit-relations) 0)
      (push (cons "relation" (vconcat gnowsys-commit-relations)) gnowsys-set-value)))


(defvar update-commit nil)

(defun ontology-update ()
  "Generates template to update ontogy"
  
  (setq gnowsys-set-value nil)
  (push (cons "subject" ontology-ssid) gnowsys-set-value)
  (push (cons "status" "public") gnowsys-set-value)
  (push (cons "language" "en") gnowsys-set-value)
  (push (cons "commit" 1) gnowsys-set-value)
  (if (equal ontology-structure nil)
      (setq ontology-structure (list context-ssid))
    (push context-ssid ontology-structure))
  
  (push (cons "structure" (vconcat ontology-structure)) gnowsys-set-value)
  
  (xml-rpc-method-call url 'Update gnowsys-set-value)
  (setq ontology-ssid (cdr (assoc gnow-ontology (xml-rpc-method-call url 'getlatestssids (list gnow-ontology) "nid"))))
  (setq ontology-structure (cdr (assoc "structure" (cdr (assoc "rendered_nbh" (elt (xml-rpc-method-call url 'getAllnbh (list ontology-ssid)) 0))))))
  )

(defvar temp-ont nil)
(defvar temp-ont-list nil)
(defvar temp-structure nil)

(defun add-to-gnowsys-ontology ()
  "To add the context node to one of the existing Ontologies."

  (interactive)
  (gnowsys-head)
  (if (member gnow-node '("gbobjecttypes" "gbrelationtypes" "gbattributetypes"))
      (progn
	(setq temp-ont (completing-read "Ontology: " (setq temp-ont-list (xml-rpc-method-call url 'getOntologies))))
	(setq gnowsys-set-value nil)
	(push (cons "subject" (cdr (assoc temp-ont temp-ont-list))) gnowsys-set-value)
	(push (cons "status" "public") gnowsys-set-value)
	(push (cons "language" "en") gnowsys-set-value)
	(push (cons "commit" 1) gnowsys-set-value)
	
	(print "Working till here")
	(setq temp-structure (cdr (assoc "structure" (cdr (assoc "rendered_nbh" (elt (xml-rpc-method-call url 'getAllnbh (list (cdr (assoc temp-ont temp-ont-list)))) 0))))))
	(print temp-structure)
	(if temp-structure
	    (push (cons "structure" (vconcat (push context-ssid temp-structure))) gnowsys-set-value)
	  (push (cons "structure" (vconcat (list context-ssid))) gnowsys-set-value))
	
	(xml-rpc-method-call url 'Update gnowsys-set-value)
	)
    (message "Tokens can not be added to an ontology. Current node is a token."))
  )

(defun update-template ()
  "Create template to commit."

  (setq relation-keys nil)
  (setq gnowsys-commit-relations nil)
  (setq gnowsys-commit-attributes nil)
  (push (cons "subject" context-ssid) gnowsys-set-value)
  (push (cons "status" "public") gnowsys-set-value)
  (push (cons "language" "en") gnowsys-set-value)
  (setq update-commit (completing-read "Commit status: " '("save" "publish")))
  (if (equal update-commit "save")
      (push (cons "commit" 0) gnowsys-set-value)
    (push (cons "commit" 1) gnowsys-set-value))
  (setq relation-keys (get-keys rel-element-list))
  (dolist (i relation-keys)
    (if (> (length (cdr (assoc i rel-element-list))) 0)
	(if (not (member i '("RelationType")))
	    (push (vector i (cdr (assoc i rel-element-list))) gnowsys-commit-relations))
      ))
  (setq attribute-keys (get-keys attr-element-list))
  (dolist (i attribute-keys)
    (if (> (length (cdr (assoc i attr-element-list))) 0)
	(if (not (member i '("AttributeType" "restrictiontype" "datatype" "regex" "length" "range" "precision" "selectiontype" "listid" "inversename" "applicablenodetypes")))
	     (push (vector i (cdr (assoc i attr-element-list))) gnowsys-commit-attributes))))
  (if (> (length gnowsys-commit-attributes) 0)
      (if (equal gnow-node "gbattributes")
	  (push (cons "instanceof" (vconcat gnowsys-commit-attributes)) gnowsys-set-value)
	(push (cons "attributes" (vconcat gnowsys-commit-attributes)) gnowsys-set-value)))
  (if (> (length gnowsys-commit-relations) 0)
      (push (cons "relation" (vconcat gnowsys-commit-relations)) gnowsys-set-value)))
	      
(defvar ind nil)
(defvar end-cond nil)
(defvar disp-text nil)
(defvar attr-ins nil)
(defvar attr-res nil)
(defvar attr-val nil)
(defvar attr-val-list nil)


(defun attribute-template (at-list)
  "This function generates template for attributes.
   It takes the list of all possible attributes as input."

  (setq temp-list nil)
  (setq attr-ins (completing-read "Select attribute: " at-list))
  (insert "\n\t\t::" attr-ins ":: ")
  (setq attr-res (cdr (assoc (cdr (assoc attr-ins at-list)) (xml-rpc-method-call url 'getRestrictions (list (cdr (assoc attr-ins at-list)))))))
  (if (equal (cdr (assoc "type" attr-res)) "0")
      (progn
	(setq attr-val-list (xml-rpc-method-call url 'getStructure (cdr (assoc "listid" attr-res))))
	(if (equal (cdr (assoc "selectiontype" attr-res)) "0")
	    (progn
	      (setq attr-val (completing-read "Select value: " attr-val-list))
	      (if (not (equal (length attr-val) 0))
		  (push attr-val temp-list))
	      (insert attr-val ";;"))
	  (progn
	    (loop
	     (setq attr-val (completing-read "Select value: " attr-val-list))
	     (insert attr-val ";")
	     (if (not (equal (length attr-val) 0))
		 (push attr-val temp-list))
	     (setq attr-val-list (remove attr-val attr-val-list))
	     (if (equal (length attr-val-list) 0)
		 (return)
	       (progn
		 (message "Type ; to stop adding values.")
		 (setq next-char (read-char))
		 
		 (if (equal next-char 59)
		     (return)))))
	    (insert ";"))))
    (progn
      (setq attr-val (read-from-minibuffer "Enter value: "))
      (if (not (equal (length attr-val) 0))
	  (push attr-val temp-list))
      (insert attr-val ";;")))
  (if temp-list
      (push (vector attr-ins (vconcat temp-list)) attr-ins-list))
  (setq at-list (remove (cons attr-ins (cdr (assoc attr-ins at-list))) at-list))
  (if (y-or-n-p "Want to add more attribute?") 
      (attribute-template at-list))
  attr-ins-list)
	     

(defun gnow-recursive-complete (text glist count)
  "This function recursively calls the autocompletion.
  Like: object's instanceof  "

  (setq end-cond nil)
  (if (equal text "instanceof")
      (setq disp-text "Member of")
    (if (equal text "subtypeof")
	(setq disp-text "Type of")
      (setq disp-text text)))
  (setq ind (completing-read (concat disp-text ": ") glist))
  (if (equal (length ind) 0)
      (progn
	(if (>= count 1)
	    (setq end-cond t)
	  (insert ";")))
    (progn
      (if (stringp (cdr (assoc ind glist)))
	  (insert " [[" link-type ":" (cdr (assoc ind glist))"]["ind"]];")
	(insert " [[" link-type ":" (number-to-string (cdr (assoc ind glist)))"]["ind"]];"))
      (if (equal count 0)
	  (progn
	    (if (or (equal text "instanceof") (equal text "subject1") (equal text "subject2") (equal text "subtypeof") (equal text "priorstatedef") (equal text "poststatedef") (equal text "priorstate") (equal text "poststate"))
		(progn
		  (if (stringp (cdr (assoc ind glist)))
		      (setq temp-list (list (cdr (assoc ind glist))))
		    (setq temp-list (list (number-to-string (cdr (assoc ind glist)))))))
	      (setq temp-list (list ind))))
	(progn
	  (if (or (equal text "instanceof") (equal text "subject1") (equal text "subject2") (equal text "subtypeof") (equal text "priorstatedef") (equal text "poststatedef") (equal text "priorstate") (equal text "poststate"))
	      (progn
		(if (stringp (cdr (assoc ind glist)))
		    (push (cdr (assoc ind glist)) temp-list)
		  (push (number-to-string (cdr (assoc ind glist))) temp-list)))
	    (push ind temp-list))))
      (setq glist (remove (cons ind (cdr (assoc ind glist))) glist))))

  (if (and (not end-cond) (not (equal (length glist) 0)))
      (progn
	(setq count (+ count 1))
	(message (concat "Type ; to stop adding " text "."))
	(setq next-char (read-char)))
    (setq next-char 59))
  (if (and (not (equal next-char 59)) (not (equal (length glist) 0)))
      (gnow-recursive-complete text glist count)
    (progn 
      (if temp-list
	  (push (cons text (vconcat temp-list)) gnowsys-set-value))
      (insert ";\n\t"))))

(defun org-open-at-point (&optional in-emacs)
  "Open link at or after point.
If there is no link at point, this function will search forward up to
the end of the current subtree.
Normally, files will be opened by an appropriate application.  If the
optional argument IN-EMACS is non-nil, Emacs will visit the file.
With a double prefix argument, try to open outside of Emacs, in the
application the system uses for this file type."
  (interactive "P")
  (org-load-modules-maybe)
  (move-marker org-open-link-marker (point))
  (setq org-window-config-before-follow-link (current-window-configuration))
  (org-remove-occur-highlights nil nil t)
  (cond
   ((org-at-timestamp-p t) (org-follow-timestamp-link))
   ((or (org-footnote-at-reference-p) (org-footnote-at-definition-p))
    (org-footnote-action))
   (t 
    (let (type path link line search (pos (point)))
      (catch 'match
	(save-excursion
	  (skip-chars-forward "^]\n\r")
	  (when (org-in-regexp org-bracket-link-regexp)
	    (setq link (org-extract-attributes
			(org-link-unescape (org-match-string-no-properties 1))))
	    (while (string-match " *\n *" link)
	      (setq link (replace-match " " t t link)))
	    (setq link (org-link-expand-abbrev link))
	    (cond
	     ((or (file-name-absolute-p link)
		  (string-match "^\\.\\.?/" link))
	      (setq type "file" path link))
	     ((string-match org-link-re-with-space3 link)
	      (setq type (match-string 1 link) path (match-string 2 link)))
	     (t (setq type "thisfile" path link)))
	    (throw 'match t)))

	(when (get-text-property (point) 'org-linked-text)
	  (setq type "thisfile"
		pos (if (get-text-property (1+ (point)) 'org-linked-text)
			(1+ (point)) (point))
		path (buffer-substring
		      (previous-single-property-change pos 'org-linked-text)
		      (next-single-property-change pos 'org-linked-text)))
	  (throw 'match t))

	(save-excursion
	  (when (or (org-in-regexp org-angle-link-re)
		    (org-in-regexp org-plain-link-re))
	    (setq type (match-string 1) path (match-string 2))
	    (throw 'match t)))
	(when (org-in-regexp "\\<\\([^><\n]+\\)\\>")
	  (setq type "tree-match"
		path (match-string 1))
	  (throw 'match t))
	(save-excursion
	  (when (org-in-regexp (org-re "\\(:[[:alnum:]_@:]+\\):[ \t]*$"))
	    (setq type "tags"
		  path (match-string 1))
	    (while (string-match ":" path)
	      (setq path (replace-match "+" t t path)))
	    (throw 'match t))))
      (unless path
	(error "No link found"))
      ;; Remove any trailing spaces in path
      (if (string-match " +\\'" path)
	  (setq path (replace-match "" t t path)))
      (if (and org-link-translation-function
	       (fboundp org-link-translation-function))
	  ;; Check if we need to translate the link
	  (let ((tmp (funcall org-link-translation-function type path)))
	    (setq type (car tmp) path (cdr tmp))))

      (cond

       ((assoc type org-link-protocols)
	(funcall (nth 1 (assoc type org-link-protocols)) path))

       ((equal type "mailto")
	(let ((cmd (car org-link-mailto-program))
	      (args (cdr org-link-mailto-program)) args1
	      (address path) (subject "") a)
	  (if (string-match "\\(.*\\)::\\(.*\\)" path)
	      (setq address (match-string 1 path)
		    subject (org-link-escape (match-string 2 path))))
	  (while args
	    (cond
	     ((not (stringp (car args))) (push (pop args) args1))
	     (t (setq a (pop args))
		(if (string-match "%a" a)
		    (setq a (replace-match address t t a)))
		(if (string-match "%s" a)
		    (setq a (replace-match subject t t a)))
		(push a args1))))
	  (apply cmd (nreverse args1))))

       ((member type '("http" "https" "ftp" "news"))
	(browse-url (concat type ":" (org-link-escape
				      path org-link-escape-chars-browser))))

       ((member type '("message"))
	(browse-url (concat type ":" path)))

       ((string= type "tags")
	(org-tags-view in-emacs path))
       ((string= type "thisfile")
	(if in-emacs
	    (switch-to-buffer-other-window
	     (org-get-buffer-for-internal-link (current-buffer)))
	  (org-mark-ring-push))
	(let ((cmd `(org-link-search
		     ,path
		     ,(cond ((equal in-emacs '(4)) 'occur)
			    ((equal in-emacs '(16)) 'org-occur)
			    (t nil))
		     ,pos)))
	  (condition-case nil (eval cmd)
	    (error (progn (widen) (eval cmd))))))

       ((string= type "tree-match")
	(org-occur (concat "\\[" (regexp-quote path) "\\]")))

       ((string= type "file")
	(if (string-match "::\\([0-9]+\\)\\'" path)
	    (setq line (string-to-number (match-string 1 path))
		  path (substring path 0 (match-beginning 0)))
	  (if (string-match "::\\(.+\\)\\'" path)
	      (setq search (match-string 1 path)
		    path (substring path 0 (match-beginning 0)))))
	(if (string-match "[*?{]" (file-name-nondirectory path))
	    (dired path)
	  (org-open-file path in-emacs line search)))

       ((string= type "news")
	(require 'org-gnus)
	(org-gnus-follow-link path))

       ((string= type "shell")
	(let ((cmd path))
	  (if (or (not org-confirm-shell-link-function)
		  (funcall org-confirm-shell-link-function
			   (format "Execute \"%s\" in shell? "
				   (org-add-props cmd nil
				     'face 'org-warning))))
	      (progn
		(message "Executing %s" cmd)
		(shell-command cmd))
	    (error "Abort"))))

       ((string= type "elisp")
	(let ((cmd path))
	  (if (or (not org-confirm-elisp-link-function)
		  (funcall org-confirm-elisp-link-function
			   (format "Execute \"%s\" as elisp? "
				   (org-add-props cmd nil
				     'face 'org-warning))))
	      (message "%s => %s" cmd
		       (if (equal (string-to-char cmd) ?\()
			   (eval (read cmd))
			 (call-interactively (read cmd))))
	    (error "Abort"))))

       ((member type '("obj" "otype" "rel" "rtype" "attr" "atype" "proc" "ptype" "usr" "utype" "mtype" "func" "class" "flow"))
	(erase-buffer)
	(setq gnow-link-info (xml-rpc-method-call url 'getnids (list path)))
	(setnode-type type)
	(setlink-type gnow-node)
	(insert "\n\n")
	(org-insert-heading)
	(insert node-symbol " [[" type ":" path "][" (cdr (assoc path gnow-link-info)) "]]\n" )
	(disp-expand path (cdr (assoc path gnow-link-info)))
	(org-cycle-hide-drawers 'all))

       ((member type '("gnow"))
	(setq gnow-link-info (xml-rpc-method-call url 'getSpecificnbh path (list (cons "attributes" (vector "title" "description")))))
	(setq gnow-link-message "")
	(setq search-attr (get-keys (cdr (assoc "attributes" gnow-link-info))))
	(dolist (i search-attr)
	  (setq gnow-link-message (concat gnow-link-message "\n" i " : " (elt (cdr (assoc i (cdr (assoc "attributes" gnow-link-info)))) 1))))
	(if (> (length gnow-link-message) 0)
	    (message gnow-link-message)
	  (message "Description not available.")))

       ((string= type "hasVersion")
	(if (get-buffer-window "*Compare*")
	    (delete-window (get-buffer-window "*Compare*"))
	  (get-buffer-create "*Compare*"))
	(pop-to-buffer "*Compare*")
	(setq buffer-read-only nil)
	(goto-char (point-max))
	(setq gnow-link-info (xml-rpc-method-call url 'getnids (list path)))
	(setq gnow-node (xml-rpc-method-call url 'getNodetype path))
;;	(setnode-type type)
	(setlink-type gnow-node)
	(insert "\n\n")
	(org-insert-heading)
	(insert node-symbol " [[" link-type ":" path "][" (cdr (assoc path gnow-link-info)) "]]\n" )
	(disp-expand path (cdr (assoc path gnow-link-info)))
	(org-mode)
	(setq buffer-read-only t))
	 
       (t
	(browse-url-at-point))))))
  (move-marker org-open-link-marker nil)
  (run-hook-with-args 'org-follow-link-hook))

(defvar gnow-result nil)
(defvar gnow-node nil)

(defun gnowsys-getnodes (&optional node)
  "To view all entries in kb of the particular node type specified bu the user."

  (interactive)
  (erase-buffer)
  (if node
      (setq gnow-node node)
    (setq gnow-node (completing-read "Node: " gnowsys-nodes)))

  (if (equal  gnow-node "process")
      (setq gnow-node (concat "gb" gnow-node))
    (setq gnow-node (concat "gb" gnow-node "s")))
  
  (setq gnow-result (xml-rpc-method-call url 'getAll gnow-node))
  (insert "\n\n")
  (dolist (i gnow-result)
    (setlink-type gnow-node)
;;    (org-insert-heading)
    (insert "* " node-symbol " [[" link-type ":" (cdr (assoc (elt i 0) gnow-result)) "][" (elt i 0) "]]\n" )
    )
  (org-overview)
  )

(defun gnowsys-getSelectionlists ()
  "To view all the selection lists availaible in the kb."

  (interactive)
  (erase-buffer)

  (setlink-type "gbobjects")
  (setq gnow-result (xml-rpc-method-call url 'getSelectionlist))
  (dolist (i gnow-result)
    (org-insert-heading)
    (insert "[[" link-type ":"(cdr (assoc "listid" i)) "][" (cdr (assoc "nid" i)) "]]" )
    (disp-expand (cdr (assoc "listid" i)) (cdr (assoc "nid" i)))
    )
  (org-overview))

(defun setlink-type (node-type)
  "Depending upon type of the node, it determines the link type."

  (cond ((equal node-type "gbobjects")
	 (setq link-type "obj")
	 (setq node-symbol "O:"))
	((equal node-type "gbobjecttypes")
	 (setq link-type "otype")
	 (setq node-symbol "OT:"))
	((equal node-type "gbattributes")
	 (setq link-type "attr")
	 (setq node-symbol "A:"))
	((equal node-type "gbattributetypes")
	 (setq link-type "atype")
	 (setq node-symbol "AT:"))
	((equal node-type "gbrelations")
	 (setq link-type "rel")
	 (setq node-symbol "R:"))
	((equal node-type "gbrelationtypes")
	 (setq link-type "rtype")
	 (setq node-symbol "RT:"))
	((equal node-type "gbusers")
	 (setq link-type "usr")
	 (setq node-symbol "U:"))
	((equal node-type "gbusertypes")
	 (setq link-type "utype")
	 (setq node-symbol "UT:"))
	((equal node-type "gbprocess")
	 (setq link-type "proc")
	 (setq node-symbol "P:"))
	((equal node-type "gbprocesstypes")
	 (setq link-type "ptype")
	 (setq node-symbol "PT:"))
	((equal node-type "gbmetatypes")
	 (setq link-type "mtype")
	 (setq node-symbol "MT:"))
	((equal node-type "gbfunctions")
	 (setq link-type "func")
	 (setq node-symbol "F:"))
	((equal node-type "gbclass")
	 (setq link-type "class")
	 (setq node-symbol "C:"))
	((equal node-type "gbflow")
	 (setq link-type "flow")
	 (setq node-symbol "FL:"))))

(defun setnode-type (link-type)
  "Depending upon type of the node, it determines the link type."

  (cond ((equal link-type "obj")
	 (setq gnow-node "gbobjects")
	 (setq add-node "object"))
	((equal link-type "otype")
	 (setq gnow-node "gbobjecttypes")
	 (setq add-node "objecttype"))
	((equal link-type "attr")
	 (setq gnow-node "gbattributes")
	 (setq add-node "attribute"))
	((equal link-type "atype")
	 (setq gnow-node "gbattributetypes")
	 (setq add-node "attributetype"))
	((equal link-type "rel")
	 (setq gnow-node "gbrelations")
	 (setq add-node "relation"))
	((equal link-type "rtype")
	 (setq gnow-node "gbrelationtypes")
	 (setq add-node "relationtype"))
	((equal link-type "usr")
	 (setq gnow-node "gbusers")
	 (setq add-node "user"))
	((equal link-type "utype")
	 (setq gnow-node "gbusertypes")
	 (setq add-node "usertype"))
	((equal link-type "proc")
	 (setq gnow-node "gbprocess")
	 (setq add-node "process"))
	((equal link-type "ptype")
	 (setq gnow-node "gbprocesstypes")
	 (setq add-node "processtype"))
	((equal link-type "mtype")
	 (setq gnow-node "gbmetatypes")
	 (setq add-node "metatype"))
	((equal link-type "funct")
	 (setq gnow-node "gbfunctions")
	 (setq add-node "function"))
	((equal link-type "class")
	 (setq gnow-node "gbclass")
	 (setq add-node "class"))
	((equal link-type "flow")
	 (setq gnow-node "gbflow")
	 (setq add-node "flow"))))

(defvar promp-rel-id nil)
(defvar promp-rel-nid nil)
(defvar promp-attr-id nil)
(defvar promp-attr-nid nil)

(defvar gnow-interanl-link nil)

(defun insert-gnow-link ()
  "Insert gnow link."

  (interactive)
  (setq add-node (completing-read "Node: " gnowsys-nodes))
  (if (equal  add-node "process")
      (setq gnow-node (concat "gb" add-node))
    (setq gnow-node (concat "gb" add-node "s")))
  (setq gnow-result (xml-rpc-method-call url 'getAll gnow-node))
  (setq gnow-internal-link (completing-read "Gnow Link: " gnow-result))
  (insert "[[gnow:" (number-to-string (cdr (assoc gnow-internal-link gnow-result))) "][" gnow-internal-link "]]"))

(defvar conetxt-headline nil)
(defvar link-text nil)
(defvar type-path nil)
(defvar type nil)
(defvar path nil)
(defvar link-nid-desc nil)
(defvar context-start-pos nil)
(defvar context-end-pos nil)
(defvar context-ssid nil)

(defun gnow-parse ()
  "Parse current context."
  
  (setq gnowsys-set-value nil)
  (if (not (outline-on-heading-p))
      (outline-back-to-heading))
  (setq context-headline (org-context))
  (beginning-of-line)
  (setq context-start-pos (+ (elt (elt context-headline 0) 1) 2))
  (goto-char context-start-pos)
  (setq context-start-pos (re-search-forward " "))
  (setq context-end-pos (elt (elt context-headline 0) 2))
  (setq link-text (buffer-substring-no-properties context-start-pos context-end-pos))
  (if (> (length link-text) 0)
      (progn
	(link-info link-text)
	(setnode-type type)
	(setq node-nid link-nid-desc)
	(setq context-ssid path)
	(setq context-start-pos context-end-pos)))
  (outline-next-heading)
  (setq context-relation-drawer-start-pos nil)
  (setq context-attribute-drawer-start-pos nil)
  (setq context-view-drawer-start-pos nil)
  (if (and (outline-on-heading-p) (not (equal (point) context-end-pos)))
      (setq context-end-pos (- (elt (elt (org-context) 0) 1) 1))
    (setq context-end-pos (point)))
  (goto-char context-end-pos)
  (setq context-text (buffer-substring-no-properties context-start-pos context-end-pos))
  (if (string-match "\:RELATIONS\:" context-text)
      (progn
	(setq context-relation-drawer-start-pos (+ context-start-pos (string-match "\:RELATIONS\:" context-text)))
	(goto-char (+ context-start-pos (string-match "\:END\:" context-text (- context-relation-drawer-start-pos context-start-pos))))
	(setq context-relation-drawer-end-pos (- (line-beginning-position) 1))))
  (if (string-match "\:ATTRIBUTES\:" context-text)
      (progn
	(setq context-attribute-drawer-start-pos (+ context-start-pos (string-match "\:ATTRIBUTES\:" context-text)))
	(goto-char (+ context-start-pos (string-match "\:END\:" context-text (- context-attribute-drawer-start-pos context-start-pos))))
	(setq context-attribute-drawer-end-pos (- (line-beginning-position) 1))))
  )
  

(defun gnowsys-head ()
  "Display current heading."
  
  (setq gnowsys-set-value nil)
  (if (not (outline-on-heading-p))
      (outline-back-to-heading))
  (setq context-headline (org-context))
  (setq context-start-pos (+ (elt (elt context-headline 0) 1) 2))
  (goto-char context-start-pos)
  (setq context-start-pos (re-search-forward " "))
  (setq context-end-pos (elt (elt context-headline 0) 2))
  (setq link-text (buffer-substring-no-properties context-start-pos context-end-pos))
  (if (> (length link-text) 0)
      (progn
	(link-info link-text)
;;	(if (not gnow-node)
	(setnode-type type)
	(setq node-nid link-nid-desc)
	(setq context-ssid path)
	(setq context-start-pos context-end-pos)))
  (context-parser)
  )

(defun link-info (link-text)
  "Extarct type path & description of link."

  (setq type-path (substring link-text 2 (string-match "\]" link-text)))
  (setq type (substring type-path 0 (string-match "\:" type-path)))
  (setq path (substring type-path (+ (string-match "\:" type-path) 1)))
  (setq link-nid-desc (substring link-text (+ (string-match "\]" link-text) 2) (string-match "\]\]" link-text)))
  )

(defvar context-end-pos nil)
(defvar context-text nil)
(defvar context-relation-drawer-start-pos nil)
(defvar context-attribute-drawer-start-pos nil)
(defvar context-view-drawer-start-pos nil)
(defvar context-view-drawer-end-pos nil)
(defvar context-content nil)

(defun context-parser ()
  "Parse the entire region of the context."
  
  (outline-next-heading)
  (setq context-relation-drawer-start-pos nil)
  (setq context-attribute-drawer-start-pos nil)
  (setq context-view-drawer-start-pos nil)
  (setq rel-element-list nil)
  (setq attr-element-list nil)
  (if (outline-on-heading-p)
      (setq context-end-pos (- (elt (elt (org-context) 0) 1) 1))
    (setq context-end-pos (point)))
  (goto-char context-end-pos)
  (setq context-text (buffer-substring-no-properties context-start-pos context-end-pos))
  (if (string-match "\:RELATIONS\:" context-text)
      (progn
	(setq context-relation-drawer-start-pos (+ context-start-pos (string-match "\:RELATIONS\:" context-text)))
	(goto-char (+ context-start-pos (string-match "\:END\:" context-text (- context-relation-drawer-start-pos context-start-pos))))
	(setq context-relation-drawer-end-pos (- (line-beginning-position) 1))
	))
  (if (string-match "\:ATTRIBUTES\:" context-text)
      (progn
	(setq context-attribute-drawer-start-pos (+ context-start-pos (string-match "\:ATTRIBUTES\:" context-text)))
	(goto-char (+ context-start-pos (string-match "\:END\:" context-text (- context-attribute-drawer-start-pos context-start-pos))))
	(setq context-attribute-drawer-end-pos (- (line-beginning-position) 1))
	))
  
  (if (string-match "\:VIEW\:" context-text)
      (progn
	(setq context-view-drawer-start-pos (+ context-start-pos (string-match "\:VIEW\:" context-text)))
	(goto-char (+ context-start-pos (string-match "\:END\:" context-text (- context-view-drawer-start-pos context-start-pos))))
	(setq context-view-drawer-end-pos (line-end-position))
	)
    (progn
      (if context-attribute-drawer-start-pos
	  (progn
	    (setq context-view-drawer-start-pos context-attribute-drawer-start-pos)
	    (goto-char (+ context-attribute-drawer-end-pos 1))
	    (setq context-view-drawer-end-pos (line-end-position))
	    )
	(if context-relation-drawer-start-pos
	    (progn
	      (setq context-view-drawer-start-pos context-relation-drawer-start-pos)
	      (goto-char (+ context-relation-drawer-end-pos 1))
	      (setq context-view-drawer-end-pos (line-end-position))
	      )
	    (progn
	      (setq context-view-drawer-start-pos context-start-pos)
	      (setq context-view-drawer-end-pos (line-end-position))
	      )))))

  (if context-relation-drawer-start-pos
      (relation-drawer-parser))
  (if context-attribute-drawer-start-pos
      (attribute-drawer-parser))
  (setq temp-list nil)
  (if (not (equal context-view-drawer-end-pos context-end-pos))
      (if (string-match "[a-zA-Z0-9]+" (buffer-substring-no-properties (+ context-view-drawer-end-pos 1) context-end-pos))
	  (progn
	    (setq temp-list (buffer-substring-no-properties (+ (string-match "[a-zA-Z0-9]+" (buffer-substring-no-properties (+ context-view-drawer-end-pos 1) context-end-pos)) (+ context-view-drawer-end-pos 1)) context-end-pos))
	    (push (cons "content" temp-list) attr-element-list))))
  )

(defvar context-relation-drawer-end-pos nil)
(defvar rel-element nil)
(defvar rel-element-value nil)
(defvar rel-element-list nil)

(defun relation-drawer-parser ()
  "Relation drawer in the context."

;;  (goto-char (+ context-start-pos (string-match "\:END\:" context-text (- context-relation-drawer-start-pos context-start-pos))))
;;  (setq context-relation-drawer-end-pos (- (line-beginning-position) 1))
  (goto-char context-relation-drawer-start-pos)
  (end-of-line)
  (loop
   (setq temp-list nil)
   (setq rel-element (buffer-substring-no-properties (- (re-search-forward "\:") 1) (re-search-forward drawer-element-end-regex)))
   (if (equal rel-element ":END:")
       (return))
   (setq rel-element-value (buffer-substring-no-properties (point) (line-end-position)))
   (dolist (i (split-string rel-element-value ";"))
     (if (> (length i) 0)
	 (progn
	   (link-info (substring i 1))
	   (if (member (substring rel-element 1 -1) '("subjecttype1" "subjecttype2" "subjecttypes"))
	       (push link-nid-desc temp-list)
	     (if (equal (substring rel-element 1 -1) "subject")
		 (setq temp-list path)
	       (if (equal (substring rel-element 1 -1) "relationtypes")
		   (setq temp-list link-nid-desc) 
		 (push path temp-list)))))))
   (if (equal (substring rel-element 1 -1) "memberof")
       (setq rel-element ":instanceof:"))
   (if (equal (substring rel-element 1 -1) "typeof")
       (setq rel-element ":subtypeof:"))
   (if (listp temp-list)
       (push (cons (substring rel-element 1 -1) (vconcat temp-list)) rel-element-list)
     (push (cons (substring rel-element 1 -1) temp-list) rel-element-list))
   (end-of-line))
  )

(defvar context-attribute-drawer-end-pos nil)
(defvar attr-element-list nil)
(defvar attr-element nil)
(defvar attr-element-value nil)
(defvar line-end-pos nil)

(defun attribute-drawer-parser ()
  "Attribute drawer in the context."
  
;;  (goto-char (+ context-start-pos (string-match "\:END\:" context-text (- context-attribute-drawer-start-pos context-start-pos))))
;;  (setq context-attribute-drawer-end-pos (- (line-beginning-position) 1))
  (goto-char context-attribute-drawer-start-pos)
  (end-of-line)
  (setq attr-res nil)
  (loop
   (setq attr-element (buffer-substring-no-properties (- (re-search-forward "\:") 1) (re-search-forward drawer-element-end-regex)))
   (if (equal attr-element ":END:")
       (return))
   (setq proc-loc (point))
   (end-of-line)
   (setq line-end-pos (point))
   (re-search-forward drawer-element-start-regex)
;; (buffer-substring-no-properties (- (re-search-forward "\:") 1) (re-search-forward drawer-element-end-regex))
   (if (not (equal (- (line-beginning-position) 1) line-end-pos))
       (progn
	 (goto-char (- (line-beginning-position) 1))
	 (setq line-end-pos (point))
	 ))
   (goto-char proc-loc)
   (setq attr-element-value (buffer-substring-no-properties (point) line-end-pos))
   (if (not (member attr-element '(":applicableto:" ":restrictiontype:" ":selectiontype:" ":listname:" ":datatype:"  ":regex:" ":length:" ":range:" ":precision:" ":AttributeType:" ":inversename:")))
       (progn
	 (setq attr-ins (xml-rpc-method-call url 'getlatestssids (list (substring attr-element 1 -1)) "nid"))
	 (setq attr-res (cdr (assoc  (cdr (assoc (substring attr-element 1 -1) attr-ins)) (xml-rpc-method-call url 'getRestrictions (list (cdr (assoc (substring attr-element 1 -1) attr-ins))))))))
     (progn
       (push (cons "type" "1") attr-res)
       (push (cons "datatype" "notype") attr-res)))

   (setq temp-list nil)
   
   (if (equal (cdr (assoc "type" attr-res)) "0")
       (progn
	 (if (equal (length (split-string attr-element-value ";")) 1)
	     (setq temp-list attr-element-value)
	   (dolist (i (split-string attr-element-value ";"))
	     (if (> (length i) 0)
		 (push (substring i 1) temp-list)))))
     (if (> (length attr-element-value) 0)
	 (if (equal (substring attr-element-value 0 1) " ")
	     (setq temp-list (substring attr-element-value 1))
	   (setq temp-list attr-element-value))
       (setq temp-list attr-element-value))
     )

   (if (equal (substring attr-element 1 -1) "listname")
       (setq attr-element ":listid:"))
   (if (equal (substring attr-element 1 -1) "applicableto")
       (setq attr-element ":applicablenodetypes:"))
   
;;   (if (equal (substring (cdr (assoc "datatype" attr-res)) -2) "[]")
;;       (push (cons (substring attr-element 1 -1) (list temp-list)) attr-element-list)
   (push (cons (substring attr-element 1 -1) temp-list) attr-element-list)
   (end-of-line))
;;  (setq attr-element-list attr-element-list)
  )

(defvar node-head-pos nil)

(defun gnowsys-insert (&optional node)
  "Adding to gnowsys through template."
  
  (interactive)
  (end-of-buffer)
  (setq gnowsys-insert-status t)
  
  (if node
      (setq add-node node)
    (setq add-node (completing-read "Node: " gnowsys-nodes)))
  (if (equal  add-node "process")
      (setq gnow-node (concat "gb" add-node))
    (setq gnow-node (concat "gb" add-node "s")))

  (setq node-nid nil)
  (setlink-type gnow-node)
  (insert "\n\n")
  (org-insert-heading)
  (insert node-symbol " ")
  (setq node-head-pos (point))
  (if (member gnow-node '("gbrelations" "gbattributes"))
      (progn
	(insert "\n")
	(setq gnowsys-insert-status nil)
	(relation-drawer))))
  
(defun drawer-template ()
  "Creating dynamic templates."
  
  (interactive)
  (if gnowsys-insert-status
      (progn
	(setq node-nid (buffer-substring-no-properties node-head-pos (point)))
	(delete-region node-head-pos (point))
	(goto-char node-head-pos)
	(if (or (equal (xml-rpc-method-call url 'nidExists node-nid) 1) (equal (length node-nid) 0))
	    (message "Node already exists!!!")
	  (progn
	    (insert "[[" link-type ":][" node-nid "]]\n")
	    (relation-drawer)
	    (if (member gnow-node '("gbattributetypes" "gbrelationtypes"))
		(att-attribute-drawer))
	    (setq gnowsys-insert-status nil)))
	)
    (insert "\n"))
  )

(defvar attribute-drawer-template (list (cons "gbattributetypes" (vector "applicableto" "restrictiontype")) (cons "selection" (vector "listname")) (cons "value" (vector "datatype" "regex" "length")) (cons "data" (vector "range" "precision")) (cons "gbrelationtypes" (vector "applicableto" "inversename"))))

(defvar relation-drawer-template (list (cons "gbobjects" (vector "memberof")) (cons "gbobjecttypes" (vector "typeof" "memberof")) (cons "ontology" (vector "typeof")) (cons "gbprocesstypes" (vector "memberof" "typeof" "priorstatedef" "poststatedef")) (cons "gbprocess" (vector "memberof" "priorstate" "poststate")) (cons "gbmetatypes" (vector "typeof")) (cons "gbrelationtypes" (vector "memberof" "typeof" "subjecttype1" "subjecttype2")) (cons "gbrelations" (vector "relationtypes" "subject1" "subject2")) (cons "gbattributetypes" (vector "memberof" "typeof" "subjecttypes")) (cons "gbattributes" (vector "subject")) ))
(defvar relation-drawer-start-pos nil)
(defvar relation-drawer-end-pos nil)

(defun relation-drawer ()
  "Create RELATIONS drawer template."
  
  (insert "  :RELATIONS:")
  (setq context-relation-drawer-start-pos (point))
  (setq relation-drawer-start-pos (point))
  (dolist (i (listify-key-sequence (cdr (assoc gnow-node relation-drawer-template))))
    (if (and ont-flag (equal gnow-node "gbobjecttypes") (equal i "memberof"))
	(insert "\n  :" i ": [[mtype:" (cdr (assoc "ONTOLOGY" (xml-rpc-method-call url 'getlatestssids '("ONTOLOGY") "nid"))) "][ONTOLOGY]]")
    (insert "\n  :" i ":")))
  (setq context-relation-drawer-end-pos (point))
  (setq relation-drawer-end-pos (point))
  (insert "\n  :END:"))

(defun att-attribute-drawer ()
  "Generate ATTRIBUTES drawer."

  (insert "\n\n  :ATTRIBUTES:")
  (setq context-attribute-drawer-start-pos (point))
  (setq attribute-drawer-start-pos (point))
  (dolist (i (listify-key-sequence (cdr (assoc gnow-node attribute-drawer-template))))
    (if (not (equal i "content"))
	(insert "\n  :" i ":")))
  (setq context-attribute-drawer-end-pos (point))
  (setq attribute-drawer-end-pos (point))
  (insert "\n  :END:")
  (setq context-end-pos (point)))

  
(defvar drawer-increment nil)
(defvar drawer-increment-start nil)
(defvar drawer-increment-end nil)
(defvar curr-pos nil)
(defvar drawer-element-start-regex "\:[a-zA-Z]")
(defvar drawer-element-end-regex "[a-zA-Z0-9]\:")

(defun template-autolist ()
  "Autolist in templates."

  (interactive)
  (setq curr-pos (point))
  (if (or (org-before-first-heading-p) (org-on-heading-p))
      (insert " ")
    (progn
      (gnow-parse)
      (goto-char curr-pos)
      (if (not context-relation-drawer-start-pos)
	  (progn
	    (setq context-relation-drawer-start-pos 0)
	    (setq context-relation-drawer-end-pos 0)
	    ))
      (if (not context-attribute-drawer-start-pos)
	  (progn
	    (setq context-attribute-drawer-start-pos 0)
	    (setq context-attribute-drawer-end-pos 0)
	    ))
  (if (and (> (point) context-relation-drawer-start-pos) (<= (point) context-relation-drawer-end-pos))
      (progn 
	(template-completion)
	(goto-char curr-pos)
	(end-of-line)
	(setq ind (completing-read "Node: " gnowsys-ots))
	(setq drawer-increment-start (point))
	(if (not link-type)
	      (setlink-type (xml-rpc-method-call url 'getNodetype (cdr (assoc ind gnowsys-ots)))))
	(if (stringp (cdr (assoc ind gnowsys-ots)))
	    (insert " [[" link-type ":" (cdr (assoc ind gnowsys-ots))"]["ind"]];")
	(insert " [[" link-type ":" (number-to-string (cdr (assoc ind gnowsys-ots)))"]["ind"]];"))
	(setq drawer-increment-end (point))
	(setq drawer-increment (- drawer-increment-end drawer-increment-start))
	(setq context-relation-drawer-end-pos (+ context-relation-drawer-end-pos drawer-increment))
	(if context-attribute-drawer-start-pos
	    (progn
	      (setq context-attribute-drawer-start-pos (+ context-attribute-drawer-start-pos drawer-increment))
	      (setq context-attribute-drawer-end-pos (+ context-attribute-drawer-end-pos drawer-increment))
	      ))
	)
    (progn
      (if (and (> (point) context-attribute-drawer-start-pos) (<= (point) context-attribute-drawer-end-pos))
	  (progn
	    (re-search-backward drawer-element-start-regex)
	    (beginning-of-line)
	    (setq attr-element (buffer-substring-no-properties (- (re-search-forward "\:") 1) (re-search-forward drawer-element-end-regex)))
	
	    (if (member attr-element '(":title:" ":description:" ":content:" ":inversename:"))
		(progn
		  (goto-char curr-pos)
		  (insert " ")
		  )
		  
	      (progn
	    (if (member attr-element '(":applicableto:" ":restrictiontype:" ":selectiontype:" ":listname:" ":datatype:" ":regex:"))
		(progn
		  (goto-char curr-pos)
		  (attr-template-completion attr-element)
		  (end-of-line)
		  (setq ind (completing-read "Value: " gnowsys-ots))
		  (setq drawer-increment-start (point))
		  (insert " " ind)
		  (setq drawer-increment-end (point))
		  (setq drawer-increment (- drawer-increment-end drawer-increment-start))
		  (setq context-attribute-drawer-end-pos (+ context-attribute-drawer-end-pos drawer-increment))
		  (if (equal attr-element ":restrictiontype:") 
		      (progn
			(if (equal ind "selectionlist")
			    (progn
			      (setq frame-template nil)
			      (push (cons "attributes" (listify-key-sequence (cdr (assoc "selection" attribute-drawer-template)))) frame-template)
			      (gnowsys-head)
			      (attribute-drawer-update)))
			(if (equal ind "valuerestriction")
			    (progn
			      (setq frame-template nil)
			      (push (cons "attributes" (listify-key-sequence (cdr (assoc "value" attribute-drawer-template)))) frame-template)
			      (gnowsys-head)
			      (attribute-drawer-update)))))
		  (if (equal attr-element ":datatype:") 
		      (if (member ind at-restrict-precision-datatype)
			  (progn
			    (setq frame-template nil)
			    (push (cons "attributes" (listify-key-sequence (cdr (assoc "data" attribute-drawer-template)))) frame-template)
			    (gnowsys-head)
			    (attribute-drawer-update))))
		  )
	      (progn
		(if (not (member attr-element '(":AttributeType:" ":restrictiontype:" ":datatype:" ":regex:" ":length:" ":range:" ":precision:" ":selectiontype:" ":listname:" ":inversename:" ":applicableto:")))
		    (progn
		      (goto-char curr-pos)
		      (setq attr-ins (xml-rpc-method-call url 'getlatestssids (list (substring attr-element 1 -1)) "nid"))
		      (setq attr-res (cdr (assoc  (cdr (assoc (substring attr-element 1 -1) attr-ins)) (xml-rpc-method-call url 'getRestrictions (list (cdr (assoc (substring attr-element 1 -1) attr-ins)))))))
		      (if (equal (cdr (assoc "type" attr-res)) "0")
			  (progn
			    (end-of-line)
			    (setq attr-val-list (xml-rpc-method-call url 'getStructure (cdr (assoc "listid" attr-res))))
			    (setq attr-val (completing-read "Select value: " attr-val-list))
			    (setq drawer-increment-start (point))
			    (insert " " attr-val ";")
			    (setq drawer-increment-end (point))
			    (setq drawer-increment (- drawer-increment-end drawer-increment-start))
			    (setq context-attribute-drawer-end-pos (+ context-attribute-drawer-end-pos drawer-increment))
			    )
			  (insert " ")
			))))))))
	  (insert " ")
	))
;;      (insert " ")
    )))       
  )

(defvar frame-template nil)

(defun generate-template ()
  "Dynamically Generates template based on the selected values."
  
  (interactive)
  (setq frame-template nil)
  (setq context-attribute-drawer-start-pos nil)
  (setq context-relation-drawer-start-pos nil)
  (gnowsys-head)
  (setq temp-list nil)
  (if (equal gnow-node "gbattributes")
	(if (member "subject" (get-keys rel-element-list))
	    (setq temp-list (cdr (assoc "subject" rel-element-list))))
    (dolist (i (get-keys rel-element-list))
      (if (and (member i '("instanceof" "subtypeof")) (> (length (cdr (assoc i rel-element-list))) 0))
	  (push (cons i (cdr (assoc i rel-element-list))) temp-list))))

  (if (and (equal gnow-node "gbattributes") temp-list)
      (push (cons "attributes" (get-keys (xml-rpc-method-call url 'getPossibleAttr temp-list))) frame-template)
    (if temp-list
	(setq frame-template (xml-rpc-method-call url 'getFrame gnow-node temp-list))
      (setq frame-template (xml-rpc-method-call url 'getFrame gnow-node))))
  
  (if (member "attributes" (get-keys frame-template))
      (if context-attribute-drawer-start-pos
	  (attribute-drawer-update)
	(attribute-drawer)))
  (if (member "relations" (get-keys frame-template))
      (if context-relation-drawer-start-pos
  	  (relation-drawer-update)
  	(progn
  	  (goto-char context-start-pos)
  	  (insert "\n\n  :RELATIONS:")
  	  (setq context-relation-drawer-end-pos (point))
  	  (relation-drawer-update)
  	  (insert "\n  :END:"))
  	)
      )
  )

(defvar existing-relation-element nil)

(defun relation-drawer-update ()
  "Update RELATIONS drawer."

  (setq existing-relation-element (get-keys rel-element-list))
  (goto-char context-relation-drawer-end-pos)
  (dolist (i (cdr (assoc "relations" frame-template)))
    (if (not (member i existing-relation-element))
	(progn
	  (if (equal i "instanceof")
	      (setq i "memberof"))
	  (if (equal i "subtypeof")
	      (setq i "typeof"))
	  (insert "\n  :" i ":"))
	  ))
  (setq relation-drawer-end-pos (point))
  (setq context-relation-drawer-end-pos (point))
  )

(defvar existing-attribute-element nil)
  
(defun attribute-drawer-update ()
  "Update RELATIONS drawer."

  (setq existing-attribute-element (get-keys attr-element-list))
  (goto-char context-attribute-drawer-end-pos)
  (dolist (i (cdr (assoc "attributes" frame-template)))
    (if (not (member i existing-attribute-element))
	(if (not (equal i "content"))
	    (insert "\n  :" i ":"))))
  (setq context-attribute-drawer-end-pos (point))
  (setq attribute-drawer-end-pos (point))
  )


(defvar attribute-drawer-start-pos nil)
(defvar attribute-drawer-end-pos nil)

(defun attribute-drawer ()
  "Generate ATTRIBUE drawer."

  (gnow-parse)
  (if context-relation-drawer-start-pos
      (progn
	(goto-char (+ context-relation-drawer-end-pos 1))
	(end-of-line))
;;    (goto-char (+ context-start-pos 1)))
    (goto-char context-start-pos))
  (insert "\n\n")
  (insert "  :ATTRIBUTES:")
  (setq context-attribute-drawer-start-pos (point))
  (setq attribute-drawer-start-pos (point))
  (dolist (i (cdr (assoc "attributes" frame-template)))
    (if (not (equal i "content"))
	(insert "\n  :" i ":")))
  (setq context-attribute-drawer-end-pos (point))
  (setq attribute-drawer-end-pos (point))
  (insert "\n  :END:")
  )


(defvar proc-loc nil)


(defun attr-template-completion (at-ele)
  "Set attribute values for autocompletion in insert-mode."
  
  (cond ((equal gnow-node "gbattributetypes")
	 (cond ((equal at-ele ":applicableto:")
		(setq gnowsys-ots gnowsys-nodetype))
	       ((equal at-ele ":restrictiontype:")
		(setq gnowsys-ots '("valuerestriction" "selectionlist")))
	       ((equal at-ele ":selectiontype:")
		(setq gnowsys-ots (list "multipleselection" "singleselection")))
	       ((equal at-ele ":listname:")
		(setq gnowsys-ots (list-from-dict (xml-rpc-method-call url 'getSelectionlist) "nid" "listid")))
	       ((equal at-ele ":datatype:")
		(setq gnowsys-ots (xml-rpc-method-call url 'getDatatypes "all")))
	       ((equal at-ele ":regex:")
		(setq gnow-reg (xml-rpc-method-call url 'getRegex))
		(setq gnowsys-ots (list-from-dict gnow-reg "exp" "id")))
	       ))
	((equal gnow-node "gbrelationtypes")
	 (cond ((equal at-ele ":applicableto:")
		(setq gnowsys-ots gnowsys-nodetype))
	       ))
	)
)


(defvar relation-instanceof (list (cons "gbobjects" "gbobjecttypes") (cons "gbobjecttypes" "gbmetatypes") (cons "gbattributes" "gbattributetypes") (cons "gbattributetypes" "gbmetatypes") (cons "gbrelations" "gbrelationtypes") (cons "gbrelationtypes" "gbmetatypes") (cons "gbprocess" "gbprocesstypes") (cons "gbprocesstypes" "gbmetatypes")))

(defvar relation-subtypeof (list (cons "gbobjecttypes" "gbobjecttypes") (cons "gbmetatypes" "gbmetatypes") (cons "gbattributetypes" "gbattributetypes") (cons "gbrelationtypes" "gbrelationtypes") (cons "gbprocesstypes" "gbprocesstypes"))) 
  
(defun template-completion ()
  "Set values for autocompletion in insert-mode."

  (cond ((equal gnow-node "gbobjects")
	 (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":memberof:")
	     (progn
	       (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbobjecttypes"))
	       (setlink-type "gbobjecttypes"))
	   (progn
	     (setq gnowsys-ots (xml-rpc-method-call url 'getRelFrame gnow-node (substring (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) 1 -1)))
	     (setq link-type nil))))

	((equal gnow-node "gbobjecttypes")
	 (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":memberof:")
	     (progn
	       (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbmetatypes"))
	       (setlink-type "gbmetatypes"))
	   (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":typeof:")
	       (progn
		 (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbobjecttypes"))
		 (setlink-type "gbobjecttypes"))
	     (progn
	       (setq gnowsys-ots (xml-rpc-method-call url 'getRelFrame gnow-node (substring (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) 1 -1)))
	       (setq link-type nil)))))

	((equal gnow-node "gbprocess")
	 (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":memberof:")
	     (progn
	       (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbprocesstypes"))
	       (setlink-type "gbprocesstypes"))
	   (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":prior state:")
	       (progn
		 (setq proc-loc (point))
		 (gnowsys-head)
		 (goto-char proc-loc)
		 (setq gnowsys-ots (xml-rpc-method-call url 'getPriorstates (cdr (assoc "instanceof" rel-element-list))))
		 (setlink-type "gbobjects"))
	     (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":post state:")
		 (progn 
		   (setq proc-loc (point))
		   (gnowsys-head)
		   (goto-char proc-loc)
		   (setq gnowsys-ots (xml-rpc-method-call url 'getPoststates (cdr (assoc "instanceof" rel-element-list))))
		   (setlink-type "gbobjects"))
	       (progn
		 (setq gnowsys-ots (xml-rpc-method-call url 'getRelFrame gnow-node (substring (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) 1 -1)))
		 (setq link-type nil))))))
	     
	
	((equal gnow-node "gbprocesstypes")
	 (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":memberof:")
	     (progn
	       (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbmetatypes"))
	       (setlink-type "gbmetatypes"))
	   (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":typeof:")
	       (progn
		 (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbprocesstypes"))
		 (setlink-type "gbprocesstypes"))
	     (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":prior state def:")
		 (progn
		   (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbobjecttypes"))
		   (setlink-type "gbobjecttypes"))
	       (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":post state def:")
		   (progn
		     (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbobjecttypes"))
		     (setlink-type "gbobjecttypes"))
		 (progn
		   (setq gnowsys-ots (xml-rpc-method-call url 'getRelFrame gnow-node (substring (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) 1 -1)))
		   (setq link-type nil)))))))

	((equal gnow-node "gbrelationtypes")
	 (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":memberof:")
	     (progn
	       (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbmetatypes"))
	       (setlink-type "gbmetatypes"))
	   (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":typeof:")
	       (progn
		 (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbrelationtypes"))

		 (setlink-type "gbrelationtypes"))
	     (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":subjecttype1:")
		 (progn
		   (setq selected-nodetype (concat "gb" (completing-read "Select type for subjecttype: "  gnowsys-type-list) "s"))
		   (setlink-type selected-nodetype)
		   (setq gnowsys-ots (xml-rpc-method-call url 'getAll selected-nodetype)))
	       (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":subjecttype2:")
		   (progn
		     (setq selected-nodetype (concat "gb" (completing-read "Select type for subjecttype: "  gnowsys-type-list) "s"))
		     (setlink-type selected-nodetype)
		     (setq gnowsys-ots (xml-rpc-method-call url 'getAll selected-nodetype)))
		 (progn
		   (setq gnowsys-ots (xml-rpc-method-call url 'getRelFrame gnow-node (substring (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) 1 -1)))
		   (setq link-type nil)))))))

	((equal gnow-node "gbrelations")
	 (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":relationtypes:")
	     (progn
	       (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbrelationtypes"))

	       (setlink-type "gbrelationtypes"))
	   (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":subject1:")
	       (progn
		 (setq proc-loc (point))
		 (gnowsys-head)
		 (goto-char proc-loc)
		 (if (member (cdr (assoc "relationtypes" rel-element-list)) '("instanceof" "subtypeof"))
		     (progn
		       (setq proc-loc (completing-read "Select type for subject: " gnowsys-nodes))
		       (if (equal proc-loc "process")
			   (setq selected-nodetype (concat "gb" proc-loc))
			 (setq selected-nodetype (concat "gb" proc-loc "s")))
		       
		       (setlink-type selected-nodetype)
		       (setq gnowsys-ots (xml-rpc-method-call url 'getAll selected-nodetype))
		       )
		   (progn
		     (setq rel-subjects (xml-rpc-method-call url 'getPossibleRel (cdr (assoc "relationtypes" rel-element-list))))
		     (if (elt rel-subjects 0)
			 (setlink-type (xml-rpc-method-call url 'getNodetype (cdr (assoc (elt (elt (elt rel-subjects 0) 0) 0) (elt rel-subjects 0))))))
		     (setq gnowsys-ots (elt rel-subjects 0)))))
	     (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":subject2:")
		 (progn
		   (setq proc-loc (point))
		   (gnowsys-head)
		   (goto-char proc-loc)
		   
		   (if (member (cdr (assoc "relationtypes" rel-element-list)) '("instanceof" "subtypeof"))
		     (progn
		       (setq proc-loc (xml-rpc-method-call url 'getNodetype (elt (cdr (assoc "subject1" rel-element-list)) 0)))
		       (if (equal (cdr (assoc "relationtypes" rel-element-list)) "instanceof")
			   (progn
			     (setq selected-nodetype (cdr (assoc proc-loc relation-instanceof)))
			     )
			 (progn
			   (setq selected-nodetype (cdr (assoc proc-loc relation-subtypeof)))
			   ))
		       (setlink-type selected-nodetype)
		       (setq gnowsys-ots (xml-rpc-method-call url 'getAll selected-nodetype))
		       )
		   (progn
		     (setq rel-subjects (xml-rpc-method-call url 'getPossibleRel (cdr (assoc "relationtypes" rel-element-list))))
		     (if (elt rel-subjects 1)
			 (setlink-type (xml-rpc-method-call url 'getNodetype (cdr (assoc (elt (elt (elt rel-subjects 1) 0) 0) (elt rel-subjects 1))))))
		     (setq gnowsys-ots (elt rel-subjects 1))))
		   )
	       (progn
		 (setq gnowsys-ots (xml-rpc-method-call url 'getRelFrame gnow-node (substring (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) 1 -1)))
		 (setq link-type nil))))))

  
	((equal gnow-node "gbattributetypes")
	 (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":memberof:")
	     (progn
	       (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbmetatypes"))
	       (setlink-type "gbmetatypes"))
	   (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":typeof:")
	       (progn
		 (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbattributetypes"))
		 (setlink-type "gbattributetypes"))
	     (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":subjecttypes:")
		 (progn
		   (setq selected-nodetype (concat "gb" (completing-read "Select type for subjecttype: "  gnowsys-type-list) "s"))
		   (setlink-type selected-nodetype)
		   (setq gnowsys-ots (xml-rpc-method-call url 'getAll selected-nodetype)))
	       (progn
		 (setq gnowsys-ots (xml-rpc-method-call url 'getRelFrame gnow-node (substring (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) 1 -1)))
		 (setq link-type nil))))))

	((equal gnow-node "gbattributes")
	 (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":subject:")
	     (progn
	       (setq proc-loc (completing-read "Select type for subject: " gnowsys-nodes))
	       (if (equal proc-loc "process")
		   (setq selected-nodetype (concat "gb" proc-loc))
		 (setq selected-nodetype (concat "gb" proc-loc "s")))

	       (setlink-type selected-nodetype)
	       (setq gnowsys-ots (xml-rpc-method-call url 'getAll selected-nodetype)))
	   (progn
	     (setq gnowsys-ots (xml-rpc-method-call url 'getRelFrame gnow-node (substring (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) 1 -1)))
	     (setq link-type nil))))


	((equal gnow-node "gbmetatypes")
	 (if (equal (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) ":typeof:")
	     (progn
	       (setq gnowsys-ots (xml-rpc-method-call url 'getAll "gbmetatypes"))
	       (setlink-type "gbmetatypes"))
	   (progn
	     (setq gnowsys-ots (xml-rpc-method-call url 'getRelFrame gnow-node (substring (buffer-substring-no-properties (re-search-backward drawer-element-start-regex) (re-search-forward drawer-element-end-regex)) 1 -1)))
	     (setq link-type nil))))))
  

(defun gnowsys-insert-object ()
  "Insert object into gnowsys kb."

  (interactive)
  (gnowsys-insert "object"))

(defun gnowsys-insert-objecttype ()
  "Insert objecttype into gnowsys kb."

  (interactive)
  (gnowsys-insert "objecttype"))

(setq ont-flag nil)

(defun gnowsys-add-ontology ()
  "Insert ontology into gnowsys kb."

  (interactive)
  (end-of-buffer)
  (setq gnowsys-insert-status t)
  
  ;; (if node
  ;;     (setq add-node node)
  ;;   (setq add-node (completing-read "Node: " gnowsys-nodes)))
  ;; (if (equal  add-node "process")
  ;;     (setq gnow-node (concat "gb" add-node))
  ;;   (setq gnow-node (concat "gb" add-node "s")))
  (setq gnow-node "gbobjecttypes")
  (setq node-nid nil)
  (setlink-type gnow-node)
  (insert "\n\n")
  (org-insert-heading)
  (insert node-symbol " ")
  (setq node-head-pos (point))
;;  (setq gnow-node "ontology")
  (setq ont-flag t)
  )

(defun gnowsys-insert-process ()
  "Insert process into gnowsys kb."

  (interactive)
  (gnowsys-insert "process"))

(defun gnowsys-insert-processtype ()
  "Insert processtype into gnowsys kb."

  (interactive)
  (gnowsys-insert "processtype"))

(defun gnowsys-insert-metatype ()
  "Insert metatype into gnowsys kb."

  (interactive)
  (gnowsys-insert "metatype"))

(defun gnowsys-insert-attribute ()
  "Insert attribute into gnowsys kb."

  (interactive)
  (gnowsys-insert "attribute"))

(defun gnowsys-insert-attributetype ()
  "Insert attributetype into gnowsys kb."

  (interactive)
  (gnowsys-insert "attributetype"))

(defun gnowsys-insert-relation ()
  "Insert relation into gnowsys kb."

  (interactive)
  (gnowsys-insert "relation"))

(defun gnowsys-insert-relationtype ()
  "Insert relationtype into gnowsys kb."

  (interactive)
  (gnowsys-insert "relationtype"))

(defun gnowsys-getnodes-object ()
  "Retrieve all objects from gnowsys kb."

  (interactive)
  (gnowsys-getnodes "object"))

(defun gnowsys-getnodes-objecttype ()
  "Retrieve all objecttypes from gnowsys kb."

  (interactive)
  (gnowsys-getnodes "objecttype"))

(defun gnowsys-getnodes-attribute ()
  "Retrieve all attributes from gnowsys kb."

  (interactive)
  (gnowsys-getnodes "attribute"))

(defun gnowsys-getnodes-attributetype ()
  "Retrieve all attributetypes from gnowsys kb."

  (interactive)
  (gnowsys-getnodes "attributetype"))

(defun gnowsys-getnodes-relation ()
  "Retrieve all relations from gnowsys kb."

  (interactive)
  (gnowsys-getnodes "relation"))

(defun gnowsys-getnodes-relationtype ()
  "Retrieve all relationtypes from gnowsys kb."

  (interactive)
  (gnowsys-getnodes "relationtype"))

(defun gnowsys-getnodes-process ()
  "Retrieve all process from gnowsys kb."

  (interactive)
  (gnowsys-getnodes "process"))

(defun gnowsys-getnodes-processtype ()
  "Retrieve all processtypes from gnowsys kb."

  (interactive)
  (gnowsys-getnodes "processtype"))

(defun gnowsys-getnodes-metatype ()
  "Retrieve all metatypes from gnowsys kb."

  (interactive)
  (gnowsys-getnodes "metatype"))

(defun gnowsys-getnodes-user ()
  "Retrieve all users from gnowsys kb."

  (interactive)
  (gnowsys-getnodes "user"))

(defun gnowsys-getnodes-usertype ()
  "Retrieve all usertypes from gnowsys kb."

  (interactive)
  (gnowsys-getnodes "usertype"))

(defun gnowsys-export ()
  "Export current buffer as html."

  (interactive)
  (setq org-export-preserve-breaks t)
  (setq org-export-with-drawers t)
  (setq buffer-file-name (concat (expand-file-name "~") "/gnowsys.org"))
  (gnowsys-export-as-html 3)
  )


(defun gnowsys-export-as-html (arg &optional hidden ext-plist
			       to-buffer body-only pub-dir)
  "Export the outline as a pretty HTML file.
If there is an active region, export only the region.  The prefix
ARG specifies how many levels of the outline should become
headlines.  The default is 3.  Lower levels will become bulleted
lists.  When HIDDEN is non-nil, don't display the HTML buffer.
EXT-PLIST is a property list with external parameters overriding
org-mode's default settings, but still inferior to file-local
settings.  When TO-BUFFER is non-nil, create a buffer with that
name and export to that buffer.  If TO-BUFFER is the symbol
`string', don't leave any buffer behind but just return the
resulting HTML as a string.  When BODY-ONLY is set, don't produce
the file header and footer, simply return the content of
<body>...</body>, without even the body tags themselves.  When
PUB-DIR is set, use this as the publishing directory."
  (interactive "P")

  ;; Make sure we have a file name when we need it.
  (when (and (not (or to-buffer body-only))
	     (not buffer-file-name))
    (if (buffer-base-buffer)
	(org-set-local 'buffer-file-name
		       (with-current-buffer (buffer-base-buffer)
			 buffer-file-name))
      (error "Need a file name to be able to export.")))

  (message "Exporting...")
  (setq-default org-todo-line-regexp org-todo-line-regexp)
  (setq-default org-deadline-line-regexp org-deadline-line-regexp)
  (setq-default org-done-keywords org-done-keywords)
  (setq-default org-maybe-keyword-time-regexp org-maybe-keyword-time-regexp)
;;  (setq export-file-name (read-no-blanks-input "File name: "))
  (let* ((opt-plist
	  (org-export-process-option-filters
	   (org-combine-plists (org-default-export-plist)
			       ext-plist
			       (org-infile-export-plist))))
	 (style (concat (if (plist-get opt-plist :style-include-default)
			    org-export-html-style-default)
			(plist-get opt-plist :style)
			(plist-get opt-plist :style-extra)
			"\n" org-export-html-scripts))
	 (html-extension (plist-get opt-plist :html-extension))
	 (link-validate (plist-get opt-plist :link-validation-function))
	 valid thetoc have-headings first-heading-pos
	 (odd org-odd-levels-only)
	 (region-p (org-region-active-p))
	 (rbeg (and region-p (region-beginning)))
	 (rend (and region-p (region-end)))
	 (subtree-p
	  (if (plist-get opt-plist :ignore-subree-p)
	      nil
	    (when region-p
	      (save-excursion
		(goto-char rbeg)
		(and (org-at-heading-p)
		     (>= (org-end-of-subtree t t) rend))))))
	 (opt-plist (if subtree-p
			(org-export-add-subtree-options opt-plist rbeg)
		      opt-plist))
	 ;; The following two are dynamically scoped into other
	 ;; routines below.
	 (org-current-export-dir
	  (or pub-dir (org-export-directory :html opt-plist)))
	 (org-current-export-file buffer-file-name)
	 (level 0) (line "") (origline "") txt todo
	 (umax nil)
	 (umax-toc nil)
	 (filename (if to-buffer nil
		     (expand-file-name
		      (concat
		       (file-name-sans-extension
			(or (and subtree-p
				 (org-entry-get (region-beginning)
						"EXPORT_FILE_NAME" t))
			    (file-name-nondirectory buffer-file-name)))
		       "." html-extension)
		      (file-name-as-directory
		       (or pub-dir (org-export-directory :html opt-plist))))))
	 (current-dir (if buffer-file-name
			  (file-name-directory buffer-file-name)
			default-directory))
	 (buffer (if to-buffer
		     (cond
		      ((eq to-buffer 'string) (get-buffer-create "*Org HTML Export*"))
		      (t (get-buffer-create to-buffer)))
		   (find-file-noselect filename)))
	 (org-levels-open (make-vector org-level-max nil))
	 (date (plist-get opt-plist :date))
	 (author      (plist-get opt-plist :author))
	 (title       (or (and subtree-p (org-export-get-title-from-subtree))
			  (plist-get opt-plist :title)
			  (and (not
				(plist-get opt-plist :skip-before-1st-heading))
			       (org-export-grab-title-from-buffer))
			  (and buffer-file-name
			       (file-name-sans-extension
				(file-name-nondirectory buffer-file-name)))
			  "UNTITLED"))
	 (html-table-tag (plist-get opt-plist :html-table-tag))
	 (quote-re0   (concat "^[ \t]*" org-quote-string "\\>"))
	 (quote-re    (concat "^\\(\\*+\\)\\([ \t]+" org-quote-string "\\>\\)"))
	 (inquote     nil)
	 (infixed     nil)
	 (inverse     nil)
	 (in-local-list nil)
	 (local-list-type nil)
	 (local-list-indent nil)
	 (llt org-plain-list-ordered-item-terminator)
	 (email       (plist-get opt-plist :email))
	 (language    (plist-get opt-plist :language))
	 (lang-words  nil)
	 (head-count  0) cnt
	 (start       0)
	 (coding-system (and (boundp 'buffer-file-coding-system)
			     buffer-file-coding-system))
	 (coding-system-for-write (or org-export-html-coding-system
				      coding-system))
	 (save-buffer-coding-system (or org-export-html-coding-system
					coding-system))
	 (charset (and coding-system-for-write
		       (fboundp 'coding-system-get)
		       (coding-system-get coding-system-for-write
					  'mime-charset)))
	 (region
	  (buffer-substring
	   (if region-p (region-beginning) (point-min))
	   (if region-p (region-end) (point-max))))
	 (lines
	  (org-split-string
	   (org-export-preprocess-string
	    region
	    :emph-multiline t
	    :for-html t
	    :skip-before-1st-heading
	    (plist-get opt-plist :skip-before-1st-heading)
	    :drawers (plist-get opt-plist :drawers)
	    :todo-keywords (plist-get opt-plist :todo-keywords)
	    :tags (plist-get opt-plist :tags)
	    :priority (plist-get opt-plist :priority)
	    :footnotes (plist-get opt-plist :footnotes)
	    :timestamps (plist-get opt-plist :timestamps)
	    :archived-trees
	    (plist-get opt-plist :archived-trees)
	    :select-tags (plist-get opt-plist :select-tags)
	    :exclude-tags (plist-get opt-plist :exclude-tags)
	    :add-text
	    (plist-get opt-plist :text)
	    :LaTeX-fragments
	    (plist-get opt-plist :LaTeX-fragments))
	   "[\r\n]"))
	 table-open type
	 table-buffer table-orig-buffer
	 ind item-type starter didclose
	 rpl path attr desc descp desc1 desc2 link
	 snumber fnc item-tag
	 footnotes footref-seen
	 id-file
	 )

    (let ((inhibit-read-only t))
      (org-unmodified
       (remove-text-properties (point-min) (point-max)
			       '(:org-license-to-kill t))))

    (message "Exporting...")

    (setq org-min-level (org-get-min-level lines))
    (setq org-last-level org-min-level)
    (org-init-section-numbers)

    (cond
     ((and date (string-match "%" date))
      (setq date (format-time-string date)))
     (date)
     (t (setq date (format-time-string "%Y-%m-%d %T %Z"))))

    ;; Get the language-dependent settings
    (setq lang-words (or (assoc language org-export-language-setup)
			 (assoc "en" org-export-language-setup)))

    ;; Switch to the output buffer
    (set-buffer buffer)
    (let ((inhibit-read-only t)) (erase-buffer))
    (fundamental-mode)

    (and (fboundp 'set-buffer-file-coding-system)
	 (set-buffer-file-coding-system coding-system-for-write))

    (let ((case-fold-search nil)
	  (org-odd-levels-only odd))
      ;; create local variables for all options, to make sure all called
      ;; functions get the correct information
      (mapc (lambda (x)
	      (set (make-local-variable (nth 2 x))
		   (plist-get opt-plist (car x))))
	    org-export-plist-vars)
      (setq umax (if arg (prefix-numeric-value arg)
		   org-export-headline-levels))
      (setq umax-toc (if (integerp org-export-with-toc)
			 (min org-export-with-toc umax)
		       umax))
      (unless body-only
	;; File header
	(insert (format
		 "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\"
               \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">
<html xmlns=\"http://www.w3.org/1999/xhtml\"
lang=\"%s\" xml:lang=\"%s\">
<head>
<title>%s</title>
<meta http-equiv=\"Content-Type\" content=\"text/html;charset=%s\"/>
<meta name=\"generator\" content=\"Org-mode\"/>
<meta name=\"generated\" content=\"%s\"/>
<meta name=\"author\" content=\"%s\"/>
%s
</head><body>
"
		 language language (org-html-expand title)
		 (or charset "iso-8859-1") date author style))

	(insert (or (plist-get opt-plist :preamble) ""))

	(when (plist-get opt-plist :auto-preamble)
	  (if title (insert (format org-export-html-title-format
				    (org-html-expand title))))))

      (if (and org-export-with-toc (not body-only))
	  (progn
	    (push (format "<h%d>%s</h%d>\n"
			  org-export-html-toplevel-hlevel
			  (nth 3 lang-words)
			  org-export-html-toplevel-hlevel)
		  thetoc)
	    (push "<div id=\"text-table-of-contents\">\n" thetoc)
	    (push "<ul>\n<li>" thetoc)
	    (setq lines
		  (mapcar '(lambda (line)
		    (if (string-match org-todo-line-regexp line)
			;; This is a headline
			(progn
			  (setq have-headings t)
			  (setq level (- (match-end 1) (match-beginning 1))
				level (org-tr-level level)
				txt (save-match-data
				      (org-html-expand
				       (org-export-cleanup-toc-line
					(match-string 3 line))))
				todo
				(or (and org-export-mark-todo-in-toc
					 (match-beginning 2)
					 (not (member (match-string 2 line)
						      org-done-keywords)))
					; TODO, not DONE
				    (and org-export-mark-todo-in-toc
					 (= level umax-toc)
					 (org-search-todo-below
					  line lines level))))
			  (if (string-match
			       (org-re "[ \t]+:\\([[:alnum:]_@:]+\\):[ \t]*$") txt)
			      (setq txt (replace-match  "&nbsp;&nbsp;&nbsp;<span class=\"tag\"> \\1</span>" t nil txt)))
			  (if (string-match quote-re0 txt)
			      (setq txt (replace-match "" t t txt)))
			  (setq snumber (org-section-number level))
			  (if org-export-with-section-numbers
			      (setq txt (concat snumber " " txt)))
			  (if (<= level (max umax umax-toc))
			      (setq head-count (+ head-count 1)))
			  (if (<= level umax-toc)
			      (progn
				(if (> level org-last-level)
				    (progn
				      (setq cnt (- level org-last-level))
				      (while (>= (setq cnt (1- cnt)) 0)
					(push "\n<ul>\n<li>" thetoc))
				      (push "\n" thetoc)))
				(if (< level org-last-level)
				    (progn
				      (setq cnt (- org-last-level level))
				      (while (>= (setq cnt (1- cnt)) 0)
					(push "</li>\n</ul>" thetoc))
				      (push "\n" thetoc)))
				;; Check for targets
				(while (string-match org-any-target-regexp line)
				  (setq line (replace-match
					      (concat "@<span class=\"target\">" (match-string 1 line) "@</span> ")
					      t t line)))
				(while (string-match "&lt;\\(&lt;\\)+\\|&gt;\\(&gt;\\)+" txt)
				  (setq txt (replace-match "" t t txt)))
				(push
				 (format
				  (if todo
				      "</li>\n<li><a href=\"#sec-%s\"><span class=\"todo\">%s</span></a>"
				    "</li>\n<li><a href=\"#sec-%s\">%s</a>")
				  snumber txt) thetoc)

				(setq org-last-level level))
			    )))
		    line)
			  lines))
	    (while (> org-last-level (1- org-min-level))
	      (setq org-last-level (1- org-last-level))
	      (push "</li>\n</ul>\n" thetoc))
	    (push "</div>\n" thetoc)
	    (setq thetoc (if have-headings (nreverse thetoc) nil))))

      (setq head-count 0)
      (org-init-section-numbers)

      (org-open-par)

      (while (setq line (pop lines) origline line)
	(catch 'nextline

	  ;; end of quote section?
	  (when (and inquote (string-match "^\\*+ " line))
	    (insert "</pre>\n")
	    (setq inquote nil))
	  ;; inside a quote section?
	  (when inquote
	    (insert (org-html-protect line) "\n")
	    (throw 'nextline nil))

	  ;; Fixed-width, verbatim lines (examples)
	  (when (and org-export-with-fixed-width
		     (string-match "^[ \t]*:\\(\\([ \t]\\|$\\)\\(.*\\)\\)" line))
	    (when (not infixed)
	      (setq infixed t)
	      (org-close-par-maybe)
	      (insert "<pre class=\"example\">\n"))
	    (insert (org-html-protect (match-string 3 line)) "\n")
	    (when (or (not lines)
		      (not (string-match "^[ \t]*\\(:.*\\)"
					 (car lines))))
	      (setq infixed nil)
	      (insert "</pre>\n"))
	    (throw 'nextline nil))

	  ;; Protected HTML
	  (when (get-text-property 0 'org-protected line)
	    (let (par)
	      (when (re-search-backward
		     "\\(<p>\\)\\([ \t\r\n]*\\)\\=" (- (point) 100) t)
		(setq par (match-string 1))
		(replace-match "\\2\n"))
	      (insert line "\n")
	      (while (and lines
			  (or (= (length (car lines)) 0)
			      (get-text-property 0 'org-protected (car lines))))
		(insert (pop lines) "\n"))
	      (and par (insert "<p>\n")))
	    (throw 'nextline nil))

	  ;; Horizontal line
	  (when (string-match "^[ \t]*-\\{5,\\}[ \t]*$" line)
	    (if org-par-open
		(insert "\n</p>\n<hr/>\n<p>\n")
	      (insert "\n<hr/>\n"))
	    (throw 'nextline nil))

	  ;; Blockquotes and verse
	  (when (equal "ORG-BLOCKQUOTE-START" line)
	    (org-close-par-maybe)
	    (insert "<blockquote>\n<p>\n")
	    (throw 'nextline nil))
	  (when (equal "ORG-BLOCKQUOTE-END" line)
	    (insert "</p>\n</blockquote>\n")
	    (throw 'nextline nil))
	  (when (equal "ORG-VERSE-START" line)
	    (org-close-par-maybe)
	    (insert "\n<p class=\"verse\">\n")
	    (setq inverse t)
	    (throw 'nextline nil))
	  (when (equal "ORG-VERSE-END" line)
	    (insert "</p>\n")
	    (setq inverse nil)
	    (throw 'nextline nil))
	  (when inverse
	    (let ((i (org-get-string-indentation line)))
	      (if (> i 0)
		  (setq line (concat (mapconcat 'identity
						(make-list (* 2 i) "\\nbsp") "")
				     " " (org-trim line))))
	      (setq line (concat line "\\\\"))))

	  ;; make targets to anchors
	  (while (string-match "<<<?\\([^<>]*\\)>>>?\\((INVISIBLE)\\)?[ \t]*\n?" line)
	    (cond
	     ((match-end 2)
	      (setq line (replace-match
			  (format
			   "@<a name=\"%s\" id=\"%s\">@</a>"
			   (org-solidify-link-text (match-string 1 line))
			   (org-solidify-link-text (match-string 1 line)))
			  t t line)))
	     ((and org-export-with-toc (equal (string-to-char line) ?*))
	      ;; FIXME: NOT DEPENDENT on TOC?????????????????????
	      (setq line (replace-match
			  (concat "@<span class=\"target\">" (match-string 1 line) "@</span> ")
;			  (concat "@<i>" (match-string 1 line) "@</i> ")
			  t t line)))
	     (t
	      (setq line (replace-match
			  (concat "@<a name=\""
				  (org-solidify-link-text (match-string 1 line))
				  "\" class=\"target\">" (match-string 1 line) "@</a> ")
			  t t line)))))

	  (setq line (org-html-handle-time-stamps line))

	  ;; replace "&" by "&amp;", "<" and ">" by "&lt;" and "&gt;"
	  ;; handle @<..> HTML tags (replace "@&gt;..&lt;" by "<..>")
	  ;; Also handle sub_superscripts and checkboxes
	  (or (string-match org-table-hline-regexp line)
	      (setq line (org-html-expand line)))

	  ;; Format the links
	  (setq start 0)
	  (while (string-match org-bracket-link-analytic-regexp++ line start)
	    (setq start (match-beginning 0))
	    (setq path (save-match-data (org-link-unescape
					 (match-string 3 line))))
	    (setq type (cond
			((match-end 2) (match-string 2 line))
			((save-match-data
			   (or (file-name-absolute-p path)
			       (string-match "^\\.\\.?/" path)))
			 "file")
			(t "internal")))
	    (setq path (org-extract-attributes (org-link-unescape path)))
	    (setq attr (get-text-property 0 'org-attributes path))
	    (setq desc1 (if (match-end 5) (match-string 5 line))
		  desc2 (if (match-end 2) (concat type ":" path) path)
		  descp (and desc1 (not (equal desc1 desc2)))
		  desc (or desc1 desc2))
	    ;; Make an image out of the description if that is so wanted
	    (when (and descp (org-file-image-p
			      desc org-export-html-inline-image-extensions))
	      (save-match-data
		(if (string-match "^file:" desc)
		    (setq desc (substring desc (match-end 0)))))
	      (setq desc (org-add-props
			     (concat "<img src=\"" desc "\"/>")
			     '(org-protected t))))
	    ;; FIXME: do we need to unescape here somewhere?
	    (cond
	     ((equal type "internal")
	      (setq rpl
		    (concat
		     "<a href=\"#"
		     (org-solidify-link-text
		      (save-match-data (org-link-unescape path)) nil)
		     "\"" attr ">"
		     (org-export-html-format-desc desc)
		     "</a>")))
	     ((and (equal type "id")
		   (setq id-file (org-id-find-id-file path)))
	      ;; This is an id: link to another file (if it was the same file,
	      ;; it would have become an internal link...)
	      (setq id-file (file-relative-name
			     id-file (file-name-directory org-current-export-file)))
	      (setq id-file (concat (file-name-sans-extension id-file)
				    "." html-extension))
	      (setq rpl (concat "<a href=\"" id-file "#" path "\""
				attr ">"
				(org-export-html-format-desc desc)
				"</a>")))
	     ((member type '("http" "https"))
	      ;; standard URL, just check if we need to inline an image
	      (if (and (or (eq t org-export-html-inline-images)
			   (and org-export-html-inline-images (not descp)))
		       (org-file-image-p
			path org-export-html-inline-image-extensions))
		  (setq rpl (org-export-html-format-image
			     (concat type ":" path) org-par-open))
		(setq link (concat type ":" path))
		(setq rpl (concat "<a href=\""
				  (org-export-html-format-href link)
				  "\"" attr ">"
				  (org-export-html-format-desc desc)
				  "</a>"))))
	     ((member type '("ftp" "mailto" "news"))
	      ;; standard URL
	      (setq link (concat type ":" path))
	      (setq rpl (concat "<a href=\""
				(org-export-html-format-href link)
				"\"" attr ">"
				(org-export-html-format-desc desc)
				"</a>")))

	     ((member type '("obj" "otype" "rel" "rtype" "attr" "atype" "proc" "ptype" "usr" "utype" "mtype" "func" "class" "flow" "gnow" "hasVersion"))
	      ;; standard URL

	      (setq rpl (concat "&nbsp;<font color=\"#dc143c\">"  (org-export-html-format-desc desc) "</font>"))
	      )
	     
	     ((string= type "coderef")

	      (setq rpl (format "<a href=\"#coderef-%s\" class=\"coderef\" onmouseover=\"CodeHighlightOn(this, 'coderef-%s');\" onmouseout=\"CodeHighlightOff(this, 'coderef-%s');\">%s</a>"
				path path path
				(format (org-export-get-coderef-format path (and descp desc))
					(cdr (assoc path org-export-code-refs))))))

	     ((functionp (setq fnc (nth 2 (assoc type org-link-protocols))))
	      ;; The link protocol has a function for format the link
	      (setq rpl
		    (save-match-data
		      (funcall fnc (org-link-unescape path) desc1 'html))))

	     ((string= type "file")
	      ;; FILE link
	      (let* ((filename path)
		     (abs-p (file-name-absolute-p filename))
		     thefile file-is-image-p search)
		(save-match-data
		  (if (string-match "::\\(.*\\)" filename)
		      (setq search (match-string 1 filename)
			    filename (replace-match "" t nil filename)))
		  (setq valid
			(if (functionp link-validate)
			    (funcall link-validate filename current-dir)
			  t))
		  (setq file-is-image-p
			(org-file-image-p
			 filename org-export-html-inline-image-extensions))
		  (setq thefile (if abs-p (expand-file-name filename) filename))
		  (when (and org-export-html-link-org-files-as-html
			     (string-match "\\.org$" thefile))
		    (setq thefile (concat (substring thefile 0
						     (match-beginning 0))
					  "." html-extension))
		    (if (and search
			     ;; make sure this is can be used as target search
			     (not (string-match "^[0-9]*$" search))
			     (not (string-match "^\\*" search))
			     (not (string-match "^/.*/$" search)))
			(setq thefile (concat thefile "#"
					      (org-solidify-link-text
					       (org-link-unescape search)))))
		    (when (string-match "^file:" desc)
		      (setq desc (replace-match "" t t desc))
		      (if (string-match "\\.org$" desc)
			  (setq desc (replace-match "" t t desc))))))
		(setq rpl (if (and file-is-image-p
				   (or (eq t org-export-html-inline-images)
				       (and org-export-html-inline-images
					    (not descp))))
			      (progn
				(message "image %s %s" thefile org-par-open)
				(org-export-html-format-image thefile org-par-open))
			    (concat "<a href=\"" thefile "\"" attr ">"
				    (org-export-html-format-desc desc)
				    "</a>")))
		(if (not valid) (setq rpl desc))))

	     (t
	      ;; just publish the path, as default
	      (setq rpl (concat "<i>&lt;" type ":"
				(save-match-data (org-link-unescape path))
				"&gt;</i>"))))
	    (setq line (replace-match rpl t t line)
		  start (+ start (length rpl))))

	  ;; TODO items
	  (if (and (string-match org-todo-line-regexp line)
		   (match-beginning 2))

	      (setq line
		    (concat (substring line 0 (match-beginning 2))
			    "<span class=\""
			    (if (member (match-string 2 line)
					org-done-keywords)
				"done" "todo")
			    "\">" (match-string 2 line)
			    "</span>" (substring line (match-end 2)))))

	  ;; Does this contain a reference to a footnote?
	  (when org-export-with-footnotes
	    (setq start 0)
	    (while (string-match "\\([^* \t].*?\\)\\[\\([0-9]+\\)\\]" line start)
	      (if (get-text-property (match-beginning 2) 'org-protected line)
		  (setq start (match-end 2))
		(let ((n (match-string 2 line)) extra a)
		  (if (setq a (assoc n footref-seen))
		      (progn
			(setcdr a (1+ (cdr a)))
			(setq extra (format ".%d" (cdr a))))
		    (setq extra "")
		    (push (cons n 1) footref-seen))
		  (setq line
			(replace-match
			 (format
			  "%s<sup><a class=\"footref\" name=\"fnr.%s%s\" href=\"#fn.%s\">%s</a></sup>"
			  (match-string 1 line) n extra n n)
			 t t line))))))

	  (cond
	   ((string-match "^\\(\\*+\\)[ \t]+\\(.*\\)" line)
	    ;; This is a headline
	    (setq level (org-tr-level (- (match-end 1) (match-beginning 1)))
		  txt (match-string 2 line))
	    (if (string-match quote-re0 txt)
		(setq txt (replace-match "" t t txt)))
	    (if (<= level (max umax umax-toc))
		(setq head-count (+ head-count 1)))
	    (when in-local-list
	      ;; Close any local lists before inserting a new header line
	      (while local-list-type
		(org-close-li (car local-list-type))
		(insert (format "</%sl>\n" (car local-list-type)))
		(pop local-list-type))
	      (setq local-list-indent nil
		    in-local-list nil))
	    (setq first-heading-pos (or first-heading-pos (point)))
	    (org-html-level-start level txt umax
				  (and org-export-with-toc (<= level umax))
				  head-count)
	    ;; QUOTES
	    (when (string-match quote-re line)
	      (org-close-par-maybe)
	      (insert "<pre>")
	      (setq inquote t)))

	   ((and org-export-with-tables
		 (string-match "^\\([ \t]*\\)\\(|\\|\\+-+\\+\\)" line))
	    (if (not table-open)
		;; New table starts
		(setq table-open t table-buffer nil table-orig-buffer nil))
	    ;; Accumulate lines
	    (setq table-buffer (cons line table-buffer)
		  table-orig-buffer (cons origline table-orig-buffer))
	    (when (or (not lines)
		      (not (string-match "^\\([ \t]*\\)\\(|\\|\\+-+\\+\\)"
					 (car lines))))
	      (setq table-open nil
		    table-buffer (nreverse table-buffer)
		    table-orig-buffer (nreverse table-orig-buffer))
	      (org-close-par-maybe)
	      (insert (org-format-table-html table-buffer table-orig-buffer))))
	   (t
	    ;; Normal lines
	    (when (string-match
		   (cond
		    ((eq llt t) "^\\([ \t]*\\)\\(\\([-+*] \\)\\|\\([0-9]+[.)]\\) \\)?\\( *[^ \t\n\r]\\|[ \t]*$\\)")
		    ((= llt ?.) "^\\([ \t]*\\)\\(\\([-+*] \\)\\|\\([0-9]+\\.\\) \\)?\\( *[^ \t\n\r]\\|[ \t]*$\\)")
		    ((= llt ?\)) "^\\([ \t]*\\)\\(\\([-+*] \\)\\|\\([0-9]+)\\) \\)?\\( *[^ \t\n\r]\\|[ \t]*$\\)")
		    (t (error "Invalid value of `org-plain-list-ordered-item-terminator'")))
		   line)
	      (setq ind (org-get-string-indentation line)
		    item-type (if (match-beginning 4) "o" "u")
		    starter (if (match-beginning 2)
				(substring (match-string 2 line) 0 -1))
		    line (substring line (match-beginning 5))
		    item-tag nil)
	      (if (and starter (string-match "\\(.*?\\) ::[ \t]*" line))
		  (setq item-type "d"
			item-tag (match-string 1 line)
			line (substring line (match-end 0))))
	      (when (and (not (equal item-type "d"))
			 (not (string-match "[^ \t]" line)))
		;; empty line.  Pretend indentation is large.
		(setq ind (if org-empty-line-terminates-plain-lists
			      0
			    (1+ (or (car local-list-indent) 1)))))
	      (setq didclose nil)
	      (while (and in-local-list
			  (or (and (= ind (car local-list-indent))
				   (not starter))
			      (< ind (car local-list-indent))))
		(setq didclose t)
		(org-close-li (car local-list-type))
		(insert (format "</%sl>\n" (car local-list-type)))
		(pop local-list-type) (pop local-list-indent)
		(setq in-local-list local-list-indent))
	      (cond
	       ((and starter
		     (or (not in-local-list)
			 (> ind (car local-list-indent))))
		;; Start new (level of) list
		(org-close-par-maybe)
		(insert (cond
			 ((equal item-type "u") "<ul>\n<li>\n")
			 ((equal item-type "o") "<ol>\n<li>\n")
			 ((equal item-type "d")
			  (format "<dl>\n<dt>%s</dt><dd>\n" item-tag))))
		(push item-type local-list-type)
		(push ind local-list-indent)
		(setq in-local-list t))
	       (starter
		;; continue current list
		(org-close-li (car local-list-type))
		(insert (cond
			 ((equal (car local-list-type) "d")
			  (format "<dt>%s</dt><dd>\n" (or item-tag "???")))
			 (t "<li>\n"))))
	       (didclose
		;; we did close a list, normal text follows: need <p>
		(org-open-par)))
	      (if (string-match "^[ \t]*\\[\\([X ]\\)\\]" line)
		  (setq line
			(replace-match
			 (if (equal (match-string 1 line) "X")
			     "<b>[X]</b>"
			   "<b>[<span style=\"visibility:hidden;\">X</span>]</b>")
			   t t line))))

	    ;; Empty lines start a new paragraph.  If hand-formatted lists
	    ;; are not fully interpreted, lines starting with "-", "+", "*"
	    ;; also start a new paragraph.
	    (if (string-match "^ [-+*]-\\|^[ \t]*$" line) (org-open-par))

	    ;; Is this the start of a footnote?
	    (when org-export-with-footnotes
	      (when (and (boundp 'footnote-section-tag-regexp)
			 (string-match (concat "^" footnote-section-tag-regexp)
				       line))
		;; ignore this line
		(throw 'nextline nil))
	      (when (string-match "^[ \t]*\\[\\([0-9]+\\)\\]" line)
		(org-close-par-maybe)
		(let ((n (match-string 1 line)))
		  (setq org-par-open t
			line (replace-match
			      (format "<p class=\"footnote\"><sup><a class=\"footnum\" name=\"fn.%s\" href=\"#fnr.%s\">%s</a></sup>" n n n) t t line)))))

	    ;; Check if the line break needs to be conserved
	    (cond
	     ((string-match "\\\\\\\\[ \t]*$" line)
	      (setq line (replace-match "<br/>" t t line)))
	     (org-export-preserve-breaks
	      (setq line (concat line "<br/>"))))

	    ;; Check if a paragraph should be started
	    (let ((start 0))
	      (while (and org-par-open
			  (string-match "\\\\par\\>" line start))
		;; Leave a space in the </p> so that the footnote matcher
		;; does not see this.
		(if (not (get-text-property (match-beginning 0)
					    'org-protected line))
		    (setq line (replace-match "</p ><p >" t t line)))
		(setq start (match-end 0))))

	    (insert line "\n")))))

      ;; Properly close all local lists and other lists
      (when inquote
	(insert "</pre>\n")
	(org-open-par))
      (when in-local-list
	;; Close any local lists before inserting a new header line
	(while local-list-type
	  (org-close-li (car local-list-type))
	  (insert (format "</%sl>\n" (car local-list-type)))
	  (pop local-list-type))
	(setq local-list-indent nil
	      in-local-list nil))
      (org-html-level-start 1 nil umax
			    (and org-export-with-toc (<= level umax))
			    head-count)
      ;; the </div> to close the last text-... div.
      (when (and (> umax 0) first-heading-pos) (insert "</div>\n"))

      (save-excursion
	(goto-char (point-min))
	(while (re-search-forward "<p class=\"footnote\">[^\000]*?\\(</p>\\|\\'\\)" nil t)
	  (push (match-string 0) footnotes)
	  (replace-match "" t t)))
      (when footnotes
	(insert (format org-export-html-footnotes-section
			(or (nth 4 lang-words) "Footnotes")
			(mapconcat 'identity (nreverse footnotes) "\n"))
		"\n"))
      (unless body-only
	(when (plist-get opt-plist :auto-postamble)
	  (insert "<div id=\"postamble\">")
	  (when (and org-export-author-info author)
	    (insert "<p class=\"author\"> "
		    (nth 1 lang-words) ": " author "\n")
	    (when email
	      (if (listp (split-string email ",+ *"))
		  (mapc (lambda(e)
			  (insert "<a href=\"mailto:" e "\">&lt;"
				  e "&gt;</a>\n"))
			(split-string email ",+ *"))
		(insert "<a href=\"mailto:" email "\">&lt;"
			email "&gt;</a>\n")))
	    (insert "</p>\n"))
	  (when (and date org-export-time-stamp-file)
	    (insert "<p class=\"date\"> "
		    (nth 2 lang-words) ": "
		    date "</p>\n"))
	  (when org-export-creator-info
	    (insert (format "<p>HTML generated by org-mode %s in emacs %s</p>\n"
			    org-version emacs-major-version)))
	  (insert "</div>"))

	(if org-export-html-with-timestamp
	    (insert org-export-html-html-helper-timestamp))
	(insert (or (plist-get opt-plist :postamble) ""))
	(insert "</body>\n</html>\n"))

      (unless (plist-get opt-plist :buffer-will-be-killed)
	(normal-mode)
	(if (eq major-mode default-major-mode) (html-mode)))

      ;; insert the table of contents
      (goto-char (point-min))
      (when thetoc
	(if (or (re-search-forward
		 "<p>\\s-*\\[TABLE-OF-CONTENTS\\]\\s-*</p>" nil t)
		(re-search-forward
		 "\\[TABLE-OF-CONTENTS\\]" nil t))
	    (progn
	      (goto-char (match-beginning 0))
	      (replace-match ""))
	  (goto-char first-heading-pos)
	  (when (looking-at "\\s-*</p>")
	    (goto-char (match-end 0))
	    (insert "\n")))
	(insert "<div id=\"table-of-contents\">\n")
	(mapc 'insert thetoc)
	(insert "</div>\n"))
      ;; remove empty paragraphs and lists
      (goto-char (point-min))
      (while (re-search-forward "<p>[ \r\n\t]*</p>" nil t)
	(replace-match ""))
      (goto-char (point-min))
      (while (re-search-forward "<li>[ \r\n\t]*</li>\n?" nil t)
	(replace-match ""))
      (goto-char (point-min))
      (while (re-search-forward "</ul>\\s-*<ul>\n?" nil t)
	(replace-match ""))
      ;; Convert whitespace place holders
      (goto-char (point-min))
      (let (beg end n)
	(while (setq beg (next-single-property-change (point) 'org-whitespace))
	  (setq n (get-text-property beg 'org-whitespace)
		end (next-single-property-change beg 'org-whitespace))
	  (goto-char beg)
	  (delete-region beg end)
	  (insert (format "<span style=\"visibility:hidden;\">%s</span>"
			  (make-string n ?x)))))
      (or to-buffer (save-buffer))
      (goto-char (point-min))
      (message "Exporting... done")
      (if (eq to-buffer 'string)
	  (prog1 (buffer-substring (point-min) (point-max))
	    (kill-buffer (current-buffer)))
	(current-buffer)))))

(defvar rdf-tripple nil)

(defun gnowsys-n3-export ()
  "Exports the context node in the RDF format."

  (interactive)
  (setq node-nid nil)
  (gnow-parse)
  (setq rdf-tripple (xml-rpc-method-call url 'n3_nbhexport context-ssid "gnowsys"))
  (goto-char context-end-pos)
  (if (get-buffer-window "*RDF*")
      (delete-window (get-buffer-window "*RDF*")))
  (get-buffer-create "*RDF*")
  (pop-to-buffer "*RDF*")
  (setq buffer-read-only nil)
  (n3-mode)
  (erase-buffer)
  (insert rdf-tripple)
  (goto-char (point-min))
  (setq buffer-read-only t)
  )

(defvar snapshots nil)  
(defvar temp-count nil)
(defvar snapshot-id nil)

(defvar version nil)

(defun gnowsys-getversions (&optional ssid)
  "This command is to get all the snapshots of 
   the context node. Prompts to select from list
   snapshots and replaces the current context with
   that particular version of the same."

  (interactive)
  (setq temp-count 1)
  (gnow-parse)
  (setq version nil)
  (setq snapshots (xml-rpc-method-call url 'getAllSnapshots "nid" node-nid))
  (outline-back-to-heading)
  (beginning-of-line)

  (setq link-text (buffer-substring-no-properties (point) context-start-pos))
  (goto-char (+ (point) (+ (string-match "\]\]" link-text) 2)))
  (if (not (equal (point) (line-end-position)))
	(setq version (buffer-substring-no-properties (re-search-forward "\\[") (- (re-search-forward "\/") 1)))
      )

  (setq temp-list nil)
  (if (> (length snapshots) 1)
      (progn
	(while (< temp-count (length snapshots))
	  (push (number-to-string temp-count) temp-list)
	  (setq temp-count (+ temp-count 1))
	  )
	(if version
	    (progn
	      (setq temp-list (remove version temp-list))
	      (push (number-to-string temp-count) temp-list)
	      ))
	(setq snapshot-id (completing-read "Snapshot: " temp-list))
	
	(beginning-of-line)
	(delete-region (point) context-end-pos)
	(setlink-type gnow-node)
	(org-insert-heading)
	(insert node-symbol " ")
	(insert "[[" link-type ":" (elt snapshots (- (string-to-number snapshot-id) 1)) "][" node-nid "]]\t[" snapshot-id "/"  (number-to-string (length snapshots)) "]\n")
	(disp-expand (elt snapshots (- (string-to-number snapshot-id) 1)) node-nid)
	(insert "\n")
	(org-cycle-hide-drawers 'all)
	)
    (message "Current node has a single snapshot."))
  )

  
(defun org-cycle (&optional arg)
  "Visibility cycling for Org-mode.

- When this function is called with a prefix argument, rotate the entire
  buffer through 3 states (global cycling)
  1. OVERVIEW: Show only top-level headlines.
  2. CONTENTS: Show all headlines of all levels, but no body text.
  3. SHOW ALL: Show everything.
  When called with two C-u C-u prefixes, switch to the startup visibility,
  determined by the variable `org-startup-folded', and by any VISIBILITY
  properties in the buffer.
  When called with three C-u C-u C-u prefixed, show the entire buffer,
  including drawers.

- When point is at the beginning of a headline, rotate the subtree started
  by this line through 3 different states (local cycling)
  1. FOLDED:   Only the main headline is shown.
  2. CHILDREN: The main headline and the direct children are shown.
               From this state, you can move to one of the children
               and zoom in further.
  3. SUBTREE:  Show the entire subtree, including body text.

- When there is a numeric prefix, go up to a heading with level ARG, do
  a `show-subtree' and return to the previous cursor position.  If ARG
  is negative, go up that many levels.

- When point is not at the beginning of a headline, execute the global
  binding for TAB, which is re-indenting the line.  See the option
  `org-cycle-emulate-tab' for details.

- Special case: if point is at the beginning of the buffer and there is
  no headline in line 1, this function will act as if called with prefix arg.
  But only if also the variable `org-cycle-global-at-bob' is t."
  (interactive "P")
  (org-load-modules-maybe)

  (if (org-on-heading-p)
      (progn
	(setq proc-loc (point))
	(gnow-parse)
	(setq link-text (buffer-substring-no-properties context-start-pos context-end-pos))
	(if (< (length link-text) 10)
	    (progn
	      (end-of-line)
	      (disp-expand context-ssid node-nid)
	      (insert "\n")
	      )
	  (goto-char proc-loc)
	  )
	(org-cycle-hide-drawers 'all)
	))

  (let* ((outline-regexp
	  (if (and (org-mode-p) org-cycle-include-plain-lists)
	      "\\(?:\\*+ \\|\\([ \t]*\\)\\([-+*]\\|[0-9]+[.)]\\) \\)"
	    outline-regexp))
	 (bob-special (and org-cycle-global-at-bob (bobp)
			   (not (looking-at outline-regexp))))
	 (org-cycle-hook
	  (if bob-special
	      (delq 'org-optimize-window-after-visibility-change
		    (copy-sequence org-cycle-hook))
	    org-cycle-hook))
	 (pos (point)))

    (if (or bob-special (equal arg '(4)))
	;; special case:  use global cycling
	(setq arg t))

    (cond

     ((equal arg '(16))
      (org-set-startup-visibility)
      (message "Startup visibility, plus VISIBILITY properties"))

     ((equal arg '(64))
      (show-all)
      (message "Entire buffer visible, including drawers"))

     ((org-at-table-p 'any)
      ;; Enter the table or move to the next field in the table
      (or (org-table-recognize-table.el)
	  (progn
	    (if arg (org-table-edit-field t)
	      (org-table-justify-field-maybe)
	      (call-interactively 'org-table-next-field)))))

     ((eq arg t) ;; Global cycling

      (cond
       ((and (eq last-command this-command)
	     (eq org-cycle-global-status 'overview))
	;; We just created the overview - now do table of contents
	;; This can be slow in very large buffers, so indicate action
	(message "CONTENTS...")
	(org-content)
	(message "CONTENTS...done")
	(setq org-cycle-global-status 'contents)
	(run-hook-with-args 'org-cycle-hook 'contents))

       ((and (eq last-command this-command)
	     (eq org-cycle-global-status 'contents))
	;; We just showed the table of contents - now show everything
	(show-all)
	(message "SHOW ALL")
	(setq org-cycle-global-status 'all)
	(run-hook-with-args 'org-cycle-hook 'all))

       (t
	;; Default action: go to overview
	(org-overview)
	(message "OVERVIEW")
	(setq org-cycle-global-status 'overview)
	(run-hook-with-args 'org-cycle-hook 'overview))))

     ((and org-drawers org-drawer-regexp
	   (save-excursion
	     (beginning-of-line 1)
	     (looking-at org-drawer-regexp)))
      ;; Toggle block visibility
      (org-flag-drawer
       (not (get-char-property (match-end 0) 'invisible))))

     ((integerp arg)
      ;; Show-subtree, ARG levels up from here.
      (save-excursion
	(org-back-to-heading)
	(outline-up-heading (if (< arg 0) (- arg)
			      (- (funcall outline-level) arg)))
	(org-show-subtree)))

     ((and (save-excursion (beginning-of-line 1) (looking-at outline-regexp))
	   (or (bolp) (not (eq org-cycle-emulate-tab 'exc-hl-bol))))
      ;; At a heading: rotate between three different views
      (org-back-to-heading)
      (let ((goal-column 0) eoh eol eos)
	;; First, some boundaries
	(save-excursion
	  (org-back-to-heading)
	  (save-excursion
	    (beginning-of-line 2)
	    (while (and (not (eobp)) ;; this is like `next-line'
			(get-char-property (1- (point)) 'invisible))
	      (beginning-of-line 2)) (setq eol (point)))
	  (outline-end-of-heading)   (setq eoh (point))
	  (org-end-of-subtree t)
	  (unless (eobp)
	    (skip-chars-forward " \t\n")
	    (beginning-of-line 1) ; in case this is an item
	    )
	  (setq eos (1- (point))))
	;; Find out what to do next and set `this-command'
	(cond
	 ((= eos eoh)
	  ;; Nothing is hidden behind this heading
	  (message "EMPTY ENTRY")
	  (setq org-cycle-subtree-status nil)
	  (save-excursion
	    (goto-char eos)
	    (outline-next-heading)
	    (if (org-invisible-p) (org-flag-heading nil))))
	 ((or (>= eol eos)
	      (not (string-match "\\S-" (buffer-substring eol eos))))
	  ;; Entire subtree is hidden in one line: open it
	  (org-show-entry)
	  (show-children)
	  (message "CHILDREN")
	  (save-excursion
	    (goto-char eos)
	    (outline-next-heading)
	    (if (org-invisible-p) (org-flag-heading nil)))
	  (setq org-cycle-subtree-status 'children)
	  (run-hook-with-args 'org-cycle-hook 'children))
	 ((and (eq last-command this-command)
	       (eq org-cycle-subtree-status 'children))
	  ;; We just showed the children, now show everything.
	  (org-show-subtree)
	  (message "SUBTREE")
	  (setq org-cycle-subtree-status 'subtree)
	  (run-hook-with-args 'org-cycle-hook 'subtree))
	 (t
	  ;; Default action: hide the subtree.
	  (hide-subtree)
	  (message "FOLDED")
	  (setq org-cycle-subtree-status 'folded)
	  (run-hook-with-args 'org-cycle-hook 'folded)))))

     ;; TAB emulation and template completion
     (buffer-read-only (org-back-to-heading))

     ((org-try-structure-completion))

     ((org-try-cdlatex-tab))

     ((and (eq org-cycle-emulate-tab 'exc-hl-bol)
	   (or (not (bolp))
	       (not (looking-at outline-regexp))))
      (call-interactively (global-key-binding "\t")))

     ((if (and (memq org-cycle-emulate-tab '(white whitestart))
	       (save-excursion (beginning-of-line 1) (looking-at "[ \t]*"))
	       (or (and (eq org-cycle-emulate-tab 'white)
			(= (match-end 0) (point-at-eol)))
		   (and (eq org-cycle-emulate-tab 'whitestart)
			(>= (match-end 0) pos))))
	  t
	(eq org-cycle-emulate-tab t))
      (call-interactively (global-key-binding "\t")))

     (t (save-excursion
	  (org-back-to-heading)
	  (org-cycle))))))

(defun org-cycle-hide-drawers (state)
  "Re-hide all drawers after a visibility state change."
  (when (not (memq state '(overview folded)))
    (save-excursion
      (let* ((globalp (memq state '(contents all)))
             (beg (if globalp (point-min) (point)))
             (end (if globalp (point-max) (org-end-of-subtree t))))
	(goto-char beg)
	(while (re-search-forward org-drawer-regexp end t)
	  (org-flag-drawer t))))))


(defun gnowsys-get-buffer ()
  "To retieve the current context buffer structure."

  (interactive)
  (gnow-parse)
  (setq buffer-structure nil)

  (if (get-buffer-window "*structure*")
      (delete-window (get-buffer-window "*structure*")))
  (get-buffer-create "*structure*")
  (pop-to-buffer "*structure*")
  (setq buffer-read-only nil)
  (erase-buffer)
  (insert (prin1-to-string (xml-rpc-method-call url 'getAllnbh (list context-ssid))))
  (setq buffer-read-only t)
  (goto-char (point-min))
  )

  
(provide 'gnowsys)